/*
 * Copyright (c) 1995 Akihiro Tominaga and the WIDE Project.
 * All rights reserved.
 *
 * Permission to use, copy and distribute this software in source and
 * binary forms is hereby granted provided that the above copyright
 * notice and this permission notice appear in all copies of the software,
 * derivative works or modified versions, and any portions thereof.
 * Furthermore, any supporting documentation, advertising materials,
 * and all other materials related to such a distribution and use must
 * acknowledge that the software was developed by Akihiro Tominaga and
 * the WIDE Project. The names of the author and the WIDE Project may
 * not be used to endorse or promote products derived from this software
 * without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <syslog.h>
#include <fcntl.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <net/route.h>
#ifdef sun
#include <sys/stropts.h>
#include <net/nit.h>
#include <net/nit_if.h>
#include <net/nit_pf.h>
#include <net/packetfilt.h>
#else
#include <net/bpf.h>
#endif
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/if_ether.h>
#include <sys/param.h>
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <arpa/inet.h>

#include "dhcp.h"
#include "dhcpc.h"
#include "common.h"
#include "dhcpc_subr.h"

#ifndef sun
#ifdef sony_news
struct bpf_insn arpf[] = {
  BPF_STMT(LdHOp, 12),                     /* A <- ETHER_TYPE */
  BPF_JUMP(EQOp, ETHERTYPE_ARP, 1, 6),     /* is it ARP ? */
  BPF_STMT(LdHOp, 20),                     /* A <- ARP_OP */
  BPF_JUMP(EQOp, ARPOP_REPLY, 1, 4),       /* is it ARP reply ? */
  BPF_STMT(LdOp, 28),                      /* A <- target IP */
  BPF_JUMP(EQOp, 0, 1, 2),                 /* is it target ? */
  BPF_STMT(RetOp, (u_int)-1),
  BPF_STMT(RetOp, 0)
};
#else /* if sony_news */
struct bpf_insn arpf[] = {
  BPF_STMT(BPF_LD+BPF_H+BPF_ABS, 12),                     /* A <- ETHER_TYPE */
  BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, ETHERTYPE_ARP, 0, 5),   /* is it ARP ? */
  BPF_STMT(BPF_LD+BPF_H+BPF_ABS, 20),                     /* A <- ARP_OP */
  BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, ARPOP_REPLY, 0, 3),     /* is it ARP reply ? */
  BPF_STMT(BPF_LD+BPF_W+BPF_ABS, 28),                     /* A <- target IP */
  BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, 0, 0, 1),               /* is it target ? */
  BPF_STMT(BPF_RET+BPF_K, (u_int)-1),
  BPF_STMT(BPF_RET+BPF_K, 0)
};
#endif /* sony_news */
#define TARGETIP  5
    
struct bpf_program arpfilter = {
  sizeof(arpf) / sizeof(struct bpf_insn),
  arpf
};
#endif

int timeout;
int outputfd;
int (*handle_param[MAXTAGNUM + 1])() = {
  NULL,            /* PAD */
  handle_ip,      /* SUBNET_MASK */
  handle_num,     /* TIME_OFFSET */
  handle_ips,     /* ROUTER */
  handle_ips,     /* TIME_SERVER */
  handle_ips,     /* NAME_SERVER */
  handle_ips,     /* DNS_SERVER */
  handle_ips,     /* LOG_SERVER */
  handle_ips,     /* COOKIE_SERVER */
  handle_ips,     /* LPR_SERVER */
  handle_ips,     /* IMPRESS_SERVER */
  handle_ips,     /* RLS_SERVER */
  handle_str,     /* HOSTNAME */
  handle_num,     /* BOOTSIZE */
  handle_str,     /* MERIT_DUMP */
  handle_str,     /* DNS_DOMAIN */
  handle_ip,      /* SWAP_SERVER */
  handle_str,     /* ROOT_PATH */
  handle_str,     /* EXTENSIONS_PATH */
  handle_bool,    /* IP_FORWARD */
  handle_bool,    /* NONLOCAL_SRCROUTE */
  handle_ippairs, /* POLICY_FILTER */
  handle_num,     /* MAX_DGRAM_SIZE */
  handle_num,     /* DEFAULT_IP_TTL */
  handle_num,     /* MTU_AGING_TIMEOUT */
  handle_nums,    /* MTU_PLATEAU_TABLE */
  handle_num,     /* IF_MTU */
  handle_bool,    /* ALL_SUBNET_LOCAL */
  handle_ip,      /* BRDCAST_ADDR */
  handle_bool,    /* MASK_DISCOVER */
  handle_bool,    /* MASK_SUPPLIER */
  handle_bool,    /* ROUTER_DISCOVER */
  handle_ip,      /* ROUTER_SOLICIT */
  handle_ippairs, /* STATIC_ROUTE */
  handle_bool,    /* TRAILER */
  handle_num,     /* ARP_CACHE_TIMEOUT */
  handle_bool,    /* ETHER_ENCAP */
  handle_num,     /* DEFAULT_TCP_TTL */
  handle_num,     /* KEEPALIVE_INTER */
  handle_bool,    /* KEEPALIVE_GARBA */
  handle_str,     /* NIS_DOMAIN */
  handle_ips,     /* NIS_SERVER */
  handle_ips,     /* NTP_SERVER */
  NULL,            /* VENDOR_SPEC */      /* XXX not implemented */
  handle_ips,     /* NBN_SERVER */
  handle_ips,     /* NBDD_SERVER */
  handle_num,     /* NB_NODETYPE */
  handle_str,     /* NB_SCOPE */
  handle_ips,     /* XFONT_SERVER */
  handle_ips,     /* XDISPLAY_MANAGER */
  NULL,            /* REQUEST_IPADDR */
  handle_num,     /* LEASE_TIME */
  NULL,            /* OPT_OVERLOAD */
  NULL,            /* DHCP_MSGTYPE */
  handle_ip,      /* SERVER_ID */
  NULL,            /* REQ_LIST */
  handle_str,     /* DHCP_ERRMSG */
  NULL,            /* DHCP_MAXMSGSIZE */
  handle_num,     /* DHCP_T1 */
  handle_num,     /* DHCP_T2  */
  NULL,            /* CLASS_ID */
  NULL             /* CLIENT_ID */
};


/*
 * generate transaction identifier
 */
long
generate_xid(intf)
  struct dhcp_if *intf;
{
  time_t current = 0;
  u_short tmp[6];
  u_short result2 = 0;
  static u_short result1 = 0;

  if (intf != NULL) {
    bcopy(intf->haddr.haddr, tmp, 6);
    result1 = cksum(tmp, 6);
    result1 = (result1 << 16);
  }

  time(&current);
  result2 = cksum((u_short *) &current, 2);

  return(result1 + result2);
}


/*
 * general timeout handler
 */
void
gen_timeout()
{
  timeout = 1;
  return;
}


/*
 * general retransmission routine
 */
void
gen_retransmit()
{
  time_t curr_epoch;

  timeout = 1;
  if (time(&curr_epoch) == -1)
    return;

  sndmsg.dhcp->secs = htons(curr_epoch - init_epoch);
  sndmsg.udp->uh_sum = 0;
  sndmsg.udp->uh_sum = udp_cksum(&sndmsg.pudph, (char *) sndmsg.udp,
				 ntohs(sndmsg.pudph.ulen));

  ether_write(outputfd, &sndmsg.buf[QWOFF], sndmsg.bufsize);
  return;
}


/*
 * retransmission routine for requesting
 */
void
req_retransmit()
{
  timeout = 1;

  sndmsg.udp->uh_sum = 0;
  sndmsg.udp->uh_sum = udp_cksum(&sndmsg.pudph, (char *) sndmsg.udp,
				 ntohs(sndmsg.pudph.ulen));

  ether_write(outputfd, &sndmsg.buf[QWOFF], sndmsg.bufsize);
  return;
}


/*
 * broadcast ARP_REQUEST and check reply
 *
 *   return -1, if there is the invalid client
 */
int
arp_check(target, intf)
  struct in_addr *target;
  struct dhcp_if *intf;
{
  int i = 0;
  long tmp = 0;
  char lsbuf[ETHERHL + sizeof(struct ether_arp)];
  struct ether_header *lsether = NULL;
  struct ether_arp *lsarp = NULL;
#ifdef sun
  struct timeval timeout;
  fd_set readfds;
  struct strioctl si;
  struct packetfilt arpf;
  register u_short *fwp = &arpf.Pf_Filter[0];
  u_short targetIP[2];

  bcopy(&target->s_addr, targetIP, sizeof(struct in_addr));
  bzero(&timeout, sizeof(timeout));
#endif
  bzero(lsbuf, ETHERHL + sizeof(struct ether_arp));
  lsether = (struct ether_header *) lsbuf;
  lsarp = (struct ether_arp *) &lsbuf[ETHERHL];

#ifdef sun
  *fwp++ = ENF_PUSHWORD + 6;
  *fwp++ = ENF_PUSHLIT | ENF_CAND;
  *fwp++ = htons(ETHERTYPE_ARP);
  *fwp++ = ENF_PUSHWORD + 10;
  *fwp++ = ENF_PUSHLIT | ENF_CAND;
  *fwp++ = htons(ARPOP_REPLY);
  *fwp++ = ENF_PUSHWORD + 14;
  *fwp++ = ENF_PUSHLIT | ENF_CAND;
  *fwp++ = targetIP[0];
  *fwp++ = ENF_PUSHWORD + 15;
  *fwp++ = ENF_PUSHLIT | ENF_CAND;
  *fwp++ = targetIP[1];
  arpf.Pf_FilterLen = fwp - &arpf.Pf_Filter[0];

  bzero(&si, sizeof(si));
  si.ic_cmd = NIOCSETF;
  si.ic_timout = INFTIM;
  si.ic_len = sizeof(arpf);
  si.ic_dp = (char *)&arpf;
  if (ioctl(intf->forarp->fd, I_STR, (char *)&si) < 0){
    Syslog(LOG_WARNING, "ioctl(NIOCSETF) error in arp_check: %m");
    return(0);
  }
  if (ioctl(intf->forarp->fd, I_FLUSH, (char *)FLUSHR) < 0){
    Syslog(LOG_WARNING, "ioctl(I_FLUSH) error in arp_check: %m");
    return(0);
  }
#else
  arpf[TARGETIP].k = ntohl(target->s_addr);     /* must be host byte order */
  if (ioctl(intf->forarp->fd, BIOCSETF, &arpfilter) < 0) {
    Syslog(LOG_WARNING, "ioctl(BIOCSETF) for arp: %m");
    return(0);
  }
#endif

  lsarp->arp_hrd = htons(ARPHRD_ETHER);
  lsarp->arp_pro = htons(ETHERTYPE_IP);
  lsarp->arp_hln = 6;
  lsarp->arp_pln = 4;
  lsarp->arp_op = htons(ARPOP_REQUEST);
  for (i = 0; i < 6; i++) {
#ifdef sun
    lsarp->arp_sha.ether_addr_octet[i] =
      lsether->ether_shost.ether_addr_octet[i] = intf->haddr.haddr[i];
    lsarp->arp_tha.ether_addr_octet[i] = 0;
    lsether->ether_dhost.ether_addr_octet[i] = 0xff;
#else
    lsarp->arp_sha[i] = lsether->ether_shost[i] = intf->haddr.haddr[i];
    lsarp->arp_tha[i] = 0;
    lsether->ether_dhost[i] = 0xff;
#endif
  }
  tmp = 0;
  bcopy(&tmp, lsarp->arp_spa, lsarp->arp_pln);
  bcopy(&target->s_addr, lsarp->arp_tpa, lsarp->arp_pln);
  lsether->ether_type = htons(ETHERTYPE_ARP);

  if (ether_write(intf->forarp->fd, lsbuf,
		  ETHERHL + sizeof(struct ether_arp)) < 0) {
    return(0);                               /* it is harmless to return 0 */
  }

#ifdef sun
  /* if timeout, there is no replay and return 0 */
  timeout.tv_sec = 0;
  timeout.tv_usec = 500000;
  FD_ZERO(&readfds);
  FD_SET(intf->forarp->fd, &readfds);
  if (select(intf->forarp->fd + 1, &readfds, NULL, NULL, &timeout) <= 0) {
    return(0);
  } else {
    return(-1);
  }
#else
  if (read(intf->forarp->fd, intf->forarp->rbuf, intf->forarp->rbufsize) <= 0)
    return(0);
  else
    return(-1);
#endif
}


/*
 * broadcast ARP_REPLY
 */
int
arp_announce(ipaddr, intf)
  struct in_addr *ipaddr;
  struct dhcp_if *intf;
{
  int i = 0;
  char lsbuf[ETHERHL + sizeof(struct ether_arp)];
  struct ether_header *lsether = NULL;
  struct ether_arp *lsarp = NULL;

  bzero(lsbuf, ETHERHL + sizeof(struct ether_arp));
  lsether = (struct ether_header *) lsbuf;
  lsarp = (struct ether_arp *) &lsbuf[ETHERHL];

  lsarp->arp_hrd = htons(ARPHRD_ETHER);
  lsarp->arp_pro = htons(ETHERTYPE_IP);
  lsarp->arp_hln = 6;
  lsarp->arp_pln = 4;
  lsarp->arp_op = htons(ARPOP_REPLY);

  for (i = 0; i < 6; i++) {
#ifdef sun
    lsarp->arp_tha.ether_addr_octet[i] = lsarp->arp_sha.ether_addr_octet[i] =
      lsether->ether_shost.ether_addr_octet[i] = intf->haddr.haddr[i];
    lsether->ether_dhost.ether_addr_octet[i] = 0xff;
#else
    lsarp->arp_tha[i] = lsarp->arp_sha[i] =
      lsether->ether_shost[i] = intf->haddr.haddr[i];
    lsether->ether_dhost[i] = 0xff;
#endif
  }

  bcopy(&ipaddr->s_addr, lsarp->arp_spa, lsarp->arp_pln);
  bcopy(&ipaddr->s_addr, lsarp->arp_tpa, lsarp->arp_pln);

  lsether->ether_type = htons(ETHERTYPE_ARP);

  if (ether_write(intf->forarp->fd, lsbuf,
		  ETHERHL + sizeof(struct ether_arp)) < 0) {
    return(-1);
  }

  return(0);
}


/*
 * conver NVT ASCII to strings
 * (actually, only remove null characters)
 */
int
nvttostr(nvt, str, length)
  char *nvt;
  char *str;
  int length;
{
  register int i = 0;
  register char *tmp = NULL;

  tmp = str;

  for (i = 0; i < length; i++) {
    if (nvt[i] != NULL) {
      *tmp = nvt[i];
      tmp++;
    }
  }

  str[length] = '\0';
  return(0);
}


#define dhcp_optmap_set(OPT, MAPP)\
  (MAPP)->bitmap[(OPT)/8] |= (1 << ((OPT) % 8))

/*
 * convert DHCP message to struct dhcp_param
 *
 *
 */
int
dhcp_msgtoparam(msg, msglen, parameter, optmap)
  struct dhcp *msg;
  int msglen;
  struct dhcp_param *parameter;
  struct dhcp_optmap *optmap;
{
  register char *optp = NULL;
  char tag = 0;
  int sname_is_opt = 0,
      file_is_opt = 0;
  int err = 0;

  char *endofopt = &msg->options[msglen - DFLTDHCPLEN + DFLTOPTLEN];
  bzero(optmap, sizeof(struct dhcp_optmap));

  for (optp = &msg->options[MAGIC_LEN]; optp <= endofopt; optp++) {
    tag = *optp;

    /* skip the PAD */
    if (tag == PAD) {
      continue;
    }

    /* reach to the END */
    if (tag == END) {
      break;
    }

    /* handle the "Option Overload" */
    if (tag == OPT_OVERLOAD) {
      optp += 2;
      switch (*optp) {
      case FILE_ISOPT:
	file_is_opt = TRUE;
	break;
      case SNAME_ISOPT:
	sname_is_opt = TRUE;
	break;
      case BOTH_AREOPT:
	file_is_opt = sname_is_opt = TRUE;
	break;
      default:
	break;
      }
      continue;
    }

    if (handle_param[(int) tag] != NULL) {
      if ((err = (*handle_param[(int) tag])(optp, parameter)) != 0) {
	return(err);
      } else {
	dhcp_optmap_set(tag, optmap);
      }
    }

    optp++;
    optp += *optp;
  }

  if (file_is_opt) {
    char *endofopt = &msg->file[MAX_FILE];
    for (optp = msg->file; optp <= endofopt; optp++) {

      tag = *optp;

      /* skip the PAD */
      if (tag == PAD) {
	continue;
      }

      /* reach to the END */
      if (tag == END) {
	break;
      }

      if (handle_param[(int) tag] != NULL) {
	if ((err = (*handle_param[(int) tag])(optp, parameter)) != 0) {
	  return(err);
	} else {
	  dhcp_optmap_set(tag, optmap);
	}
      }

      optp++;
      optp += *optp;
    }
  } else {
    if ((parameter->file = calloc(1, strlen(msg->file))) == NULL) {
      return(-1);
    }
    strcpy(parameter->file, msg->file);
  }


  if (sname_is_opt) {
    char *endofopt = &msg->sname[MAX_SNAME];
    for (optp = msg->file; optp <= endofopt; optp++) {

      tag = *optp;

      /* skip the PAD */
      if (tag == PAD) {
	continue;
      }

      /* reach to the END */
      if (tag == END) {
	break;
      }

      if (handle_param[(int) tag] != NULL) {
	if ((err = (*handle_param[(int) tag])(optp, parameter)) != 0) {
	  return(err);
	} else {
	  dhcp_optmap_set(tag, optmap);
	}
      }

      optp++;
      optp += *optp;
    }
  } else {
    if ((parameter->sname = calloc(1, strlen(msg->sname))) == NULL) {
      return(-1);
    }
    strcpy(parameter->sname, msg->sname);
  }

  parameter->ciaddr.s_addr = msg->ciaddr.s_addr;
  parameter->yiaddr.s_addr = msg->yiaddr.s_addr;
  parameter->siaddr.s_addr = msg->siaddr.s_addr;
  parameter->giaddr.s_addr = msg->giaddr.s_addr;

  if (parameter->dhcp_t1 == 0) {
    parameter->dhcp_t1 = parameter->lease_duration / 2;
    dhcp_optmap_set(DHCP_T1, optmap);
  }
  if (parameter->dhcp_t2 == 0) {
    double tmp = (double) parameter->lease_duration * 0.875;
    parameter->dhcp_t2 = (unsigned short) tmp;
    dhcp_optmap_set(DHCP_T2, optmap);
  }

  return(0);
}


/*
 * free memory which is used in struct dhcp_param
 */
int
clean_param(param)
  struct dhcp_param *param;
{
  if (param == NULL)
    return(0);

  if (param->sname != NULL) free(param->sname);
  if (param->file != NULL) free(param->file);
  if (param->hostname != NULL) free(param->hostname);
  if (param->merit_dump != NULL) free(param->merit_dump);
  if (param->dns_domain != NULL) free(param->dns_domain);
  if (param->root_path != NULL) free(param->root_path);
  if (param->extensions_path != NULL) free(param->extensions_path);
  if (param->nis_domain != NULL) free(param->nis_domain);
  if (param->nb_scope != NULL) free(param->nb_scope);
  if (param->errmsg != NULL) free(param->errmsg);

  if (param->mtu_plateau_table != NULL) {
    if (param->mtu_plateau_table->value != NULL)
      free(param->mtu_plateau_table->value);
    free(param->mtu_plateau_table);
  }
  if (param->server_id != NULL) free(param->server_id);
  if (param->subnet_mask != NULL) free(param->subnet_mask);
  if (param->swap_server != NULL) free(param->swap_server);
  if (param->brdcast_addr != NULL) free(param->brdcast_addr);
  if (param->router_solicit != NULL) free(param->router_solicit);
  if (param->router != NULL) {
    if (param->router->addr != NULL)
      free(param->router->addr);
    free(param->router);
  }
  if (param->time_server != NULL) {
    if (param->time_server->addr != NULL)
      free(param->time_server->addr);
    free(param->time_server);
  }
  if (param->name_server != NULL) {
    if (param->name_server->addr != NULL)
      free(param->name_server->addr);
    free(param->name_server);
  }
  if (param->dns_server != NULL) {
    if (param->dns_server->addr != NULL)
      free(param->dns_server->addr);
    free(param->dns_server);
  }
  if (param->log_server != NULL) {
    if (param->log_server->addr != NULL)
      free(param->log_server->addr);
    free(param->log_server);
  }
  if (param->cookie_server != NULL) {
    if (param->cookie_server->addr != NULL)
      free(param->cookie_server->addr);
    free(param->cookie_server);
  }
  if (param->lpr_server != NULL) {
    if (param->lpr_server->addr != NULL)
      free(param->lpr_server->addr);
    free(param->lpr_server);
  }
  if (param->impress_server != NULL) {
    if (param->impress_server->addr != NULL)
      free(param->impress_server->addr);
    free(param->impress_server);
  }
  if (param->rls_server != NULL) {
    if (param->rls_server->addr != NULL)
      free(param->rls_server->addr);
    free(param->rls_server);
  }
  if (param->policy_filter != NULL) {
    if (param->policy_filter->addr != NULL)
      free(param->policy_filter->addr);
    free(param->policy_filter);
  }
  if (param->static_route != NULL) {
    if (param->static_route->addr != NULL)
      free(param->static_route->addr);
    free(param->static_route);
  }
  if (param->nis_server != NULL) {
    if (param->nis_server->addr != NULL)
      free(param->nis_server->addr);
    free(param->nis_server);
  }
  if (param->ntp_server != NULL) {
    if (param->ntp_server->addr != NULL)
      free(param->ntp_server->addr);
    free(param->ntp_server);
  }
  if (param->nbn_server != NULL) {
    if (param->nbn_server->addr != NULL)
      free(param->nbn_server->addr);
    free(param->nbn_server);
  }
  if (param->nbdd_server != NULL) {
    if (param->nbdd_server->addr != NULL)
      free(param->nbdd_server->addr);
    free(param->nbdd_server);
  }
  if (param->xfont_server != NULL) {
    if (param->xfont_server->addr != NULL)
      free(param->xfont_server->addr);
    free(param->xfont_server);
  }
  if (param->xdisplay_manager != NULL) {
    if (param->xdisplay_manager->addr != NULL)
      free(param->xdisplay_manager->addr);
    free(param->xdisplay_manager);
  }

  bzero(param, sizeof(struct dhcp_param));
  return(0);
}


void
make_decline(intf, reqspec, lsbuf)
  struct dhcp_if *intf;
  struct dhcp_reqspec *reqspec;
  char *lsbuf;
{
  int offopt = 0;    /* offset in options field */
  int i = 0;
  u_long tmpul = 0;
  struct dhcp    *lsdhcp = NULL;
  struct udphdr  *lsudp = NULL;
  struct ip      *lsip = NULL;
  struct ether_header *lsether = NULL;
  struct ps_udph pudph;

  bzero(&pudph, sizeof(pudph));

  lsether = (struct ether_header *) &lsbuf[QWOFF];
  lsip = (struct ip *) &lsbuf[QWOFF + ETHERHL];
  lsudp = (struct udphdr *) &lsbuf[QWOFF + ETHERHL + IPHL];
  lsdhcp = (struct dhcp *) &lsbuf[QWOFF + ETHERHL + IPHL + UDPHL];

  /*
   * construct dhcp part
   */
  lsdhcp->op = BOOTREQUEST;
  lsdhcp->htype = intf->haddr.htype;
  lsdhcp->hlen = intf->haddr.hlen;
  lsdhcp->xid = generate_xid(NULL);
  bcopy(intf->haddr.haddr, lsdhcp->chaddr, lsdhcp->hlen);

  /* insert magic cookie */
  bcopy(magic_c, lsdhcp->options, MAGIC_LEN);
  offopt = MAGIC_LEN;

  /* insert message type */
  lsdhcp->options[offopt++] = DHCP_MSGTYPE;
  lsdhcp->options[offopt++] = 1;
  lsdhcp->options[offopt++] = DHCPDECLINE;

  /* insert server identifier */
  if (CHKOFF(4)) {
    lsdhcp->options[offopt++] = SERVER_ID;
    lsdhcp->options[offopt++] = 4;
    bcopy(&reqspec->srvaddr, &lsdhcp->options[offopt], 4);
    offopt += 4;
  }

  /* insert requested IP */
  if (reqspec->ipaddr.s_addr == 0) {
    return;
  } else if (CHKOFF(4)) {
    lsdhcp->options[offopt++] = REQUEST_IPADDR;
    lsdhcp->options[offopt++] = 4;
    bcopy(&reqspec->ipaddr, &lsdhcp->options[offopt], 4);
    offopt += 4;
  }


  /* insert client identifier */
  if (intf->cltype->client_id_value != NULL &&
      CHKOFF(intf->cltype->client_id_len + 1)) {
    lsdhcp->options[offopt++] = CLIENT_ID;
    lsdhcp->options[offopt++] = intf->cltype->client_id_len + 1;
    lsdhcp->options[offopt++] = intf->cltype->client_id_type;
    bcopy(intf->cltype->client_id_value, &lsdhcp->options[offopt],
	  intf->cltype->client_id_len);
    offopt += intf->cltype->client_id_len;
  }

  /* if necessary, insert error message */
  if (reqspec->dhcp_errmsg != NULL && CHKOFF(strlen(reqspec->dhcp_errmsg))) {
    lsdhcp->options[offopt++] = DHCP_ERRMSG;
    lsdhcp->options[offopt++] = strlen(reqspec->dhcp_errmsg);
    bcopy(reqspec->dhcp_errmsg, &lsdhcp->options[offopt],
	  strlen(reqspec->dhcp_errmsg));
    offopt += strlen(reqspec->dhcp_errmsg);
  }
  lsdhcp->options[offopt] = END;

  /*
   * construct udp part
   */
  lsudp->uh_sport = dhcpc_port;
  lsudp->uh_dport = dhcps_port;
  lsudp->uh_ulen = htons(DFLTDHCPLEN + UDPHL);
  lsudp->uh_sum = 0;

  /* fill pseudo udp header */
  pudph.srcip.s_addr = 0;
  pudph.dstip.s_addr = 0xffffffff;
  pudph.zero = 0;
  pudph.prto = IPPROTO_UDP;
  pudph.ulen = lsudp->uh_ulen;
  lsudp->uh_sum = udp_cksum(&pudph, (char *) lsudp, ntohs(pudph.ulen));

  /*
   * construct ip part
   */
  lsip->ip_v = IPVERSION;
  lsip->ip_hl = IPHL >> 2;
  lsip->ip_tos = 0;
  lsip->ip_len = htons(DFLTDHCPLEN + UDPHL + IPHL);
  tmpul = generate_xid(NULL);
  tmpul += (tmpul >> 16);
  lsip->ip_id = (u_short) (~tmpul);
  lsip->ip_off = htons(IP_DF);                         /* XXX */
  lsip->ip_ttl = 0x20;                                 /* XXX */
  lsip->ip_p = IPPROTO_UDP;
  lsip->ip_src.s_addr = 0;
  lsip->ip_dst.s_addr = 0xffffffff;
  lsip->ip_sum = 0;
  lsip->ip_sum = cksum((u_short *)lsip, lsip->ip_hl * 2);

  /*
   * construct ether part
   */
  for (i = 0; i < 6; i++) {
#ifdef sun
    lsether->ether_dhost.ether_addr_octet[i] = 0xff;
    lsether->ether_shost.ether_addr_octet[i] = intf->haddr.haddr[i];
#else
    lsether->ether_dhost[i] = 0xff;
    lsether->ether_shost[i] = intf->haddr.haddr[i];
#endif
  }
  lsether->ether_type = htons(ETHERTYPE_IP);

  return;
}


void
make_release(intf, reqspec, lsdhcp)
  struct dhcp_if *intf;
  struct dhcp_reqspec *reqspec;
  struct dhcp *lsdhcp;
{
  int offopt = 0;    /* offset in options field */

  /*
   * construct dhcp part
   */
  lsdhcp->op = BOOTREQUEST;
  lsdhcp->htype = intf->haddr.htype;
  lsdhcp->hlen = intf->haddr.hlen;
  lsdhcp->xid = generate_xid(NULL);
  lsdhcp->ciaddr = reqspec->ipaddr;
  bcopy(intf->haddr.haddr, lsdhcp->chaddr, lsdhcp->hlen);

  /* insert magic cookie */
  bcopy(magic_c, lsdhcp->options, MAGIC_LEN);
  offopt = MAGIC_LEN;

  /* insert message type */
  lsdhcp->options[offopt++] = DHCP_MSGTYPE;
  lsdhcp->options[offopt++] = 1;
  lsdhcp->options[offopt++] = DHCPRELEASE;

  /* insert server identifier */
  if (CHKOFF(4)) {
    lsdhcp->options[offopt++] = SERVER_ID;
    lsdhcp->options[offopt++] = 4;
    bcopy(&reqspec->srvaddr, &lsdhcp->options[offopt], 4);
    offopt += 4;
  }

  /* insert client identifier */
  if (intf->cltype->client_id_value != NULL &&
      CHKOFF(intf->cltype->client_id_len + 1)) {

    lsdhcp->options[offopt++] = CLIENT_ID;
    lsdhcp->options[offopt++] = intf->cltype->client_id_len + 1;
    lsdhcp->options[offopt++] = intf->cltype->client_id_type;
    bcopy(intf->cltype->client_id_value, &lsdhcp->options[offopt],
	  intf->cltype->client_id_len);
    offopt += intf->cltype->client_id_len;
  }

  /* if necessary, insert error message */
  if (reqspec->dhcp_errmsg != NULL && CHKOFF(strlen(reqspec->dhcp_errmsg))) {
    lsdhcp->options[offopt++] = DHCP_ERRMSG;
    lsdhcp->options[offopt++] = strlen(reqspec->dhcp_errmsg);
    bcopy(reqspec->dhcp_errmsg, &lsdhcp->options[offopt],
	  strlen(reqspec->dhcp_errmsg));
    offopt += strlen(reqspec->dhcp_errmsg);
  }
  lsdhcp->options[offopt] = END;

  return;
}


/*
 * send DHCPDECLINE
 */
void
dhcp_decline(intf, reqspec)
  struct dhcp_if *intf;
  struct dhcp_reqspec *reqspec;
{
  int  lsbufsize = ETHERHL + IPHL + UDPHL + DFLTDHCPLEN;
  char lsbuf[ETHERHL + IPHL + UDPHL + DFLTDHCPLEN + QWOFF];

  if (reqspec->srvaddr.s_addr == 0)
    return;

  /* clean memory for send DHCP message */
  bzero(lsbuf, lsbufsize + QWOFF);
  make_decline(intf, reqspec, lsbuf);

  if (ether_write(intf->fordhcp->fd, &lsbuf[QWOFF], lsbufsize) < 0) {
    return;
  }

  errno = 0;
  Syslog(LOG_INFO, "send DHCPDECLINE");
  return;
}


/*
 * send DHCPRELEASE
 */
void
dhcp_release(intf, reqspec)
  struct dhcp_if *intf;
  struct dhcp_reqspec *reqspec;
{
  int  lsbufsize = DFLTDHCPLEN;
  char lsbuf[DFLTDHCPLEN];
  struct dhcp *lsdhcp = NULL;

  bzero(lsbuf, lsbufsize);

  lsdhcp = (struct dhcp *) lsbuf;
  if (reqspec->srvaddr.s_addr == 0)
    return;

  /* send DHCP message */
  make_release(intf, reqspec, lsdhcp);

  if (send_unicast(&reqspec->srvaddr, lsdhcp) < 0) {
    return;
  }

  errno = 0;
  Syslog(LOG_INFO, "send DHCPRELEASE(%s)", inet_ntoa(reqspec->ipaddr));

  return;
}


/*
 * construct DHCPDISCOVER
 */
int
make_discover(intf, optlist, reqspec)
  struct dhcp_if *intf;
  struct dhcp_optlist *optlist;
  struct dhcp_reqspec *reqspec;
{
  int i = 0;
  int offopt = 0;                   /* offset in options field */
  u_long tmpul = 0;
  u_short tmpus = 0;

  if (intf == NULL) {
    errno = 0;
    Syslog(LOG_WARNING, "Invalid argument #1 in make_discover");
    dhcp_errno = EMISC;
    return(-1);
  }
  if (reqspec == NULL) {
    errno = 0;
    Syslog(LOG_WARNING, "Invalid argument #3 in make_discover");
    dhcp_errno = EMISC;
    return(-1);
  }

  /*
   * construct dhcp part
   */
  bzero(sndmsg.buf, sndmsg.bufsize + QWOFF);
  sndmsg.dhcp->op = BOOTREQUEST;
  sndmsg.dhcp->htype = intf->haddr.htype;
  sndmsg.dhcp->hlen = intf->haddr.hlen;
  sndmsg.dhcp->xid = generate_xid(NULL);
  bcopy(intf->haddr.haddr, sndmsg.dhcp->chaddr, sndmsg.dhcp->hlen);

  /* insert magic cookie */
  bcopy(magic_c, sndmsg.dhcp->options, MAGIC_LEN);
  offopt = MAGIC_LEN;

  /* insert message type */
  sndmsg.dhcp->options[offopt++] = DHCP_MSGTYPE;
  sndmsg.dhcp->options[offopt++] = 1;
  sndmsg.dhcp->options[offopt++] = DHCPDISCOVER;

  /* insert class identifier */
  if (intf->cltype->class_id_value != NULL &&
      CHKOFF(intf->cltype->class_id_len)) {
    sndmsg.dhcp->options[offopt++] = CLASS_ID;
    sndmsg.dhcp->options[offopt++] = intf->cltype->class_id_len;
    bcopy(intf->cltype->class_id_value, &sndmsg.dhcp->options[offopt],
	  intf->cltype->class_id_len);
    offopt += intf->cltype->class_id_len;
  }

  /* insert client identifier */
  if (intf->cltype->client_id_value != NULL &&
      CHKOFF(intf->cltype->client_id_len + 1)) {
    sndmsg.dhcp->options[offopt++] = CLIENT_ID;
    sndmsg.dhcp->options[offopt++] = intf->cltype->client_id_len + 1;
    sndmsg.dhcp->options[offopt++] = intf->cltype->client_id_type;
    bcopy(intf->cltype->client_id_value, &sndmsg.dhcp->options[offopt],
	  intf->cltype->client_id_len);
    offopt += intf->cltype->client_id_len;
  }

  /* insert requesting lease */
  if (reqspec->lease != 0 && CHKOFF(4)) {
    sndmsg.dhcp->options[offopt++] = LEASE_TIME;
    sndmsg.dhcp->options[offopt++] = 4;
    tmpul = htonl(reqspec->lease);
    bcopy(&tmpul, &sndmsg.dhcp->options[offopt], 4);
    offopt += 4;
  }

  /* insert requesting ipaddr */
  if (reqspec->ipaddr.s_addr != 0 && CHKOFF(4)) {
    sndmsg.dhcp->options[offopt++] = REQUEST_IPADDR;
    sndmsg.dhcp->options[offopt++] = 4;
    bcopy(&reqspec->ipaddr.s_addr, &sndmsg.dhcp->options[offopt], 4);
    offopt += 4;
  }

  /* insert Maximum DHCP message size */
  if (CHKOFF(2)) {
    sndmsg.dhcp->options[offopt++] = DHCP_MAXMSGSIZE;
    sndmsg.dhcp->options[offopt++] = 2;
    tmpus = htons(DFLTDHCPLEN);
    bcopy(&tmpus, &sndmsg.dhcp->options[offopt], 2);
    offopt += 2;
  }

  /* if necessary, insert request list */
  if (optlist != NULL && CHKOFF(optlist->len)) {
    sndmsg.dhcp->options[offopt++] = REQ_LIST;
    sndmsg.dhcp->options[offopt++] = optlist->len;
    bcopy(optlist->list, &sndmsg.dhcp->options[offopt], optlist->len);
    offopt += optlist->len;
  }

  sndmsg.dhcp->options[offopt] = END;

  /*
   * make udp part
   */
  /* fill udp header */
  sndmsg.udp->uh_sport = dhcpc_port;
  sndmsg.udp->uh_dport = dhcps_port;
  sndmsg.udp->uh_ulen = htons(DFLTDHCPLEN + UDPHL);

  /* fill pseudo udp header */
  sndmsg.pudph.srcip.s_addr = 0;
  sndmsg.pudph.dstip.s_addr = 0xffffffff;
  sndmsg.pudph.zero = 0;
  sndmsg.pudph.prto = IPPROTO_UDP;
  sndmsg.pudph.ulen = sndmsg.udp->uh_ulen;

  /*
   * make ip part
   */
  /* fill ip header */
  sndmsg.ip->ip_v = IPVERSION;
  sndmsg.ip->ip_hl = IPHL >> 2;
  sndmsg.ip->ip_tos = 0;
  sndmsg.ip->ip_len = htons(DFLTDHCPLEN + UDPHL + IPHL);
  tmpul = generate_xid(NULL);
  tmpul += (tmpul >> 16);
  sndmsg.ip->ip_id = (u_short) (~tmpul);
  sndmsg.ip->ip_off = htons(IP_DF);                         /* XXX */
  sndmsg.ip->ip_ttl = 0x20;                                 /* XXX */
  sndmsg.ip->ip_p = IPPROTO_UDP;
  sndmsg.ip->ip_src.s_addr = 0;
  sndmsg.ip->ip_dst.s_addr = 0xffffffff;
  sndmsg.ip->ip_sum = 0;
  sndmsg.ip->ip_sum = cksum((u_short *)sndmsg.ip, sndmsg.ip->ip_hl * 2);

  /*
   * make ether part
   */
  /* fill ether frame header */
  for (i = 0; i < 6; i++) {
#ifdef sun
    sndmsg.ether->ether_dhost.ether_addr_octet[i] = 0xff;
    sndmsg.ether->ether_shost.ether_addr_octet[i] = intf->haddr.haddr[i];
#else
    sndmsg.ether->ether_dhost[i] = 0xff;
    sndmsg.ether->ether_shost[i] = intf->haddr.haddr[i];
#endif
  }
  sndmsg.ether->ether_type = htons(ETHERTYPE_IP);

  return(0);
}


/*
 * construct DHCPREQUEST
 */
int
make_request(intf, optlist, reqspec, type)
  struct dhcp_if *intf;
  struct dhcp_optlist *optlist;
  struct dhcp_reqspec *reqspec;
  int type;
{
  int i = 0;
  int offopt = 0;                   /* offset in options field */
  u_long tmpul = 0;
  u_short tmpus = 0;

  if (intf == NULL) {
    errno = 0;
    Syslog(LOG_WARNING, "Invalid argument #1 in make_request");
    dhcp_errno = EMISC;
    return(-1);
  }
  if (reqspec == NULL) {
    errno = 0;
    Syslog(LOG_WARNING, "Invalid argument #3 in make_request");
    dhcp_errno = EMISC;
    return(-1);
  }
  /*
   * construct dhcp part
   */
  bzero(sndmsg.buf, sndmsg.bufsize + QWOFF);
  sndmsg.dhcp->op = BOOTREQUEST;
  sndmsg.dhcp->htype = intf->haddr.htype;
  sndmsg.dhcp->hlen = intf->haddr.hlen;
  sndmsg.dhcp->xid = generate_xid(NULL);
  if (type == REQUESTING || type == REBOOTING || type == VERIFYING)
    sndmsg.dhcp->ciaddr.s_addr = 0;
  else
    sndmsg.dhcp->ciaddr.s_addr = reqspec->ipaddr.s_addr;

  bcopy(intf->haddr.haddr, sndmsg.dhcp->chaddr, sndmsg.dhcp->hlen);

  /* insert magic cookie */
  bcopy(magic_c, sndmsg.dhcp->options, MAGIC_LEN);
  offopt = MAGIC_LEN;

  /* insert message type */
  sndmsg.dhcp->options[offopt++] = DHCP_MSGTYPE;
  sndmsg.dhcp->options[offopt++] = 1;
  sndmsg.dhcp->options[offopt++] = DHCPREQUEST;

  /* insert class identifier */
  if (intf->cltype->class_id_value != NULL &&
      CHKOFF(intf->cltype->class_id_len)) {
    sndmsg.dhcp->options[offopt++] = CLASS_ID;
    sndmsg.dhcp->options[offopt++] = intf->cltype->class_id_len;
    bcopy(intf->cltype->class_id_value, &sndmsg.dhcp->options[offopt],
	  intf->cltype->class_id_len);
    offopt += intf->cltype->class_id_len;
  }

  /* insert client identifier */
  if (intf->cltype->client_id_value != NULL &&
      CHKOFF(intf->cltype->client_id_len + 1)) {
    sndmsg.dhcp->options[offopt++] = CLIENT_ID;
    sndmsg.dhcp->options[offopt++] = intf->cltype->client_id_len + 1;
    sndmsg.dhcp->options[offopt++] = intf->cltype->client_id_type;
    bcopy(intf->cltype->client_id_value, &sndmsg.dhcp->options[offopt],
	  intf->cltype->client_id_len);
    offopt += intf->cltype->client_id_len;
  }

  /* insert requesting lease */
  if (type != VERIFYING && reqspec->lease != 0 && CHKOFF(4)) {
    sndmsg.dhcp->options[offopt++] = LEASE_TIME;
    sndmsg.dhcp->options[offopt++] = 4;
    tmpul = htonl(reqspec->lease);
    bcopy(&tmpul, &sndmsg.dhcp->options[offopt], 4);
    offopt += 4;
  }

  /* insert requesting ipaddr */
  if (type == REQUESTING || type == REBOOTING || type == VERIFYING) {
    if (reqspec->ipaddr.s_addr != 0 && CHKOFF(4)) {
      sndmsg.dhcp->options[offopt++] = REQUEST_IPADDR;
      sndmsg.dhcp->options[offopt++] = 4;
      bcopy(&reqspec->ipaddr.s_addr, &sndmsg.dhcp->options[offopt], 4);
      offopt += 4;
    }
  }

  /* insert server identifier */
  if (type == REQUESTING) {
    if (reqspec->srvaddr.s_addr == 0) {
      return(-1);
    }
    if (CHKOFF(4)) {
      sndmsg.dhcp->options[offopt++] = SERVER_ID;
      sndmsg.dhcp->options[offopt++] = 4;
      bcopy(&reqspec->srvaddr.s_addr, &sndmsg.dhcp->options[offopt], 4);
      offopt += 4;
    }
  }

  /* insert Maximum DHCP message size */
  if (CHKOFF(2)) {
    sndmsg.dhcp->options[offopt++] = DHCP_MAXMSGSIZE;
    sndmsg.dhcp->options[offopt++] = 2;
    tmpus = htons(DFLTDHCPLEN);
    bcopy(&tmpus, &sndmsg.dhcp->options[offopt], 2);
    offopt += 2;
  }

  /* if necessary, insert request list */
  if (optlist != NULL && CHKOFF(optlist->len)) {
    sndmsg.dhcp->options[offopt++] = REQ_LIST;
    sndmsg.dhcp->options[offopt++] = optlist->len;
    bcopy(optlist->list, &sndmsg.dhcp->options[offopt], optlist->len);
    offopt += optlist->len;
  }

  sndmsg.dhcp->options[offopt] = END;

  if (type == RENEWING)    /* RENEWING is unicast with the normal socket */
    return(0);

  /*
   * make udp part
   */
  /* fill udp header */
  sndmsg.udp->uh_sport = dhcpc_port;
  sndmsg.udp->uh_dport = dhcps_port;
  sndmsg.udp->uh_ulen = htons(DFLTDHCPLEN + UDPHL);

  /* fill pseudo udp header */
  sndmsg.pudph.zero = 0;
  sndmsg.pudph.prto = IPPROTO_UDP;
  sndmsg.pudph.ulen = sndmsg.udp->uh_ulen;

  /*
   * make ip part
   */
  /* fill ip header */
  sndmsg.ip->ip_v = IPVERSION;
  sndmsg.ip->ip_hl = IPHL >> 2;
  sndmsg.ip->ip_tos = 0;
  sndmsg.ip->ip_len = htons(DFLTDHCPLEN + UDPHL + IPHL);
  tmpul = generate_xid(NULL);
  tmpul += (tmpul >> 16);
  sndmsg.ip->ip_id = (u_short) (~tmpul);
  sndmsg.ip->ip_off = htons(IP_DF);                         /* XXX */
  sndmsg.ip->ip_ttl = 0x20;                                 /* XXX */
  sndmsg.ip->ip_p = IPPROTO_UDP;

  switch (type) {
  case REQUESTING:
  case REBOOTING:
  case VERIFYING:
    sndmsg.ip->ip_src.s_addr = sndmsg.pudph.srcip.s_addr = 0;
    sndmsg.ip->ip_dst.s_addr = sndmsg.pudph.dstip.s_addr = 0xffffffff;
    break;
  case RENEWING:
    sndmsg.ip->ip_src.s_addr = sndmsg.pudph.srcip.s_addr =
      reqspec->ipaddr.s_addr;
    sndmsg.ip->ip_dst.s_addr = sndmsg.pudph.dstip.s_addr =
      reqspec->srvaddr.s_addr;
  case REBINDING:
    sndmsg.ip->ip_src.s_addr = sndmsg.pudph.srcip.s_addr =
      reqspec->ipaddr.s_addr;
    sndmsg.ip->ip_dst.s_addr = sndmsg.pudph.dstip.s_addr = 0xffffffff;
    break;
  }
  sndmsg.ip->ip_sum = 0;
  sndmsg.ip->ip_sum = cksum((u_short *)sndmsg.ip, sndmsg.ip->ip_hl * 2);

  /*
   * make ether part
   */
  /* fill ether frame header */
  for (i = 0; i < 6; i++) {
#ifdef sun
    sndmsg.ether->ether_dhost.ether_addr_octet[i] = 0xff;
    sndmsg.ether->ether_shost.ether_addr_octet[i] = intf->haddr.haddr[i];
#else
    sndmsg.ether->ether_dhost[i] = 0xff;
    sndmsg.ether->ether_shost[i] = intf->haddr.haddr[i];
#endif
  }
  sndmsg.ether->ether_type = htons(ETHERTYPE_IP);

  return(0);
}


/*
 * send DHCP message within unicast
 */
int
send_unicast(dstip, sdhcp)
  struct in_addr *dstip;
  struct dhcp *sdhcp;
{
  int sockfd = 0;
  struct sockaddr_in client, server;
  int bufsize = 0;

  bzero(&client, sizeof(client));
  bzero(&server, sizeof(server));

  if ((sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
    return(-1);
  client.sin_family = AF_INET;
  client.sin_addr.s_addr = htonl(INADDR_ANY);
  client.sin_port = dhcpc_port;
  server.sin_family = AF_INET;
  server.sin_addr.s_addr = dstip->s_addr;
  server.sin_port = dhcps_port;
  if (bind(sockfd, (struct sockaddr *) &client, sizeof(client)) < 0) {
    close(sockfd);
    return(-1);
  }
  if (connect(sockfd, (struct sockaddr *) &server, sizeof(server)) < 0) {
    close(sockfd);
    return(-1);
  }
  bufsize = DFLTDHCPLEN;
  if (setsockopt(sockfd, SOL_SOCKET, SO_SNDBUF, &bufsize, sizeof(bufsize)) < 0) {
    close(sockfd);
    return(-1);
  }

  if (write(sockfd, sdhcp, bufsize) < bufsize) {
    close(sockfd);
    return(-1);
  }

  close(sockfd);
  return(0);
}


/*
 * pickup the IP address and set it into appropriate field of dhcp_param
 */
int
handle_ip(buf, param)
  char *buf;
  struct dhcp_param *param;
{
  struct in_addr *addr = NULL;

  if ((addr = (struct in_addr *)calloc(1, sizeof(struct in_addr))) == NULL) {
    return(-1);
  }

  switch (*buf) {
  case SERVER_ID:
    if (param->server_id != NULL) free(param->server_id);
    param->server_id = addr;
    break;
  case SUBNET_MASK:
    if (param->subnet_mask != NULL) free(param->subnet_mask);
    param->subnet_mask = addr;
    break;
  case SWAP_SERVER:
    if (param->swap_server != NULL) free(param->swap_server);
    param->swap_server = addr;
    break;
  case BRDCAST_ADDR:
    if (param->brdcast_addr != NULL) free(param->brdcast_addr);
    param->brdcast_addr = addr;
    break;
  case ROUTER_SOLICIT:
    if (param->router_solicit != NULL) free(param->router_solicit);
    param->router_solicit = addr;
    break;
  default:
    free(addr);
    return(EINVAL);
  }

  bcopy(OPTBODY(buf), addr, OPTLEN(buf));
  buf += OPTLEN(buf) + 1;
  return(0);
}


/*
 * pickup the number and set it into appropriate field of dhcp_param
 */
int
handle_num(buf, param)
  char *buf;
  struct dhcp_param *param;
{
  char   charnum = 0;
  short  shortnum = 0;
  long   longnum = 0;

  switch (OPTLEN(buf)) {
  case 1:
    charnum = *OPTBODY(buf);
    break;
  case 2:
    shortnum = GETHS(OPTBODY(buf));
    break;
  case 4:
    longnum = GETHL(OPTBODY(buf));
    break;
  default:
    return(-1);
  }

  switch (*buf) {
  case TIME_OFFSET:
    param->time_offset = longnum;
    break;
  case BOOTSIZE:
    param->bootsize = (unsigned short) shortnum;
    break;
  case MAX_DGRAM_SIZE:
    param->max_dgram_size = (unsigned short) shortnum;
    break;
  case DEFAULT_IP_TTL:
    param->default_ip_ttl = (unsigned char) charnum;
    break;
  case MTU_AGING_TIMEOUT:
    param->mtu_aging_timeout = (unsigned long) longnum;
    break;
  case IF_MTU:
    param->if_mtu = (unsigned short) shortnum;
    break;
  case ARP_CACHE_TIMEOUT:
    param->arp_cache_timeout = (unsigned long) longnum;
    break;
  case DEFAULT_TCP_TTL:
    param->default_tcp_ttl = (unsigned char) charnum;
    break;
  case KEEPALIVE_INTER:
    param->keepalive_inter = (unsigned long) longnum;
    break;
  case NB_NODETYPE:
    param->nb_nodetype = (unsigned) charnum;
    break;
  case LEASE_TIME:
    param->lease_duration = (unsigned long) longnum;
    break;
  case DHCP_T1:
    param->dhcp_t1 = (unsigned long) longnum;
    break;
  case DHCP_T2:
    param->dhcp_t2 = (unsigned long) longnum;
    break;
  default:
    return(EINVAL);
  }

  buf += OPTLEN(buf) + 1;
  return(0);
}


/*
 * pickup IP addresses and set them into appropriate field of dhcp_param
 */
int
handle_ips(buf, param)
  char *buf;
  struct dhcp_param *param;
{
  struct in_addr  *addr = NULL;
  struct in_addrs *addrs = NULL;
  unsigned char  num = 0;

  num = OPTLEN(buf) / 4;
  if ((addr = (struct in_addr *)calloc((int) num, sizeof(struct in_addr))) == NULL) {
    return(-1);
  }

  if ((addrs = (struct in_addrs *) calloc(1, sizeof(struct in_addrs))) == NULL) {
    free(addr);
    return(-1);
  }

  switch (*buf) {
  case ROUTER:
    if (param->router != NULL) {
      if (param->router->addr != NULL) free(param->router->addr);
    free(param->router);
    }
    param->router = addrs;
    param->router->num = num;
    param->router->addr = addr;
    break;
  case TIME_SERVER:
    if (param->time_server != NULL) {
      if (param->time_server->addr != NULL) free(param->time_server->addr);
      free(param->time_server);
    }
    param->time_server = addrs;
    param->time_server->num = num;
    param->time_server->addr = addr;
    break;
  case NAME_SERVER:
    if (param->name_server != NULL) {
      if (param->name_server->addr != NULL) free(param->name_server->addr);
      free(param->name_server);
    }
    param->name_server = addrs;
    param->name_server->num = num;
    param->name_server->addr = addr;
    break;
  case DNS_SERVER:
    if (param->dns_server != NULL) {
      if (param->dns_server->addr != NULL) free(param->dns_server->addr);
      free(param->dns_server);
    }
    param->dns_server = addrs;
    param->dns_server->num = num;
    param->dns_server->addr = addr;
    break;
  case LOG_SERVER:
    if (param->log_server != NULL) {
      if (param->log_server->addr != NULL) free(param->log_server->addr);
      free(param->log_server);
    }
    param->log_server = addrs;
    param->log_server->num = num;
    param->log_server->addr = addr;
    break;
  case COOKIE_SERVER:
    if (param->cookie_server != NULL) {
      if (param->cookie_server->addr != NULL) free(param->cookie_server->addr);
      free(param->cookie_server);
    }
    param->cookie_server = addrs;
    param->cookie_server->num = num;
    param->cookie_server->addr = addr;
    break;
  case LPR_SERVER:
    if (param->lpr_server != NULL) {
      if (param->lpr_server->addr != NULL) free(param->lpr_server->addr);
      free(param->lpr_server);
    }
    param->lpr_server = addrs;
    param->lpr_server->num = num;
    param->lpr_server->addr = addr;
    break;
  case IMPRESS_SERVER:
    if (param->impress_server != NULL) {
      if (param->impress_server->addr != NULL) free(param->impress_server->addr);
      free(param->impress_server);
    }
    param->impress_server = addrs;
    param->impress_server->num = num;
    param->impress_server->addr = addr;
    break;
  case RLS_SERVER:
    if (param->rls_server != NULL) {
      if (param->rls_server->addr != NULL) free(param->rls_server->addr);
      free(param->rls_server);
    }
    param->rls_server = addrs;
    param->rls_server->num = num;
    param->rls_server->addr = addr;
    break;
  case NIS_SERVER:
    if (param->nis_server != NULL) {
      if (param->nis_server->addr != NULL) free(param->nis_server->addr);
      free(param->nis_server);
    }
    param->nis_server = addrs;
    param->nis_server->num = num;
    param->nis_server->addr = addr;
    break;
  case NTP_SERVER:
    if (param->ntp_server != NULL) {
      if (param->ntp_server->addr != NULL) free(param->ntp_server->addr);
      free(param->ntp_server);
    }
    param->ntp_server = addrs;
    param->ntp_server->num = num;
    param->ntp_server->addr = addr;
    break;
  case NBN_SERVER:
    if (param->nbn_server != NULL) {
      if (param->nbn_server->addr != NULL) free(param->nbn_server->addr);
      free(param->nbn_server);
    }
    param->nbn_server = addrs;
    param->nbn_server->num = num;
    param->nbn_server->addr = addr;
    break;
  case NBDD_SERVER:
    if (param->nbdd_server != NULL) {
      if (param->nbdd_server->addr != NULL) free(param->nbdd_server->addr);
      free(param->nbdd_server);
    }
    param->nbdd_server = addrs;
    param->nbdd_server->num = num;
    param->nbdd_server->addr = addr;
    break;
  case XFONT_SERVER:
    if (param->xfont_server != NULL) {
      if (param->xfont_server->addr != NULL) free(param->xfont_server->addr);
      free(param->xfont_server);
    }
    param->xfont_server = addrs;
    param->xfont_server->num = num;
    param->xfont_server->addr = addr;
    break;
  case XDISPLAY_MANAGER:
    if (param->xdisplay_manager != NULL) {
      if (param->xdisplay_manager->addr != NULL) free(param->xdisplay_manager->addr);
      free(param->xdisplay_manager);
    }
    param->xdisplay_manager = addrs;
    param->xdisplay_manager->num = num;
    param->xdisplay_manager->addr = addr;
    break;
  default:
    free(addr);
    free(addrs);
    return(EINVAL);
  }

  bcopy(OPTBODY(buf), addr, OPTLEN(buf));
  buf += OPTLEN(buf) + 1;
  return(0);
}


/*
 * pickup the strings and set it into appropriate field of dhcp_param
 */
int
handle_str(buf, param)
  char *buf;
  struct dhcp_param *param;
{
  char *str = NULL;

  if ((str = calloc(1, (OPTLEN(buf) + 1))) == NULL) { /* +1 for null terminate */
    return(-1);
  }

  switch (*buf) {
  case HOSTNAME:
    if (param->hostname != NULL) free(param->hostname);
    param->hostname = str;
    break;
  case MERIT_DUMP:
    if (param->merit_dump != NULL) free(param->merit_dump);
    param->merit_dump = str;
    break;
  case DNS_DOMAIN:
    if (param->dns_domain != NULL) free(param->dns_domain);
    param->dns_domain = str;
    break;
  case ROOT_PATH:
    if (param->root_path != NULL) free(param->root_path);
    param->root_path = str;
    break;
  case EXTENSIONS_PATH:
    if (param->extensions_path != NULL) free(param->extensions_path);
    param->extensions_path = str;
    break;
  case NIS_DOMAIN:
    if (param->nis_domain != NULL) free(param->nis_domain);
    param->nis_domain = str;
    break;
  case NB_SCOPE:
    if (param->nb_scope != NULL) free(param->nb_scope);
    param->nb_scope = str;
    break;
  case DHCP_ERRMSG:
    if (param->errmsg != NULL) free(param->errmsg);
    param->errmsg = str;
    break;
  default:
    free(str);
    return(EINVAL);
  }

  bcopy(OPTBODY(buf), str, OPTLEN(buf));
  str[OPTLEN(buf)] = '\0';
  buf += OPTLEN(buf) + 1;
  return(0);
}


/*
 * pickup the boolean value and set it into appropriate field of dhcp_param
 */
int
handle_bool(buf, param)
  char *buf;
  struct dhcp_param *param;
{
  switch (*buf) {
  case IP_FORWARD:
    param->ip_forward = *OPTBODY(buf);
    break;
  case NONLOCAL_SRCROUTE:
    param->nonlocal_srcroute = *OPTBODY(buf);
    break;
  case ALL_SUBNET_LOCAL:
    param->all_subnet_local = *OPTBODY(buf);
    break;
  case MASK_DISCOVER:
    param->mask_discover = *OPTBODY(buf);
    break;
  case MASK_SUPPLIER:
    param->mask_supplier = *OPTBODY(buf);
    break;
  case ROUTER_DISCOVER:
    param->router_discover = *OPTBODY(buf);
    break;
  case TRAILER:
    param->trailer = *OPTBODY(buf);
    break;
  case ETHER_ENCAP:
    param->ether_encap = *OPTBODY(buf);
    break;
  case KEEPALIVE_GARBA:
    param->keepalive_garba = *OPTBODY(buf);
    break;
  default:
    return(EINVAL);
  }

  buf += OPTLEN(buf) + 1;
  return(0);
}


/*
 * pickup pairs of IP address and set them into appropriate field of dhcp_param
 */
int
handle_ippairs(buf, param)
  char *buf;
  struct dhcp_param *param;
{
  struct in_addr  *addr = NULL;
  struct in_addrs *addrs = NULL;
  unsigned char   num = 0;

  num = OPTLEN(buf) / 4;
  if ((addr = (struct in_addr *)calloc((int) num, sizeof(struct in_addr))) == NULL) {
    return(-1);
  }

  if ((addrs = (struct in_addrs *) calloc(1, sizeof(struct in_addrs))) == NULL) {
    free(addr);
    return(-1);
  }

  switch (*buf) {
  case POLICY_FILTER:
    if (param->policy_filter != NULL) {
      if (param->policy_filter->addr != NULL) free(param->policy_filter->addr);
      free(param->policy_filter);
    }
    param->policy_filter = addrs;
    param->policy_filter->num = num / 2;
    param->policy_filter->addr = addr;
    break;
  case TIME_SERVER:
    if (param->static_route != NULL) {
      if (param->static_route->addr != NULL) free(param->static_route->addr);
      free(param->static_route);
    }
    param->static_route = addrs;
    param->static_route->num = num / 2;
    param->static_route->addr = addr;
    break;
  default:
    free(addr);
    free(addrs);
    return(EINVAL);
  }

  bcopy(OPTBODY(buf), addr, OPTLEN(buf));
  buf += OPTLEN(buf) + 1;
  return(0);
}


/*
 * pickup the number and set it into appropriate field of dhcp_param
 */
int
handle_nums(buf, param)
  char *buf;
  struct dhcp_param *param;
{
  int i = 0, max = 0;

  if (*buf != MTU_PLATEAU_TABLE) {
    return(EINVAL);
  } else {
    if (param->mtu_plateau_table != NULL) {
      if (param->mtu_plateau_table->value != NULL)
	free(param->mtu_plateau_table->value);
      free(param->mtu_plateau_table);
    }
    if ((param->mtu_plateau_table =
	 (struct shorts *) calloc(1, sizeof(struct shorts))) == NULL) {
      return(-1);
    }
    max = param->mtu_plateau_table->num = OPTLEN(buf) / 2;
    if ((param->mtu_plateau_table->value =
	 (u_short *) calloc(max, sizeof(unsigned short))) == NULL) {
      return(-1);
    }
    for (i = 0; i < max; i++) {
      param->mtu_plateau_table->value[i] = ntohs(buf[i * 2 + 2]);
    }
  }

  buf += OPTLEN(buf) + 1;
  return(0);
}


/*
 * setup the service port
 */
void
set_srvport()
{
  struct servent *srv = NULL;

  if ((srv = getservbyname("dhcpc", "udp")) != NULL) {
    dhcpc_port = srv->s_port;
  } else {
    Syslog(LOG_WARNING, "udp/dhcpc: unknown service -- use default port %d",
	   DHCPC_PORT);
    dhcpc_port = htons(DHCPC_PORT);
  }
  srv = getservbyname("dhcps", "udp");
  if (srv != NULL) {
    dhcps_port = srv->s_port;
  } else {
    Syslog(LOG_WARNING, "udp/dhcps: unknown service -- use default port %d",
	   DHCPS_PORT);
    dhcps_port = htons(DHCPS_PORT);
  }

  return;
}


/**************************************************************
 *  pickup option from dhcp message                           *
 *                                                            *
 *  If there are many options which has the same tag,         *
 *  this function returns the first one.                      *
 *  searching sequence is :                                   *
 *                                                            *
 *    options field -> 'file' field -> 'sname' field          *
 **************************************************************/
char *
pickup_opt(msg, msglen, tag)
  struct dhcp *msg;
  int msglen;
  char tag;
{
  int sname_is_opt = 0;
  int file_is_opt = 0;
  int i = 0;
  char *opt = NULL;
  char *found = NULL;

  /*
   *  search option field.
   */
  opt = &msg->options[MAGIC_LEN];
  /* DHCPLEN - OPTLEN == from 'op' to 'file' field */
  for (i = 0; i < msglen - DFLTDHCPLEN + DFLTOPTLEN; i++) {
    if (*(opt + i) == tag) {
      found = (opt + i);
      break;
    }
    else if (*(opt+i) == END) {
      break;
    }
    else if (*(opt+i) == OPT_OVERLOAD) {
      i += 2 ;
      if (*(opt+i) == 1)
	file_is_opt = 1;
      else if (*(opt+i) == 2)
	sname_is_opt = 1;
      else if (*(opt+i) == 3)
	file_is_opt = sname_is_opt = 1;
      continue;
    }
    else if (*(opt+i) == PAD) {
      continue;
    }
    else {
      i += *(opt + i + 1) + 1;
    }
  }
  if (found != NULL)
    return(found);

  /*
   *  if necessary, search file field
   */
  if (file_is_opt) {
    opt = msg->file;
    for (i = 0; i < sizeof(msg->file); i++) {
      if (*(opt+i) == PAD) {
	continue;
      }
      else if (*(opt+i) == END) {
	break;
      }
      else if (*(opt + i) == tag) {
	found = (opt + i);
	break;
      }
      else {
	i += *(opt + i + 1) + 1;
      }
    }
    if (found != NULL)
      return(found);
  }

  /*
   *  if necessary, search sname field
   */
  if (sname_is_opt) {
    opt = msg->sname;
    for (i = 0; i < sizeof(msg->sname); i++) {
      if (*(opt+i) == PAD) {
	continue;
      }
      else if (*(opt+i) == END) {
	break;
      }
      else if (*(opt + i) == tag) {
	found = (opt + i);
	break;
      }
      else {
	i += *(opt + i + 1) + 1;
      }
    }
    if (found != NULL)
      return(found);
  }

  return(NULL);
}


/*
 * check IP checksum
 */
int
check_ipsum(ipp)
  struct ip *ipp;
{
  u_short ripcksum;                   /* recv IP checksum */
  ripcksum = ipp->ip_sum;

  return(ripcksum == get_ipsum(ipp));
}


/*
 * return IP checksum in network byte order
 */
u_short
get_ipsum(ipp)
  struct ip *ipp;
{
  ipp->ip_sum = 0;

  return(cksum((u_short *)ipp, ipp->ip_hl * 2));
}


/*
 * check UDP checksum
 */
int
check_udpsum(ipp, udpp)
  struct ip *ipp;
  struct udphdr *udpp;
{
  u_short rudpcksum;                /* recv UDP checksum */

  if (udpp->uh_sum == 0) 
    return(TRUE);
  rudpcksum = udpp->uh_sum;

  return(rudpcksum == get_udpsum(ipp, udpp));
}


/*
 * return the UDP checksum in network byte order
 */
u_short
get_udpsum(ipp, udpp)
  struct ip *ipp;
  struct udphdr *udpp;
{
  struct ps_udph pudph;

  bzero(&pudph, sizeof(pudph));

  pudph.srcip.s_addr = ipp->ip_src.s_addr;
  pudph.dstip.s_addr = ipp->ip_dst.s_addr;
  pudph.zero = 0;
  pudph.prto = IPPROTO_UDP;
  pudph.ulen = udpp->uh_ulen;
  udpp->uh_sum = 0;

  return(udp_cksum(&pudph, (char *) udpp, ntohs(pudph.ulen)));
}


/*
 * caluculate the udp check sum
 *        n is length in bytes
 */
u_short
udp_cksum(pudph, buf, n)
  struct ps_udph *pudph;
  char *buf;
  int n;
{
    u_long sum = 0;
    u_short *tmp = NULL,
            result;
    register int i = 0;
    unsigned char pad[2];

    tmp = (u_short *) pudph;
    for (i = 0; i < 6; i++) {
      sum += *tmp++;
    }      

    tmp = (u_short *) buf;
    while (n > 1) {
      sum += *tmp++;
      n -= 2;
    }

    if (n == 1) {      /* n % 2 == 1, have to do padding */
      pad[0] = *tmp;
      pad[1] = 0;
      tmp = (u_short *) pad;
      sum += *tmp;
    }

    sum = (sum >> 16) + (sum & 0xffff);
    sum += (sum >> 16);
    result = (u_short) ~sum;
    if (result == 0)
      result = 0xffff;

    return(result);
}


/*
 * caluculate the check sum for IP suit
 */
u_short
cksum(buf, n)
  u_short *buf;
  int n;
{
    u_long sum;
    u_short result;

    for (sum = 0; n > 0; n--) {
      sum += *buf++;
    }

    sum = (sum >> 16) + (sum & 0xffff);
    sum += (sum >> 16);
    result = (u_short) (~sum);
    if (result == 0)
      result = 0xffff;

    return(result);
}


#ifdef sun
int
ether_write(fd, buf, size)
  int fd;
  char *buf;
  int size;
{
  struct sockaddr sa;
  struct strbuf cbuf, dbuf;

  bzero(&sa, sizeof(sa));
  bzero(&cbuf, sizeof(cbuf));
  bzero(&dbuf, sizeof(dbuf));
  sa.sa_family = AF_UNSPEC;
  bcopy(buf, sa.sa_data, ETHERHL);

  cbuf.maxlen = cbuf.len = sizeof(sa);
  cbuf.buf = (char *)&sa;

  dbuf.maxlen = dbuf.len = size - ETHERHL;
  dbuf.buf = &buf[ETHERHL];

  if (putmsg(fd, &cbuf, &dbuf, 0) == 0)
    return(size);
  else
    return(-1);
}


int
ether_writev(fd, bufvec, nvec)
  int fd;
  struct iovec *bufvec;
  int nvec;
{
  int i;
  int len = 0;
  int result = 0;
  char *buf;

  for (i = 0; i < nvec; i++) {
    len += bufvec[i].iov_len;
  }
  if ((buf = (char *) calloc(1, len)) == NULL) {
    syslog(LOG_WARNING, "calloc error in ether_writev(): %m");
    return(-1);
  }
  len = 0;
  for (i = 0; i < nvec; i++) {
    bcopy(bufvec[i].iov_base, &buf[len], bufvec[i].iov_len);
    len += bufvec[i].iov_len;
  }

  result = ether_write(fd, buf, len);
  free(buf);

  return(result);
}
#endif


#ifdef sun
void
ptrtomsg(cp)
  char *cp;
{
  rcvmsg.ether = (struct ether_header *) cp;
  rcvmsg.ip = (struct ip *) &cp[ETHERHL];
  if ((ntohs(rcvmsg.ip->ip_off) & 0x1fff) == 0 &&
      ntohs(rcvmsg.ip->ip_len) >= DFLTBOOTPLEN + UDPHL + IPHL) {
    rcvmsg.udp = (struct udphdr *) &cp[ETHERHL+rcvmsg.ip->ip_hl*WORD];
    rcvmsg.dhcp = (struct dhcp *) &cp[ETHERHL+rcvmsg.ip->ip_hl*WORD+UDPHL];
  } else {
    rcvmsg.udp = NULL;
    rcvmsg.dhcp = NULL;
  }

  return;
}


void
mvptrtonext()
{
  rcvmsg.remain -= ntohs(rcvmsg.ip->ip_len);
  if (rcvmsg.remain >= 0) {
    rcvmsg.bptr += ntohs(rcvmsg.ip->ip_len);
  } else {
    rcvmsg.remain = 0;
    rcvmsg.bptr = NULL;
  }

  return;
}

#else  /* if it's not sun */

void
ptrtomsg(cp)
  char *cp;
{
  rcvmsg.bpf = (struct bpf_hdr *) cp;
  rcvmsg.ether = (struct ether_header *) &cp[rcvmsg.bpf->bh_hdrlen];
  rcvmsg.ip = (struct ip *) &cp[rcvmsg.bpf->bh_hdrlen+ETHERHL];
  if (ntohs(rcvmsg.ip->ip_len) >= DFLTBOOTPLEN + UDPHL + IPHL) {
    rcvmsg.udp = (struct udphdr *)
      &cp[rcvmsg.bpf->bh_hdrlen+ETHERHL+rcvmsg.ip->ip_hl*WORD];
    rcvmsg.dhcp = (struct dhcp *)
      &cp[rcvmsg.bpf->bh_hdrlen+ETHERHL+rcvmsg.ip->ip_hl*WORD+UDPHL];
  } else {
    rcvmsg.udp = NULL;
    rcvmsg.dhcp = NULL;
  }

  return;
}


void
mvptrtonext()
{
  rcvmsg.bptr += BPF_WORDALIGN(rcvmsg.bpf->bh_hdrlen + rcvmsg.bpf->bh_caplen);
  rcvmsg.remain -= (char *)rcvmsg.bptr - (char *)rcvmsg.bpf;

  if (rcvmsg.remain <= 0) {
    rcvmsg.remain = 0;
    rcvmsg.bptr = NULL;
  }

  return;
}
#endif /* sun */
