/*
 * Copyright (c) 1995 Akihiro Tominaga and the WIDE Project.
 * All rights reserved.
 *
 * Permission to use, copy and distribute this software in source and
 * binary forms is hereby granted provided that the above copyright
 * notice and this permission notice appear in all copies of the software,
 * derivative works or modified versions, and any portions thereof.
 * Furthermore, any supporting documentation, advertising materials,
 * and all other materials related to such a distribution and use must
 * acknowledge that the software was developed by Akihiro Tominaga and
 * the WIDE Project. The names of the author and the WIDE Project may
 * not be used to endorse or promote products derived from this software
 * without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <netdb.h>
#include <syslog.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#ifdef sun
#include <sys/stropts.h>
#endif
#include <net/if.h>
#ifdef sun
#include <net/nit.h>
#include <net/nit_if.h>
#include <net/nit_pf.h>
#include <net/packetfilt.h>
#else
#include <net/bpf.h>
#endif
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/if_ether.h>
#include <netinet/ip.h>
#include <netinet/udp.h>

#include "dhcp.h"
#include "dhcpc.h"
#include "common.h"

long generate_xid();
void reset_if();
#if !defined(sony_news) && !defined(__FreeBSD__)
int getmac();
#endif

int dhcp_errno;
int dhcp_log;
unsigned char magic_c[MAGIC_LEN] = RFC1048_MAGIC;
u_short dhcps_port, dhcpc_port;
struct msgbuf sndmsg;
struct msgbuf rcvmsg;
time_t init_epoch;
#ifdef sun
struct packetfilt arpf;
struct packetfilt dhcpf;
#else
/*
 * bpf filter program
 */
#ifdef sony_news
struct bpf_insn dhcpf[] = {
  BPF_STMT(LdHOp, 12),                    /* A <- ETHER_TYPE */
  BPF_JUMP(EQOp, ETHERTYPE_IP, 1, 10),    /* is it IP ? */
  BPF_STMT(LdHOp, 20),                    /* A <- IP FRAGMENT */
  BPF_STMT(AndIOp, 0x1fff),               /* A <- A & 0x1fff */
  BPF_JUMP(EQOp, 0, 1, 7),                /* OFFSET == 0 ? */
  BPF_STMT(LdBOp, 23),                    /* A <- IP_PROTO */
  BPF_JUMP(EQOp, IPPROTO_UDP, 1, 5),      /* UDP ? */
  BPF_STMT(LdxmsOp, 14),                  /* X <- IPHDR LEN */
  BPF_STMT(ILdHOp, 16),                   /* A <- UDP DSTPORT */
  BPF_JUMP(EQOp, 0, 1, 2),                /* check DSTPORT */
  BPF_STMT(RetOp, (u_int)-1),             /* return all*/
  BPF_STMT(RetOp, 0)                      /* ignore */
};
#define DSTPORTIS  9
#else
struct bpf_insn dhcpf[] = {
  BPF_STMT(BPF_LD+BPF_H+BPF_ABS, 12),                     /* A <- ETHER_TYPE */
  BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, ETHERTYPE_IP, 0, 8),    /* is it IP ? */
  BPF_STMT(BPF_LD+BPF_H+BPF_ABS, 20),                     /* A <- IP FRAGMENT */
  BPF_JUMP(BPF_JMP+BPF_JSET+BPF_K, 0x1fff, 6, 0),         /* OFFSET == 0 ? */
  BPF_STMT(BPF_LD+BPF_B+BPF_ABS, 23),                     /* A <- IP_PROTO */
  BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, IPPROTO_UDP, 0, 4),     /* UDP ? */
  BPF_STMT(BPF_LDX+BPF_B+BPF_MSH, 14),                    /* X <- IPHDR LEN */
  BPF_STMT(BPF_LD+BPF_H+BPF_IND, 16),                     /* A <- UDP DSTPORT */
  BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, 0, 0, 1),               /* check DSTPORT */
  BPF_STMT(BPF_RET+BPF_K, (u_int)-1),                     /* return all*/
  BPF_STMT(BPF_RET+BPF_K, 0)                              /* ignore */
};
#define DSTPORTIS  8
#endif /* sony_news */

struct bpf_program dhcpfilter = {
  sizeof(dhcpf) / sizeof(struct bpf_insn),
  dhcpf
};

#endif /* not sun */

/*
 * get service port for DHCP
 */
void
get_srvport()
{
  struct servent *srv = NULL;

  if ((srv = getservbyname("dhcpc", "udp")) != NULL) {
    dhcpc_port = srv->s_port;
  } else {
    Syslog(LOG_WARNING, "udp/dhcpc: unknown service -- use default port %d",
           DHCPC_PORT);
    dhcpc_port = htons(DHCPC_PORT);
  }
  srv = getservbyname("dhcps", "udp");
  if (srv != NULL) {
    dhcps_port = srv->s_port;
  } else {
    Syslog(LOG_WARNING, "udp/dhcps: unknown service -- use default port %d",
           DHCPS_PORT);
    dhcps_port = htons(DHCPS_PORT);
  }

  return;
}


#ifdef sun
struct dhcp_if *
dhcp_init(ifname, cltype)
  char *ifname;
  struct dhcp_clienttype *cltype;
{
  u_short *fwp;
  struct dhcp_if *intf;
  struct strioctl si;
  struct ifreq ifreq;

  dhcp_errno = SUCCESS;

#ifndef LOG_CONS
#define LOG_CONS    0
#endif
#ifndef LOG_PERROR
#define LOG_PERROR  0
#endif

  openlog("dhcpc", LOG_PID | LOG_CONS | LOG_PERROR, LOG_LOCAL0);

  if ((intf = (struct dhcp_if *) calloc(1, sizeof(struct dhcp_if))) == NULL) {
    Syslog(LOG_ERR, "calloc error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }

  bcopy(ifname, intf->name, sizeof(intf->name));
  get_srvport();
  reset_if(intf);

  if ((intf->fordhcp =
       (struct netif_info *) calloc(1, sizeof(struct netif_info))) == NULL) {
    Syslog(LOG_ERR, "calloc error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }
  if ((intf->forarp =
       (struct netif_info *) calloc(1, sizeof(struct netif_info))) == NULL) {
    Syslog(LOG_ERR, "calloc error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }

  intf->fordhcp->fd = open("/dev/nit", O_RDWR);
  if (intf->fordhcp->fd < 0) {
    Syslog(LOG_WARNING, "Can't open nit for DHCP in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }
  intf->forarp->fd = open("/dev/nit", O_RDWR);
  if (intf->forarp->fd < 0) {
    Syslog(LOG_WARNING, "Can't open nit for ARP in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }

  if (ioctl(intf->fordhcp->fd, I_SRDOPT, (char *)RMSGD) < 0) {
    Syslog(LOG_WARNING, "ioctl(I_SRDOPT) error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }
  if (ioctl(intf->forarp->fd, I_SRDOPT, (char *)RMSGD) < 0) {
    Syslog(LOG_WARNING, "ioctl(I_SRDOPT) error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }

  bzero(&si, sizeof(si));
  bzero(&ifreq, sizeof(ifreq));
  strncpy(ifreq.ifr_name, ifname, sizeof(ifreq.ifr_name));
  ifreq.ifr_name[sizeof(ifreq.ifr_name) - 1] = '\0';
  si.ic_cmd = NIOCBIND;
  si.ic_len = sizeof(ifreq);
  si.ic_dp = (char *) &ifreq;
  if (ioctl(intf->fordhcp->fd, I_STR, (char *) &si) < 0) {
    Syslog(LOG_WARNING, "ioctl(NIOCBIND) error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }
  bzero(&si, sizeof(si));
  bzero(&ifreq, sizeof(ifreq));
  strncpy(ifreq.ifr_name, ifname, sizeof(ifreq.ifr_name));
  ifreq.ifr_name[sizeof(ifreq.ifr_name) - 1] = '\0';
  si.ic_cmd = NIOCBIND;
  si.ic_len = sizeof(ifreq);
  si.ic_dp = (char *) &ifreq;
  if (ioctl(intf->forarp->fd, I_STR, (char *) &si) < 0) {
    Syslog(LOG_WARNING, "ioctl(NIOCBIND) error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }

  /* enable packet filter */
  if (ioctl(intf->forarp->fd, I_PUSH, "pf") < 0) {
    Syslog(LOG_WARNING, "ioctl(I_PUSH) error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }
  if (ioctl(intf->fordhcp->fd, I_PUSH, "pf") < 0) {
    Syslog(LOG_WARNING, "ioctl(I_PUSH) error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }
  fwp = &dhcpf.Pf_Filter[0];
  *fwp++ = ENF_PUSHWORD + 6;
  *fwp++ = ENF_PUSHLIT | ENF_CAND;
  *fwp++ = htons(ETHERTYPE_IP);
  *fwp++ = ENF_PUSHWORD + 11;
  *fwp++ = ENF_PUSHLIT | ENF_AND;
  *fwp++ = htons(0x00ff);
  *fwp++ = ENF_PUSHLIT | ENF_CAND;
  *fwp++ = htons(IPPROTO_UDP);
  dhcpf.Pf_FilterLen = fwp - &dhcpf.Pf_Filter[0];

  /* set packet filter */
  bzero(&si, sizeof(si));
  si.ic_cmd = NIOCSETF;
  si.ic_timout = INFTIM;
  si.ic_len = sizeof(dhcpf);
  si.ic_dp = (char *)&dhcpf;
  if (ioctl(intf->fordhcp->fd, I_STR, (char *)&si) < 0){
    Syslog(LOG_WARNING, "ioctl(NIOCSETF) error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }

  sndmsg.bufsize = ETHERHL + IPHL + UDPHL + DFLTDHCPLEN;
  /* allocate buffer for send DHCP message */
  if ((sndmsg.buf = (char *) calloc(1, sndmsg.bufsize + QWOFF)) == NULL) {
    Syslog(LOG_WARNING, "calloc error for sndmsg.buf in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }
  sndmsg.ether = (struct ether_header *) &sndmsg.buf[QWOFF];
  sndmsg.ip = (struct ip *) &sndmsg.buf[QWOFF + ETHERHL];
  sndmsg.udp = (struct udphdr *) &sndmsg.buf[QWOFF + ETHERHL + IPHL];
  sndmsg.dhcp = (struct dhcp *) &sndmsg.buf[QWOFF + ETHERHL + IPHL + UDPHL];

  if (cltype != NULL) {
    intf->cltype = cltype;
  } else {
    if ((intf->cltype = (struct dhcp_clienttype *)
	 calloc(1, sizeof(struct dhcp_clienttype))) == NULL) {
      Syslog(LOG_ERR, "calloc error in dhcp_init: %m");
      dhcp_errno = EMISC;
      return(NULL);
    }

    intf->cltype->client_id_type = ETHER;
    intf->cltype->client_id_len = 6;
    if ((intf->cltype->client_id_value =
	 calloc(1, intf->cltype->client_id_len)) == NULL) {
      Syslog(LOG_ERR, "calloc error in dhcp_init: %m");
      dhcp_errno = EMISC;
      return(NULL);
    }
    if (getmac(intf->name, intf->cltype->client_id_value) < 0) {
      Syslog(LOG_ERR, "getmac error in dhcp_init");
      dhcp_errno = EMISC;
      return(NULL);
    }
  }

  /* get haddr of interface */
  intf->haddr.htype = ETHER;
  intf->haddr.hlen = 6;
  if (getmac(intf->name, intf->haddr.haddr) < 0) {
    dhcp_errno = EMISC;
    return(NULL);
  }

  srandom(generate_xid(intf));
  return(intf);
}

#else /* sun */

struct dhcp_if *
dhcp_init(ifname, cltype)
  char *ifname;
  struct dhcp_clienttype *cltype;
{
  char dev[sizeof "/dev/bpf00"];
  int n;
  struct dhcp_if *intf;
  struct ifreq ifreq;
  struct timeval timeout;

  dhcp_errno = SUCCESS;

#ifndef LOG_CONS
#define LOG_CONS    0
#endif
#ifndef LOG_PERROR
#define LOG_PERROR  0
#endif

  openlog("dhcpc", LOG_PID | LOG_CONS | LOG_PERROR, LOG_LOCAL0);

  if ((intf = (struct dhcp_if *) calloc(1, sizeof(struct dhcp_if))) == NULL) {
    Syslog(LOG_ERR, "calloc error in dhcp_init");
    dhcp_errno = EMISC;
    return(NULL);
  }

  bcopy(ifname, intf->name, sizeof(intf->name));
  get_srvport();
  reset_if(intf);

  if ((intf->fordhcp =
       (struct netif_info *) calloc(1, sizeof(struct netif_info))) == NULL) {
    Syslog(LOG_ERR, "calloc error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }
  if ((intf->forarp =
       (struct netif_info *) calloc(1, sizeof(struct netif_info))) == NULL) {
    Syslog(LOG_ERR, "calloc error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }

  /* open bpf to read/write dhcp messages */
  n = 0;
  do {
    sprintf(dev, "/dev/bpf%d", n++);
    intf->fordhcp->fd = open(dev, O_RDWR);
  } while (intf->fordhcp->fd < 0 && n < NBPFILTER);
  if (intf->fordhcp->fd < 0) {
    Syslog(LOG_WARNING, "Can't open bpf read for DHCP in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }

  /* open bpf to read/write arp messages */
  n = 0;
  do {
    sprintf(dev, "/dev/bpf%d", n++);
    intf->forarp->fd = open(dev, O_RDWR);
  } while (intf->forarp->fd < 0 && n < NBPFILTER);
  if (intf->forarp->fd < 0) {
    Syslog(LOG_WARNING, "Can't open bpf for ARP in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }

  /* set immediate mode */
  n = 1;
  if (ioctl(intf->fordhcp->fd, BIOCIMMEDIATE, &n) < 0) {
    Syslog(LOG_WARNING, "ioctl(BIOCIMMEDIATE) error in dhcp_init: %m");
    return(NULL);
  }
  if (ioctl(intf->forarp->fd, BIOCIMMEDIATE, &n) < 0) {
    Syslog(LOG_WARNING, "ioctl(BIOCIMMEDIATE) error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }

  /* bind interface to the bpfs */
  bzero(&ifreq, sizeof(ifreq));
  strncpy(ifreq.ifr_name, ifname, sizeof(ifreq.ifr_name));
  ifreq.ifr_name[sizeof(ifreq.ifr_name) - 1] = '\0';
  if (ioctl(intf->fordhcp->fd, BIOCSETIF, &ifreq) < 0) {
    Syslog(LOG_WARNING, "ioctl(BIOCSETIF) error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }
  if (ioctl(intf->forarp->fd, BIOCSETIF, &ifreq) < 0) {
    Syslog(LOG_WARNING, "ioctl(BIOCSETIF) error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }

  /* get buffer length for read and allocate buffer actually */
  if (ioctl(intf->fordhcp->fd, BIOCGBLEN, &intf->fordhcp->rbufsize) < 0) {
    Syslog(LOG_WARNING, "ioctl(BIOCGBLEN) for dhcp: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }
  if ((intf->fordhcp->rbuf =
       (char *)calloc(1, intf->fordhcp->rbufsize + QWOFF)) == NULL) {
    Syslog(LOG_WARNING, "calloc error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }
  if (ioctl(intf->forarp->fd, BIOCGBLEN, &intf->forarp->rbufsize) < 0) {
    Syslog(LOG_WARNING, "ioctl(BIOCGBLEN) error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }
  if ((intf->forarp->rbuf =
       (char *)calloc(1, intf->forarp->rbufsize + QWOFF)) == NULL) {
    Syslog(LOG_WARNING, "calloc error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }

  dhcpf[DSTPORTIS].k = ntohs(dhcpc_port);

  /* set filter program to the interfaces */
  if (ioctl(intf->fordhcp->fd, BIOCSETF, &dhcpfilter) < 0) {
    Syslog(LOG_WARNING, "ioctl(BIOCSETF) error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }

  /* set timeout */
  timeout.tv_sec = 0;
  timeout.tv_usec = 500000;
  if (ioctl(intf->forarp->fd, BIOCSRTIMEOUT, &timeout) < 0) {
    syslog(LOG_WARNING, "ioctl(BIOCSRTIMEOUT) error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }

  sndmsg.bufsize = ETHERHL + IPHL + UDPHL + DFLTDHCPLEN;
  if ((sndmsg.buf = (char *) calloc(1, sndmsg.bufsize + QWOFF)) == NULL) {
    Syslog(LOG_WARNING, "calloc error in dhcp_init: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }
  sndmsg.ether = (struct ether_header *) &sndmsg.buf[QWOFF];
  sndmsg.ip = (struct ip *) &sndmsg.buf[QWOFF + ETHERHL];
  sndmsg.udp = (struct udphdr *) &sndmsg.buf[QWOFF + ETHERHL + IPHL];
  sndmsg.dhcp = (struct dhcp *) &sndmsg.buf[QWOFF + ETHERHL + IPHL + UDPHL];

  if (cltype != NULL) {
    intf->cltype = cltype;
  } else {
    if ((intf->cltype = (struct dhcp_clienttype *)
	 calloc(1, sizeof(struct dhcp_clienttype))) == NULL) {
      Syslog(LOG_ERR, "calloc error in dhcp_init: %m");
      dhcp_errno = EMISC;
      return(NULL);
    }

    intf->cltype->client_id_type = ETHER;
    intf->cltype->client_id_len = 6;
    if ((intf->cltype->client_id_value =
	 calloc(1, intf->cltype->client_id_len)) == NULL) {
      Syslog(LOG_ERR, "calloc error in dhcp_init: %m");
      dhcp_errno = EMISC;
      return(NULL);
    }
#if defined(sony_news) || defined(__FreeBSD__)
    if (ioctl(intf->fordhcp->fd, SIOCGIFADDR, &ifreq) < 0) {
      Syslog(LOG_ERR, "ioctl(SIOCGIFADDR) error in dhcp_init: %m");
      dhcp_errno = EMISC;
      return(NULL);
    }
    bcopy(ifreq.ifr_addr.sa_data, intf->cltype->client_id_value, 6);
#else
    if (getmac(intf->name, intf->cltype->client_id_value) < 0) {
      Syslog(LOG_ERR, "getmac error in dhcp_init");
      dhcp_errno = EMISC;
      return(NULL);
    }
#endif
  }

  /* get haddr of interface */
  intf->haddr.htype = ETHER;
  intf->haddr.hlen = 6;

#if defined(sony_news) || defined(__FreeBSD__)
  bcopy(ifreq.ifr_addr.sa_data, intf->haddr.haddr, 6);
#else
  if (getmac(intf->name, intf->haddr.haddr) < 0) {
    dhcp_errno = EMISC;
    return(NULL);
  }
#endif

  srandom(generate_xid(intf));
  return(intf);
}
#endif
