/*
 * Copyright (c) 1995 Akihiro Tominaga and the WIDE Project.
 * All rights reserved.
 *
 * Permission to use, copy and distribute this software in source and
 * binary forms is hereby granted provided that the above copyright
 * notice and this permission notice appear in all copies of the software,
 * derivative works or modified versions, and any portions thereof.
 * Furthermore, any supporting documentation, advertising materials,
 * and all other materials related to such a distribution and use must
 * acknowledge that the software was developed by Akihiro Tominaga and
 * the WIDE Project. The names of the author and the WIDE Project may
 * not be used to endorse or promote products derived from this software
 * without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <strings.h>
#include <fcntl.h>
#include <netdb.h>
#include <signal.h>
#include <errno.h>
#include <syslog.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#ifdef sun
#include <sys/stropts.h>
#endif
#include <net/if.h>
#ifdef sun
#include <net/nit.h>
#include <net/nit_if.h>
#include <net/nit_pf.h>
#include <net/packetfilt.h>
#endif
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/if_ether.h>
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <arpa/inet.h>

#include "dhcp.h"
#include "dhcpc.h"
#include "common.h"
#include "dhcpc_subr.h"
#include "config.h"

#define REQUEST_RETRANS 3
struct dhcp_param *_param;

/*
 * check OFFER. return 1 if there is no problem
 */
int
check_offer()
{
  char *option = NULL;
  u_short ripcksum = 0,
          rudpcksum = 0;
  struct ps_udph rpudph;

  /*
   * already checked at ptrtomsg(), and apply result at here
   *    ntohs(rcvmsg.ip->ip_len) >= DFLTBOOTPLEN + UDPHL + IPHL
   */
  if (rcvmsg.udp == NULL || rcvmsg.dhcp == NULL)
    return(0);

  bzero(&rpudph, sizeof(rpudph));

  ripcksum = rcvmsg.ip->ip_sum;
  rcvmsg.ip->ip_sum = 0;
  rudpcksum = rcvmsg.udp->uh_sum;
  rcvmsg.udp->uh_sum = 0;
  rpudph.zero = 0;
  rpudph.prto = IPPROTO_UDP;
  rpudph.srcip.s_addr = rcvmsg.ip->ip_src.s_addr;
  rpudph.dstip.s_addr = rcvmsg.ip->ip_dst.s_addr;
  rpudph.ulen = rcvmsg.udp->uh_ulen;

  /*
   * check the message.
   */

  /* check ip's checksum */
  if (ripcksum != cksum((u_short *) rcvmsg.ip, rcvmsg.ip->ip_hl * 2))
    return(0);

  /* check udp length */
  if (ntohs(rcvmsg.udp->uh_ulen) < DFLTBOOTPLEN + UDPHL)
    return(0);

#ifdef sun
  if (rcvmsg.udp->uh_dport != dhcpc_port)
    return(0);
#endif 

  /* check udp's checksum */
  if (rcvmsg.udp->uh_sum != 0 &&
      rudpcksum != udp_cksum(&rpudph, (char *) rcvmsg.udp, ntohs(rpudph.ulen)))
    return(0);

  /* check dhcp.op and dhcp.xid */
  if (rcvmsg.dhcp->op != BOOTREPLY || rcvmsg.dhcp->xid != sndmsg.dhcp->xid)
    return(0);

  /* and magic cookie */
  if (bcmp(rcvmsg.dhcp->options, magic_c, MAGIC_LEN) != 0)
    return(0);

  /* check DHCP message type */
  if ((option = (char *)
       pickup_opt(rcvmsg.dhcp, DHCPLEN(rcvmsg.udp), DHCP_MSGTYPE)) == NULL ||
      *OPTBODY(option) != DHCPOFFER)
    return(0);

  return(1);
}


/*
 * check reply to request. return 1 if there is no problem
 */
int
check_reply()
{
  u_short ripcksum = 0,
          rudpcksum = 0;
  struct ps_udph rpudph;

  /*
   * already checked at ptrtomsg(), and apply result at here
   *    ntohs(rcvmsg.ip->ip_len) >= DFLTBOOTPLEN + UDPHL + IPHL
   */
  if (rcvmsg.udp == NULL || rcvmsg.dhcp == NULL)
    return(0);

  bzero(&rpudph, sizeof(rpudph));

  ripcksum = rcvmsg.ip->ip_sum;
  rcvmsg.ip->ip_sum = 0;
  rudpcksum = rcvmsg.udp->uh_sum;
  rcvmsg.udp->uh_sum = 0;
  rpudph.zero = 0;
  rpudph.prto = IPPROTO_UDP;
  rpudph.srcip.s_addr = rcvmsg.ip->ip_src.s_addr;
  rpudph.dstip.s_addr = rcvmsg.ip->ip_dst.s_addr;
  rpudph.ulen = rcvmsg.udp->uh_ulen;

  /*
   * check the message.
   */

  /* check ip's checksum */
  if (ripcksum != cksum((u_short *) rcvmsg.ip, rcvmsg.ip->ip_hl * 2))
    return(0);

  /* check udp length */
  if (ntohs(rcvmsg.udp->uh_ulen) < DFLTBOOTPLEN + UDPHL)
    return(0);

#ifdef sun
  if (rcvmsg.udp->uh_dport != dhcpc_port)
    return(0);
#endif 

  /* check udp's checksum */
  if (rcvmsg.udp->uh_sum != 0 &&
      rudpcksum != udp_cksum(&rpudph, (char *) rcvmsg.udp, ntohs(rpudph.ulen)))
    return(0);

  /* check dhcp.op and dhcp.xid */
  if (rcvmsg.dhcp->op != BOOTREPLY || rcvmsg.dhcp->xid != sndmsg.dhcp->xid)
    return(0);

  /* and magic cookie */
  if (bcmp(rcvmsg.dhcp->options, magic_c, MAGIC_LEN) != 0)
    return(0);

  return(1);
}


struct dhcp_param *
initandwait(intf, optmap)
  struct dhcp_if *intf;
  struct dhcp_optmap *optmap;
{
  int timer;
  int retry;
  fd_set readfds;
  struct dhcp_param *paramp;

  if ((paramp = (struct dhcp_param *)
       calloc(1, sizeof(struct dhcp_param))) == NULL) {
    Syslog(LOG_WARNING, "calloc() error in dhcp_getparam: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }

  sndmsg.dhcp->secs = 0;
  sndmsg.udp->uh_sum = 0;
  sndmsg.udp->uh_sum = udp_cksum(&sndmsg.pudph, (char *) sndmsg.udp,
				 ntohs(sndmsg.pudph.ulen));
  if (time(&init_epoch) == -1) {
    Syslog(LOG_WARNING, "time() error in dhcp_getparam: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }
  outputfd = intf->fordhcp->fd;
  if (ether_write(outputfd, &sndmsg.buf[QWOFF], sndmsg.bufsize) <
      sndmsg.bufsize) {
    Syslog(LOG_WARNING, "Can't send DHCPDISCOVER");
    free(paramp);
    dhcp_errno = EMISC;
    return(NULL);
  }

  timer = FIRSTTIMER;
  retry = 0;
  timeout = 1;
  rcvmsg.remain = 0;
  while (1) {
    if (rcvmsg.remain <= 0) {
      if (timeout == 1) {
	timeout = 0;
	alarm(0);
	if (retry < DISCOVER_RETRANS) {
	  ualarm(((timer - 1) * 1000000) + (random() % 2000000), 0);
	  signal(SIGALRM, (void *) gen_retransmit);
	}
	else if (retry == DISCOVER_RETRANS) {
	  ualarm(((timer - 1) * 1000000) + (random() % 2000000), 0);
	  signal(SIGALRM, (void *) gen_timeout);
	} else {
	  Syslog(LOG_WARNING,
		 "Can't get reply to DHCPDISCOVER from any server");
	  free(paramp);
	  alarm(0);
	  dhcp_errno = ENOREPLY;
	  return(NULL);
	}
	retry++;
	if (timer < MAXTIMER) {
	  timer *= 2;
	}
      }

      /*
       * read from interface
       */
      FD_ZERO(&readfds);
      FD_SET(intf->fordhcp->fd, &readfds);
      if (select(intf->fordhcp->fd + 1, &readfds, NULL, NULL, NULL) < 0) {
	if (errno == EINTR) {
	  continue;
	}
      }
      else if (!FD_ISSET(intf->fordhcp->fd, &readfds) ||
	       (rcvmsg.remain = read(intf->fordhcp->fd,
				     &intf->fordhcp->rbuf[QWOFF],
				     intf->fordhcp->rbufsize)) < 0) {
	Syslog(LOG_WARNING, "can't read from interface: %m");
	alarm(0);
	free(paramp);
	dhcp_errno = EMISC;
	return(NULL);
      }

      rcvmsg.bptr = &intf->fordhcp->rbuf[QWOFF];
    }

    ptrtomsg(rcvmsg.bptr);
    mvptrtonext();
    if (check_offer() == 0) {   /* bad OFFER */
      continue;
    }

    if (dhcp_msgtoparam(rcvmsg.dhcp, DHCPLEN(rcvmsg.udp),
			paramp, optmap) == OK) {
      if (paramp->server_id != NULL && paramp->server_id->s_addr != 0) {
        alarm(0);
	paramp->lease_origin = init_epoch;
	return(paramp);
      } else {
	continue;
      }
    }
  }
}


struct dhcp_param *
selecting(intf, reqspec, firstparam, optmap)
  struct dhcp_if *intf;
  struct dhcp_reqspec *reqspec;
  struct dhcp_param *firstparam;
  struct dhcp_optmap *optmap;
{
  fd_set readfds;
  struct dhcp_param *result, tmpparam;

  bzero(&tmpparam, sizeof(tmpparam));
  result = firstparam;
  ualarm(reqspec->wait_usec, 0);
  signal(SIGALRM, (void *) gen_timeout);

  while (timeout == 0) {
    if (rcvmsg.remain <= 0) {

      /*
       * read from interface
       */
      FD_ZERO(&readfds);
      FD_SET(intf->fordhcp->fd, &readfds);
      if (select(intf->fordhcp->fd + 1, &readfds, NULL, NULL, NULL) < 0) {
	if (errno == EINTR) {
	  continue;
	}
      }
      else if (!FD_ISSET(intf->fordhcp->fd, &readfds) ||
	       (rcvmsg.remain = read(intf->fordhcp->fd,
				     &intf->fordhcp->rbuf[QWOFF],
				     intf->fordhcp->rbufsize)) < 0) {
	continue;
      }

      rcvmsg.bptr = &intf->fordhcp->rbuf[QWOFF];
    }

    ptrtomsg(rcvmsg.bptr);
    mvptrtonext();
    if (check_offer() == 0) {   /* bad OFFER */
      continue;
    }

    if (dhcp_msgtoparam(rcvmsg.dhcp, DHCPLEN(rcvmsg.udp),
			&tmpparam, optmap) == OK) {
      if (tmpparam.server_id != NULL && tmpparam.server_id->s_addr != 0) {
	if (tmpparam.lease_duration > result->lease_duration) {
	  clean_param(result);
	  *result = tmpparam;    /* it means firstparam = tmpparam */
	  result->lease_origin = init_epoch;
	} else {
	  clean_param(&tmpparam);
	}
      } else {
	continue;
      }
    }
  }

  alarm(0);
  return(result);
}


struct dhcp_param *
requesting(intf, reqspec, bestparam, optmap)
  struct dhcp_if *intf;
  struct dhcp_reqspec *reqspec;
  struct dhcp_param *bestparam;
  struct dhcp_optmap *optmap;
{
  char *option;
  char errmsg[255];
  int timer;
  int retry;
  fd_set readfds;
  struct dhcp_param *result, tmpparam;

  bzero(&tmpparam, sizeof(tmpparam));
  result = bestparam;
  if (result == NULL) {
    if ((result = (struct dhcp_param *)
	 calloc(1, sizeof(struct dhcp_param))) == NULL) {
      Syslog(LOG_WARNING, "calloc() error in requesting: %m");
      dhcp_errno = EMISC;
      return(NULL);
    }
  }
  sndmsg.udp->uh_sum = 0;
  sndmsg.udp->uh_sum = udp_cksum(&sndmsg.pudph, (char *) sndmsg.udp,
				 ntohs(sndmsg.pudph.ulen));

  outputfd = intf->fordhcp->fd;
  if (ether_write(outputfd, &sndmsg.buf[QWOFF], sndmsg.bufsize) <
      sndmsg.bufsize) {
    Syslog(LOG_WARNING, "Can't send DHCPREQUEST");
    clean_param(bestparam);
    free(bestparam);
    dhcp_errno = EMISC;
    return(NULL);
  }

  timer = FIRSTTIMER;
  retry = 0;
  timeout = 1;
  rcvmsg.remain = 0;
  while (1) {
    if (rcvmsg.remain <= 0) {
      if (timeout == 1) {
	timeout = 0;
	alarm(0);
	if (retry < REQUEST_RETRANS) {
	  ualarm(((timer - 1) * 1000000) + (random() % 2000000), 0);
	  signal(SIGALRM, (void *) req_retransmit);
	}
	else if (retry == REQUEST_RETRANS) {
	  ualarm(((timer - 1) * 1000000) + (random() % 2000000), 0);
	  signal(SIGALRM, (void *) gen_timeout);
	} else {
	  Syslog(LOG_WARNING,
		 "Can't get reply to DHCPREQUEST from the server");
	  clean_param(bestparam);
	  free(bestparam);
	  dhcp_errno = ENOREPLY;
	  return(NULL);
	}
	retry++;
	if (timer < MAXTIMER) {
	  timer *= 2;
	}
      }

      /*
       * read from interface
       */
      FD_ZERO(&readfds);
      FD_SET(intf->fordhcp->fd, &readfds);
      if (select(intf->fordhcp->fd + 1, &readfds, NULL, NULL, NULL) < 0) {
	if (errno == EINTR) {
	  continue;
	}
      }
      else if (!FD_ISSET(intf->fordhcp->fd, &readfds) ||
	       (rcvmsg.remain = read(intf->fordhcp->fd,
				     &intf->fordhcp->rbuf[QWOFF],
				     intf->fordhcp->rbufsize)) < 0) {
	continue;
      }

      rcvmsg.bptr = &intf->fordhcp->rbuf[QWOFF];
    }

    ptrtomsg(rcvmsg.bptr);
    mvptrtonext();
    if (check_reply() == 0) {   /* bad reply */
      continue;
    }

    if ((option =
	 pickup_opt(rcvmsg.dhcp, DHCPLEN(rcvmsg.udp), DHCP_MSGTYPE)) == NULL) {
      continue;
    }

    if (*OPTBODY(option) == DHCPNAK) {
      syslog(LOG_NOTICE, "Got DHCPNAK in requesting()");
      if ((option = (char *) pickup_opt(rcvmsg.dhcp, DHCPLEN(rcvmsg.udp),
					DHCP_ERRMSG)) != NULL &&
          nvttostr(OPTBODY(option), errmsg, (int) OPTLEN(option)) == 0) {
        syslog(LOG_WARNING, "DHCPNAK contains the error message \"%s\"",
	       errmsg);
      }
      clean_param(bestparam);
      free(bestparam);
      dhcp_errno = ENAK;
      return(NULL);
    }
    else if (*OPTBODY(option) != DHCPACK) {
      continue;
    }

    /* got DHCP ACK */

    if (dhcp_msgtoparam(rcvmsg.dhcp, DHCPLEN(rcvmsg.udp),
			&tmpparam, optmap) == OK) {
      if (reqspec->ck_ipdup == DHCP_CHKADDR_OFF ||
	  arp_check(&tmpparam.yiaddr, intf) == OK) {
	alarm(0);
	clean_param(result);
	*result = tmpparam;
	result->lease_origin = init_epoch;
	Syslog(LOG_INFO, "Got DHCPACK (IP = %s, duration = %d secs)",
	       inet_ntoa(result->yiaddr), result->lease_duration);
	arp_announce(&result->yiaddr, intf);
	dhcp_errno = SUCCESS;
	return(result);
      } else {
	dhcp_decline(intf, reqspec);
	clean_param(&tmpparam);
	clean_param(bestparam);
	free(bestparam);
	dhcp_errno = EIPDUP;
	return(NULL);
      }
    }
  }
}


struct dhcp_param *
dhcp_getparam(intf, optlist, reqspec, optmap)
  struct dhcp_if *intf;
  struct dhcp_optlist *optlist;
  struct dhcp_reqspec *reqspec;
  struct dhcp_optmap *optmap;
{
  struct dhcp_param *firstparam, *bestparam;

  alarm(0);
  if (_param != NULL) {
    clean_param(_param);
  }

  /* make a skeleton of DHCPDISCOVER */
  if (make_discover(intf, optlist, reqspec) != 0) {
    return(NULL);
  }

  if ((firstparam = initandwait(intf, optmap)) == NULL) {
    return(NULL);
  }
  if (reqspec->select_offer == DHCPOFFER_SELECT_LONG) {
    bestparam = selecting(intf, reqspec, firstparam, optmap);
  } else {
    bestparam = firstparam;
  }

  /* make a skeleton of DHCPREQEUST */
  reqspec->srvaddr.s_addr = bestparam->server_id->s_addr;
  reqspec->ipaddr.s_addr = bestparam->yiaddr.s_addr;
  if (make_request(intf, optlist, reqspec, REQUESTING) != 0) {
    return(NULL);
  }

  _param = requesting(intf, reqspec, bestparam, optmap);
  signal(SIGALRM, SIG_IGN);
  return(_param);
}
