/*
 * Copyright (c) 1995 Akihiro Tominaga and the WIDE Project.
 * All rights reserved.
 *
 * Permission to use, copy and distribute this software in source and
 * binary forms is hereby granted provided that the above copyright
 * notice and this permission notice appear in all copies of the software,
 * derivative works or modified versions, and any portions thereof.
 * Furthermore, any supporting documentation, advertising materials,
 * and all other materials related to such a distribution and use must
 * acknowledge that the software was developed by Akihiro Tominaga and
 * the WIDE Project. The names of the author and the WIDE Project may
 * not be used to endorse or promote products derived from this software
 * without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <strings.h>
#include <fcntl.h>
#include <netdb.h>
#include <signal.h>
#include <errno.h>
#include <syslog.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#ifdef sun
#include <sys/stropts.h>
#endif
#include <net/if.h>
#ifdef sun
#include <net/nit.h>
#include <net/nit_if.h>
#include <net/nit_pf.h>
#include <net/packetfilt.h>
#endif
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/if_ether.h>
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <arpa/inet.h>

#include "dhcp.h"
#include "dhcpc.h"
#include "common.h"
#include "dhcpc_subr.h"

extern struct dhcp_param *_param;

int check_reply();

struct dhcp_param *
verify_extend(intf, reqspec, prevparam, optmap)
  struct dhcp_if *intf;
  struct dhcp_reqspec *reqspec;
  struct dhcp_param *prevparam;
  struct dhcp_optmap *optmap;
{
  char *option;
  char errmsg[255];
  fd_set readfds;
  struct dhcp_param *result, tmpparam;

  bzero(&tmpparam, sizeof(tmpparam));
  result = prevparam;
  if (result == NULL) {
    if ((result = (struct dhcp_param *)
	 calloc(1, sizeof(struct dhcp_param))) == NULL) {
      Syslog(LOG_WARNING, "calloc() error in verify_extend: %m");
      dhcp_errno = EMISC;
      return(NULL);
    }
  }
  sndmsg.udp->uh_sum = 0;
  sndmsg.udp->uh_sum = udp_cksum(&sndmsg.pudph, (char *) sndmsg.udp,
				 ntohs(sndmsg.pudph.ulen));

  outputfd = intf->fordhcp->fd;

  if (reqspec->extend_phase == RENEWING) {
    if (send_unicast(&reqspec->srvaddr, sndmsg.dhcp) < 0) {
      Syslog(LOG_WARNING, "Can't send DHCPREQUEST");
      dhcp_errno = EMISC;
      return(NULL);
    }
  } else {
    if (ether_write(outputfd, &sndmsg.buf[QWOFF], sndmsg.bufsize) <
	sndmsg.bufsize) {
      Syslog(LOG_WARNING, "Can't send DHCPREQUEST: %m");
      dhcp_errno = EMISC;
      return(NULL);
    }
  }

  timeout = 0;
  rcvmsg.remain = 0;
  alarm(0);
  ualarm(reqspec->wait_usec, 0);
  signal(SIGALRM, (void *) gen_timeout);
  while (timeout == 0) {
    if (rcvmsg.remain <= 0) {

      /*
       * read from interface
       */
      FD_ZERO(&readfds);
      FD_SET(intf->fordhcp->fd, &readfds);
      if (select(intf->fordhcp->fd + 1, &readfds, NULL, NULL, NULL) < 0) {
	if (errno == EINTR) {
	  dhcp_errno = EMISC;
	  continue;
	}
      }
      else if (!FD_ISSET(intf->fordhcp->fd, &readfds) ||
	       (rcvmsg.remain = read(intf->fordhcp->fd,
				     &intf->fordhcp->rbuf[QWOFF],
				     intf->fordhcp->rbufsize)) < 0) {
	dhcp_errno = EMISC;
	continue;
      }

      rcvmsg.bptr = &intf->fordhcp->rbuf[QWOFF];
    }

    ptrtomsg(rcvmsg.bptr);
    mvptrtonext();
    if (check_reply() == 0) {   /* bad reply */
      continue;
    }

    if ((option =
	 pickup_opt(rcvmsg.dhcp, DHCPLEN(rcvmsg.udp), DHCP_MSGTYPE)) == NULL) {
      dhcp_errno = EMISC;
      continue;
    }

    if (*OPTBODY(option) == DHCPNAK) {
      syslog(LOG_NOTICE, "Got DHCPNAK in verify_extend()");
      if ((option = (char *) pickup_opt(rcvmsg.dhcp, DHCPLEN(rcvmsg.udp),
					DHCP_ERRMSG)) != NULL &&
          nvttostr(OPTBODY(option), errmsg, (int) OPTLEN(option)) == 0) {
        syslog(LOG_WARNING, "DHCPNAK contains the error message \"%s\"",
	       errmsg);
      }
      alarm(0);
      dhcp_errno = ENAK;
      return(NULL);
    }
    else if (*OPTBODY(option) != DHCPACK) {
      continue;
    }

    /* got DHCP ACK */

    if (dhcp_msgtoparam(rcvmsg.dhcp, DHCPLEN(rcvmsg.udp),
			&tmpparam, optmap) == OK) {
      if (reqspec->ck_ipdup == DHCP_CHKADDR_OFF ||
	  arp_check(&tmpparam.yiaddr, intf) == OK) {
	alarm(0);
	clean_param(result);
	*result = tmpparam;
	result->lease_origin = init_epoch;
	Syslog(LOG_INFO, "Got DHCPACK (IP = %s, duration = %d secs)",
	       inet_ntoa(result->yiaddr), result->lease_duration);
	arp_announce(&result->yiaddr, intf);
	dhcp_errno = SUCCESS;
	return(result);
      } else {
	dhcp_decline(intf, reqspec);
	clean_param(&tmpparam);
	alarm(0);
	dhcp_errno = EIPDUP;
	return(NULL);
      }
    }
  }

  dhcp_errno = ENOREPLY;
  alarm(0);
  return(NULL);
}


struct dhcp_param *
dhcp_extend(intf, optlist, reqspec, optmap)
  struct dhcp_if *intf;
  struct dhcp_optlist *optlist;
  struct dhcp_reqspec *reqspec;
  struct dhcp_optmap *optmap;
{
  time_t curr_epoch;
  static int first_time = 1;
  struct dhcp_param *result;

  alarm(0);
  if (time(&curr_epoch) == -1) {
    Syslog(LOG_WARNING, "time() error in dhcp_extend: %m");
    dhcp_errno = EMISC;
    return(NULL);
  }

  if (reqspec->extend_phase != RENEWING &&
      reqspec->extend_phase != REBINDING) {
    dhcp_errno = EMISC;
    return(NULL);
  }

  if (first_time == 1) {
    first_time = 0;
    if (time(&init_epoch) == -1) {
      Syslog(LOG_WARNING, "time() error in dhcp_extend: %m");
      dhcp_errno = EMISC;
      return(NULL);
    }
  }

  /* make a skeleton of DHCPREQEUST */
  if (make_request(intf, optlist, reqspec, reqspec->extend_phase) != 0) {
    return(NULL);
  }

  sndmsg.dhcp->secs = htons(curr_epoch - init_epoch);
  result = verify_extend(intf, reqspec, NULL, optmap);
  if (result != NULL) {
    first_time = 1;
    if (_param != NULL) {
      clean_param(_param);
      free(_param);
    }
  }
  signal(SIGALRM, SIG_IGN);
  _param = result;

  return(_param);
}
