# get-sumrn: an alternate version of get by Bob Sum (sumrn@dssv01.crd.ge.com)
#
# This service retrieve files from an archive.  Patterns describing the desired
# files are in switches, or in the single body; all matching archive files are
# bundled into the output message
#
#########################################################################
#                                                                       #
# Copyright (C) 1993 General Electric. All rights reserved.             #
#                                                                       #
# Permission to use, copy, modify, and distribute this                  #
# software and its documentation for any purpose and without            #
# fee is hereby granted, provided that the above copyright              #
# notice appear in all copies and that both that copyright              #
# notice and this permission notice appear in supporting                #
# documentation, and that the name of General Electric not be used in   #
# advertising or publicity pertaining to distribution of the            #
# software without specific, written prior permission.                  #
#                                                                       #
# General Electric makes no representations about the suitability of    #
# this software for any purpose.  It is provided ``as is''              #
# without express or implied warranty.                                  #
#                                                                       #
# This work was supported by the DARPA Initiative in Concurrent         #
# Engineering (DICE) through DARPA Contract MDA972-92-C-0027.           #
#                                                                       #
#########################################################################
#
# Enhancements initially by Robert Sum (sumrn@crd.ge.com) for the
# Microwave and Millimeter-wave Pilot Sites (MMPS) DICE program.
#
# $Id: archive-request.tcl,v 1.8 1993/02/16 01:18:39 sumrn Exp $
#
# $Log: archive-request.tcl,v $
# Revision 1.8  1993/02/16  01:18:39  sumrn
# Files are returned in alpha sorted order.
#
# Revision 1.7  1993/02/15  17:25:49  sumrn
# New setmimetype (with audio).
#
# Revision 1.6  1993/02/10  21:02:53  sumrn
# Just starting to periodically include my setmimetype from ArchiveServices.
# It has more type stuff in it.
#
# Revision 1.5  1993/02/10  15:41:31  sumrn
# Added file extension recognition to for gif and jp[e]g.
#
# Revision 1.4  1993/02/09  22:10:45  sumrn
# Enhanced setmimetype with respect to C, Fortran, tcl, and a couple others
# to add name of the originating file and use x-subtype.
#
# Revision 1.3  1993/01/22  23:37:00  sumrn
# 1. Fixed a bug where switches/request from body were not handled
# properly.  (My own bug.)
#
# Revision 1.2  1993/01/22  19:08:55  sumrn
# 1. Enhanced the error messages returned to the requestor so that
# if anything file searches go wrong he is notified.
# (The only exception might be special devices, but there should not be
# any of those in the archive anyway, right?)
# 2. Fixed bug of trying to send non-existent files which would happen
# if a fixed string is handed to glob.
# 3. Enhanced (I think at the moment anyway) the error checking for patterns
# that try to stray outside the archive for files.
#
#

proc dofetch {switches envelope inputs} {

	set messages ""

	#
	# if no switches in subject, get from body--if any.
	#
	if {[llength $switches] == 0} {
		set switches [exec cat [getfield $inputs FILE]]
	}

	#
	# Determine request, use info.txt if no switches
	#
	set request $switches
	if {[llength $request] == 0} then {
		set request "info.txt"
		set messages \
			"$messages\nNo specific request:  Sending information."
	}

	#
	# change to archive directory
	#
	cd ~/archive

	#
	# check that files are within the archive
	#
	set hits {}
	foreach pattern $request {

		#
		# check obvious straying outside the archive
		#
		if { [string match /* $pattern]
				|| [string match ~* $pattern]
				|| [string match ../* $pattern]
				|| [regexp /\.\./ $pattern]
		} then {
			set messages \
			"$messages\nImproper pattern:  $pattern."
			continue
		}

		#
		# expand to almost get the actual files
		#
		set expansions [glob -nocomplain $pattern]
		if { $expansions == {} } then {
			set messages \
				"$messages\nNo match: $pattern."
			continue
		}

		#
		# check for not so obvious wandering
		#
		foreach expan $expansions {
			# check straying outside the archive
			if { [string match /* $expan]
					|| [string match ../* $expan]
					|| [regexp /\.\./ $expan]
			} then {
				set messages \
				"$messages\nImproper pattern:  $pattern."
				break
			}
			set hits [concat $hits $expan]
		}
	}

	#
	# check that files exist and are processable:
	#   globbing without pattern can let non-file through, and
	#   only simple files can be sent.
	#
	set filelist {}
	foreach h $hits {
		if {![file exists $h] } then {
			set messages \
				"$messages\nNo such file: $h."
			continue
		}
		if {![file isfile $h]} then {
			set messages \
				"$messages\nFile can not be processed: $h."
			continue
		}
		set filelist [concat $filelist $h]
	}

	#
	# Note:  One could regard returning error messages about searches
	#   as a security risk.  The messages here take a modest effort
	#   to use to determine file existence external to the archive.
	#   Benevolent users will much appreciate them, however.
	#
	set filelist [lsort $filelist]
	case [llength $filelist] {
	0 {
		setfield response DESCRIPTION "No files filled your request."
		set messages "$messages\nEnd of messages.\n"
		setfield response \
			STRING "Messages for request: $switches.\n$messages"
	}
	1 {
		setfield response DESCRIPTION "the archive file you requested"
		if { $messages != "" } then {

			set messages "$messages\nEnd of messages.\n"

			setfield response TYPE multipart
			setfield response SUBTYPE mixed
			set parts {}

			set part {}
			setfield part TYPE text
			setfield part STRING \
				"Messages for request: $switches.\n$messages"
			lappend parts $part

			set part {}
			setfield part FILE $filelist
			setmimetype part
			lappend parts $part

			setfield response PARTS $parts
		} else {
			setfield response FILE $filelist;
			setmimetype response
		}
	}
	default {
		setfield response TYPE multipart
		setfield response SUBTYPE mixed
		setfield response DESCRIPTION "the archive files you requested"
		set parts {}
		if { $messages != "" } then {

			set messages "$messages\nEnd of messages.\n"

			set part {}
			setfield part TYPE text
			setfield part STRING \
				"Messages for request: $switches.\n$messages"
			lappend parts $part
		}
		foreach f $filelist {
			set part {}
			setfield part FILE $f
			setmimetype part
			lappend parts $part
		}
		setfield response PARTS $parts
	}
	}

	return [mailout [turnaround $envelope] $response]
}

# Id: setmimetype.tcl,v 1.5 1993/02/15 17:22:12 sumrn Exp
#
# Enhancements initially by Robert Sum (sumrn@crd.ge.com) for the
# Microwave and Millimeter-wave Pilot Sites (MMPS) DICE program.
#
#
# Log: setmimetype.tcl,v
# Revision 1.5  1993/02/15  17:22:12  sumrn
# Added audio type to filename typing.
#
# Revision 1.4  1993/02/10  20:35:49  sumrn
# Added type information for Express (*.exp,text/x-express),
# tex dvi output  (*.dvi,image/x-dvi), Framemaker (*.mif,text/x-frame),
# and AutoCAD dxf (*.dxf,text/x-dxf).
#
# Revision 1.3  1993/02/10  15:44:03  sumrn
# Added file extension recognition to for gif and jp[e]g.
#
# Revision 1.2  1993/02/09  22:22:25  sumrn
# Enhanced setmimetype with respect to C, Fortran, tcl, and a couple others
# to add name of the originating file and use x-subtype.
#
# Revision 1.1  1993/02/04  19:43:09  sumrn
# Initial revision
#
# setmimetype
#   sets the appropriate type information for a file.
#   input is the name of a variable that contains a message part needed
#   type information.
#
proc setmimetype {objectname} {

	# set up filename as call-by-name
	upvar $objectname object

	set filename [getfield $object FILE]
	case $filename {
	*.au {
		setfield object TYPE audio
		setfield object SUBTYPE basic
		setfield params name $filename
		setfield object PARAMS $params
	}
	*.c {
		setfield object TYPE text
		setfield object SUBTYPE x-c
		setfield params charset us-ascii
		setfield params name $filename
		setfield object PARAMS $params
	}
	*.dvi {
		setfield object TYPE image
		setfield object SUBTYPE x-dvi
		setfield params name $filename
		setfield object PARAMS $params
	}
	*.dxf {
		setfield object TYPE text
		setfield object SUBTYPE x-dxf
		setfield params charset us-ascii
		setfield params name $filename
		setfield object PARAMS $params
	}
	*.exp {
		setfield object TYPE text
		setfield object SUBTYPE x-express
		setfield params charset us-ascii
		setfield params name $filename
		setfield object PARAMS $params
	}
	{*.f *.ftn} {
		setfield object TYPE text
		setfield object SUBTYPE x-fortran
		setfield params charset us-ascii
		setfield params name $filename
		setfield object PARAMS $params
	}
	{*.GIF *.gif} {
		setfield object TYPE image
		setfield object SUBTYPE gif
		setfield params name $filename
		setfield object PARAMS $params
	}
	{*.JPG *.JPEG *.jpg *.jpeg} {
		setfield object TYPE image
		setfield object SUBTYPE jpeg
		setfield params name $filename
		setfield object PARAMS $params
	}
	*.mif {
		setfield object TYPE text
		setfield object SUBTYPE x-frame
		setfield params charset us-ascii
		setfield params name $filename
		setfield object PARAMS $params
	}
	*.ps {
		setfield object TYPE application
		setfield object SUBTYPE postscript
	}
	*.sh {
		setfield object TYPE application
		setfield object SUBTYPE x-sh
	}
	*.tar.Z {
		setfield object TYPE application
		setfield object SUBTYPE octet-stream
		setfield params name $filename
		setfield params type tar
		setfield params conversions compress
		setfield object PARAMS $params
	}
	*.tar {
		setfield object TYPE application
		setfield object SUBTYPE octet-stream
		setfield params name $filename
		setfield params type tar
		setfield object PARAMS $params
	}
	*.tex {
		setfield object TYPE text
		setfield object SUBTYPE x-latex
		setfield params charset us-ascii
		setfield params name $filename
		setfield object PARAMS $params
	}
	*.tcl {
		setfield object TYPE text
		setfield object SUBTYPE x-tcl
		setfield params charset us-ascii
		setfield params name $filename
		setfield object PARAMS $params
	}
	}
}
