#! /usr/bin/perl
#
# This is a primitive script for benchmarking a NFS server implementation.
# It's not rocket science, it does not use anything fancy like characteristic
# traffic patterns etc, and it does not cover all NFS calls (much less
# common mixtures of NFS calls). It only servers to provide a rough estimate
# so that I'm able to locate potential bottlenecks when modifying the server.
#
# Copyright (C) 1995 O. Kirch <okir@monad.swb.de>

$flat_dirs	= 2000;
$creat_files	= 500;
$write_files	= 100;
$write_size	= 4;		# KBytes
$rec_levels	= 5;
$rec_dirs	= 4;

die ("usage: nfsbench directory\n") if ($#ARGV != 0);
($target) = @ARGV;

die ("$target is not a directory!\n")
	unless (-d $target);
$target="$target/nfsbench";

print "\nBenchmark           :    user     system\n";
&bench_mkdir($target, $flat_dirs);
&bench_rmdir($target, $flat_dirs);

&bench_creat($target, $creat_files);
&bench_write($target, $creat_files, $write_size);
&bench_read($target, $creat_files, $write_size);
&delete_hierarchy($target);

&bench_mkdir_rec($target,  $rec_levels, $rec_dirs);
&bench_lookup_rec($target, $rec_levels, $rec_dirs);
&bench_rmdir_rec($target,  $rec_levels, $rec_dirs);


sub benchmark {

	local ($tag, $cmd) = @_;
	local (@now, @then);

	@then = times;
	eval "$cmd" || warn("failed to eval $cmd\n");
	@now  = times;
	printf ("%-20s: %7.2f    %7.2f\n",
			$tag,
			(@now)[0] - (@then)[0],
			(@now)[1] - (@then)[1]
			);
}

sub bench_mkdir {

	local ($target, $nodes) = @_;

	&benchmark(
		"mkdir ($nodes)", 
		"&create_hierarchy(\"$target\", 1, $nodes, \"\")"
		);
}

sub bench_rmdir {

	local ($target, $nodes) = @_;

	&benchmark(
		"rmdir ($nodes)",
		"&delete_hierarchy(\"$target\")"
		);
}

sub bench_creat {

	local ($target, $nodes) = @_;

	&benchmark(
		"creat ($nodes)",
		"&write_files(\"$target\", $nodes, 0)"
		);
}

sub bench_write {

	local ($target, $nodes, $size) = @_;

	&benchmark(
		"write ($nodes x $size K)",
		"&write_files(\"$target\", $nodes, $size * 1024)"
		);
}

sub bench_read {

	local ($target, $nodes, $size) = @_;

	&benchmark(
		"read  ($nodes x $size K)",
		"&read_files(\"$target\", $nodes, $size * 1024)"
		);
}

sub bench_mkdir_rec {

	local ($target, $levels, $nodes) = @_;

	&benchmark(
		"mkdir rec ($levels x $nodes)", 
		"&create_hierarchy(\"$target\", $levels, $nodes, \"\")"
		);
}

sub bench_lookup_rec {

	local ($target, $levels, $nodes) = @_;

	&benchmark(
		"lookup rec ($levels x $nodes)", 
		"&traverse_hierarchy(\"$target\", \"\", \"\")"
		);
}

sub bench_rmdir_rec {

	local ($target, $levels, $nodes) = @_;

	&benchmark(
		"rmdir rec ($levels x $nodes)", 
		"&delete_hierarchy(\"$target\")"
		);
}

#
# Create directory hierarchy
#
sub create_hierarchy {
	local ($target, $levels, $nodes, $leafcmd) = @_;
	local ($subdir, $tmp, $j);

	mkdir ($target, 0755) || die("could not mkdir $target: $!\n");
	$levels--;

	for ($j = 0; $j < $nodes; $j++) {
		$subdir="$target/$j";

		if ($levels > 0) {
			&create_hierarchy($subdir, $levels, $nodes, $leafcmd);
		} elsif ("$leafcmd" ne "") {
			eval "$leafcmd($subdir, $levels-1, $nodes, $leafcmd)";
		}
	}

	1;
}

#
# Traverse directory hierarchy
#
sub traverse_hierarchy {

	local ($target, $dircmd, $filecmd) = @_;
	# local (DIR);

	opendir(DIR, "$target") || die ("Unable to read $target: $!.");
	local (@files) = readdir(DIR);
	closedir(DIR);

	# print "traverse($target, $dircmd, $filecmd)\n";
	for (@files) {
		$object="$target/$_";
		next if ($_ eq "." || $_ eq "..");
		if (-d $object) {
			if ("$dircmd" ne "") {
				eval "$dircmd(\"$object\");" || 
					print "failed to exec $dircmd: $!\n";
				next;
			}

			&traverse_hierarchy($object, $dircmd, $filecmd);
		} elsif (-f $object) {
			if ("$filecmd" ne "") {
				eval "$filecmd(\"$object\");" || 
					print "failed to exec $filecmd: $!\n";
				next;
			}
		}
	}

	1;
}

#
# Delete directory hierarchy
#
sub delete_hierarchy {

	local($target) = @_;

	&traverse_hierarchy($target, "&delete_hierarchy", "unlink");
	rmdir $target;

	1;
}

#
# Write a number of files in a directory
#
sub write_files {

	local ($target, $nodes, $size) = @_;
	local ($j, $file, $buffer);

	mkdir ($target, 0755);
	if ($size > 0) {
		$buffer = pack("a$size", "");
	}
	for ($j = 0; $j < $nodes; $j++) {
		$file = "$target/$j";
		unless (open(FILE, ">$file")) {
			warn("failed to open $file for writing: $!\n");
			return 0;
		}
		if ($size > 0) {
			$written = syswrite(FILE, $buffer, $size, 0);
			warn ("write error on $file: $!\n")
				unless defined $written;
		}
		close(FILE);
	}

	1;
}

#
# Read a number of files in a directory
#
sub read_files {

	local ($target, $nodes, $size) = @_;
	local ($j, $file, $buffer);

	for ($j = 0; $j < $nodes; $j++) {
		$file = "$target/$j";
		unless (open(FILE, "<$file")) {
			warn("failed to open $file for reading: $!\n");
			return 0;
		}
		if ($size > 0) {
			$read = sysread(FILE, $buffer, $size, 0);
			warn ("read error on $file: $!\n")
				unless defined $read;
		}
		close(FILE);
	}

	1;
}
