/*
 *   $Id: device-linux.c,v 1.3 1997/01/16 21:14:38 lf Exp $
 *
 *   Authors:
 *    Lars Fenneberg		<lf@elemental.net>	 
 *
 *   This software is Copyright 1996 by the above mentioned author(s), 
 *   All Rights Reserved.
 *
 *   The license which is distributed with this software in the file COPYRIGHT
 *   applies to this software. If your distribution is missing this file, you
 *   may request it from <lf@elemental.net>.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet6/in6.h>
#include <netinet6/ipv6.h>

#ifdef HAVE_LINUX_IF_ARP_H
#include <linux/if_arp.h>
#endif

#include "radvd.h"
#include "defaults.h"
#include "pathnames.h"		/* for PATH_PROC_NET_IF_INET6 */

#ifndef IPV6_ADDR_LINKLOCAL
#define IPV6_ADDR_LINKLOCAL   0x0020U
#endif

/*
 * this function gets the hardware type and address of an interface,
 * determines the link layer token length and checks it against
 * the defined prefixes
 */
int
setup_deviceinfo(int sock, struct Interface *iface)
{
	struct ifreq	ifr;
	struct AdvPrefix *prefix;
	
	strcpy(ifr.ifr_name, iface->Name);
	
	if (ioctl(sock, SIOCGIFHWADDR, &ifr) < 0)
	{
		log(LOG_ERR, "ioctl(SIOCGIFHWADDR) failed for %s: %s",
			iface->Name, strerror(errno));
		return (-1);
	}

	dlog(LOG_DEBUG, 3, "hardware type for %s is %d", iface->Name,
		ifr.ifr_hwaddr.sa_family); 

	switch(ifr.ifr_hwaddr.sa_family)
        {
	case ARPHRD_ETHER:
		iface->if_tokenlen = 48;
		iface->if_maxmtu = 1500;
		break;
#ifdef ARPHRD_FDDI
	case ARPHRD_FDDI:
		iface->if_tokenlen = 48;
		iface->if_maxmtu = 4352;
		break;
#endif /* ARPHDR_FDDI */
#ifdef ARPHRD_ARCNET
	case ARPHRD_ARCNET:
		iface->if_tokenlen = 8;
		iface->if_maxmtu = -1;
		break;
#endif /* ARPHDR_ARCNET */
	default:
		iface->if_tokenlen = -1;
		iface->if_maxmtu = -1;
		break;
	}

	dlog(LOG_DEBUG, 3, "link layer token length for %s is %d", iface->Name,
		iface->if_tokenlen);

	if (iface->if_tokenlen != -1)
		memcpy(iface->if_hwaddr, ifr.ifr_hwaddr.sa_data, (iface->if_tokenlen + 7) >> 3);

	prefix = iface->AdvPrefixList;
	while (prefix)
	{
		if ((iface->if_tokenlen != -1) &&
		   ((iface->if_tokenlen + prefix->PrefixLen) != MAX_PrefixLen))
		{
			log(LOG_WARNING, "prefix length should be %d for %s",
				MAX_PrefixLen - iface->if_tokenlen, iface->Name);
 			return (-1);
 		}
 			
 		prefix = prefix->next;
	}
                
	return (0);
}

/*
 * this function extracts the link local address and interface index
 * from PATH_PROC_NET_IF_INET6
 */
int setup_linklocal_addr(int sock, struct Interface *iface)
{
	FILE *fp;
	char str_addr[40];
	int plen, scope, dad_status, if_idx;
	char devname[IFNAMSIZ];

	if ((fp = fopen(PATH_PROC_NET_IF_INET6, "r")) == NULL)
	{
		log(LOG_ERR, "can't open %s: %s", PATH_PROC_NET_IF_INET6,
			strerror(errno));
		return (-1);	
	}
	
	while (fscanf(fp, "%32s %02x %02x %02x %02x %s\n",
		      str_addr, &if_idx, &plen, &scope, &dad_status,
		      devname) != EOF)
	{
		if (scope == IPV6_ADDR_LINKLOCAL &&
		    strcmp(devname, iface->Name) == 0)
		{
			struct in6_addr addr;
			unsigned int ap;
			int i;
			
			for (i=0; i<16; i++)
			{
				sscanf(str_addr + i * 2, "%02x", &ap);
				addr.s6_addr[i] = (unsigned char)ap;
			}
			memcpy(&iface->if_addr, &addr,
			       sizeof(struct in6_addr));

			iface->if_index = if_idx;
			fclose(fp);
			return 0;
		}
	}

	log(LOG_ERR, "no linklocal address configured for %s", iface->Name);
	fclose(fp);
	return (-1);
}
