/*
 *   $Id: device.c,v 1.3 1996/12/21 18:04:20 lf Exp $
 *
 *   Authors:
 *    Lars Fenneberg		<lf@elemental.net>	 
 *
 *   This software is Copyright 1996 by the above mentioned author(s), 
 *   All Rights Reserved.
 *
 *   The license which is distributed with this software in the file COPYRIGHT
 *   applies to this software. If your distribution is missing this file, you
 *   may request it from <lf@elemental.net>.
 *
 */

#include <syslog.h>
#include <errno.h>

#include <sys/ioctl.h>
#include <linux/if_arp.h>

#include "radvd.h"
#include "defaults.h"

int ether_fillin(struct Interface *, void *);

static struct Device DeviceList[] = {
	{ARPHRD_ARCNET	,	 8,	0,		NULL},
	{ARPHRD_ETHER	,	48,  1500,	ether_fillin},
#ifdef ARPHRD_FDDI
	{ARPHRD_FDDI	,	48,  4352,	ether_fillin},
#endif
	{-1, -1, -1, NULL}
};

/*
 * this function gets the hardware type and address of an interface,
 * determines the link layer token length and checks it against
 * the defined prefixes
 */
int
setup_deviceinfo(int sock, struct Interface *iface)
{
	int i;
	struct ifreq	ifr;
	struct AdvPrefix *prefix;
	
	strcpy(ifr.ifr_name, iface->Name);
	
	if (ioctl(sock, SIOCGIFHWADDR, &ifr) < 0)
	{
		log(LOG_ERR, "ioctl(SIOCGIFHWADDR) failed for %s: %s",
			iface->Name, strerror(errno));
		return (-1);
	}

	dlog(LOG_DEBUG, 3, "hardware type for %s is %d", iface->Name,
		ifr.ifr_hwaddr.sa_family); 
	
	memcpy(&iface->if_hwaddr, &ifr.ifr_hwaddr, sizeof(struct sockaddr));
	
	for(i=0; DeviceList[i].ifhwtype != -1; i++)
		if (iface->if_hwaddr.sa_family == DeviceList[i].ifhwtype)
			break;

	if (i != -1)
	{
		dlog(LOG_DEBUG, 3, "link layer token length for %s is %d", iface->Name, 
			DeviceList[i].tokenlen);
		
		iface->dev = &DeviceList[i];
	}
	else
	{
		iface->dev = NULL;
	}

	if (iface->dev && (iface->dev->tokenlen != -1))
	{
		prefix = iface->AdvPrefixList;
		while (prefix)
		{
			if ((iface->dev->tokenlen + prefix->PrefixLen) 
					!= MAX_PrefixLen)
			{
				
				log(LOG_WARNING, "prefix length should be %d"
					" for %s", MAX_PrefixLen - iface->dev->tokenlen,
					iface->Name);
				return (-1);
			}
			prefix = prefix->next;
		}
	}
	else
	{
		log(LOG_WARNING, "can't determine lenght of link layer token"
			" for %s", iface->Name);
	}
		
	return (0);
}

int
check_device(int sock, struct Interface *iface)
{
	struct ifreq	ifr;
	
	strcpy(ifr.ifr_name, iface->Name);
	
	if (ioctl(sock, SIOCGIFFLAGS, &ifr) < 0)
	{
		log(LOG_ERR, "ioctl(SIOCGIFFLAGS) failed for %s: %s", 
			iface->Name, strerror(errno));
		return (-1);
	}

	if (!(ifr.ifr_flags & IFF_UP))
	{
		log(LOG_ERR, "interface %s is not UP", iface->Name);
		return (-1);
	}
	
	if (!(ifr.ifr_flags & IFF_MULTICAST))
	{
		log(LOG_WARNING, "interface %s does not support multicast",
			iface->Name);
	}

	if (!(ifr.ifr_flags & IFF_BROADCAST))
	{
		log(LOG_WARNING, "interface %s does not support broadcast",
			iface->Name);
	}

	return 0;
}

/*
 * this function fills in an ethernet hardware address.
 * (return_value + 2) must always be a multiple of eight.
 */

int
ether_fillin(struct Interface *iface, void *buff)
{
	memcpy(buff, iface->if_hwaddr.sa_data, 6);
	return 6;
}
