/* 
 * $Id: config.c,v 1.11 1997/03/21 10:32:52 masaki Exp $
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <mrt.h>
#include <trace.h>
#include <io.h>
#include <select.h>
#include <stdarg.h>
#include <interface.h>
#include <bgp.h>
#include "config.h"
#include "bgpsim.h"

/* globals */
config_master_t CONFIG;

int config_ip_route (uii_connection_t *uii);
int config_router_network (uii_connection_t *uii);
int config_router_neighbor (uii_connection_t *uii);
int bgp_debug (uii_connection_t *uii);
int parse_line (char *line, char *format, ...);
network_t* config_find_network (int num);

/* Comments...
 * Config can work from a file or from user configuration via a
 * socket.
 */

/* start_config
 * call by uii thread. change state so all future uii commands are
 * processed by process_config
 */
void start_config (uii_connection_t *uii) {
  
  uii->state = 2;
  CONFIG.state = 0;
  CONFIG.sockfd = uii->sockfd;
}



/* config_from_file
 */
int config_from_file (char *filename) {
  FILE *fd;

  uii_connection_t *uii = New (uii_connection_t);
  uii->state = 2;
  MRT->config_file_name = strdup (filename);

  CONFIG.line = 0;

  if ((fd = fopen (filename, "r")) == NULL)
    return (-1);

  while (fgets ((char *)uii->buffer, MAXLINE, fd) != 0) {
    CONFIG.line++;
    /* process_config (0, line);*/
    uii_proccess_command (uii);
  }

  Delete (uii);
  
  fclose (fd);
  return (0);
}


int config_router_bgp (uii_connection_t *uii) {
  u_int as;

  parse_line (uii->cp, "%d", &as);
  set_BGP (BGP_MY_AS, as, NULL);

  trace (NORM, BGP->trace, "CONFIG BGP AS%d\n", as);
  uii->state = 3;
  return (1);
}

int config_network_list (uii_connection_t *uii) {
  prefix_t *p;
  int num, stability, change, set, localpref;
  network_t *network;
  char aspath[100];
  char community[100];
  u_int as, val;

  network = New (network_t);
  network->ll_bgp_routes = LL_Create (0);
  
  if (parse_line (uii->cp, "%d range %p %p", &network->num, 
		  &network->start_prefix,
		  &network->end_prefix) >= 3) {
    LL_Add (SIMULATION.ll_networks, network);
    return (1);
  }
  else {
    LL_Destroy (network->ll_bgp_routes);
    Delete (network);
  }
  
  /* set next hop */
  if (parse_line (uii->cp, "%d nexthop %d %p", &num, &set, &p) >= 2) {
    if ((network = config_find_network (num)) != NULL) {
      network->nexthops[set] = (prefix_t *) copy_prefix (p);
      if (set > network->max) network->max = set;
      return (1);
    }
  }

  /* set aspath */
  if (parse_line (uii->cp, "%d aspath %d %S", &num, &set, aspath) >= 2) {
    if ((network = config_find_network (num)) != NULL) {
      /*network->nexthop = copy_prefix (p);*/
      network->aspaths[set] = (aspath_t *) 
	aspth_from_string (aspath);
      if (set > network->max) network->max = set;
      return (1);
    }
  }

  /* community */
  if (parse_line (uii->cp, "%d community %d %S", &num, &set, community) >= 2) {
    if ((network = config_find_network (num)) != NULL) {
      if (network->communities[set] = (community_t *) 
	community_from_string (community))
        if (set > network->max) network->max = set;
      return (1);
    }
  }

  /* stability */
  if (parse_line (uii->cp, "%d stability %d", &num, &stability) >= 2) {
    if ((network = config_find_network (num)) != NULL) {
      network->stability = stability;
      network->timer_stability = 
	New_Timer ((void *) network_schedule_stability,
		   network->stability,
		   "Network Stability", (void *) network);
      return (1);
    }
  }

  /* set dpa */
  if (parse_line (uii->cp, "%d dpa %d as %d %d", &num, &set, &as, &val) >= 3) {
    if ((network = config_find_network (num)) != NULL) {
      network->dpas[set] = (dpa_t *) New_DPA (as, val);
      if (set > network->max) network->max = set;
      return (1);
    }
  }

  /* localpref */
  if (parse_line (uii->cp, "%d localpref %d %d", &num, &set, &localpref) >= 2) {
    if ((network = config_find_network (num)) != NULL) {
      network->localprefs[set] = localpref;
      if (set > network->max) network->max = set;
      return (1);
    }
  }


  if (parse_line (uii->cp, "%d change %d", &num, &change) >= 2) {
    if ((network = config_find_network (num)) != NULL) {
      network->change_interval = change;
      network->timer_change = New_Timer ((void *) network_schedule_change,
					 network->change_interval,
					 "Network Change", (void *) network);
      return (1);
    }
  }

  /* unrecognized option */
  return (-1);
}


/* config_router_neighbor 
 * just used by BGP to configure peer (at least for now)
 */
int config_router_neighbor (uii_connection_t *uii) {
  u_int as, weight, view = 0;
  prefix_t *prefix;
  char tmp[MAXLINE];
  bgp_peer_t *peer;
  
  if (parse_line (uii->cp, "%p remote-as %d view %d", 
		  &prefix, &as, &view) > 2) {
    peer = Add_BGP_Peer (prefix, (short ) as, BGP->trace);
    trace (NORM, BGP->trace, "CONFIG BGP neighbor %s AS%d\n",
	   rinet_ntoa (tmp, MAXLINE, prefix), as);

    /* by default, all peers use view 0 */
    if (view == 0) {return (1);}

    if ((view >32) || (view <0)) {
      trace (NORM, BGP->trace, 
	     "*ERROR* CONFIG invalid view %d for BGP neighbor %s\n",
	     view, rinet_ntoa (tmp, MAXLINE, prefix));
      uii_send_data (uii, "Invalid view %d [0-32]\r\n", view);
      return (-1);
    }

    /* a new view */
    BGP->views[view] = New_View (32);
    peer->view = view;
  }

  /* neighbor x.x.x.x weight xx */
  if (parse_line (uii->cp, "%p weight %d", &prefix, &weight) >= 3) {
    trace (NORM, BGP->trace, "CONFIG BGP neighbor %s weight %d\n",
	   rinet_ntoa (tmp, MAXLINE, prefix), weight);
    return (1);
  }

  if (parse_line (uii->cp, "%p transparent-as", &prefix) >= 2) {
    gateway_t * gateway;
    trace (NORM, BGP->trace, "CONFIG BGP neighbor %s transparent-as\n",
	   rinet_ntoa (tmp, MAXLINE, prefix), weight);
    gateway = (gateway_t *) find_gateway (prefix, 0);
    peer = (bgp_peer_t *) find_bgp_peer (gateway);

    if (peer != NULL) {
      peer->options |= BGP_TRANSPARENT_AS;
      return (1);
    }
  }

  return (-1);
}


int show_config (uii_connection_t *uii) {
  char tmp[100];
  FILE *fd;
  
  if ((fd = fopen (MRT->config_file_name, "r")) == NULL) {return (-1);}

  while (fgets (tmp, 100, fd)) {
    uii_send_data (uii, "%s\r", tmp);
  }
  
}


int config_comment (uii_connection_t *uii) {
  if (strlen ((const char *)uii->cp) < 2) {
    trace (NORM, BGP->trace, "COMMENT -- STATE EOF %s\n", uii->cp);
    uii->state = 2;
  }
  
  trace (NORM, BGP->trace, "COMMENT %s\n", uii->cp);
  return (1);
}

int init_config () {

  uii_add_command (1, "show config", (void *) show_config);

  /* config top level */
  uii_add_command (2, "!", (void *) config_comment);
  uii_add_command (2, "router bgp", (void *) config_router_bgp);
  uii_add_command (2, "network-list", (void *) config_network_list);

  /* router [rip|bgp|ripng] */
  uii_add_command (3, "neighbor", (void *) config_router_neighbor);
  uii_add_command (3, "!", (void *) config_comment);

  return (1);
}





network_t* config_find_network (int num) {
  network_t *network;

  LL_Iterate (SIMULATION.ll_networks, network) {
    if (network->num == num)
      return (network);
  }

  return (NULL);
}
