/*
 *     $Id: linux.c,v 1.4 1997/03/21 09:12:00 masaki Exp $
 */

#include <config.h>

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <syslog.h>
#include <fcntl.h>

#include <sys/socket.h>
#include <sys/socketio.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <linux/route.h>

#include <version.h>
#include <New.h>
#include <linked_list.h>
#include <interface.h>
#include <mrt.h>
#include <trace.h>
#include <select.h>
#include <api6.h>

#ifdef HAVE_IPV6
#include <netinet6/in6.h>
#include <netinet6/ipv6.h>
#include <netinet6/ipv6_route.h>
#include <netinet6/icmpv6.h>
#include <asm/bitops.h>
#endif /* HAVE_IPV6 */

/*
 *     read_interfaces called from lib init routine
 *     must call new_interface to register configuration.
 */

interface_t *ifstatus (char *name);
int ifaddress (interface_t * interface);
#ifdef HAVE_IPV6
void iface_init6 ();
#endif /* HAVE_IPV6 */

void
read_interfaces ()
{
    struct ifconf ifc;
    struct ifreq *ifptr, *end;
    int s, num;
    char *buffer;
    interface_t *interface;

    if ((s = socket (AF_INET, SOCK_DGRAM, 0)) < 0) {
	trace (ERROR, INTERFACE_MASTER->trace,
	       "socket for AF_INET failed (%s)\n", strerror (errno));
	return;
    }

    for (num = 64; ;num = (num << 1)) {
	buffer = (char *) NewArray (struct ifreq, num);
        ifc.ifc_len = num * sizeof (struct ifreq);
        ifc.ifc_buf = buffer;

        if (ioctl (s, SIOCGIFCONF, (char *) &ifc) < 0) {
	    trace (ERROR, INTERFACE_MASTER->trace, "SIOCGIFCONF failed: %s\n",
	           strerror (errno));
	    close (s);
	    return;
        }

	if (ifc.ifc_len < num * sizeof (struct ifreq))
	    break;
	Delete (buffer);
    }
    close (s);

    end = (struct ifreq *) (ifc.ifc_buf + ifc.ifc_len);
    ifptr = ifc.ifc_req;

    while (ifptr < end) {
	interface = ifstatus (ifptr->ifr_name);
	if (interface == NULL)
	    continue;
	(void) ifaddress (interface);
	ifptr++;
    }
    Delete (buffer);
#ifdef HAVE_IPV6
    iface_init6 ();
#endif /* HAVE_IPV6 */
}


/* get interface configuration */
interface_t *
ifstatus (char *name)
{
    struct ifreq ifr;
    int s;
    u_long flags, mtu, index = 0;
    interface_t *interface;

    memcpy (ifr.ifr_name, name, sizeof (ifr.ifr_name));

    if ((interface = find_interface_byname (ifr.ifr_name)))
	return (interface);

    if ((s = socket (AF_INET, SOCK_DGRAM, 0)) < 0) {
	trace (ERROR, INTERFACE_MASTER->trace,
	       "socket for AF_INET failed (%s)\n", strerror (errno));
	return (NULL);
    }

    if (ioctl (s, SIOCGIFFLAGS, (caddr_t) & ifr) < 0) {
	trace (ERROR, INTERFACE_MASTER->trace,
	       "SIOCGIFFLAGS failed for %s (%s)\n", ifr.ifr_name,
	       strerror (errno));
	close (s);
	return (NULL);
    }
    assert (sizeof (ifr.ifr_flags) == 2);
    flags = ifr.ifr_flags & 0x0000ffff;		/* short */

    /*
     * I hate this kind of coding, but no way for now.
     * tunnel vif seems to be something like a p-to-p if,
     * but it needs special treatment
     */
    if (strncmp (name, "sit", 3) == 0)
	flags |= IFF_TUNNEL;

    if (ioctl (s, SIOCGIFMTU, (caddr_t) & ifr) < 0) {
	trace (ERROR, INTERFACE_MASTER->trace,
	       "SIOCSIFMTU failed for %s, use 576 instead (%s)\n",
	       ifr.ifr_name, strerror (errno));
	mtu = 576;
    }
    else {
	mtu = ifr.ifr_mtu;
    }

#ifdef HAVE_IPV6
#ifdef SIOGIFINDEX
    close (s);
    if ((s = socket (AF_INET6, SOCK_DGRAM, 0)) < 0) {
	trace (ERROR, INTERFACE_MASTER->trace,
	       "socket for AF_INET6 failed (%s)\n", strerror (errno));
	return (NULL);
    }

    if (ioctl (s, SIOGIFINDEX, (caddr_t) & ifr) < 0) {
	trace (ERROR, INTERFACE_MASTER->trace,
	       "SIOGIFINDEX failed for %s, being assigned by mrt (%s)\n",
	       ifr.ifr_name, strerror (errno));
    }
    else {
	index = ifr.ifr_ifindex;
	assert (index != 0);
    }
#endif /* SIOGIFINDEX */
#endif /* HAVE_IPV6 */

    close (s);
    interface = new_interface (name, flags, mtu, index);
    return (interface);
}


/* get interface configuration for IPv4 */
int
ifaddress (interface_t * interface)
{
    struct ifreq ifr;
    struct sockaddr_in addr, mask, dest;
    int s;

    if ((s = socket (AF_INET, SOCK_DGRAM, 0)) < 0) {
	trace (ERROR, INTERFACE_MASTER->trace,
	       "socket for AF_INET failed (%s)\n", strerror (errno));
	return (-1);
    }

    strncpy (ifr.ifr_name, interface->name, sizeof (ifr.ifr_name));

    if (ioctl (s, SIOCGIFADDR, (caddr_t) & ifr) < 0) {
#ifdef notdef
	trace (ERROR, INTERFACE_MASTER->trace,
	       "SIOCGIFADDR failed for %s (%s)\n", ifr.ifr_name,
	       strerror (errno));
#endif
	close (s);
	return (-1);
    }
    memcpy (&addr, &ifr.ifr_addr, sizeof (addr));

    /*
     * for now, tunnel vif has only destination address
     */
    if (!BIT_TEST (interface->flags, IFF_TUNNEL) &&
	(addr.sin_family != AF_INET || addr.sin_addr.s_addr == INADDR_ANY)) {
	close (s);
	return (1);
    }

    if (ioctl (s, SIOCGIFNETMASK, (caddr_t) & ifr) < 0) {
	trace (ERROR, INTERFACE_MASTER->trace,
	       "SIOCGIFNETMASK failed for %s (%s)\n", ifr.ifr_name,
	       strerror (errno));
	close (s);
	return (-1);
    }
    memcpy (&mask, &ifr.ifr_addr, sizeof (mask));

    if (BIT_TEST (interface->flags, IFF_POINTOPOINT)) {
	if (ioctl (s, SIOCGIFDSTADDR, (caddr_t) & ifr) < 0) {
	    trace (ERROR, INTERFACE_MASTER->trace,
		   "SIOCGIFDSTADDR failed for %s (%s)\n", ifr.ifr_name,
		   strerror (errno));
	    close (s);
	    return (-1);
	}
    }
    else {
	if (ioctl (s, SIOCGIFBRDADDR, (caddr_t) & ifr) < 0) {
	    trace (ERROR, INTERFACE_MASTER->trace,
		   "SIOCGIFBRDADDR failed for %s (%s)\n", ifr.ifr_name,
		   strerror (errno));
	    close (s);
	    return (-1);
	}
    }
    memcpy (&dest, &ifr.ifr_addr, sizeof (dest));

    add_addr_to_interface (interface, AF_INET,
			   (char *) &addr.sin_addr.s_addr,
			   mask2len ((char *) &mask.sin_addr.s_addr, 4),
			   (char *) &dest.sin_addr.s_addr);
    close (s);
    return (1);
}


/*
 *     read intefaces via /proc
 *     it will also be possible to do it via a SIOCGIFCONF-like ioctl
 *     but the API is still being worked out.
 *     (ioctl isn't, IMHO, a better method than /proc anyway)
 */

#ifdef HAVE_IPV6
#define PROC_PATH "/proc/net/if_inet6"
void
iface_init6 ()
{
    FILE *fp;
    char str_addr[40];
    int plen, scope, dad_status, if_idx;
    char devname[10];
    interface_t *interface;

    if ((fp = fopen (PROC_PATH, "r")) == NULL) {
	trace (ERROR, INTERFACE_MASTER->trace,
	       "fopen (%s) failed: (%s)\n", PROC_PATH, strerror (errno));
	return;
    }

    while (fscanf (fp, "%32s %02x %02x %02x %02x %s\n",
		   str_addr, &if_idx, &plen, &scope, &dad_status,
		   devname) != EOF) {
	struct in6_addr addr;
	int i, x;

	for (i = 0; i < 16; i++) {
	    sscanf (str_addr + i * 2, "%02x", &x);
	    addr.s6_addr[i] = x & 0xff;
	}
	interface = find_interface_byname (devname);
	assert (interface);
	assert (interface->index == if_idx);
	if (plen == 0)		/* may be fixed in the future */
	    plen = 128;
	add_addr_to_interface (interface, AF_INET6, (char *) &addr, plen, 0);
    }
    fclose (fp);
}

#endif /* HAVE_IPV6 */


#ifdef HAVE_IPV6
#define ICMPV6_FILTER_WILLPASS(type, filterp) \
	 (test_bit(type, filterp) == 0)

#define ICMPV6_FILTER_WILLBLOCK(type, filterp) \
	 test_bit(type, filterp)

#define ICMPV6_FILTER_SETPASS(type, filterp) \
	 clear_bit(type & 0x1f, &((filterp)->data[type >> 5]))

#define ICMPV6_FILTER_SETBLOCK(type, filterp) \
	 set_bit(type & 0x1f, &((filterp)->data[type >> 5]))

#define ICMPV6_FILTER_SETPASSALL(filterp) \
	 memset(filterp, 0, sizeof(struct icmp6_filter));

#define ICMPV6_FILTER_SETBLOCKALL(filterp) \
	 memset(filterp, 0xFF, sizeof(struct icmp6_filter));

#endif /* HAVE_IPV6 */

#define MRT_RT_ADD     0
#define MRT_RT_CHANGE  1
#define MRT_RT_DELETE  2


static int rt4_fd = -1;
static int ctl_sk_4 = -1;
#ifdef HAVE_IPV6
static int rt6_fd = -1;
static int rt6_sk = -1;
static int ctl_sk_6 = -1;
#endif /* HAVE_IPV6 */


/* Pedro:
 *     masaki: maybe the select interface could have (mask, void *) 
 *     as args so that the code could now in advance which even
 *     awoke select. Not critical at all.
 */
static void krt4_rtmsg_rcv (void *);
#ifdef HAVE_IPV6
static void krt6_rtmsg_rcv (void *);
static void krt6_icmp_rcv (void *);
#endif /* HAVE_IPV6 */

#if 0

/* Pedro:
 *     Maybe move this to include/select.h
 */

int MRT_kernel_add_route (int family, void *dest, int destlen, void *nexthop,
			  int index);
int MRT_kernel_del_route (int family, void *dest, int destlen, void *nexthop);
#endif

#define SEL_MASK_READ  1
#define SEL_MASK_WRITE  2
#define SEL_MASK_ERR   4

int
kernel_init (void)
{
#ifdef HAVE_IPV6
    struct icmp6_filter filter;
#endif /* HAVE_IPV6 */
    int mask = (SEL_MASK_READ | SEL_MASK_ERR);
    int val = 1;
    int err;

    rt4_fd = open ("/dev/route", O_RDWR);
    if (rt4_fd == -1) {
	trace (ERROR, MRT->trace, "[krt_in4]: error opening"
	       "route control device: %s\n"
	       "/dev/route should be major 36, minor 0\n",
	       strerror (errno));
    }
    else {
	err = ioctl (rt4_fd, FIONBIO, &val);
	if (err) {
	    trace (ERROR, MRT->trace, "[krt_in4]: error in ioctl %s\n",
		   strerror (errno));
	}
	else {
	    select_add_fd (rt4_fd, mask, krt4_rtmsg_rcv, NULL);
	}
    }

    ctl_sk_4 = socket (AF_INET, SOCK_DGRAM, 0);
    if (ctl_sk_4 < 0) {
	trace (ERROR, MRT->trace, "[krt_in4]: error creating "
	       "control socket: %s\n", strerror (errno));
    }

#ifdef HAVE_IPV6
    rt6_fd = open ("/dev/ipv6_route", O_RDWR);
    if (rt6_fd == -1) {
	trace (ERROR, MRT->trace, "[krt_in6]: error opening"
	       "route control device: %s\n"
	       "/dev/in6_route should be major 36, minor 11\n",
	       strerror (errno));
    }
    else {
	err = ioctl (rt6_fd, FIONBIO, &val);
	if (err) {
	    trace (ERROR, MRT->trace, "[krt_in6]: error in ioctl %s\n", 
                   strerror (errno));
	}
	else {
	    select_add_fd (rt6_fd, mask, krt6_rtmsg_rcv, NULL);
	}
    }

    rt6_sk = socket (AF_INET6, SOCK_RAW, IPPROTO_ICMPV6);
    if (rt6_sk < 0) {
	trace (ERROR, MRT->trace, "[krt_in6]: icmpv6 socket %s\n",
	       strerror (errno));
    }
    else {
	ICMPV6_FILTER_SETBLOCKALL (&filter);
	ICMPV6_FILTER_SETPASS (ICMPV6_DEST_UNREACH, &filter);
#ifdef ND6_REDIRECT
	ICMPV6_FILTER_SETPASS (ND6_REDIRECT, &filter);
#endif
	err = setsockopt (rt6_sk, IPPROTO_ICMPV6, ICMPV6_FILTER, &filter,
			  sizeof (struct icmp6_filter));

	if (err < 0) {
	    trace (ERROR, MRT->trace, "[krt_in6]: icmpv6 filter %s\n",
		   strerror (errno));
	}
	else {
	    select_add_fd (rt6_sk, mask, krt6_icmp_rcv, NULL);
	}
    }

    ctl_sk_6 = socket (AF_INET6, SOCK_DGRAM, 0);
    if (ctl_sk_6 < 0) {
	trace (ERROR, MRT->trace, "[krt_in6]: error creating "
	       "control socket: %s\n", strerror (errno));
    }

#endif /* HAVE_IPV6 */
    return 0;
}


#ifdef HAVE_IPV6

#define ADDRCOPY(D, S) memcpy(D, S, sizeof(struct in6_addr));

/*
 *     This is a temporary version until i code the moral equivalent of
 *     a routing socket in Linux IPv6 code.
 */

static int 
krt_update_v6 (int cmd, prefix_t * dest, prefix_t * nexthop,
	       int index, int metric)
{
    struct in6_rtmsg rt;
    int ret;

    memset (&rt, 0, sizeof (struct in6_rtmsg));

    ADDRCOPY (&rt.rtmsg_dst, prefix_tochar (dest));
    ADDRCOPY (&rt.rtmsg_gateway, prefix_tochar (nexthop));
    rt.rtmsg_prefixlen = dest->bitlen;

#define SAFE_DEFAULT_ROUTE
#ifdef SAFE_DEFAULT_ROUTE
    if (dest->bitlen == 0) {
	/* This is only for Provider-Based Unicast Address */
	/* Geographic-Based Unicast Addresses and 
	   other reserved address spaces are not included */
	memset (&rt.rtmsg_dst, 0, sizeof (rt.rtmsg_dst));
	rt.rtmsg_dst.s6_addr[0] = 0x40;
	rt.rtmsg_prefixlen = 3;
/* This will cause inconsistency with a route read from the kernel */
/* should move to somewhere */
    }
#endif
    rt.rtmsg_metric = metric;
    rt.rtmsg_flags = RTF_UP;
    /*
       * Pedro suggests that the route doesn't have a gateway
       * if the gateway address is in_addr_any (all zero)
     */
    if (!ipv6_any_addr (&rt.rtmsg_gateway)) {
	rt.rtmsg_flags |= RTF_GATEWAY;
    }
    if (dest->bitlen == 128)
	rt.rtmsg_flags |= RTF_HOST;

    if (cmd == MRT_RT_ADD) {

	rt.rtmsg_ifindex = index;

#ifdef notdef
/*
 * Linux ipv6 requires special treatment for tunnel vif.
 * replace a gateway with its ipv4 address and specify the vif
 */
	if ((interface = find_interface_byindex (index))
	    && BIT_TEST (interface->flags, IFF_TUNNEL)
	    && BIT_TEST (interface->flags, IFF_POINTOPOINT)) {
	    /* replace the nexthop with a compatible address of 
	       configured tunnel destination */
	    ADDRCOPY (&rt.rtmsg_gateway, prefix_tochar (nexthop));
	    memset (&rt.rtmsg_gateway.s6_addr, 0, 12);
	    memcpy (&rt.rtmsg_gateway.s6_addr + 12,
		    prefix_tochar (interface->primary->broadcast), 4);
	}
	else
#endif
	    ADDRCOPY (&rt.rtmsg_gateway, prefix_tochar (nexthop));
    }
#if 0
    printf ("DEST: %s/%d ", inet_ntop (AF_INET6, &rt.rtmsg_dst, tmp6, sizeof tmp6),
	    rt.rtmsg_prefixlen);
    printf ("GATE: %s ", inet_ntop (AF_INET6, &rt.rtmsg_gateway, tmp6, sizeof tmp6));
    if (rt.rtmsg_device[0])
	printf ("DEVICE: %s ", rt.rtmsg_device);
    printf ("\n");
#endif

    switch (cmd) {
    case MRT_RT_ADD:
	rt.rtmsg_type = RTMSG_NEWROUTE;
	break;

    case MRT_RT_DELETE:
	rt.rtmsg_type = RTMSG_DELROUTE;
	break;

    default:
	trace (ERROR, MRT->trace, "[krt_linux]: "
	       "unknown or unimplemented command\n");
	return -1;
    };

    if (rt6_fd >= 0) {
	ret = write (rt6_fd, &rt, sizeof (rt));
	if (ret == -1) {
	    trace (ERROR, MRT->trace, "[krt_v6_write]: %s\n",
		   strerror (errno));
	}
    }
    if (rt6_fd < 0 || (rt6_fd >= 0 && ret < 0)) {	/* try old interface */
	int s_op = (rt.rtmsg_type == RTMSG_NEWROUTE) ? SIOCADDRT : SIOCDELRT;
	assert (ctl_sk_6 >= 0);
	assert (s_op == SIOCADDRT || s_op == SIOCDELRT);
	if ((ret = ioctl (ctl_sk_6, s_op, &rt)) < 0) {
	    trace (ERROR, MRT->trace, "[krt_in6]: error in ioctl: %s\n",
		   strerror (errno));
	}
    }
    return (ret);
}
#endif /* HAVE_IPV6 */


static int 
krt_update_v4 (int cmd, prefix_t * dest, prefix_t * nexthop,
	       int index, int metric)
{
    struct rtentry rt;
    int s_op = 0;
    struct sockaddr_in *sin;

    memset (&rt, 0, sizeof (struct rtentry));
    sin = (struct sockaddr_in *) &rt.rt_dst;
    memcpy (&sin->sin_addr, prefix_tochar (dest), 4);
    sin->sin_family = AF_INET;
    sin = (struct sockaddr_in *) &rt.rt_gateway;
    memcpy (&sin->sin_addr, prefix_tochar (nexthop), 4);
    sin->sin_family = AF_INET;
    sin = (struct sockaddr_in *) &rt.rt_genmask;
    len2mask (dest->bitlen, (char *) &sin->sin_addr, 4);
    sin->sin_family = 0;
    rt.rt_metric = metric;

    rt.rt_flags = RTF_UP;
    if (((struct sockaddr_in *) &rt.rt_gateway)->sin_addr.s_addr !=
	INADDR_ANY) {
	rt.rt_flags |= RTF_GATEWAY;
    }
    if (dest->bitlen == 32)
	rt.rt_flags |= RTF_HOST;


    switch (cmd) {
    case MRT_RT_ADD:
	s_op = SIOCADDRT;
	break;

    case MRT_RT_DELETE:
	s_op = SIOCDELRT;
	break;

    default:
	trace (ERROR, MRT->trace, "[krt_linux]: "
	       "unknown or unimplemented command\n");
	return -1;
    };

    if (ioctl (ctl_sk_4, s_op, &rt) < 0) {
	trace (ERROR, MRT->trace, "[krt_in]: error in ioctl: %s\n",
	       strerror (errno));
	return -1;
    }
    return 1;
}


int 
kernel_update_route (int cmd, prefix_t * dest,
		     prefix_t * nexthop, int index /* , int metric */ )
{
    int res = -1;
    int metric = 1;

    switch (dest->family) {
    case AF_INET:
	res = krt_update_v4 (cmd, dest, nexthop, index, metric);
	break;
#ifdef HAVE_IPV6
    case AF_INET6:
	res = krt_update_v6 (cmd, dest, nexthop, index, metric);
	break;
#endif /* HAVE_IPV6 */
    default:
	trace (ERROR, MRT->trace, "[krt_linux]: "
	       "unknown protocol family\n");
    }

    return res;
}


static void 
krt4_rtmsg_rcv (void *arg)
{
#ifdef notdef
    struct netlink_rtinfo rt;
    int err;

    if ((err = read (rt4_fd, &rt, sizeof (struct netlink_rtinfo))) > 0) {
	switch (rt.rtmsg_type) {
	case RTMSG_NEWROUTE:
	case RTMSG_DELROUTE:
	case RTMSG_NEWDEVICE:
	case RTMSG_DELDEVICE:
	    trace (NORM, MRT->trace, "[krt_rtmsg_rcv]: "
		   "message type %d\n", rt.rtmsg_type);
	    break;
	default:
	    trace (ERROR, MRT->trace, "[krt_rtmsg_rcv]: "
		   "unknown message type %d\n", rt.rtmsg_type);
	}
    }

    if (err == -1) {
	trace (ERROR, MRT->trace, "[krt_rtmsg_rcv]: %s\n",
	       strerror (errno));
    }
    select_enable_fd (rt4_fd);
#endif
}


#ifdef HAVE_IPV6
static void 
krt6_rtmsg_rcv (void *arg)
{
    struct in6_rtmsg rt;
    int err;

    if ((err = read (rt6_fd, &rt, sizeof (struct in6_rtmsg))) > 0) {
	switch (rt.rtmsg_type) {
	case RTMSG_NEWROUTE:
	case RTMSG_DELROUTE:
	case RTMSG_NEWDEVICE:
	case RTMSG_DELDEVICE:
	    /*
	     *      Address Resolution failed
	     */
/*             case RTMSG_AR_FAILED: why is this the same as RTMSG_NEWROUTE ? */

	    trace (NORM, MRT->trace, "[krt6_rtmsg_rcv]: "
		   "message type %d\n", rt.rtmsg_type);
	    break;
	    /*
	     *      What else do you routing folks need
	     *      the kernel to tell you ? [Pedro]
	     */
	default:
	    trace (NORM, MRT->trace, "[krt6_rtmsg_rcv]: "
		   "unknown message type %d\n", rt.rtmsg_type);
	}
    }

    if (err == -1) {
	trace (ERROR, MRT->trace, "[krt_rtmsg_rcv]: %s\n",
	       strerror (errno));
    }
    select_enable_fd (rt6_fd);
}

#define BUFF_SIZ       2048

static void 
krt6_icmp_rcv (void *arg)
{

    struct sockaddr_in6 from;
    struct msghdr mhdr;
    struct iovec iov;
    char buff[BUFF_SIZ];
    int err;

    memset (&mhdr, 0, sizeof (struct msghdr));
    memset (&from, 0, sizeof (struct sockaddr_in6));
    mhdr.msg_name = (void *) &from;
    mhdr.msg_namelen = sizeof (struct sockaddr_in6);
    mhdr.msg_iov = &iov;
    mhdr.msg_iovlen = 1;

    iov.iov_base = buff;
    iov.iov_len = BUFF_SIZ;

    if ((err = recvmsg (rt6_sk, &mhdr, O_NONBLOCK)) > 0) {
	struct icmpv6hdr *hdr;

	hdr = (struct icmpv6hdr *) buff;
	trace (NORM, MRT->trace, "[krt_icmp_rcv]: type=%d, code=%d\n",
	       hdr->type, hdr->code);
	/*
	   * process icmp here
	 */
    }

    if (err == -1) {
	trace (ERROR, MRT->trace, "[krt_icmp_rcv]: %s\n",
	       strerror (errno));
    }
    select_enable_fd (rt6_sk);
}

#endif /* HAVE_IPV6 */


typedef void (*add_rt_f) (int family, unsigned char *dest,
			  unsigned char *nexthop, int masklen, int index);

/*
 *    read /proc/net/route6
 */
#ifdef HAVE_IPV6
#define PROC_NET_ROUTE6 "/proc/net/ipv6_route"
static void
krt_read_table_v6 (add_rt_f add_kernel_route)
{
    FILE *fp;
    char dest_str[40], nexthop_str[40], devname[16];
    int masklen, metric, use, refcnt, flags, index = 0;
    struct in6_addr dest6, nexthop6;
    int num;
    char buff[1024];
    interface_t *interface;

    if ((fp = fopen (PROC_NET_ROUTE6, "r")) == NULL) {
	perror ("error opening /proc file");
	return;
    }
    while (fgets (buff, sizeof (buff) - 1, fp)) {
	num = sscanf (buff,
		      "%4s%4s%4s%4s%4s%4s%4s%4s %02x %4s%4s%4s%4s%4s%4s%4s%4s %02x %02x %02x %04x %s",
		      dest_str, dest_str + 5, dest_str + 10, dest_str + 15,
		 dest_str + 20, dest_str + 25, dest_str + 30, dest_str + 35,
		      &masklen,
	   nexthop_str, nexthop_str + 5, nexthop_str + 10, nexthop_str + 15,
		      nexthop_str + 20, nexthop_str + 25, nexthop_str + 30, nexthop_str + 35,
		      &metric, &use, &refcnt, &flags, devname);

	if (num < 22)
	    continue;
	dest_str[4] = dest_str[9] = dest_str[14] = dest_str[19] =
	    dest_str[24] = dest_str[29] = dest_str[34] = ':';
	nexthop_str[4] = nexthop_str[9] = nexthop_str[14] = nexthop_str[19] =
	    nexthop_str[24] = nexthop_str[29] = nexthop_str[34] = ':';

	if (inet_pton (AF_INET6, dest_str, &dest6) < 0)
	    continue;
	if (inet_pton (AF_INET6, nexthop_str, &nexthop6) < 0)
	    continue;
	if (masklen < 0 || masklen > 128)
	    continue;
	if ((interface = find_interface_byname (devname)))
	    index = interface->index;
	(*add_kernel_route) (AF_INET6,
		       (char *) &dest6, (char *) &nexthop6, masklen, index);
    }
    fclose (fp);
}

#endif /* HAVE_IPV6 */


#define PROC_NET_ROUTE "/proc/net/route"
static void 
krt_read_table_v4 (add_rt_f add_kernel_route)
{
    FILE *fp;
    char dest_str[9], nexthop_str[9], mask_str[9], devname[16];
    int masklen, metric, use, refcnt, flags, index = 0;
    struct in_addr dest, nexthop, mask;
    int num;
    char buff[1024];
    interface_t *interface;

    if ((fp = fopen (PROC_NET_ROUTE, "r")) == NULL) {
	perror ("error opening /proc file");
	return;
    }

    /*
     * skip the first line
     */
    if (fgets (buff, sizeof (buff) - 1, fp) == NULL) {
	fclose (fp);
	return;
    }

    while (fgets (buff, sizeof (buff) - 1, fp)) {
	num = sscanf (buff, "%s %s %s %x %d %d %d %s",
	     devname, dest_str, nexthop_str, &flags, &refcnt, &use, &metric,
		      mask_str);

	if (num < 8)
	    continue;

#if 1
	if (!BIT_TEST (flags, RTF_UP))
	    continue;
	if (!BIT_TEST (flags, RTF_GATEWAY))
	    continue;
#endif

#ifdef notdef
	dest.s_addr = htonl (atox (dest_str));
	nexthop.s_addr = htonl (atox (nexthop_str));
	mask.s_addr = htonl (atox (nexthop_str));
#else
	/* This file contains in host byte order */
	dest.s_addr = atox (dest_str);
	nexthop.s_addr = atox (nexthop_str);
	mask.s_addr = atox (mask_str);
#endif
	masklen = mask2len ((char *) &mask, 4);

	if (masklen < 0 || masklen > 32)
	    continue;

	if ((interface = find_interface_byname (devname)))
	    index = interface->index;

	(*add_kernel_route) (AF_INET,
			 (char *) &dest, (char *) &nexthop, masklen, index);
    }
    fclose (fp);
}

void 
kernel_read_rt_table (void (*add_kernel_route) ())
{
    krt_read_table_v4 (add_kernel_route);
#ifdef HAVE_IPV6
    krt_read_table_v6 (add_kernel_route);
#endif /* HAVE_IPV6 */
}
