/*
 * $Id: view.c,v 1.11 1997/03/21 10:32:05 masaki Exp $
 */

#include <stdio.h>
#include <sys/types.h>
#include <stack.h>
#include <mrt.h>
#include <view.h>
#include <bgp.h>

u_int view_hash_fn (prefix_t *prefix, u_int size);
u_int view_lookup_fn (prefix_t *tmp1, prefix_t *tmp2);
bgp_route_head_t *New_Bgp_Route_Head (prefix_t *prefix);
bgp_route_t *New_Bgp_Route (bgp_route_head_t *bgp_route_head,
			      int pref, bgp_attr_t *attr);
int Delete_Bgp_Route (bgp_route_t *);
void delete_update_bucket (update_bucket_t *update_bucket);
bgp_route_t *bgp_implicit_withdraw (view_t *view, bgp_route_t* bgp_route, 
			   prefix_t* prefix, bgp_attr_t* attr, int pref);

int Delete_Bgp_Route_Head (bgp_route_head_t *head);

/* view_get_route_head
 * get "head" of routing table entry for a given prefix. Create a head
 * and radix-tree entry if one does not aleady exist
 */
bgp_route_head_t *view_get_route_head (view_t *view, prefix_t *prefix) 
{
  bgp_route_head_t *bgp_route_head;

  bgp_route_head = HASH_Lookup (view->hash, prefix);

  if (bgp_route_head == NULL) {
    bgp_route_head = (bgp_route_head_t *) New_Bgp_Route_Head (prefix);
    HASH_Insert (view->hash, bgp_route_head);
  }

  return (bgp_route_head);

}


/* view_find_bgp_active
 * given a prefix AND gateway, find route node entry
 * return NULL if one does not exist
 */
bgp_route_t *view_find_bgp_active (view_t *view,  prefix_t *prefix)
{
  bgp_route_head_t *bgp_route_head;
  bgp_route_t *bgp_route;

  bgp_route_head = HASH_Lookup (view->hash, prefix);

  return (bgp_route_head->active);
}


/* view_find_bgp_route
 * given a prefix AND gateway, find route node entry
 * return NULL if one does not exist
 */
bgp_route_t *view_find_bgp_route (view_t *view,
				   prefix_t *prefix, gateway_t *gateway) 
{
  bgp_route_head_t *bgp_route_head;
  bgp_route_t *bgp_route;

  bgp_route_head = view_get_route_head (view, prefix);
    
  LL_Iterate (bgp_route_head->ll_routes, bgp_route) {
    if (bgp_route->attr->gateway == gateway) {
      return (bgp_route);
    }
  }

  return (NULL);
}


/* bgp_add_route 
 */
bgp_route_t *bgp_add_route (view_t *view, prefix_t *prefix, 
			    bgp_attr_t *attr, int pref)
{
  bgp_route_t *bgp_route = NULL;
  bgp_route_head_t *bgp_route_head = NULL;

  if (pthread_mutex_trylock (&view->mutex_lock) == 0) {
    printf ("\nERROR -- I should be locked in bgp_add_route\n");
    exit(0);
  }

  bgp_route_head = view_get_route_head (view, prefix);

  /* check to see if we already got this route from the gateway 
   *  if we have, then this is an implicit withdraw of the old route and
   *  we add the new route */
  LL_Iterate (bgp_route_head->ll_routes, bgp_route) {
    if (bgp_route->attr->gateway == attr->gateway) {
      return (bgp_implicit_withdraw (view, bgp_route, prefix, attr, pref));
      /* set implicit withdraw flag */
      /* we need a special flag because a host due to policy may not
	 accept the new route, then we have to explicityl withdraw this
	 prefix
	 */
    }
  }

  /* first time we've heard from this gateway */    
  bgp_route = New_Bgp_Route (bgp_route_head, pref, attr);
  LL_Add (bgp_route_head->ll_routes, bgp_route);

  /* is this new route better than the active route ? */
  if ((bgp_route_head->active == NULL) || 
      (bgp_route->pref <= bgp_route_head->active->pref)) {


    /* check for tie breaking */
    if ((bgp_route_head->active != NULL) && 
	(bgp_route->pref == bgp_route_head->active->pref)) {
      int a1, a2;
      
      /* shorter aspath wins */
      a1 = aspath_length (bgp_route->attr->aspath, 0);
      a2 = aspath_length (bgp_route_head->active->attr->aspath, 0);
      if (a1 > a2) {return (bgp_route);}

      /* lowest ip address wins */
      if (a1 == a2) {
	int l1, l2;
	l1 = prefix_tolong (bgp_route->attr->gateway->prefix);
	l2 = prefix_tolong (bgp_route_head->active->attr->gateway->prefix);
	if (l1 > l2) 
	  return (bgp_route);
      }
    }

    bgp_route_head->last_active = bgp_route_head->active;
    bgp_route_head->active = bgp_route;
    
    /* add the route head to the change list */
    LL_Add (view->ll_changed_routes, bgp_route_head);

    trace (NORM, view->trace, "BGP view Change: %s\t%d\n",
 	   prefix_toa (prefix), pref);
  }

  return (bgp_route);
}


int view_close (view_t *view) {

  pthread_mutex_unlock (&view->mutex_lock);
  return (1);
}


int view_open (view_t *view) {

#ifdef HAVE_LIBPTHREAD
   if (pthread_mutex_trylock (&view->mutex_lock) != 0) {
      trace (TR_DEBUG, view->trace, "Going to block in view_open\n");
      pthread_mutex_lock (&view->mutex_lock);
   }
#endif /* HAVE_LIBPTHREAD */

   return (1);
}


/* view_delete_bgp_route
 * delete a route from view. If active route, set change list
 * and modify route_head
 */
int view_delete_bgp_route (view_t *view, bgp_route_t *bgp_route) 
{
  bgp_route_head_t *bgp_route_head;
  bgp_route_t *best, *tmp;

  best = NULL;
  bgp_route_head = bgp_route->head;

  /* remove from list in any case */
  LL_Remove (bgp_route_head->ll_routes, bgp_route);

  /* we are NOT the active route - just delete */
  if (bgp_route_head->active != bgp_route) {
    Delete_Bgp_Route (bgp_route);
    return (1);
  }

  bgp_route_head->active = NULL;
  bgp_route_head->last_active = bgp_route;

  /* find new active route */
  LL_Iterate (bgp_route_head->ll_routes, tmp) {
    if ((best == NULL) || ((tmp->pref < best->pref))){
      best = tmp;
    }
  }
  
  /* prefix withdrawn with nothing to replace it */
  if (best == NULL) {
    trace (NORM, BGP->trace, "VIEW %s route withdrawn (no other route)\n",
	   prefix_toa (&bgp_route_head->prefix));
    /* flag route head for deletion */
    bgp_route_head->state_bits |= VRTS_DELETE;
    /*bgp_route_head->change_bits |= RT_WITHDRAWN;*/
    HASH_Remove (view->hash, bgp_route_head);
  }
  else {
    char tmp1[MAXLINE], stmp[MAXLINE];
    sprintf (tmp1, "%s", prefix_toa (&bgp_route_head->prefix));
    trace (NORM, BGP->trace, "VIEW %s route withdrawn (best now from %s)\n",
	   tmp1, gateway_toa (stmp, best->attr->gateway));
    bgp_route_head->state_bits |= VRTS_CHANGE;
    /*bgp_route_head->change_bits |= RT_WITHDRAWN;*/
  }

  /* set to new best */
  bgp_route_head->active = best;
  LL_Add (view->ll_changed_routes, bgp_route_head);
  return (1);
}


bgp_route_t *New_Bgp_Route (bgp_route_head_t *bgp_route_head,
			    int pref, bgp_attr_t *attr) {

   bgp_route_t *tmp = (bgp_route_t *) New (bgp_route_t);
   tmp->head = bgp_route_head;
   tmp->pref = pref;
   tmp->attr = attr;
   
   attr->ref_count++;
   return (tmp);
}



bgp_route_head_t *New_Bgp_Route_Head (prefix_t *prefix) {
  bgp_route_head_t *tmp = New (bgp_route_head_t);
  
  /*tmp->prefix = prefix;*/
  memcpy (&tmp->prefix, prefix, sizeof (prefix_t));
  tmp->ll_routes = LL_Create (0);
  tmp->active = NULL;
   
   return (tmp);
}


int Delete_Bgp_Route (bgp_route_t *rt_node) {
   generic_attr_t *attr;
   
   attr = (generic_attr_t *) rt_node->attr;

   attr->ref_count--;
   if (attr->ref_count <= 0) {
      /*lbgpprint_attr (rt_node->p_attr);*/
      /*printf ("\nRef count now %d", rt_node->p_attr->ref_count);*/
     /*rm_delete_attr (rt_node->p_attr);*/
   }
   Delete (rt_node);

}

/* Delete_Bgp_Route_Head
 */
int Delete_Bgp_Route_Head (bgp_route_head_t *head) {
  if (head->active != NULL) 
    Delete_Bgp_Route (head->active);
  if (head->last_active != NULL) 
    Delete_Bgp_Route (head->last_active);
  LL_Destroy (head->ll_routes);
  Delete (head);
}

view_t *New_View (int maxbitlen)
{
   bgp_route_head_t head;
   view_t *tmp = New (view_t);

   /*tmp->ll_bgp_routes = LL_Create (0);*/
   tmp->ll_changed_routes = LL_Create (0);
   /*tmp->ll_withdrawn_routes = LL_Create (0);*/
   
   /*if (LL_ATTR_MEMORY == NULL)
      init_attr_memory ();*/

   pthread_mutex_init (&tmp->mutex_lock, NULL);
   tmp->hash = HASH_Create (100, 
			    HASH_EmbeddedKey, True,
			    HASH_KeyOffset, HASH_Offset(&head, &head.prefix),
			    HASH_LookupFunction, view_lookup_fn,
			    HASH_HashFunction, view_hash_fn,
			    NULL);
   return (tmp);
}


u_int view_hash_fn (prefix_t *prefix, u_int size) {
  u_int tmp;
  u_char *dest  = prefix_tochar (prefix);
  

  tmp = dest[0] + dest[1] + dest[2] + dest[3];
  tmp = tmp % size;
  return (tmp);
}

u_int view_lookup_fn (prefix_t *tmp1, prefix_t *tmp2) {
  u_int tmp;
  tmp = prefix_compare (tmp1, tmp2);
  return (tmp);
}


update_bucket_t *New_Update_Bucket () {
  update_bucket_t *tmp = New (update_bucket_t);
  
  /* NOTE: update bucket memory belongs to view */
  tmp->ll_prefix =  LL_Create (0);

  return (tmp);
}


/* view_delete_peer
 * withdraw all routes sent by a peer when peer leaves
 * established state
 */
int view_delete_peer (view_t *view, gateway_t *gateway) {
  bgp_route_head_t *rt_head;
  bgp_route_t* route;

  LINKED_LIST *ll_route_with = LL_Create (0);

  view_open (view);

  /* find all routes announced by this peer */
  HASH_Iterate (view->hash, rt_head) {
    LL_Iterate (rt_head->ll_routes, route) {
      if (route->attr->gateway == gateway)
	LL_Add (ll_route_with, route);
    }
  }

  LL_Iterate (ll_route_with, route) {
    view_delete_bgp_route (view, route);
  }

  bgp_process_changes (view);

  view_close (view);
}



/* build_update_buckets
 */

LINKED_LIST *build_update_buckets (LINKED_LIST *ll_routes) {
   LINKED_LIST *ll_update_buckets;
   update_bucket_t *update_bucket;
   bgp_route_t *route;
   int found;

   ll_update_buckets = LL_Create (LL_DestroyFunction, delete_update_bucket, 
				  0);

   /* replace this with a hash, or something a bit more reasonable */
   LL_Iterate (ll_routes, route) {
      found = 0;
      LL_Iterate (ll_update_buckets, update_bucket) {
	 if (update_bucket->attr == route->attr) {
	    LL_Add (update_bucket->ll_prefix, 
		    &route->head->prefix);
	    found = 1;
	    break;
	 }
      }

      if (found == 0) {
	 update_bucket = New_Update_Bucket ();
	 update_bucket->attr = route->attr;

	 LL_Add (ll_update_buckets, update_bucket);
	 LL_Add (update_bucket->ll_prefix, 
		 &route->head->prefix);
      }
   }

   return (ll_update_buckets);
}


/* view_announce_peer
 * here is where we actually build update packets and 
 * schedule them to be sent off. NOTE: It is is the responsibility of
 * bgp peer to delete buffer memory!
 * 
 */
int view_announce_peer (view_t *view, bgp_peer_t *peer,
			LINKED_LIST *ll_with_prefixes, 
			update_bucket_t *update_bucket) 
{
   u_char *p_total_attrib_len, *p_unfeasible_len;
   prefix_t *p_with_prefix = NULL;
   prefix_t *p_ann_prefix = NULL;
   buffer_t *p_buffer;
   short len;
   char tmp[MAXLINE];
   short AS = BGP->my_as;

   /* if transparent option is set, do not add our AS to aspath 
    *  on announcements */
   if (peer->options & BGP_TRANSPARENT_AS) {
     AS = 0;
   }

   /* log route announces */
   trace_open (peer->trace);

   trace (TR_DEBUG, peer->trace, "Building update packet for %s\n",
	  gateway_toa (tmp, peer->gateway));

   /* trace this info */
   if (update_bucket != NULL) {
     bgp_trace_attr ("BGP build", peer->trace,update_bucket->attr);

     trace_prefix_list ("BGP build announce ", peer->trace, 
			update_bucket->ll_prefix);
   }
   if (ll_with_prefixes)
      trace_prefix_list ("BGP build withdraw ", peer->trace, 
			 ll_with_prefixes);
   trace_close (peer->trace);


   if (ll_with_prefixes)
      p_with_prefix = (prefix_t *) LL_ContGetHead (ll_with_prefixes);
   if (update_bucket)
      p_ann_prefix = (prefix_t *) LL_ContGetHead (update_bucket->ll_prefix);
   
   while (p_with_prefix || p_ann_prefix) {
      u_char *cp, *start_pdu, *maxend;
      
      cp = start_pdu = NewArray (u_char, BGPMAXPACKETSIZE);
      maxend = cp + BGPMAXPACKETSIZE - BGP_HEADER_LEN;

      p_unfeasible_len = (u_char *) cp;
      cp +=2; /* skip unfeasible route length */


      /* 
       * add withdrawn prefixes 
       */
      while (p_with_prefix) {
	 int nbytes =  (p_with_prefix->bitlen + 7) >> 3;
	 if (cp + nbytes < maxend) {
	    BGP_PUT_PREFIX(p_with_prefix->bitlen, 
			   prefix_tochar (p_with_prefix), cp);
	    p_with_prefix = 
	       (prefix_t *) LL_ContGetNext (ll_with_prefixes, p_with_prefix);
	 }
	 else
	    break;
      }

      /* put in withdrawn routes length */
      len = cp - start_pdu - 2;
      BGP_PUT_SHORT (len, p_unfeasible_len);

      /* special case when only have withdarws */ 
      if (p_ann_prefix == NULL) {
	BGP_PUT_SHORT (0, cp);
      }

      /* 
       * add announced routes if no more withdrawn routes 
       */
      if ((p_ann_prefix) && (p_with_prefix == NULL)) {
	 short len;

	 /* skip total path attribute length for now */
	 p_total_attrib_len = cp;
	 cp += 2; 

	 cp = (u_char *) bgp_add_attributes (cp, &len, update_bucket->attr, AS);
	 
	 if (cp > maxend)
	    break;

	 /* Now put in attribute length */
	 BGP_PUT_SHORT (len, p_total_attrib_len);

	 /* Now stick in prefixes */
	 while (p_ann_prefix) {
	    int nbytes = (p_ann_prefix->bitlen + 7) >> 3;
	    if (cp + nbytes < maxend) {
	       BGP_PUT_PREFIX (p_ann_prefix->bitlen, 
			       prefix_tochar (p_ann_prefix), cp);
	       p_ann_prefix = 
		  LL_ContGetNext (update_bucket->ll_prefix, p_ann_prefix);
	    }
	    else
	       break;
	 }
      }


      p_buffer = New(buffer_t);
      p_buffer->len = cp - start_pdu;
      p_buffer->data = start_pdu;

      /* ASSERT ((p_buffer->len < BGPMAXPACKETSIZE));*/


      /* schedule for sending it */
      schedule_event (peer->schedule, bgp_send_update, 3, peer,
		      p_buffer->len, p_buffer->data);


      /* this does NOT delete data which is left	
       * to bgp_send_update to delete			*/
      Delete (p_buffer);
   }
}



/* bgp_establish_peer
 * when a peer is first established, run through view and build
 * BGP updates for all routes per policy
 */
int bgp_establish_peer (view_t *view, bgp_peer_t *peer)
{
  bgp_route_head_t *head;
  LINKED_LIST *ll_ann, *ll_buckets;
  update_bucket_t *bucket;

  ll_ann = LL_Create (0);

  view_open (view);

  HASH_Iterate (view->hash, head) {
    if (head->active != NULL) {

      head->peer_mask |= peer->bit; /* set announce bit */
      LL_Add (ll_ann, head->active);
    }
  }

  ll_buckets = build_update_buckets (ll_ann);

  LL_Iterate (ll_buckets, bucket) {
    view_announce_peer (view, peer, NULL, bucket);
  }
  LL_Destroy (ll_ann);
  LL_Destroy (ll_buckets);

  view_close (view);
  return (1);
}



int show_view (uii_connection_t *uii)
{
  u_int view = 0;

  if (((parse_line (uii->cp, "%d", &view)) != 1)  ||
      (view < 0) || (view > 32) || (BGP->views[view] == NULL)) {
    uii_send_data (uii, "ERROR: invalid or unconfigured view %d\r\n", view);
    return (-1);
  }

  bgp_dump_view (uii, view);

  return (1);
}



/* delete_update_bucket
 */
void delete_update_bucket (update_bucket_t *update_bucket) {
  LL_Destroy (update_bucket->ll_prefix);
  Delete (update_bucket);
}



/* bgp_process_changes
 * process route changes - run policy, and then build and schedule 
 * updates to peers. This is call after receipt of update packets
 */
int bgp_process_changes (view_t *view) 
{
  bgp_route_head_t *head, *lhead;
  LINKED_LIST *ll_ann = NULL;
  LINKED_LIST *ll_with = NULL;
  LINKED_LIST *ll_buckets = NULL;
  bgp_peer_t *peer;
  update_bucket_t *bucket = NULL;;

  /* maybe check that view is open? */

  /* update each peer of view with changes */
  LL_Iterate (BGP->ll_bgp_peers, peer) {

    /* see if peer is established */
    if (peer->state != BGPSTATE_ESTABLISHED) continue;

    ll_ann = LL_Create (0);
    ll_with = LL_Create (0);

    /* build announce list and withdraw list */
    LL_Iterate (view->ll_changed_routes, head) {

      /* 
       * Add
       */
      if (!(head->state_bits & VRTS_DELETE)) {

	/* do not announce to peer who gave us route */
	if (head->active->attr->gateway == peer->gateway) {continue;}

	/* policy */
	/* if policy fails and route announced, withdraw the puppy */

	head->peer_mask |= peer->bit; /* set announce bit */
	LL_Add (ll_ann, head->active);
      }
	  
      /* 
       * Withdraw 
       */
      else if (head->state_bits & VRTS_DELETE) {

	/* do not give withdraw to person who gave us the route */
	if (head->last_active->attr->gateway == peer->gateway) {continue;}

	/* if not announced, then do not withdraw */
	if (!(head->peer_mask & peer->bit)) {continue;}

	LL_Add (ll_with, &head->prefix);
      }
    }

    /* just withdraws */
    if ((LL_GetCount (ll_ann) <= 0) && (LL_GetCount (ll_with) > 0)) {
      view_announce_peer (view, peer, ll_with, bucket);
    }

    /* organize by attr type for easy building of packets */
    ll_buckets = build_update_buckets (ll_ann);

    LL_Iterate (ll_buckets, bucket) {

      /* don't announce if not sent */
      /* don't announce if we learned it from this gateway */
      if (bucket->attr->gateway == peer->gateway) continue;

      view_announce_peer (view, peer, ll_with, bucket);
    }

    LL_Destroy (ll_buckets);

    LL_Destroy (ll_ann);
    LL_Destroy (ll_with);
  }

  LL_Iterate (view->ll_changed_routes, head) {
    if (head->state_bits & VRTS_DELETE) {
      lhead = LL_GetPrev (view->ll_changed_routes, head);
      LL_Remove (view->ll_changed_routes, head);
      Delete_Bgp_Route_Head (head);
      head = lhead;
    }
  }
  LL_Clear (view->ll_changed_routes);
  
  /* from hash if withdrawn */
  /* reset stuff */
  return (1);
}



bgp_route_t *bgp_implicit_withdraw (view_t *view, bgp_route_t* bgp_route, 
				    prefix_t* prefix, bgp_attr_t* attr, int pref)
{
  bgp_route_head_t *bgp_route_head;
  bgp_route_t *best, *tmp;

  best = NULL;
  bgp_route_head = bgp_route->head;

  bgp_route_head->state_bits = VRTS_ACTIVE;
  /*bgp_route_head->change_bits = RT_CHANGE;*/

  /* remove old route from list in any case */
  LL_Remove (bgp_route_head->ll_routes, bgp_route);

  /* if we were active route, wipe clean active route */
  if (bgp_route_head->active == bgp_route) {
    bgp_route_head->active = NULL;
    bgp_route_head->last_active = bgp_route;
  }
  /* old route was not ative, so we can just delete it! yip! */
  else {
    Delete_Bgp_Route (bgp_route);
  }

  /* and add new route */
  bgp_route = New_Bgp_Route (bgp_route_head, pref, attr);
  LL_Add (bgp_route_head->ll_routes, bgp_route);

  /* find new active route */
  LL_Iterate (bgp_route_head->ll_routes, tmp) {
    if ((best == NULL) || ((tmp->pref < best->pref))){
      best = tmp;
    }
  }
  
  if (bgp_route_head->active == best) {
    trace (NORM, BGP->trace, "VIEW no change for %s\n", 
	   prefix_toa (&bgp_route_head->prefix));
    /* delete memory for last active ?*/
    return (bgp_route);
  }

  trace (NORM, BGP->trace, "VIEW implicit widthdraw change for %s\n",
 	  prefix_toa (&bgp_route_head->prefix));
  bgp_route_head->state_bits = VRTS_ACTIVE;
  /*bgp_route_head->change_bits = RT_CHANGE;*/
  
  /* set to new best */
  bgp_route_head->active = best;
  LL_Add (view->ll_changed_routes, bgp_route_head);
  return (bgp_route);
}
