/*
 * $Id: bgp_pdu.c,v 1.4 1997/02/01 01:19:36 masaki Exp $
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <varargs.h>
#include <netdb.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <memory.h> 
#ifndef __linux__
#include <sys/conf.h>  /* BSDism */
#endif
#include <errno.h>
#include <mrt.h>
#include <bgp_proto.h>
#include <timer.h>
#include <trace.h>
#include <bgp.h>
#include <lbgp.h>
#include <mrt.h>
#include <io.h>


/* bgp_get_pdu
 * called after mrt_select notices change on 
 * descriptor. Gets packet from peer and starts processing
 * If status is  1 - more packets in buffer, 
 * If status is -1 - error (peer is down)
 */
int bgp_get_pdu (bgp_peer_t *peer) {
   u_char *tmp;
   char buf[MAXLINE];
   int status;

   if ((tmp = (u_char *) get_packet (peer, &status)) != NULL) {
      process_pdu (peer, peer->state, tmp);
   }

   /* read returned 0 or less -- something has happened to peer */
   if (status == -1) {
      trace (NORM, peer->trace, "Error reading packet for %s\n",
	     gateway_toa (buf, peer->gateway));
      bgp_sm_process_event (peer, peer->state, BGPEVENT_ERROR);
      return (-1);
   }

   select_enable_fd (peer->sockfd);
   return (1);
}


/* process_pdu
 * munge bgp packet
 * Process notifications here -- deal with errors immediately
 */
int process_pdu (bgp_peer_t *peer, int state, u_char *cp)
{
   int type;
   int length;
   int event; 
   char tmp[MAXLINE];

   /*peer->packet = cp;*/

   BGP_GET_HDRTYPE(type, cp);
   BGP_GET_HDRLEN(length, cp);

   trace (TR_PACKET, peer->trace, "\n");   
   trace (TR_PACKET, peer->trace, "BGP recv %s from %s (%d bytes)\n",
	  sbgp_pdus[type], gateway_toa (tmp, peer->gateway), length);
   
   if (type == BGP_OPEN)
      event = BGPEVENT_RECVOPEN;
   else if (type == BGP_UPDATE)
      event = BGPEVENT_RECVUPDATE;
   else if (type == BGP_NOTIFY) {
      event = BGPEVENT_RECVNOTIFY;
      bgp_process_notify (peer, peer->packet);
   }
   else if (type == BGP_KEEPALIVE)
      event = BGPEVENT_RECVKEEPALIVE;

   bgp_sm_process_event (peer, state, event);

   return (1);
}




/* get_packet
 * fill in buffer and return packet if full one has arrived
 */
u_char *get_packet (bgp_peer_t *peer, int *status)
{
   int len = 0;
   int size = 0;
   int pdu_header_len = 0;
   int have_packet_flag = 0;
   u_char* tmp;

   tmp = peer->packet;
   *status = 0;

   /* check if packets already in buffer */
   if (peer->read_ptr - peer->buffer >= BGP_HEADER_LEN) {
      BGP_GET_HDRLEN(pdu_header_len, peer->buffer);
      if ((peer->read_ptr - peer->buffer) >= pdu_header_len) 
	 have_packet_flag = 1;
   }

   if (!have_packet_flag) {

      len = read (peer->sockfd, peer->read_ptr, peer->room_in_buffer);

      if (len <= 0) {
	char _tmp[MAXLINE];
	 if (errno == EINTR) {
	    perror ("READ FAILED -- OKAY TO IGNORE\n"); 
	    return (NULL);
	 }
	 trace (NORM, peer->trace, "read file from %s\n", 
		gateway_toa (_tmp, peer->gateway));
	 *status = -1; 
	 return (NULL);
      }
      
      peer->read_ptr += len;
      peer->room_in_buffer -= len;

      size = peer->read_ptr - peer->buffer;

      /* okay, we at least read enough to get the header */
      BGP_GET_HDRLEN(pdu_header_len, peer->buffer);

      /* sanity checking */
      if (pdu_header_len > BGPMAXPACKETSIZE) {
	 trace (FATAL, NULL, "BOGUS HEADER SIZE %d\n", pdu_header_len);
	 printf ("BOGUS HEADER SIZE %d\n", pdu_header_len);
	 exit (0);
      }
      
      if (pdu_header_len > size) {
	 return (NULL);
      }
   }
      
   memcpy (tmp, peer->buffer, pdu_header_len);

   /* yes, yes, yes -- I know there are more efficient ways of 
    * doing things... */
   memcpy (peer->buffer, peer->buffer + pdu_header_len,
	  BGPMAXPACKETSIZE - peer->room_in_buffer - pdu_header_len);
   peer->room_in_buffer += pdu_header_len;
   peer->read_ptr -= pdu_header_len;
  
   if ((len - pdu_header_len) >= BGP_HEADER_LEN) {
      BGP_GET_HDRLEN(pdu_header_len, peer->buffer);
      if ((peer->read_ptr - peer->buffer) >= pdu_header_len)
	 *status = 1;
   }

   return (tmp);
}

/* bgp_process_update
 * munge update pdu info into withdrawn routes,  announced routes and 
 * aspath
 */
int bgp_process_update (bgp_peer_t *peer, u_char *cp)
{
   short length;

   BGP_GET_HDRLEN(length, cp);
   BGP_SKIP_HEADER(cp);
   length -= BGP_HEADER_LEN;
   
   if (BGP->update_call_fn)
      BGP->update_call_fn (peer, cp, length);   

   return (1);
}


/* bgp_process_open
 */
int bgp_process_open (bgp_peer_t *peer, u_char *cp)
{
   int version;
   int as;
   int holdtime;
   int authcode;
   int id;
   struct in_addr in;

   BGP_SKIP_HEADER(cp);
   BGP_GET_OPEN_V4(version, as, holdtime, id, authcode, cp);

   if (version != DEFAULT_BGP_VERSION) {
      trace (NORM, NULL, 
	     "BGP server is running BGP 3 or earlier! (version = %d)\n", 
	     version);
      bgp_send_notification (peer, peer->sockfd, &peer->peer_addr,
			     BGP_ERR_OPEN, BGP_ERROPN_VERSION);
      return (-1);
   }

   in.s_addr = id;
   trace (TR_PACKET, peer->trace, 
	  "BGP recv AS %d, HOLDTIME %d, ID %s, AUTH %d\n", 
	  as, (u_int) holdtime, inet_ntoa (in), authcode);
   trace (TR_PACKET, peer->trace, "");

   /* set as if default (peer_as = 0) and check if valid AS */
   if (peer->gateway->AS == 0) 
      peer->gateway->AS = as;
   else if (peer->gateway->AS != as) {
      bgp_send_notification (peer, peer->sockfd, &peer->peer_addr,
			     BGP_ERR_OPEN, BGP_ERROPN_AS);
      return (-1);
   }

      peer->HoldTime_Interval = holdtime;

   if (peer->HoldTime_Interval != 0) {
      Timer_Turn_ON (peer->timer_KeepAlive);
      Timer_Turn_ON (peer->timer_HoldTime);
   }
   
   return (1);
}




/* bgp_process_notify
 */
int bgp_process_notify (bgp_peer_t *peer, u_char *cp) {
   u_char code, subcode;

   BGP_SKIP_HEADER(cp);

   UTIL_GET_BYTE(code, cp);
   UTIL_GET_BYTE(subcode, cp);

   peer->code = code;
   peer->subcode = subcode;

   trace (NORM | TR_PACKET, peer->trace, "BGP recv Notification %d/%d: %s\n",
	  code, subcode, bgp_notify_to_string (code, subcode));

   return (1);
}


/* bgp_send_open
 * send an open pdu
 */
void bgp_send_open (bgp_peer_t *peer)
{
   register u_char *cp;
   u_char *buffer = NewArray (u_char, BGPMAXPACKETSIZE);
   int len;
   char tmp[MAXLINE];

   u_char authcode = 0;
   u_long addr;
   struct in_addr in;

   memset(buffer, 0, BGPMAXPACKETSIZE);

   /* set auth field */
   memset (buffer, 255, BGP_HEADER_MARKER_LEN);

      
   /*
    * Skip the header for now, we'll do this last.
    */
   cp = buffer;
   BGP_SKIP_HEADER(cp);

   BGP_PUT_OPEN(DEFAULT_BGP_VERSION, BGP->my_as, DEFAULT_HOLDTIME_INTERVAL,
		BGP->my_id, authcode, cp);

   len = cp - buffer;
   BGP_PUT_HDRLEN(len, buffer);
   BGP_PUT_HDRTYPE(BGP_OPEN, buffer);
   
   write (peer->sockfd, buffer, len);

   in.s_addr = BGP->my_id;
   trace (TR_PACKET, peer->trace, "\n");
   trace (TR_PACKET, peer->trace, 
	"BGP send OPEN to %s with own AS %d, HOLDTIME %d, ID %s, AUTH %d\n", 
	 gateway_toa (tmp, peer->gateway), BGP->my_as, 
	 DEFAULT_HOLDTIME_INTERVAL, inet_ntoa (in), authcode);
   trace (TR_PACKET, peer->trace, "\n");

   Delete (buffer);
}



/* bgp_send_keepalive
 */
void bgp_send_keepalive (bgp_peer_t *peer)
{
   u_char buffer[BGP_HEADER_LEN];
   char tmp[MAXLINE];

   /* set auth field */
   memset (buffer, 255, BGP_HEADER_MARKER_LEN);
   
   BGP_PUT_HDRLEN(BGP_HEADER_LEN, buffer);
   BGP_PUT_HDRTYPE(BGP_KEEPALIVE, buffer);
   
   if (write (peer->sockfd, buffer, BGP_HEADER_LEN) < 1) {
      trace (NORM, BGP->trace, "BGP Error Writing Keepalive to %s",
	     prefix_toa (peer->gateway->prefix));;
      bgp_sm_process_event (peer, peer->state, BGPEVENT_ERROR);
      return;
   }

   trace (TR_PACKET, peer->trace, "BGP send KEEPALIVE to %s\n", 
	  gateway_toa (tmp, peer->gateway));
}


/* bgp_send_notification
 * Send notification error message to BGP peer. If
 * peer argument is NULL, use sockfd argument.
 */
void bgp_send_notification (bgp_peer_t *peer,
			    int sockfd, struct sockaddr_in *addr,
			    u_char code, u_char subcode)
{
   u_char buffer[BGPMAXPACKETSIZE];
   u_char *cp = buffer;

   /* set auth field */
   memset (buffer, 255, BGP_HEADER_MARKER_LEN);
   
   BGP_PUT_HDRLEN(BGP_HEADER_LEN + 2, buffer);
   BGP_PUT_HDRTYPE(BGP_NOTIFY, buffer);
   
   cp += BGP_HEADER_LEN;
   BGP_PUT_BYTE(code, cp);
   BGP_PUT_BYTE(subcode, cp);

   write (sockfd, buffer, cp - buffer);
   trace (NORM, BGP->trace, "");
   trace (NORM, BGP->trace, "BGP send NOTIFICATION to %s (%d)\n",
	  inet_ntoa (addr->sin_addr), ntohs (addr->sin_port));
   trace (NORM, BGP->trace, "BGP send %s\n", 
	  bgp_notify_to_string (code, subcode));
   trace (NORM, BGP->trace, "\n");
   

}



/*-----------------------------------------------------------
 *  Name: 	bgp_send_update
 *  Created:	Tue Jan 24 09:59:38 1995
 *  Author: 	Craig Labovitz   <labovit@snoopy.merit.net>
 *  DESCR:  	
 */

void bgp_send_update (bgp_peer_t *peer, int len, u_char *data)
{
   u_char *buffer = NewArray (u_char, BGPMAXPACKETSIZE);
   u_char *cp = buffer;
   int n;
   char tmp[MAXLINE];

   /* set auth field */
   memset (buffer, 255, BGP_HEADER_MARKER_LEN);
   
   if ((len < 0) || (len > BGPMAXPACKETSIZE)) {
     printf ("\nlen = %d", len);
     exit (0);
   }

   n = BGP_HEADER_LEN + len;
   BGP_PUT_HDRLEN(n, buffer);
   BGP_PUT_HDRTYPE(BGP_UPDATE, buffer);
   
   cp += BGP_HEADER_LEN;
   memcpy (cp, data, len);

   if (write (peer->sockfd, buffer, BGP_HEADER_LEN + len) < 0) {
     trace (NORM, BGP->trace, "BGP Update Write Failed to %s",
	    prefix_toa (peer->gateway->prefix));
     bgp_sm_process_event (peer, peer->state, BGPEVENT_ERROR);
   }

   trace (TR_PACKET, peer->trace, "\n");
   trace (NORM | TR_PACKET, peer->trace, "BGP send UPDATE to %s (%d bytes)\n",
	  gateway_toa (tmp, peer->gateway), BGP_HEADER_LEN + len);
   trace (TR_PACKET, peer->trace, "\n");

   Delete (data);
   Delete (buffer);
}


