
#include <stdio.h>
#include <mrt.h>
#include <trace.h>
#include <interface.h>
#include <rip.h>
#include <bgp.h>
#include <signal.h>
#include "rib.h"

rip_t *RIP;
rib_t *RIB;
rib_t *BGPRIB;
bgp_t *BGP;

void rip_recv_update ();
void quit ();
int rip_uii (char *str);
void show_routing_table (int fd);
void start_config ();
void show_rip ();
int show_ip_routes ();
int show_rip_routing_table();
int show_bgp_routing_table();
int rip_debug ();
int show_interfaces ();
int show_threads ();
int show_view ();
int rib_add_rip_route (prefix_t *prefix, rip_attr_t *attr);
int rib_add_rib_route (prefix_t *prefix, bgp_attr_t *attr);
int process_bgp_update ();
int show_bgp ();
int kill_peer ();

main (int argc, char *argv[]) {
  char c;
  extern char *optarg;	/* getopt stuff */
  extern int optind;	/* getopt stuff */
  int errors = 0;

  char *usage ="Usage: srip\n";
  char *config_file = "./config";
  trace_t *default_trace = NULL;
  
  while ((c = getopt(argc, argv, "vf:")) != -1) 
     switch (c) {
     case 'v': /* verbose */
       VERBOSE_ERROR_FLAG = 1;
       default_trace = New_Trace ();
       set_trace (default_trace, TRACE_FLAGS, TR_ALL, 
		  TRACE_LOGFILE, "stdout",
		  NULL);
       break;
     case 'f': /* config file */
       config_file = strdup ((char *) (optarg));
       break;
     case 'h':
     default:
       errors++;
       break;
     }


   if (errors) {
     fprintf(stderr, usage);
     printf ("\nMRT version (%s) compiled on %s\n\n",
	     MRT_VERSION, __DATE__);
     exit (0);
   }
  
   RIB = New_Rib (32);

   init_mrt (default_trace);
   init_rip (default_trace);
   init_BGP (default_trace);

   /* install the route in the kernel */
   RIP->add_call_fn = rib_add_rip_route;
   set_BGP (BGP_RECV_UPDATE_FN, process_bgp_update, NULL);

   /* add interface routes -- this should probably go somewhere else */
   {
     interface_t *interface;
     generic_attr_t *attr = New (generic_attr_t);
     attr->type = PROTO_CONNECTED;

     rib_open (RIB);
     LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
       rib_add_route (RIB, interface->prefix, attr, 
		      CONNECTED_PREF, 0);
     }
     rib_close (RIB);
   }

   init_config ();
   config_from_file (config_file);

   /* distribute statics into bgp */
   redistribute ();

   uii_add_command (1, "config", (void *) start_config);
   uii_add_command (2, "debug rip", (void *) rip_debug);
   uii_add_command (1, "show bgp", (void *) show_bgp);
   uii_add_command (1, "show rip", (void *) show_rip);
   uii_add_command (1, "show view", (void *) show_view);
   uii_add_command (1, "show interfaces", (void *) show_interfaces);
   uii_add_command (1, "show rip routes", (void *) show_rip_routing_table);
   uii_add_command (1, "show bgp routes", (void *) show_bgp_routing_table);
   uii_add_command (1, "clear bgp", (void *) kill_peer);
   uii_add_command (1, "show ip", (void *) show_ip_routes);

   start_rip ();
   start_bgp ();

#ifndef THREADS
   while (1) {
     process_all_schedules();
     mrt_select ();
   }
#endif /* THREADS */

   /* loop */
   while (1);
}


int rib_add_rip_route (prefix_t *prefix, rip_attr_t *attr) {
  rib_open (RIB);
  rib_add_route (RIB, prefix, (generic_attr_t *) attr, RIP_PREF, 0);
  rib_close (RIB);
  return (1);
}

int rib_add_bgp_route (prefix_t *prefix, bgp_attr_t *attr) {
  rib_open (RIB);
  rib_add_route (RIB, prefix, (generic_attr_t *) attr, BGP_PREF, 0);
  rib_close (RIB);
  return (1);
}

int show_ip_routes (int fd, char *line) {

  show_rib_routes (fd, line, RIB);

}


int process_bgp_update (bgp_peer_t *peer, u_char *cp, int length) {
  bgp_attr_t *tmp_attr, *attr;
  LINKED_LIST *ll_with_prefixes;
  LINKED_LIST *ll_ann_prefixes;
  prefix_t *prefix;
  int i;

  bgp_process_update_packet (cp, length, peer->gateway, &tmp_attr,
			     &ll_with_prefixes, &ll_ann_prefixes);

  attr = bgp_get_attr (tmp_attr);

  bgp_trace_attr ("BGP", peer->trace, attr);
  trace_prefix_list ("BGP recv", peer->trace, ll_ann_prefixes);

  /* itterate through views adding the announce/withdraws */
  for (i=0; i<32; i++) {
    if (BGP->views[i] == NULL) {continue;}

    view_open (BGP->views[i]);
  
    if (ll_ann_prefixes) {
      LL_Iterate (ll_ann_prefixes, prefix) {
	/* apply policy */
	bgp_add_route (BGP->views[i], prefix, attr, 0);
      }
    }

    /* process change list -- send out updates to peers */
    bgp_process_changes (BGP->views[i]);

    view_close (BGP->views[i]);
  }

  return (1);
}

int kill_peer (uii_connection_t *uii)
{
  prefix_t *prefix;
  int asl;
  short as;
  gateway_t *gateway;
  bgp_peer_t *peer;

  if (parse_line (uii->cp, "%p %d", &prefix, &asl) != 2) {
    uii_send_data (uii, "ERROR: close peer d.d.d.d as\r\n");
    return (-1);
  }

  as = (short) asl;
  if ((gateway = find_gateway (prefix, as)) == NULL) {return (-1);}

  peer = find_bgp_peer (gateway);
  /*bgp_peer_dead (peer);
  view_delete_peer (BGP->view, gateway);*/
  bgp_sm_process_event (peer, peer->state, BGPEVENT_ERROR);
  return (1);
}


/* redistribute
 */
int redistribute () {
  route_head_t *head;
  bgp_attr_t *attr;

  rib_open (RIB);
  view_open (BGP->view);

  HASH_Iterate (RIB->hash, head) {

    /* redistribute static */
    if (head->active->attr->type == PROTO_STATIC) {
      attr = New (bgp_attr_t);
      attr->gateway = head->active->attr->gateway;
      attr->nexthop = head->active->attr->nexthop;

      bgp_add_route (BGP->view, head->prefix, 
		     attr, 0);
    }
  }

    rib_close (RIB);  
  view_close (BGP->view);
}
