/*
 * $Id: rip_proto.c,v 1.13 1997/03/21 10:32:36 masaki Exp $
 */

#include <mrt.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>
#include <stdlib.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <string.h>
#include <netdb.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <trace.h>
#include <timer.h>
#include <select.h>
#include <interface.h>
#include <route_manager.h>
#include <rip.h>

void	rip_send_routes (interface_t *interface);
void	rip_timeout_routes (void);
void	rip_receive_update (void);
int	rip_update_routing_table (void);

void	rip_timer_fire (void);
int	init_rip_listen (void);
int	rip_send_request (LINKED_LIST *ll_prefixes);
int	rip_process_update (LINKED_LIST *ll_prefixes, LINKED_LIST *ll_attr);
int	rip_process_packet (gateway_t *gateway, char *update, int bytes,
			    LINKED_LIST **ll_prefixes, LINKED_LIST **ll_attr);
void	show_rip ();

void rip_schedule_receive_update () {
  schedule_event (RIP->schedule, rip_receive_update, 0, NULL);
}

void rip_schedule_timer () {
  schedule_event (RIP->schedule, rip_timer_fire, 0, NULL);
}

u_int rip_hash_fn (prefix_t *prefix, u_int size) {
  u_int tmp;
  u_char *dest;

  dest = (u_char *) prefix_tochar (prefix);
  tmp = dest[0] + dest[1] + dest[2] + dest[3];
  tmp = tmp % size;
  return (tmp);
}

u_int rip_lookup_fn (prefix_t *tmp1, prefix_t *tmp2) {
  u_int tmp;
  tmp = prefix_compare (tmp1, tmp2);
  return (tmp);
}


/*
 * init_rip
 */
int init_rip (trace_t *ltrace)
{
   rip_route_t tmp;

   RIP = (rip_t *) New (rip_t);
   RIP->trace = trace_copy (ltrace);
   RIP->compat_switch = COMPAT_RIP1_COMPAT; /* default */

#if 0
   /* moved to init_mrt() */
   init_select ();
   init_timer ();
#endif

   RIP->timer = (mtimer_t *)
      New_Timer (rip_schedule_timer, RIP_UPDATE_INTERVAL, 
		 "RIP route timer", NULL);

   RIP->schedule = New_Schedule ("RIP", ltrace);

   RIP->hash = HASH_Create (100, 
			    HASH_KeyOffset, HASH_Offset(&tmp, &tmp.prefix),
			    HASH_LookupFunction, rip_lookup_fn,
			    HASH_HashFunction, rip_hash_fn,
			    NULL);

   
   /* uii_add_command (UII, "show rip", show_rip); */

   return (1);
}


/* start_rip_thread 
 * the thread is alive! Start the timers and wait for events to process
 */
void start_rip_main () {
  trace (NORM, RIP->trace, "RIP thread created\n");
  
  init_rip_listen ();
  Timer_Turn_ON (RIP->timer);

  /* send initial request to routers */
  rip_send_request (NULL);

#ifndef HAVE_LIBPTHREAD
 return;
#endif /* THREADS */

  while (1) {
    schedule_wait_for_event (RIP->schedule);
  }
}


/* start_rip
 * just call thread create routine
 */
int start_rip () {
  pthread_t thread;

  mrt_thread_create ("RIP thread", RIP->schedule, 
		     (void *) start_rip_main, NULL);

  /*if (pthread_create (&thread, NULL, (void *) start_rip_main, NULL) < 0) {
    trace (NORM, RIP->trace, "ERROR -- Could not start RIP thread: %s",
	   strerror (errno));
    return (-1);
  }*/
  
  return (1);
}


/* init_rip_listen
 * start listening for broadcast rip updates
 */
int init_rip_listen () {
   struct sockaddr_in serv_addr;
   char optval = 0;

   /* check that NOT listening to portmaster! */
   serv_addr.sin_port = htons (RIP_DEFAULT_PORT);
   serv_addr.sin_family = AF_INET;
   serv_addr.sin_addr.s_addr = INADDR_ANY;

   if ( (RIP->sockfd = socket (AF_INET, SOCK_DGRAM, 0)) < 0) {
      trace (NORM, RIP->trace, "ERROR -- Could not get socket (%s)\n",
	     strerror (errno));
      return (-1);
   }

   optval = 1;
   setsockopt (RIP->sockfd,  SOL_SOCKET, SO_REUSEADDR,
	       &optval, sizeof (optval));

   if (bind (RIP->sockfd, 
	     (struct sockaddr *) &serv_addr, sizeof (serv_addr)) < 0) {
      trace (NORM, RIP->trace, "ERROR -- Could not bind to port %d (%s)\n",
	     RIP_DEFAULT_PORT, strerror (errno));
      RIP->sockfd = -1; /* not listening on port */
      return (-1);
   }

   trace (NORM, RIP->trace, "RIP listening for updates on port %d\n", 
	  RIP_DEFAULT_PORT);

   select_add_fd (RIP->sockfd, 1, rip_schedule_receive_update, NULL);

   return (1);
}



/* _rip_timer
 * fire periodic rip timer and 
 * 1) timeout/garbage collect and
 * 2) announce routes
 */
void rip_timer_fire () {
  interface_t *interface;

  trace (TRACE, RIP->trace, "RIP timer fired\n");

  rip_timeout_routes ();

  /* announce routes */
  LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
    /* check if rip is configured for this interface */
    if (interface->bit & RIP->interface_mask)
      rip_send_routes (interface);
  }
}

/* rip_timeout_routes
 * 1) timoute routes (change metric to 16 and set change bit)
 * 2) garbage collect (delete route from hash, free memory and notify
 *    routing table 
 */
void rip_timeout_routes () {
  rip_route_t *route;
  char tmp1[MAXLINE];
  char tmp2[MAXLINE];
  LINKED_LIST *ll_delete = NULL;

  HASH_Iterate (RIP->hash, route) {
    /* garbage collect and delete route */
    if (time (NULL) - route->time >= RIP_GARBAGE_INTERVAL) {
      trace (NORM, RIP->trace, 
	     "RIP deleted %s \tvia %s  garbage collection\n",
	     rinet_ntoa (tmp1, MAXLINE, route->prefix),
	     rinet_ntoa (tmp2, MAXLINE, route->attr->gateway->prefix));
      if (ll_delete == NULL)
	ll_delete = LL_Create (0);
      LL_Add (ll_delete, route);
    }
    /* timeout route -- set metric to 16 and set change flag */
    else if (time (NULL) - route->time >= RIP_TIMEOUT_INTERVAL) {
      route->attr->metric = RIP_HOPCNT_INFINITY;
      route->flags |= RT_RIP_CHANGE;
    }
  }

  /* we have to use a linked list cause we can't delete hash items
   * while iterating through hash table */
  if (ll_delete != NULL) {
    LL_Process (ll_delete, (void *) Delete_Rip_Route);
    LL_Destroy (ll_delete);
  }
}


/* rip_send_request
 * broadcast request for list of prefixes
 * if NULL, send a request for complete routing table 
 */
int rip_send_request (LINKED_LIST *ll_prefixes)
{
   char buffer[RIP_MAX_PDU];
   char *cp;
   interface_t *interface;
   struct sockaddr_in serv_addr;

   serv_addr.sin_port = htons (RIP_DEFAULT_PORT);
   serv_addr.sin_family = AF_INET;

   LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
     serv_addr.sin_addr.s_addr = (long) interface->broadcast;

     trace (NORM, RIP->trace, "send RIP initial request on %s\n",
	    interface->name);

     cp = buffer;
     memset (buffer, 0, 512);


     UTIL_PUT_BYTE (RIP_REQUEST, cp);
     UTIL_PUT_BYTE (RIP_VERSION, cp);
     cp += 2;
     UTIL_PUT_SHORT (0, cp);
     cp+=2;
     UTIL_PUT_LONG (0, cp);
     cp += 8;
   
     UTIL_PUT_LONG (htonl (RIP_HOPCNT_INFINITY), cp);
     cp += 2;

     sendto (RIP->sockfd, buffer, 24, 0, (struct sockaddr *) &serv_addr,
	     sizeof (serv_addr));
   }

   return (1);
}



/* rip_send_routes
 * given an interface, broadcast rip routes (according to policy) 
 * we do this every RIP_UPDATE_INTERVAL
 */
void rip_send_routes (interface_t *interface) {
  rip_route_t *route;
  prefix_t *prefix;
  char buffer[RIP_MAX_PDU];
  char tmp[MAXLINE];
  char *cp;
  struct sockaddr_in serv_addr;

  serv_addr.sin_port = htons (RIP_DEFAULT_PORT);
  serv_addr.sin_family = AF_INET;
  serv_addr.sin_addr.s_addr = interface->broadcast;
  cp = buffer;
  
  UTIL_PUT_BYTE (RIP_RESPONSE, cp);
  UTIL_PUT_BYTE (RIP_VERSION, cp);
  cp += 2; /* must be zero */
  cp += 2; /* address familly ? */
  cp += 2; /* must be zero */

  trace (NORM, RIP->trace, "send RIP via interface %s %s\n",
	 interface->name, 
	 r_inet_ntoa (tmp, MAXLINE, (u_char *) &(interface->broadcast), 32));

  HASH_Iterate (RIP->hash, route) {
    /* do not send to the same interface on which we got the route */
    if (route->attr->gateway->interface == interface) return;

      prefix = route->prefix;
      trace (NORM, RIP->trace, "send RIP %s %d/%d\n",
	     prefix_toa (prefix),
	     route->attr->metric + 1, 0);
      UTIL_PUT_LONG (htonl ((long) prefix_tolong (prefix)), cp);
      cp += 8; /* must be zero */
      UTIL_PUT_LONG (htonl (route->attr->metric + 1), cp);

      /* see if we have filled up the buffer */
      if (cp - buffer > RIP_MAX_PDU - 17) {
	sendto (RIP->sockfd, buffer, RIP_MAX_PDU, 0, 
		(struct sockaddr *) &serv_addr, sizeof (serv_addr));
	cp = buffer;
	memset (buffer, 0, RIP_MAX_PDU);
	UTIL_PUT_BYTE (RIP_RESPONSE, cp);
	UTIL_PUT_BYTE (RIP_VERSION, cp);
	cp += 2; /* must be zero */
	cp += 2; /* address familly ? */
	cp += 2; /* must be zero */
      }
  }
}

/* rip_receive_update
 * read and process and RIP update packet recieved on
 * our interface
 */
void rip_receive_update ()
{
   char *ptr;
   int n;
   struct sockaddr_in from;
   int fromlen;
   prefix_t *prefix;
   LINKED_LIST *ll_prefixes, *ll_attr;
   gateway_t *gateway;
   char tmp[MAXLINE];
   interface_t *interface;

   fromlen = sizeof (from);
   ptr = NewArray (char, 512);

   if ((n = recvfrom(RIP->sockfd, ptr, 512, 0, 
		     (struct sockaddr *) &from, &fromlen)) <= 0) {
      trace (NORM, RIP->trace, "recv Error reading update\n");
      select_enable_fd (RIP->sockfd);
      Delete (ptr);
      return;
   }

   /* don't listen to things broadcast from our own interface */
   if (local_interface (AF_INET, (u_char *) &from.sin_addr)) {
     select_enable_fd (RIP->sockfd);
     Delete (ptr);
     return;
   }

   prefix = (prefix_t *) New_Prefix (AF_INET, (u_char *) &from.sin_addr, 32);
   interface = find_interface (prefix);

   /* check if this interface is configured for RIP */
   if (!(interface->bit & RIP->interface_mask)) {
     select_enable_fd (RIP->sockfd);
     return;
   }

   trace (TR_PACKET, RIP->trace, "\n");   
   trace (TR_PACKET, RIP->trace, "recv RIP packet from %s (%d bytes)\n",
	  inet_ntoa (from.sin_addr), n);


   gateway = add_gateway (prefix, 0, interface);

   /* munge the rip packet and return list of prefixes and
    * attributes 
    */
   rip_process_packet (gateway, ptr, n, &ll_prefixes, &ll_attr);

   /* update our tables */
   rip_process_update (ll_prefixes, ll_attr);

   Delete (ptr);

   select_enable_fd (RIP->sockfd);

}

/* rip_process_update
 * process linked list of routes (prefix & attribute combinations)
 * check if 1) just updating time on already received route
 *	    2) or a change (different attributes from same gateway,
 *	       or better metric and attributes from new gateway
 */
int rip_process_update (LINKED_LIST *ll_prefixes, 
			LINKED_LIST *ll_attr) {
  prefix_t *prefix;
  rip_attr_t *rip_attr;
  char tmp1[MAXLINE], tmp2[MAXLINE];
  rip_route_t *route;
  int better_metric;

  rip_attr = LL_GetHead (ll_attr);
  LL_Iterate (ll_prefixes, prefix) {

    better_metric = FALSE;
    route = HASH_Lookup (RIP->hash, prefix);

    /* we already have a route for this prefix. Two cases:
    *  1) from same gateway -- just update time, or implicit withdraw
    *  2) from different gateway -- if metric better use this, otherwise
    *     just ignore it. We'll hear about it again in 30 seconds...
    */
    if (route != NULL) {
      /* route from same gateway -- maybe just updating time */
      if (route->attr->gateway == rip_attr->gateway) {
	/* check attributes */
	route->time = time (NULL);
      }
      /* from different gateway -- is metric better ? */
      else {
	if (rip_attr->metric < route->attr->metric) {
	  better_metric = TRUE;
	  HASH_Remove (RIP->hash, route);
	  /* can't delete yet because it is in the rib */
	  /*Delete_Rip_Attr (route->attr);*/
	  route = NULL;
	  /* free memory */
	}
      }
    }
	  
    /* no route exists for this prefix, or a better route */    
    if (route == NULL) {
      char *status[] = {"new", "better"};
      int add = 1;	

      /* if add <= 0, then this was not installed */
      if (RIP->add_call_fn != NULL) 
	add = RIP->add_call_fn (prefix, rip_attr);

      if (add) {
	char tmp[MAXLINE];
	sprintf(tmp, "%s", prefix_toa (rip_attr->gateway->prefix));

	route = New_Rip_Route (prefix, rip_attr);
	trace (NORM, RIP->trace, "RIP change %s via %s\t%d %s\n",
	       prefix_toa (prefix), tmp, 
	       rip_attr->metric, status[better_metric]);
      }
    }
    
    rip_attr = LL_GetNext (ll_attr, rip_attr);
  }
}

    

/* set_rip
 * change/set rip attributes.
 */
int set_rip (int first, ...) {
  va_list    		ap;
  enum RIP_ATTR	attr;

  /* Process the Arguments */
   va_start(ap, first);
   for (attr = (enum RIP_ATTR)first; attr; attr = va_arg(ap, enum RIP_ATTR)) {
      switch (attr) {
      case  RIP_RT_ADD_FN:
	 RIP->add_call_fn = va_arg (ap, void*);
	 break;
      case RIP_TRACE_STRUCT:
	 RIP->trace = va_arg (ap, trace_t *);
	 break;
      default:
	trace (NORM, RIP->trace, "Unknown RIP attribute %s:(%d)", 
	       __FILE__, __LINE__);
	 break;
      }
   }
   va_end(ap);
}


/* New_Rip_Route
 * create/allocate new rip_route_t structure and insert
 * it in the RIP prefix route hash table
 */
rip_route_t *New_Rip_Route (prefix_t *prefix, rip_attr_t *attr) 
{
  rip_route_t *route;

  route = New (rip_route_t);
  route->prefix = prefix; /* copy prefix memory */
  route->attr = attr;    
  route->time = time (NULL);
  route->flags |= RT_RIP_CHANGE;

  HASH_Insert (RIP->hash, route);
  return (route);
}


/* Delete_Rip_Route
 * free route memory and remove from hash
 */
int Delete_Rip_Route (rip_route_t *route) {
  HASH_Remove (RIP->hash, route);
  Delete (route);

  return (1);
}




/* show_rip 
 * dump various rip stats to a socket
 * usually called by UII (user interactive interface 
 */
void show_rip (uii_connection_t *uii) {
  uii_send_data (uii, "\r\nRouting Protocol is \"rip\"\r\n");

  if (RIP->sockfd < 0) 
    uii_send_data (uii, "Not listening for RIP announcements\r\n",
		   RIP_DEFAULT_PORT, RIP->sockfd);
  else 
    uii_send_data (uii, "Listening on port %d (socket %d)\r\n",
		   RIP_DEFAULT_PORT, RIP->sockfd);

  uii_send_data (uii, "Sending updates every %d seconds, next due in %d seconds\r\n",
		 RIP_UPDATE_INTERVAL, 
		 RIP->timer->time_next_fire - time (NULL));
  uii_send_data (uii, "Invalid after 180 seconds, hold down 180, flushed after 240\r\n");
}


/* show_rip_routing_table 
 * dump RIP routing table to socket. Usually called by user
 * interactive interface
 */
void show_rip_routing_table (uii_connection_t *uii) {
  rip_route_t *route;
  char tmp1[MAXLINE], tmp2[MAXLINE];
  
  uii_send_data (uii, "  %-17s     %-17s %s  Time\r\n", 
		 "Prefix", "Next Hop", "Metric"); 

  HASH_Iterate (RIP->hash, route) {
    uii_send_data (uii, "%s %-17s via %-17s %d\t  %d\r\n", 
		   "R",
		   rinet_ntoa (tmp1, MAXLINE, route->prefix),
		   rinet_ntoa (tmp2, MAXLINE, route->attr->gateway->prefix),
		   route->attr->metric, 
		   time (NULL) - route->time);
  }
}


/* rip_debug
 * set the debug flags or file for rip
 * called by config routines
 */
int rip_debug (uii_connection_t *uii)
{
  char *token, *line;
  u_long flag = 0;
  line = (char *)uii->cp;

  /* get flag */
  if ((token = (char *) uii_parse_line (&line)) == NULL) return (-1);
  flag = trace_flag (token);
  set_trace (RIP->trace, TRACE_FLAGS, flag, NULL);

  /* get file name (OPTIONAL) */
  if ((token = (char *) uii_parse_line (&line)) == NULL) return (1);
  set_trace (RIP->trace, TRACE_LOGFILE, token, NULL);

  return (1);
}
