/*
 * $Id: schedule.c,v 1.6 1997/03/21 10:32:26 masaki Exp $
 */		

#include <mrt.h>
#include <signal.h>
#include <string.h>
#include <stdarg.h>
#include <schedule.h>
#include <timer.h>

schedule_master_t *SCHEDULE_MASTER;

void Delete_Event (event_t *event);


/* init_schedule
 */
int init_schedules (trace_t *tr) {
  assert (SCHEDULE_MASTER == NULL);
  SCHEDULE_MASTER = New (schedule_master_t);
  SCHEDULE_MASTER->ll_schedules = LL_Create (0);
  SCHEDULE_MASTER->trace = trace_copy (tr);
  return (1);
}


/* New_Schedule
 */
schedule_t *New_Schedule (char *description, trace_t *tr) {
   schedule_t *tmp;

   tmp = New (schedule_t);
   tmp->ll_events = LL_Create (0);

   if (description != NULL)
      tmp->description = (char *) strdup (description);
   else
      tmp->description = (char *) strdup ("unknown");

   /* this trace condition is delegated from the caller */
   tmp->trace = tr;

   /* tmp->self = pthread_self (); */
   pthread_cond_init (&tmp->cond_new_event, NULL);
   pthread_mutex_init (&tmp->mutex_lock, NULL);
   pthread_mutex_init (&tmp->mutex_cond_lock, NULL);

   LL_Add (SCHEDULE_MASTER->ll_schedules, tmp);

   return (tmp);

}



/* schedule_event
 */
int schedule_event (schedule_t *schedule, void (*call_fn)(), 
		    int narg, ...) {

   va_list    	ap;
   event_t	*event;
#ifndef HAVE_LIBPTHREAD
sigset_t old;
#endif /* HAVE_LIBPTHREAD */

   event = New (event_t);

   va_start(ap, narg);

   if (narg >= 1)
      event->arg1 = va_arg(ap, void *);
   if (narg >= 2)
      event->arg2 = va_arg(ap, void *);
   if (narg >= 3)
      event->arg3 = va_arg(ap, void *);
   if (narg >= 4)
      event->arg4 = va_arg(ap, void *);
   
   event->call_fn = call_fn;
   event->narg = narg;

#ifndef HAVE_LIBPTHREAD
old = block_signal (SIGALRM);
#endif /* HAVE_LIBPTHREAD */
   /* obtain lock */
   pthread_mutex_lock (&schedule->mutex_cond_lock);

   LL_Add (schedule->ll_events, event);
   schedule->new_event_flag ++;

   pthread_cond_signal (&schedule->cond_new_event);

   /* release lock */
   pthread_mutex_unlock (&schedule->mutex_cond_lock);
   trace (TR_THREAD, schedule->trace, "Event %x scheduled (%d events) for %s\n", 
	  call_fn, LL_GetCount (schedule->ll_events),
	  schedule->description);
#ifndef HAVE_LIBPTHREAD
recover_signal (old);
#endif /* HAVE_LIBPTHREAD */


   if (LL_GetCount (schedule->ll_events) > 5000) {
      trace (TR_THREAD, schedule->trace, "Over 5000 events !!!!!!");
      trace (TR_THREAD, schedule->trace, "*** SCHEDULE OVERFLOW ***");
      return (-1);
   }
   
   return (1);
}



/* schedule_wait_for_event
 */
int schedule_wait_for_event (schedule_t *schedule) {
   event_t *event;

#ifndef HAVE_LIBPTHREAD
	return (1);
#endif

   pthread_mutex_lock (&schedule->mutex_cond_lock);

   while (schedule->new_event_flag == 0)
      pthread_cond_wait (&schedule->cond_new_event, 
			 &schedule->mutex_cond_lock);

   /*trace (TR_THREAD, schedule->trace, "Processing event queue for %s", 
	  schedule->description);*/

   event = (event_t *) LL_GetHead (schedule->ll_events);

   if (event != NULL) {
     LL_Remove (schedule->ll_events, event);
     schedule->new_event_flag --;
   }
   else {
     trace (TR_THREAD, schedule->trace, "NULL event for %s -- strange...", 
	    schedule->description);
   }

   /* unlock */
   pthread_mutex_unlock (&schedule->mutex_cond_lock);

   if (event == NULL) {return (-1);}

   if (event->narg == 0)
     event->call_fn ();
   else if (event->narg == 1)
      event->call_fn (event->arg1);
   else if (event->narg == 2)
      event->call_fn (event->arg1, event->arg2);
   else if (event->narg == 3)
      event->call_fn (event->arg1, event->arg2, event->arg3);

   Delete_Event (event);

   return (1);
}



/* Delete_Event
 */
void Delete_Event (event_t *event) {

   Delete (event);
}



/* clear_schedule
 */
int clear_schedule (schedule_t *schedule) {

   /* obtain lock */
  if (pthread_mutex_trylock (&schedule->mutex_cond_lock) != 0) {
      trace (TR_THREAD, schedule->trace, 
	     "Going to block while clearing schedule for %s",
	     schedule->description);
      pthread_mutex_lock  (&schedule->mutex_cond_lock);
   }

   LL_Clear (schedule->ll_events);
   schedule->new_event_flag = 0;
   trace (TR_THREAD, schedule->trace, 
	  "Cleared schedule for %s\n",
	  schedule->description);

   

   pthread_mutex_unlock  (&schedule->mutex_cond_lock);

   return (1);
}



#ifndef HAVE_LIBPTHREAD

/* process_all_schedules
 */
int process_all_schedules () {
  schedule_t *schedule;
  event_t *event = NULL;
sigset_t old;

old = block_signal (SIGALRM);
  LL_Iterate (SCHEDULE_MASTER->ll_schedules, schedule) {
    if (schedule->new_event_flag > 0) {
      event = (event_t *) LL_GetHead (schedule->ll_events);
      break;
    }
  }

  if (event == NULL) {
recover_signal (old);
    return (0);
  }

  LL_Remove (schedule->ll_events, event);
  schedule->new_event_flag --;
  if (schedule->self)
    set_thread_id (schedule->self);
  trace (TR_THREAD, schedule->trace, "Event %x now run (%d events left) for %s\n", 
    event->call_fn, LL_GetCount (schedule->ll_events),
    schedule->description);
recover_signal (old);

   if (event->narg == 0)
      event->call_fn ();
   else if (event->narg == 1)
      event->call_fn (event->arg1);
   else if (event->narg == 2)
      event->call_fn (event->arg1, event->arg2);
   else if (event->narg == 3)
      event->call_fn (event->arg1, event->arg2, event->arg3);

   Delete_Event (event);
   return (1);
}


static int current_thread_id = 1;

pthread_t get_thread_id () {
   return (current_thread_id);
}

pthread_t set_thread_id (pthread_t id) {
   current_thread_id = id;
   return (current_thread_id);
}
#endif /* HAVE_LIBPTHREAD */


int schedule_count (schedule_t *schedule) {
  int num;
#ifndef HAVE_LIBPTHREAD
sigset_t old;
#endif /* HAVE_LIBPTHREAD */

  if (schedule == NULL) {return (-1);}
#ifndef HAVE_LIBPTHREAD
old = block_signal (SIGALRM);
#endif /* HAVE_LIBPTHREAD */
  /* obtain lock */
  pthread_mutex_lock (&schedule->mutex_cond_lock);
  num = LL_GetCount (schedule->ll_events);
  pthread_mutex_unlock (&schedule->mutex_cond_lock);
#ifndef HAVE_LIBPTHREAD
recover_signal (old);
#endif /* HAVE_LIBPTHREAD */

  return (num);
}
