/* 
 * $Id: io.c,v 1.8 1997/03/21 10:32:09 masaki Exp $
 */

/* TODO:
 *	- provide multiple inputs/outputs per process or per thread?
 *	  right now each process has only one input and one output stream.
 */

#include <stdio.h>
#include <stdarg.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <New.h>
#include <signal.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <io.h>

static int io_file_write (io_t *io, long tstamp, short type, short subtype,
			  int length, char *value);
static int io_msgq_write(io_t *io, long tstamp, short type, short subtype,
			 int length, char *value);
static int io_setup(enum IO_ATTR attr, io_t *IO, void *arg);
static int close_input(io_t *io);
static int close_output(io_t *io);
static int add_myq(char *client);
static int del_myq(int mqid, char *client);
static int get_destq(char *client);
static mrt_msg_t *io_file_read (io_t *io);
static mrt_msg_t *io_msgq_read (int mqid);
static int _io_start (io_t *io);
static int io_recv_mesg ();
extern io_t *MASTER_IO;
extern int errno;

char *S_MRT_MSG_TYPES[] = {
   "MSG_NULL",
   "START",
   "DIE",		
   "I_AM_DEAD",	
   "PEER_DOWN",	
   "BGP",
   "RIP",
   "IDRP"
};

char *S_MRT_MGS_BGP_TYPES[] = {
   "NULL",
   "UPDATE",	
   "PREF_UPDATE"
};




/* io_start
 * Start async IO on its own thread. Responsibility of
 * called func to delete mesg
 */
int 
io_start (io_t *io) {
  sigset_t set;
  int thread;

  sigemptyset (&set);
  sigaddset (&set, SIGALRM);
  sigaddset (&set, SIGHUP);
  pthread_sigmask (SIG_BLOCK, &set, NULL);

#ifdef  HAVE_PTHREAD_H
  if (pthread_create (&thread, NULL, (void *) _io_start, io) < 0) {
    perror ("thr_create failed"); 
    return (-1);
  }
#endif /* HAVE_PTHREAD_H */

  return (1);
}



int _io_start (io_t *io) {

  select_add_fd (io->in.fd, 1, (void *) io_recv_mesg, 
		 (void *) io);
  
#ifdef HAVE_LIBPTHREAD
  trace (TR_THREAD, io->trace, "THREAD starting for IO\n");
  while (1) 
    schedule_wait_for_event (io->schedule);
  exit (0);
#else
  return (0);
#endif /* HAVE_LIBPTHREAD */
}


static int io_recv_mesg (io_t *io) {
  mrt_msg_t *msg;

  while (1) {
    msg = io_read (io);
    
    if (msg == NULL) 
      return (-1);
    
     if (io->call_fn)
       io->call_fn (msg);
     else 
       free (msg);
  }

  exit (0);
}

      
/* io_read
 * Read and return the next message packet.
 */
mrt_msg_t *
io_read (io_t *io)
{
   switch (io->io_input_type) {
   case IO_NONE:
      return (NULL);
      break;
   case IO_FILE:
      return (io_file_read (io));
      break;
   case IO_MSGQ:
      return (io_msgq_read (io->in.mq.mqid));
      break;
   default:
      /* XXX printf should be removed in production code */
      printf ("\nUnknown io type %d", MASTER_IO->io_input_type);
      return NULL;
   }
}


static int
io_file_write (io_t *io, long tstamp, short type, short subtype,
	       int length, char *value) 
{
   int l = length;

  pthread_mutex_lock (&io->mutex_lock);

   tstamp = htonl (tstamp);
   type = htons (type);
   subtype = htons (subtype);
   length = htonl (length);

   if ((write (io->out.fd, &tstamp, 4) != 4) ||
       (write (io->out.fd, &type, 2) != 2) || 
       (write (io->out.fd, &subtype, 2) != 2) || 
       (write (io->out.fd, &length, 4) != 4) ||
       (write (io->out.fd, value, l) == -1)) {
     pthread_mutex_unlock (&io->mutex_lock);

      return (-1);
   }

     pthread_mutex_unlock (&io->mutex_lock);

   return (0);
}






static int
io_msgq_write(io_t *io, long tstamp, short type, short subtype,
	      int length, char *value)
{
   mrt_msg_t msg;

   msg.priority = 1;
   msg.tstamp = tstamp;
   msg.type = type;
   msg.subtype = subtype;
   msg.length = length;
   memcpy(msg.value, value, MAX_MSG_SIZE);

   fprintf(stderr, "io_msgq_write: sending msg (len %d) to mqid %d\n",
	   length, io->out.mq.mqid);

   if(msgsnd(io->out.mq.mqid, (struct msgbuf *)&msg, sizeof(mrt_msg_t), 0) < 0)
      return -1;
   else
      return 0;
}


/* do I need to buffer? probaly */
static mrt_msg_t *
io_file_read (io_t *io)
{
   mrt_msg_t *tmp;
   

   if ((tmp = (mrt_msg_t *)malloc (sizeof (mrt_msg_t))) == NULL)
      return NULL;

   if ((read (io->in.fd, &tmp->tstamp, 4) != 4) ||
       (read (io->in.fd, &tmp->type, 2) != 2) || 
       (read (io->in.fd, &tmp->subtype, 2) != 2) || 
       (read (io->in.fd, &tmp->length, 4) != 4)) {
     free (tmp);
      return NULL;
   }

    tmp->tstamp = ntohl (tmp->tstamp);
    tmp->type = ntohs (tmp->type);
    tmp->subtype = ntohs (tmp->subtype);
    tmp->length = ntohl (tmp->length);

   /* sanity checks */
   if ((tmp->length < 0) || (tmp->length > MAX_MSG_SIZE)) {
      free (tmp);
      return NULL;
   }
      
   if (read (io->in.fd, tmp->value, tmp->length) != tmp->length) {
      free (tmp);
      return NULL;
   }

   return (tmp);
}


static mrt_msg_t *
io_msgq_read(int mqid)
{
   mrt_msg_t *msg;

   fprintf(stderr, "io_msgq_read: invoked with mqid=%d\n", mqid);

   if((msg = (mrt_msg_t *)malloc (sizeof (mrt_msg_t))) == NULL)
      return NULL;

   if(msgrcv(mqid, (struct msgbuf *)msg, sizeof(mrt_msg_t), 0, 0) < 0) {
     free (msg);
      return NULL;
   }

   return msg;
}


static int
io_setup(enum IO_ATTR attr, io_t *IO, void *arg)
{
   char *name;

   switch(attr) {
   case IO_INNONE:
      if(close_input(IO) != 0)
	 return -1;
      break;

   case IO_OUTNONE:
      if(close_output(IO) != 0)
	 return -1;
      break;

   case IO_OUTFILE:
      name = (char *) arg;
      if(close_output(IO) != 0)
	 return -1;
      IO->io_output_type = IO_FILE;
      if (strcasecmp(name, "stdout") == 0) {
	 IO->out.fd = 1;
      } else {
	 if((IO->out.fd = open(name, (O_CREAT | O_WRONLY), 0666)) < 0)
	    return -1;
      }
      IO->io_out_name = strdup (name);
      break;

   case IO_OUTAPPEND:
      name = (char *) arg;
      if(close_output(IO) != 0)
	 return -1;
      IO->io_output_type = IO_FILE;
      if((IO->out.fd = open(name, (O_APPEND | O_WRONLY), 0666)) < 0)
	 return -1;
      break;

   case IO_INFILE:
      name = (char *) arg;
      if(close_input(IO) != 0)
	 return -1;
      IO->io_input_type = IO_FILE;
      if (strcasecmp(name, "stdin") == 0) {
	 IO->in.fd = 0;
      } else {
	 if ((IO->in.fd = open(name, O_RDONLY)) < 0)
	    return -1;
      }
      break;

   case IO_INMSGQ:
      name = (char *) arg;
      if(close_input(IO) != 0)
	 return -1;
      IO->io_input_type = IO_MSGQ;
      if((IO->in.mq.mqid = add_myq(name)) < 0)
	 return -1;
      strncpy(IO->in.mq.clientid, name, CLIENTLEN);
      break;

   case IO_OUTMSGQ:
      name = (char *) arg;
      if(close_output(IO) != 0)
	 return -1;
      IO->io_output_type = IO_MSGQ;
      if((IO->out.mq.mqid = get_destq(name)) < 0)
	 return -1;
      strncpy(IO->out.mq.clientid, name, CLIENTLEN);
      break;

   case IO_RECV_CALL_FN:
     IO->call_fn = arg;
     break;

   default:
      return -1;
   }

   return 0;
}


/* New_IO
 * Create a New IO object and initialize
 */
io_t *
New_IO(trace_t *tr) {
  
  io_t *tmp = (io_t *) malloc (sizeof (io_t));
  
  pthread_mutex_init (&tmp->mutex_lock, NULL);

  tmp->io_input_type = IO_NONE;
  tmp->io_output_type = IO_NONE;
  tmp->trace = trace_copy (tr);

  return (tmp);
}


/* io_set
 * Attach the input and output streams.
 */
int
io_set(io_t *io, int first, ...) 
{
  va_list    		ap;
  enum IO_ATTR		attr;

   /* Process the Arguments */
  va_start(ap, first);
   for (attr = (enum IO_ATTR)first; attr != IO_NULL;
	attr = va_arg(ap, enum IO_ATTR)) {
     if(io_setup(attr, io, va_arg(ap, char *)) != 0)
       return -1;
   }
   va_end(ap);

   return 0;
}


/* io_write
 * Place the given information into a message packet and
 * send it.
 */
int
io_write (io_t *io, long tstamp, short type, short subtype, int length, char *value) 
{

   if (tstamp <= 0)
      tstamp = time (NULL);

   switch (io->io_output_type) {
   case IO_NONE:
      return (0);
      break;
   case IO_FILE:
      return (io_file_write (io, tstamp, type, subtype, length, value));
      break;
   case IO_MSGQ:
      return (io_msgq_write(io, tstamp, type, subtype, length, value));
      break;
   default:
      /* XXX printf should be removed in production code */
      printf ("\nUnknown io type");
      return -1;
   }
}


/* Close input stream. */
static int
close_input(io_t *IO)
{
   switch(IO->io_input_type) {
   case IO_NONE:
      break;

   case IO_FILE:
      if(IO->in.fd > 2) {
	 if(close(IO->in.fd) != 0) {
	    return -1;
	 }
      }
      break;

   case IO_MSGQ:
      printf("close_input: removing qid %d, clientid %s\n",
	     IO->in.mq.mqid, IO->in.mq.clientid);
      if(del_myq(IO->in.mq.mqid, IO->in.mq.clientid) != 0) {
	 return -1;
      }
      break;

   default:
      return -1;
      break;
   }

   IO->io_input_type = IO_NONE;
   return 0;
}


/* Close output stream. */
static int
close_output(io_t *IO)
{
   switch(IO->io_output_type) {
   case IO_NONE:
      break;

   case IO_FILE:
      if(IO->out.fd > 2) {
	 if(close(IO->out.fd) != 0) {
	    return -1;
	 }
      }
      break;

   case IO_MSGQ:
      break;

   default:
      return -1;
      break;
   }

   IO->io_output_type = IO_NONE;
   return 0;
}


/* Create and register a message queue with the key registry.  Returns
   the MQID for our message queue, or -1 on error. */
static int
add_myq(char *client)
{
   int mqid;
   int server_qid;
   ARB_MSG_Struct msg;

   /* create queue */
   /*if((mqid = msgget(GETKEY(), (IPC_CREAT | IPC_EXCL | 0622))) < 0)*/
   if((mqid = msgget(GETKEY(), (IPC_CREAT | IPC_EXCL | 0666))) < 0)
      return -1;

   /* get server's qid */
   if((server_qid = msgget(MSGSERVER_KEY, 0222)) < 0)
      return -1;

   /* build message */
   msg.priority = 1;
   msg.type = MSG_SETMBOX;
   msg.sender = GETKEY();
   strncpy(msg.mqinfo.client, client, CLIENTLEN);
   msg.mqinfo.key = GETKEY();

   /* register */
   if(msgsnd(server_qid, (struct msgbuf *)&msg, sizeof(ARB_MSG_Struct), 0) < 0)
      return -1;

   return mqid;
}


/* De-allocate our message queue, and unregister from the client-ID
   registry. */
static int
del_myq(int mqid, char *client)
{
   int server_qid;
   ARB_MSG_Struct msg;

   /* get the server's qid */
   if((server_qid = msgget(MSGSERVER_KEY, 0222)) < 0)
      return -1;
  
   /* assemble the query */
   msg.priority = 1;
   msg.type = MSG_CLRMBOX;
   msg.sender = GETKEY();
   strncpy(msg.mqinfo.client, client, CLIENTLEN);
   msg.mqinfo.key = GETKEY();
  
   /* send query */
   if(msgsnd(server_qid, (struct msgbuf *)&msg, sizeof(ARB_MSG_Struct), 0) < 0)
      return -1;

  /* if that worked, remove our message queue */
   if(msgctl(mqid, IPC_RMID, (struct msqid_ds *) NULL) < 0)
      return -1;

   return 0;
}

/* Query the message queue key registry for the specified client ID,
   and open a message queue on the returned key.  Returns the open
   MQID, or -1 on error. */
static int
get_destq(char *client)
{
   int mqid;
   int server_qid;
   int need_rm = 1;
   ARB_MSG_Struct msg;
   int errno;

   /* Get a mailbox endpoint to receive the message from the server.
      Try to create one first; if it already exists, assume we made it
      and use it. */
   if((mqid = msgget(GETKEY(), (IPC_CREAT | IPC_EXCL | 0666))) < 0) {
     if(errno == EEXIST) {
       /*if((mqid = msgget(GETKEY(), 0622)) < 0) {*/
       if((mqid = msgget(GETKEY(), 0666)) < 0) {
	    return -1;
	 } else {
	    need_rm = 0;
	 }
      } else {
	 return -1;
      }
   }
  
   /* get the server's mailbox */
   if((server_qid = msgget(MSGSERVER_KEY, 0222)) < 0)
      return -1;
  
   /* assemble the query */
   msg.priority = 1;
   msg.type = MSG_GETMBOX;
   msg.sender = GETKEY();
   strncpy(msg.mqinfo.client, client, CLIENTLEN);
   msg.mqinfo.key = 0;
  
   /* send query */
   if(msgsnd(server_qid, (struct msgbuf *)&msg, sizeof(ARB_MSG_Struct), 0) < 0)
      return -1;
  
   /* read reply */
   if(msgrcv(mqid, (struct msgbuf *)&msg, sizeof(ARB_MSG_Struct), 0, 0) < 0)
      return -1;
  
   /* clean up if necessary */
   if(need_rm)
      msgctl(mqid, IPC_RMID, (struct msqid_ds *) NULL);

   /* finally, open the message queue with the client key */
   if((mqid = msgget(msg.mqinfo.key, 0222)) < 0)
      return -1;

   return mqid;
}




/* io_set_notify
 */
int io_set_notify (io_t *io, int method, void (*call_fn)()) {

  if (method == 1) {
    select_add_fd (io->in.fd, 1, call_fn, NULL);
    return (1);
  }
  else
    return (-1);

}






