/*
 * $Id: ripng.h,v 1.13 1997/03/21 09:11:48 masaki Exp $
 */

#ifndef _RIPNG_H
#define _RIPNG_H

#include <config.h>
#ifdef HAVE_IPV6
#include <linked_list.h>
#include <trace.h>
#include <timer.h>
#include <schedule.h>
#include <hash.h>
#include <mrt.h>
#include <proto.h>
#include <interface.h>


#define RIPNG_DEFAULT_PORT	521	/* this is correct one defined in ID */
#define RIPNG_IPV6_PRIORITY	(7<<24)

#undef RIPNG_POISONED_REVERSE	/* define if poisoned reverse */

#if 0				/* test */
#define RIPNG_TIMEOUT_INTERVAL	60
#define RIPNG_GARBAGE_INTERVAL	40
#define RIPNG_UPDATE_INTERVAL	15
#define RIPNG_UPDATE_JITTER 	0
#else
#define RIPNG_TIMEOUT_INTERVAL	180
#define RIPNG_GARBAGE_INTERVAL	120	/* after timeout */
#define RIPNG_UPDATE_INTERVAL	30
#define RIPNG_UPDATE_JITTER 	15
#endif

/* These values come from gated */
#define RIPNG_FLASH_DELAY_MIN 1
#define RIPNG_FLASH_DELAY_MAX 5

#define RIPNG_MIN_BUFSIZE ((INTERFACE_MASTER->max_mtu > 8 * 1024) ? \
                            INTERFACE_MASTER->max_mtu : 8 * 1024)

#define RIPNG_INFINITY		16

#define RIPNG_REQUEST	1
#define RIPNG_RESPONSE	2

#define RIPNG_VERSION	1

typedef struct _ripng_t {
    u_long interface_mask;	/* mask of interfaces ripng is configed for */
    int sockfd;			/* socket we are listeing on */
    HASH_TABLE *hash;		/* hash of prefixes */
    trace_t *trace;

    int (*update_call_fn) ();
    schedule_t *schedule;
    mtimer_t *timer;		/* timer used for sending update */
    mtimer_t *age;		/* aging routes */
    mtimer_t *flash;		/* flash update */

    pthread_mutex_t send_mutex_lock;
    pthread_mutex_t route_table_mutex_lock;
    pthread_mutex_t flash_update_mutex_lock;
} ripng_t;

typedef struct _ripng_attr_t {
    u_char type;
    prefix_t *nexthop;
    int ref_count;
    gateway_t *gateway;
    u_long tag;

    u_char metric;
} ripng_attr_t;


typedef struct _ripng_route_t {
    u_long flags;
    prefix_t *prefix;
    ripng_attr_t *attr;
    time_t time;		/* time received or updated */
    time_t delete;		/* time started deletion process */
} ripng_route_t;


/* route flags */
#define RT_RIPNG_CHANGE	0x1
#define RT_RIPNG_DELETE	0x2

extern ripng_t *RIPNG;

/* user settable attributes */
enum RIPNG_ATTR {
    RIPNG_NULL = 0,
    RIPNG_TRACE_STRUCT,
    RIPNG_RT_UPDATE_FN,		/* function to call when ripng routes change */
    RIPNG_USE_PORTSERVER	/* use portserver library for listening */
};


/* public functions */

int ripng_init (trace_t *trace);
void ripng_start ();
int ripng_init_listen (void);

void ripng_timeout_routes (void);
void ripng_receive_update (void);
int ripng_process_update (LINKED_LIST *ll_prefixes, LINKED_LIST *ll_attr);

void ripng_send_request (LINKED_LIST *ll_prefixes);
void ripng_send_routes (interface_t *interface, struct sockaddr_in6 *host,
			int change);
void ripng_output_processing (int change);

int ripng_policy (prefix_t *prefix, ripng_attr_t *attr, interface_t *out);
int ripng_update_route (prefix_t *prefix, generic_attr_t *new,
		    generic_attr_t *old);

void ripng_timer_update ();
void ripng_flash_update ();
void ripng_set_flash_update ();
void ripng_show ();
void ripng_set (int first, ...);
void ripng_show_routing_table (uii_connection_t *uii);

int ripng_sendmsgto (int sockfd, char *buffer, int buflen, int flag,
	   struct sockaddr_in6 *addr, int addrlen, interface_t *interface);
int ripng_recvmsgfrom (int sockfd, char *buffer, int buflen, int flag,
	       struct sockaddr_in6 *from, int *fromlen, interface_t **ifp);

ripng_route_t *ripng_new_route (prefix_t *prefix6, ripng_attr_t *attr);
void ripng_delete_route (ripng_route_t *route);
ripng_attr_t *ripng_new_attr (int metric);
ripng_attr_t *ripng_ref_attr (ripng_attr_t *attr);
void ripng_deref_attr (ripng_attr_t *attr);

/* packet.c */
void ripng_process_packet_response (gateway_t *gateway, char *update,
	       int bytes, LINKED_LIST *ll_prefixes, LINKED_LIST *ll_attr);
int ripng_process_packet_request (struct sockaddr_in6 *from, char *update,
				  int bytes, interface_t *interface);

#endif /* HAVE_IPV6 */
#endif /* _RIPNG_H */
