/*
 * property list management code.
 *
 * G. Robert Malan (rmalan@eecs.umich.edu)
 * 2/20/97
 *
 * $Id: propertyList.c,v 1.1 1997/03/31 01:17:31 labovit Exp $
 */

#include <stdio.h>
#include <string.h>
#include "propertyList.h"

/*****************************************************************************
 *
 *  Property List library routines:
 *
 *****************************************************************************/


plist_t 
createPropertyList(void)
{
	plist_t  plist;

	plist = (plist_t)malloc(sizeof(struct plist));
	if (plist) 
		plist->head = 0;

	return plist;
}

void
destroyPropertyList(plist_t plist)
{
	plist_entry_t entry, next;

	if (!plist)
		return;

	entry = plist->head;
	while(entry) {
		if (entry->key)
			free(entry->key);
		if (entry->value)
			free(entry->value);
	
		next = entry->next;
		free(entry);
		entry = next;
	}
	free(plist);
}



/* Returns 1 if success, 0 if fails */
int
updateProperty(plist_t plist, char * key, char * value)
{
	plist_entry_t found;
	plist_entry_t new_entry;

	if (!key)
		return 1;

        for (found = plist->head; found != NULL; found = found->next) 
		if ((strcmp(found->key, key) == 0))
                        break;

	if (found != NULL) {

		/*
		 * Already have a key, so replace it's value.
		 */
		if (found->value)
			free(found->value);
		if (!(found->value = (char *)malloc(strlen(value) + 1)))
			return 0;
		strcpy(found->value, value);

	} else {
		/*
		 * Not there, so just append it to beginning of the 
		 * header's linked list.
		 */

		/* Allocate it... */
		if (!(new_entry = (plist_entry_t)
					malloc(sizeof(struct plist_entry))))
			return 0;

		/* Fill it in... */
		if (!(new_entry->key = (char *)malloc(strlen(key) + 1))) {
			free(new_entry);
			return 0;
		}
		strcpy(new_entry->key, key);
		if (!(new_entry->value = (char *)malloc(strlen(value) + 1))) {
			free(new_entry->key);
			free(new_entry);
			return 0;
		}
		strcpy(new_entry->value, value);

		/* Link it up... */
		new_entry->next = plist->head;
		if (new_entry->next)
			new_entry->next->previous = new_entry;
		new_entry->previous = NULL;
		plist->head = new_entry;
	}

	return 1;
}


plist_t
copyPropertyList(plist_t from)
{
	plist_t retplist;
	plist_entry_t entry;

	if (!from) 
		return (NULL);

	if (!(retplist = createPropertyList()))
		return (NULL);

	for (entry = from->head; entry; entry = entry->next) {
		if (!updateProperty(retplist, entry->key, entry->value)) {
			destroyPropertyList(retplist);
			return 0;
		}
	}

	return (retplist);
}


void
deleteProperty(plist_t plist, char * key)
{ 
	plist_entry_t entry;

	if (!key)
		return;

	for (entry = plist->head; entry; entry = entry->next) 
		if (!strcmp(entry->key, key)) 
			break;

	if (!entry)
		return;

	free(entry->key);
	if (entry->value)
		free(entry->value);

	if (entry->previous == NULL) {
		plist->head = entry->next;
		if (entry->next)
			entry->next->previous = NULL;
	}else{
		entry->previous->next = entry->next;
	}
	if (entry->next != NULL) {
		entry->next->previous = entry->previous;
	}

	free(entry);
}


char *
getProperty(plist_t plist, char * key)
{
	plist_entry_t entry;

	for (entry = plist->head; entry; entry = entry->next) {

		if (strcmp(entry->key, key) == 0) 
			return (strdup(entry->value));
	}

	return (NULL);
}


/*
 * iterateProperty:
 *
 * cookie = 0 --> starting at beginning.
 * ret 1 if key value are valid.
 * ret 0 if end of list.
 *
 */
int
iteratePropertyList(plist_t plist, plist_entry_t * cookie,  
		    char ** key, char ** value)
{
	if (!plist)
		return 0;

	if (!*cookie) {
		*cookie = plist->head;
	}else{
		*cookie = (*cookie)->next;
	}

	if (!*cookie)
		return 0;
		
	*key = strdup((*cookie)->key);
	*value = strdup((*cookie)->value);

	return 1;
}


/**
 * For debugging....
 */
void
dumpPropertyList(plist_t plist)
{
	int ret;
	char * curkey;
	char * curval;
	plist_entry_t	cookie;

	fprintf(stderr, "plist dump:\n");

	if (!plist) {
		fprintf(stderr, "plist empty.\n");
		return;
	}

	cookie = 0;
	ret = iteratePropertyList(plist, &cookie, &curkey,
&curval);
	while(ret) {

		fprintf(stderr, "key '%s' val '%s'\n", curkey, curval);

		free(curkey);
		free(curval);
		ret = iteratePropertyList(plist, &cookie, &curkey,
						&curval);
	}

	fprintf(stderr, "\n");
}

