/*
 * $Id: select.c,v 1.13 1997/04/03 15:43:33 labovit Exp $
 */

#include <assert.h>
#include <stdio.h>
#include <config.h>
#include <sys/types.h>
#include <sys/time.h>
#include <signal.h>
#include <New.h>
#include <unistd.h>
#include <linked_list.h>
#include <mrt.h>
#include <trace.h>
#include <select.h>
#include <version.h>
#include <errno.h>
#ifdef __linux__
#include <sys/resource.h>
#endif

#ifndef HAVE_PTHREAD_H
#include <pthread_fake.h>
#else
#include <pthread.h>
#endif /* HAVE_PTHREAD_H */

Select_Struct *SELECT_MASTER;

static void __init_select ();


/* init_select
 */
int init_select (trace_t *tr) {
  int ret;
  sigset_t set;
  int fd[2];

  assert (SELECT_MASTER == NULL);

  SELECT_MASTER = New (Select_Struct);
  SELECT_MASTER->ll_descriptors = LL_Create (0);
  SELECT_MASTER->trace = trace_copy (tr);

  pthread_mutex_init (&SELECT_MASTER->mutex_lock, NULL);

  pipe (fd);
  SELECT_MASTER->interrupt_fd[1] = fd[1];
  SELECT_MASTER->interrupt_fd[0] = fd[0];
  FD_SET(fd[0], &SELECT_MASTER->fdvar_read);

  if (mrt_thread_create 
      ("Select Thread", NULL, 
       (void *) __init_select, NULL) == NULL) {return (-1);}

  return (1);
}


static void __init_select () {
  sigset_t new;

#ifndef HAVE_LIBPTHREAD
  return;
#else

  SELECT_MASTER->self = pthread_self (); 

  sigemptyset(&new);
  sigaddset(&new, SIGALRM);
  pthread_sigmask(SIG_BLOCK, &new, NULL);

  while (1) 
    mrt_select ();
#endif /* THREAD */

}





void __clear_select_interrupt () {
  char buf[2];

  printf ("\nhere in interrupt!\n");
  read (SELECT_MASTER->interrupt_fd[0], buf, 1);
  select_enable_fd (SELECT_MASTER->interrupt_fd[0]);
  
  /*exit (0);*/

}


/* select_add_fd
 * Add a file descriptor to the list of fds which we are blocking on in a 
 * select call.
 */
int select_add_fd (int fd, int type_mask, void (*call_fn)(), void *arg) {
   Descriptor_Struct *tmp = New (Descriptor_Struct);
   char buf[1];
   
   /* save us from ourselves */ /* no more -- masaki */
   assert (SELECT_MASTER);

   tmp->fd = fd;
   tmp->call_fn = call_fn;
   tmp->arg = arg;
   tmp->marked_for_deletion = 0;

   pthread_mutex_lock (&SELECT_MASTER->mutex_lock);

   if (type_mask & 1)
       FD_SET(fd, &SELECT_MASTER->fdvar_read);

   if (type_mask & 2)
       FD_SET(fd, &SELECT_MASTER->fdvar_write);

   if (type_mask & 4)
       FD_SET(fd, &SELECT_MASTER->fdvar_except);

   LL_Add (SELECT_MASTER->ll_descriptors, tmp);

   pthread_mutex_unlock (&SELECT_MASTER->mutex_lock);

   write (SELECT_MASTER->interrupt_fd[1], buf, 1);

   return (1);
}


/* select_delete_fd
 */
int select_delete_fd (int fd) {
   Descriptor_Struct *tmp;
   char buf[2];

   pthread_mutex_lock (&SELECT_MASTER->mutex_lock);

   LL_Iterate (SELECT_MASTER->ll_descriptors, tmp) 
      if (tmp->fd == fd) {
	 tmp->marked_for_deletion = 1;
	 FD_CLR(fd, &SELECT_MASTER->fdvar_read);
	 FD_CLR(fd, &SELECT_MASTER->fdvar_write);
	 FD_CLR(fd, &SELECT_MASTER->fdvar_except);

	 pthread_mutex_unlock (&SELECT_MASTER->mutex_lock);
	 close (fd);

	 write (SELECT_MASTER->interrupt_fd[1], buf, 1);
	 return (1);
      }

   pthread_mutex_unlock (&SELECT_MASTER->mutex_lock);

   return (-1);
}


/* mrt_select
 */
int mrt_select ()
{
   fd_set fdvar_read, fdvar_write, fdvar_except;
   Descriptor_Struct *tmp; 
   int mask_block, mask;
   struct timeval *timeout, tv;

#ifdef HAVE_LIBPTHREAD
   timeout = NULL;
#else
   tv.tv_sec = 5; /* XXX */
   tv.tv_usec = 0;
   timeout = &tv;
#endif /* HAVE_LIBPTHREAD */
   fdvar_read = SELECT_MASTER->fdvar_read;
   fdvar_write = SELECT_MASTER->fdvar_write;
   fdvar_except = SELECT_MASTER->fdvar_except;

   if (select (FD_SETSIZE, &fdvar_read, 
	       &fdvar_write, &fdvar_except, timeout) < 0) {
     return (-1);
   }
   
   pthread_mutex_lock (&SELECT_MASTER->mutex_lock);

   if (FD_ISSET (SELECT_MASTER->interrupt_fd[0], &fdvar_read)) {
     char buf[2];
     read (SELECT_MASTER->interrupt_fd[0], buf, 1);
   }

   LL_Iterate (SELECT_MASTER->ll_descriptors, tmp) {
      if (tmp->marked_for_deletion == 1) {
	    Descriptor_Struct *prev = 
	       LL_GetPrev (SELECT_MASTER->ll_descriptors, tmp);
	    LL_Remove (SELECT_MASTER->ll_descriptors, tmp);
	    tmp = prev;
      }
      
      if ((tmp != NULL) &&
	  ((FD_ISSET (tmp->fd, &fdvar_read)) ||
	   (FD_ISSET (tmp->fd, &fdvar_write)) ||
	   (FD_ISSET (tmp->fd, &fdvar_except)))) {

	pthread_mutex_unlock (&SELECT_MASTER->mutex_lock);

/* #ifdef HAVE_LIBPTHREAD */
/* Clearing the bit is required because sometimes this routine is called 
   again before reading data waiting. */
	FD_CLR(tmp->fd, &SELECT_MASTER->fdvar_read);
	/*FD_CLR(tmp->fd, &SELECT_MASTER->fdvar_write);*/
/* #endif /* HAVE_LIBPTHREAD */

	trace (TRACE, SELECT_MASTER->trace, 
	  "mrt_select (%d,%x,%x,%x) calling %x(%x)\n", tmp->fd, 
	  *(int *)&fdvar_read, *(int *)&fdvar_write, *(int *)&fdvar_except, 
	  tmp->call_fn, tmp->arg);
	tmp->call_fn(tmp->arg);
	break; 
      }
   }

#ifdef notdef
/*
 * I'm afraid the following part is wrong, at least mask is not initialized
 *    -- masaki */ 
 */
   /* turn alarm back on */
#ifdef HAVE_SIGRELSE
   sigrelse (SIGALRM);
#else
   sigsetmask (mask);
#endif /* HAVE_SIGRELSE */
#endif 

   pthread_mutex_unlock (&SELECT_MASTER->mutex_lock);
}





int select_disable_fd (int fd) {
   Descriptor_Struct *tmp;
   char buf[2];

   pthread_mutex_lock (&SELECT_MASTER->mutex_lock);

   LL_Iterate (SELECT_MASTER->ll_descriptors, tmp) 
      if (tmp->fd == fd) {
	 FD_CLR(fd, &SELECT_MASTER->fdvar_read);
	 FD_CLR(fd, &SELECT_MASTER->fdvar_write);
	 FD_CLR(fd, &SELECT_MASTER->fdvar_except);

	 pthread_mutex_unlock (&SELECT_MASTER->mutex_lock);

	 write (SELECT_MASTER->interrupt_fd[1], buf, 1);

	 return (1);
      }

   pthread_mutex_unlock (&SELECT_MASTER->mutex_lock);

   return (-1);
}


/* select_enable_fd
 * the mrt_select master must be notified that a socket is ready for selection again
 * after the select call routine has fired
 */
int select_enable_fd (int fd) {
   Descriptor_Struct *tmp;
   char buf[2];

   pthread_mutex_lock (&SELECT_MASTER->mutex_lock);

   LL_Iterate (SELECT_MASTER->ll_descriptors, tmp) 
      if (tmp->fd == fd) {
	 FD_SET(fd, &SELECT_MASTER->fdvar_read);
	 /*FD_SET(fd, &SELECT_MASTER->fdvar_write);*/
	 /*FD_SET(fd, &SELECT_MASTER->fdvar_except);*/

	 pthread_mutex_unlock (&SELECT_MASTER->mutex_lock);

	 write (SELECT_MASTER->interrupt_fd[1], buf, 1);
	 return (1);
      }

   /* this fd does not exist! */
   pthread_mutex_unlock (&SELECT_MASTER->mutex_lock);
   return (-1);
}


/* called on exit */
int
select_shutdown () 
{
  Descriptor_Struct *tmp;

  LL_Iterate (SELECT_MASTER->ll_descriptors, tmp) {
    printf ("Shutting down %d\n", tmp->fd);
    /*if (shutdown (tmp->fd, 2) != 0) {
      perror ("error in shutdown");
    }
    t_snddis (tmp->fd, NULL);*/
    if (close (tmp->fd) != 0) 
      perror ("error in close");
  }

  return (1);
}


