/*
 * $Id: rib.c,v 1.15 1997/03/21 17:24:37 masaki Exp $
 */


#include <stdio.h>
#include <sys/types.h>
#include <stack.h>
#include <mrt.h>
#include <interface.h>
#include <rib.h>
#include <bgp.h>
#include <rip.h>
#include <ripng.h>

void Delete_Route_Node (route_node_t * rt_node);
void Delete_Route_Head (route_head_t * route_head);
u_int rib_hash_fn (prefix_t * prefix, u_int size);
u_int rib_lookup_fn (prefix_t * tmp1, prefix_t * tmp2);

char *
route_toa (prefix_t * prefix, generic_attr_t * attr, int pref)
{
    static char tmp1[MAXLINE];
    char tmp2[MAXLINE], *buff = tmp1;

    sprintf (buff, "%s/%d nh %s",
	     prefix_toa (prefix), prefix->bitlen,
	     prefix_toa2 (attr->nexthop, tmp2));
    buff = buff + strlen (buff);

    if (attr->gateway) {
	if (!prefix_compare (attr->gateway->prefix,
			     attr->nexthop)) {
	    sprintf (buff, " gw %s", prefix_toa (attr->gateway->prefix));
	    buff = buff + strlen (buff);
	}
    }
    sprintf (buff, " proto %s pref %d",
	     proto2string (attr->type), pref);
    return (tmp1);
}


/* rib_get_route_head
 * get "head" of routing table entry for a given prefix. Create a head
 * and radix-tree entry if one does not aleady exist
 */
route_head_t *
rib_get_route_head (rib_t * rib, prefix_t * prefix)
{
    route_head_t *route_head;

#ifdef HAVE_LIBPTHREAD
    assert (pthread_mutex_trylock (&rib->mutex_lock));
#endif /* HAVE_LIBPTHREAD */

    route_head = HASH_Lookup (rib->hash, prefix);

    if (route_head == NULL) {
	route_head = New_Route_Head (prefix);
	HASH_Insert (rib->hash, route_head);
	trace (NORM, rib->trace, "RIB new route head for %s/%d\n",
	       prefix_toa (prefix), prefix->bitlen);
    }

    return (route_head);
}


/* rib_find_route_node
 * given a prefix AND gateway AND type, find route node entry
 * return NULL if one does not exist
 */
route_node_t *
rib_find_route_node (rib_t * rib,
		     prefix_t * prefix, generic_attr_t * attr)
{
    route_head_t *route_head;
    route_node_t *route_node;

#ifdef HAVE_LIBPTHREAD
    assert (pthread_mutex_trylock (&rib->mutex_lock));
#endif /* HAVE_LIBPTHREAD */

    if (route_head = HASH_Lookup (rib->hash, prefix)) {
	LL_Iterate (route_head->ll_route_nodes, route_node) {
	    if (route_node->attr->gateway == attr->gateway &&
		/* a route from the kernel always matches */
		(route_node->attr->type == PROTO_KERNEL ||
		 route_node->attr->type == attr->type)) {
		return (route_node);
	    }
	}
    }

    return (NULL);
}


/* rib_add_route_node
 * create a route_node and add it to the appropriate route_head,
 * or create a route_head if not already exist. Return
 * the new route_node
 */

/*
 * I'd like to add an additional argument that means which protocol
 * calls this routine.
 */

route_node_t *
rib_add_route (rib_t * rib, prefix_t * prefix,
	       generic_attr_t * attr, int pref, int pref2)
{
    route_node_t *route_node;
    route_head_t *route_head;
    char tmp1[MAXLINE], tmp2[MAXLINE], tmp3[MAXLINE];

#ifdef HAVE_LIBPTHREAD
    assert (pthread_mutex_trylock (&rib->mutex_lock));
#endif /* HAVE_LIBPTHREAD */

    route_head = rib_get_route_head (rib, prefix);

    /*
     * for system like Linux that has multiple routes for the same destination
     * less prefered (but metric currently unavailable) will be deleted here
     * regardless of its gateway or nexthop
     */
    if (attr->type == PROTO_KERNEL) {
	LL_Iterate (route_head->ll_route_nodes, route_node) {
	    if (route_node->attr->type == PROTO_KERNEL) {
		if (pref < route_node->pref) {
		    /* remove the existing */

		    rib_kernel_update_route (KERNEL_UPDATE_DEL, route_head->prefix,
					     route_node->attr);
		    LL_Remove (route_head->ll_route_nodes, route_node);
		    Delete_Route_Node (route_node);
		    break;
		}
		else {
		    /* ignore this */

		    rib_kernel_update_route (KERNEL_UPDATE_DEL, prefix, attr);
		    return (NULL);
		}
	    }
	}
    }

    /* check to see if we already got this route 
       from the gateway in the same proto or kernel */

    route_node = rib_find_route_node (rib, prefix, attr);

    if (route_node) {

	if (pref < route_node->pref) {
	    /* the same gateway or better preference */
	    /* implicit withdarw of old route -- delete it */
	    /* on deleting, do I have to notify the proto engine which supplied? */

	    trace (NORM, rib->trace,
		   "RIB update: %s -> %d\n",
	      route_toa (prefix, route_node->attr, route_node->pref), pref);

#if 1
	    /* I don't like to have the same kernel routes in RIB */
	    if (route_node->attr->type == PROTO_KERNEL &&
		attr->type == PROTO_KERNEL) {
		rib_kernel_update_route (KERNEL_UPDATE_DEL, route_head->prefix,
					 route_node->attr);
	    }
#endif

	    rib_inter_proto_update (prefix, attr, route_node->attr);

	    /* overwrite. is it OK? */
	    route_node->pref = pref;
	    Deref_Generic_Attr (route_node->attr);
	    route_node->attr = Ref_Generic_Attr (attr);

	    if (route_head->active != route_node)
		rib_replace_active (rib, route_head, NULL);

	    return (route_node);
	}
	else {
	    /* same or bad .. ignore it */

	    if (attr->type == PROTO_KERNEL) {
		rib_kernel_update_route (KERNEL_UPDATE_DEL, prefix, attr);
	    }

	    trace (NORM, rib->trace,
		   "RIB ignore: %s (now %d)\n",
		   route_toa (prefix, attr, pref), route_node->pref);

	    return (route_node);
	}
    }

    /* first time we've heard from this gateway in this proto */
    route_node = New_Route_Node (route_head, pref, attr);
    LL_Add (route_head->ll_route_nodes, route_node);

    rib_inter_proto_update (prefix, attr, NULL);

    /* is this new route better than the active route ? */
    if ((route_head->active == NULL) ||
	(route_node->pref < route_head->active->pref)) {

	rib_replace_active (rib, route_head, route_node);
    }
    else {
	trace (NORM, rib->trace,
	       "RIB add: %s\n", route_toa (prefix, attr, pref));
    }

    return (route_node);
}


int 
rib_close (rib_t * rib)
{
#ifdef HAVE_LIBPTHREAD
    pthread_mutex_unlock (&rib->mutex_lock);
#endif /* HAVE_LIBPTHREAD */
    return (1);
}


int 
rib_open (rib_t * rib)
{
#ifdef HAVE_LIBPTHREAD
    if (pthread_mutex_trylock (&rib->mutex_lock) != 0) {
	trace (TR_DEBUG, rib->trace, "Going to block in rib_open\n");
	pthread_mutex_lock (&rib->mutex_lock);
    }
#endif /* HAVE_LIBPTHREAD */
    return (1);
}


void
rib_del_route (rib_t * rib, prefix_t * prefix, generic_attr_t * attr)
{
    route_node_t *route_node;

#ifdef HAVE_LIBPTHREAD
    assert (pthread_mutex_trylock (&rib->mutex_lock));
#endif /* HAVE_LIBPTHREAD */
    route_node = rib_find_route_node (rib, prefix, attr);
    assert (route_node);
    rib_inter_proto_update (prefix, NULL, route_node->attr);
    rib_delete_route_node (rib, route_node);
}


void 
rib_hold_route (rib_t * rib, prefix_t * prefix, generic_attr_t * attr,
		int pref)
{
    route_node_t *route_node;
    char tmp1[MAXLINE], tmp2[MAXLINE];

#ifdef HAVE_LIBPTHREAD
    assert (pthread_mutex_trylock (&rib->mutex_lock));
#endif /* HAVE_LIBPTHREAD */
    route_node = rib_find_route_node (rib, prefix, attr);
    assert (route_node);
    if (route_node->route_head->active == route_node) {
	if (route_node->pref >= HOLD_PREF && pref < HOLD_PREF) {
	    /* recovering from holddown */
	    rib_kernel_update_route (KERNEL_UPDATE_ADD, prefix, attr);
	    route_node->pref = pref;
	    trace (NORM, rib->trace,
		   "RIB holddown recovery: %s\n",
		   route_toa (route_node->attr->nexthop,
			      route_node->attr, route_node->pref));
/*  rib_inter_proto_hold (prefix, route_node->attr); */
	    return;
	}
	else if (route_node->pref < HOLD_PREF && pref >= HOLD_PREF) {
	    /* going into holddown */
	    trace (NORM, rib->trace,
		   "RIB holddown: %s\n",
		   route_toa (route_node->attr->nexthop,
			      route_node->attr, route_node->pref));
	    rib_kernel_update_route (KERNEL_UPDATE_DEL, prefix, attr);
	    route_node->pref = pref;
/*  rib_inter_proto_hold (prefix, route_node->attr); */
	}
    }
    else {
	route_node->pref = pref;
    }
    rib_replace_active (rib, route_node->route_head, NULL);
}


void 
rib_delete_route_node (rib_t * rib, route_node_t * route_node)
{
    route_head_t *route_head;
    route_node_t *best, *tmp;
    char tmp1[MAXLINE], tmp2[MAXLINE];

#ifdef HAVE_LIBPTHREAD
    assert (pthread_mutex_trylock (&rib->mutex_lock));
#endif /* HAVE_LIBPTHREAD */

    route_head = route_node->route_head;
    LL_Remove (route_head->ll_route_nodes, route_node);

    trace (NORM, rib->trace,
	   "RIB delete: %s\n",
	route_toa (route_head->prefix, route_node->attr, route_node->pref));

    if (route_node->protocol_mask == 0) {
	/* what's protocol_mask? */
    }

    if (route_head->active == route_node) {
	rib_replace_active (rib, route_head, NULL);
    }

    /* delete here */
    Delete_Route_Node (route_node);
}


void 
rib_replace_active (rib_t * rib, route_head_t * route_head,
		    route_node_t * best)
{
    route_node_t *tmp;
    char tmp1[MAXLINE], tmp2[MAXLINE];
    int deleted = 0;

    if (best == NULL) {
	/* find new active route */
	LL_Iterate (route_head->ll_route_nodes, tmp) {
	    if ((best == NULL) || ((tmp->pref < best->pref) &&
				   (tmp->pref2 < best->pref2))) {
		best = tmp;
	    }
	}
    }

    if (route_head->active == NULL ||
	route_head->active->pref < HOLD_PREF)
	deleted++;

    if (route_head->active &&
	route_head->active->pref < HOLD_PREF &&
	(best == NULL || (
			     route_head->active != best &&
			  prefix_compare (route_head->active->attr->nexthop,
					  best->attr->nexthop) == 0))) {
	route_node_t *route_node = route_head->active;

	/* before replacing the active route, 
	   remove the current active route from the kernel */

	assert (route_node->attr->nexthop);
	if (route_node->attr->type != PROTO_CONNECTED) {
	    rib_kernel_update_route (KERNEL_UPDATE_DEL, route_head->prefix,
				     route_node->attr);
	    deleted++;
	}
    }

    if (best == NULL) {		/* there is no route nodes */
	trace (NORM, rib->trace, "RIB delete route head for %s/%d\n",
	       prefix_toa (route_head->prefix), route_head->prefix->bitlen);
	HASH_Remove (rib->hash, route_head);
	Delete_Route_Head (route_head);
	return;
    }

    if (route_head->active == best)
	return;

    route_head->last_active = route_head->active;
    route_head->active = best;

    trace (NORM, rib->trace,
	   "RIB active: %s\n",
	   route_toa (route_head->prefix, best->attr, best->pref));

    assert (best->attr->nexthop);
    if (best->attr->type != PROTO_KERNEL &&
	best->attr->type != PROTO_CONNECTED && deleted &&
	best->pref < HOLD_PREF) {

	rib_kernel_update_route (KERNEL_UPDATE_ADD, route_head->prefix,
				 best->attr);
    }
}


void 
rib_inter_proto_update (prefix_t * prefix, generic_attr_t * new,
			generic_attr_t * old)
{

    if (new && old && (MRT->redist[new->type] & MRT->redist[old->type])) {
	if (prefix->family == AF_INET) {
	    if (BIT_TEST (MRT->redist[old->type], (1 << PROTO_BGP))) {
		assert (old->type != PROTO_BGP);
		/* XXX I don't know how to update it */
	    }
	    if (BIT_TEST (MRT->redist[old->type], (1 << PROTO_RIP))) {
		assert (old->type != PROTO_RIP);
		/* XXX I don't know how to update it */
	    }
	}
#ifdef HAVE_IPV6
	else if (prefix->family == AF_INET6) {
	    if (BIT_TEST (MRT->redist[old->type], (1 << PROTO_RIPNG))) {
		assert (old->type != PROTO_RIPNG);
		ripng_update_route (prefix, new, old);
	    }
	}
#endif /* HAVE_IPV6 */
	return;
    }

    if (old) {
	if (prefix->family == AF_INET) {
	    if (BIT_TEST (MRT->redist[old->type], (1 << PROTO_BGP))) {
		assert (old->type != PROTO_BGP);
		/* XXX I don't know how to delete it */
	    }
	    if (BIT_TEST (MRT->redist[old->type], (1 << PROTO_RIP))) {
		assert (old->type != PROTO_RIP);
		/* XXX I don't know how to delete it */
	    }
	}
#ifdef HAVE_IPV6
	else if (prefix->family == AF_INET6) {
	    /* avoid looping, don't go to the same proto */
	    if ((new == NULL || new->type != PROTO_RIPNG) &&
	        (BIT_TEST (MRT->redist[old->type], (1 << PROTO_RIPNG)))) {
		assert (old->type != PROTO_RIPNG);
		ripng_update_route (prefix, NULL, old);
	    }
	}
#endif /* HAVE_IPV6 */
    }

    if (new) {
	if (prefix->family == AF_INET) {
	    if (BIT_TEST (MRT->redist[new->type], (1 << PROTO_BGP))) {
		bgp_attr_t *attr = New (bgp_attr_t);
		assert (new->type != PROTO_BGP);
		attr->type = new->type;
		attr->gateway = new->gateway;
		attr->nexthop = Ref_Prefix (new->nexthop);
		attr->ref_count = 1;
		view_open (BGP->view);
		bgp_add_route (BGP->view, prefix, attr, 0);
		view_close (BGP->view);
	    }
	    if (BIT_TEST (MRT->redist[new->type], (1 << PROTO_RIP))) {
		rip_attr_t *attr = New_Rip_Attr (1);
		assert (new->type != PROTO_RIP);
		attr->type = new->type;
		attr->nexthop = Ref_Prefix (new->nexthop);
		attr->gateway = new->gateway;
		attr->metric = 1;
		attr->tag = 0;
		attr->ref_count = 1;	/* should be done in New_Rip_Attr */
		New_Rip_Route (prefix, attr);
	    }
	}
#ifdef HAVE_IPV6
	else if (prefix->family == AF_INET6) {
	    if (BIT_TEST (MRT->redist[new->type], (1 << PROTO_RIPNG))) {
		assert (new->type != PROTO_RIPNG);
		ripng_update_route (prefix, new, NULL);
	    }
	}
#endif /* HAVE_IPV6 */
    }
}


route_node_t *
New_Route_Node (route_head_t * route_head,
		int pref, generic_attr_t * attr)
{

    route_node_t *tmp = (route_node_t *) New (route_node_t);
    tmp->route_head = route_head;
    tmp->pref = pref;
    tmp->attr = Ref_Generic_Attr (attr);
    return (tmp);
}


route_head_t *
New_Route_Head (prefix_t * prefix)
{
    route_head_t *tmp = New (route_head_t);

    tmp->prefix = Ref_Prefix (prefix);
    tmp->ll_route_nodes = LL_Create (0);
    tmp->active = NULL;

    return (tmp);
}


void 
Delete_Route_Head (route_head_t * tmp)
{

    Deref_Prefix (tmp->prefix);
    LL_Destroy (tmp->ll_route_nodes);
    Delete (tmp);
}


void 
Delete_Route_Node (route_node_t * rt_node)
{
    Deref_Generic_Attr (rt_node->attr);
    Delete (rt_node);
}


rib_t *
New_Rib (int maxbitlen)
{
    route_head_t head;
    rib_t *tmp = New (rib_t);

    /*tmp->ll_route_nodes = LL_Create (0); */
    tmp->ll_changed_route_nodes = LL_Create (0);

    /*if (LL_ATTR_MEMORY == NULL)
       init_attr_memory (); */

#ifdef HAVE_LIBPTHREAD
    if (pthread_mutex_init (&tmp->mutex_lock, NULL) != 0) {
	perror ("rib pthread_mutex_init");
	exit (0);
    }
#endif /* HAVE_LIBPTHREAD */
    tmp->hash = HASH_Create (100,
			  HASH_KeyOffset, HASH_Offset (&head, &head.prefix),
			     HASH_LookupFunction, rib_lookup_fn,
			     HASH_HashFunction, rib_hash_fn,
#if 0
			     HASH_ReportChange, True,
			     HASH_ReportAccess, True,
#endif
			     NULL);
    return (tmp);
}

int 
rib_kernel_route_sweep (rib_t * rib)
{
    route_head_t *route_head;
    route_node_t *route_node;
    LINKED_LIST *ll;

#ifdef HAVE_LIBPTHREAD
    assert (pthread_mutex_trylock (&rib->mutex_lock));
#endif /* HAVE_LIBPTHREAD */

    ll = LL_Create (0);
    HASH_Iterate (rib->hash, route_head) {
	LL_Iterate (route_head->ll_route_nodes, route_node) {
	    if (route_node->attr->type == PROTO_KERNEL)
		LL_Add (ll, route_node);
	}
    }
    LL_Iterate (ll, route_node) {
	rib_delete_route_node (rib, route_node);
    }
    LL_Destroy (ll);
}


int 
show_rib_routes (uii_connection_t * uii, rib_t * rib)
{
    route_head_t *route_head;
    route_node_t *route_node;
    char tmp1[MAXLINE], tmp2[MAXLINE];

    HASH_Iterate (rib->hash, route_head) {
	sprintf (tmp1, "%s/%d", prefix_toa (route_head->prefix),
		 route_head->prefix->bitlen);
	LL_Iterate (route_head->ll_route_nodes, route_node) {
	    sprintf (tmp2, "%d", route_node->pref);
	    uii_send_data (uii, "%c%c %7s  %-30s %-30s %s\r\n",
			   proto2string (route_node->attr->type)[0],
			   (route_head->active == route_node) ? '*' : ' ',
		      (route_node->pref >= HOLD_PREF) ? "HOLD" : tmp2, tmp1,
			   prefix_toa (route_node->attr->gateway->prefix),
			   (route_node->attr->gateway->interface) ?
			   route_node->attr->gateway->interface->name : "");
	}
    }
}


u_int 
rib_hash_fn (prefix_t * prefix, u_int size)
{
    u_int tmp;
    u_int buff[4];
    int len = 16;

    memset (buff, 0, sizeof (buff));
    if (prefix->family == AF_INET)
	len = 4;
    memcpy (buff, prefix_tochar (prefix), len);
    netmasking (prefix->family, (char *) buff, prefix->bitlen);

    tmp = buff[0] ^ buff[1] ^ buff[2] ^ buff[3];
    tmp ^= (tmp >> 16);
    tmp = tmp % size;
    return (tmp);
}


u_int 
rib_lookup_fn (prefix_t * tmp1, prefix_t * tmp2)
{
    return (prefix_compare (tmp1, tmp2));
}


rib_kernel_update_route (int cmd, prefix_t * dest, generic_attr_t * attr)
{
    int index = attr->gateway->interface->index;
    prefix_t *nexthop = attr->nexthop;
    extern int kernel_install_flag;
    int rc;
    char tmp[MAXLINE];

    if (!kernel_install_flag)
	return;

    rc = kernel_update_route (cmd, dest, nexthop, index);

    trace (NORM, MRT->trace, "KERNEL %s %s/%d nexthop %s index %d %s\n",
	   (cmd == KERNEL_UPDATE_ADD) ? "ADD" : (cmd == KERNEL_UPDATE_DEL) ? "DEL" : "???",
	   prefix_toa (dest), dest->bitlen,
	   prefix_toa2 (nexthop, tmp), index,
	   (rc < 1) ? "FAILED" : "OK");
}


generic_attr_t *
Ref_Generic_Attr (generic_attr_t * attr)
{
    attr->ref_count++;
    assert (attr->ref_count > 0);
    return (attr);
}

void 
Deref_Generic_Attr (generic_attr_t * attr)
{
    if (--attr->ref_count <= 0) {
	Deref_Prefix (attr->nexthop);
	Delete (attr);
    }
}


char *
proto2string (int proto)
{
    static char *string;

    switch (proto) {
    case PROTO_STATIC:
	string = "STATIC";
	break;
    case PROTO_CONNECTED:
	string = "CONNECTED";
	break;
    case PROTO_KERNEL:
	string = "KERNEL";
	break;
    case PROTO_RIP:
	string = "RIP";
	break;
    case PROTO_BGP:
	string = "BGP";
	break;
    case PROTO_RIPNG:
	string = "RIPNG";
	break;
    default:
	string = "UNKNOWN";
	break;
    }
    return (string);
}
