/*
 * $Id: route_manager.c,v 1.1.1.1 1996/04/09 15:50:27 labovit Exp $
 */

#include <stdio.h>
#include <sys/types.h>
#include <stack.h>
#include <mrt.h>
#include <radix.h>
#include <route_manager.h>


/* rm_get_route_head
 * get "head" of routing table entry for a given prefix. Create a head
 * and radix-tree entry if one does not aleady exist
 */
route_head_t *rm_get_route_head (route_manager_t *rm, prefix_t *prefix) 
{
  radix_node_t *radix_node;
  route_head_t *route_head;
  radix_node = (radix_node_t *) radix_insert (rm->radix, prefix);

  if ((route_head = radix_node->data) == NULL) {
    route_head = (route_head_t *) New_Route_Head (radix_node);
    radix_node->data = route_head;
  }

  return (route_head);

}

/* rm_find_route_node
 * given a prefix AND gateway, find route node entry
 * return NULL if one does not exist
 */
route_node_t *rm_find_route_node (route_manager_t *rm,
				  prefix_t *prefix, gateway_t *gateway) 
{
  route_head_t *route_head;
  route_node_t *route_node;

  route_head = rm_get_route_head (rm, prefix);
    
  LL_Iterate (route_head->ll_route_nodes, route_node) {
    if (route_node->attr->gateway == gateway) {
      return (route_node);
    }
  }

  return (NULL);
}


/* rm_add_route_node
 * create a route_node and add it to the appropriate route_head,
 * or create a route_head if not already exist. Return
 * the new route_node
 */
route_node_t *rm_add_route_node (route_manager_t *rm,
				 prefix_t *prefix, 
				 generic_attr_t *attr, 
				 int pref, int pref2)
{
  route_node_t *route_node = NULL;
  route_head_t *route_head = NULL;
  int best = NO_PREF;

  route_head = rm_get_route_head (rm, prefix);

  /* check to see if we already got this route from the gateway */
  LL_Iterate (route_head->ll_route_nodes, route_node) {
    if (route_node->attr->gateway == attr->gateway) {
      if ((pref < route_node->pref) && (pref2 <= route_node->pref2)) {
	/* implicit withdarw of old route -- delete it */
	printf ("Implicit withdraw\n");
	exit (0);
      }
      /*printf ("Same pref -- ignorming\n");*/
      return (route_node);
    }
  }

  /* first time we've heard from this gateway */    
  route_node = New_Route_Node (route_head, pref, attr);
  LL_Add (route_head->ll_route_nodes, route_node);

  /* is this new route better than the active route ? */
  if ((route_head->active == NULL) || 
      (route_node->pref < route_head->active->pref)) {
    char tmp1[MAXLINE], tmp2[MAXLINE];
    route_head->last_active = route_head->active;
    route_head->active = route_node;
    trace (NORM, rm->trace, "RT Change: %s\t%s\t%d",
	   r_inet_ntoa (tmp1, MAXLINE, prefix->dest, prefix->bitlen),
	   r_inet_ntoa (tmp2, MAXLINE, attr->gateway->prefix->dest,
			attr->gateway->prefix->bitlen),
	   pref);
	   
	   
  }

  return (route_node);
}



int *rm_iter (route_manager_t *rm, void *call_fn())
{
   STACK *stack;
   radix_node_t *rnode, *current;

   stack = Create_STACK (2000);
   Set_STACK_Dynamic (stack, True);
   current = rm->radix->head;
   rnode = current;

   while (1) {
      Push (stack, rnode);
      
      if (rnode->l != NULL)
	 rnode = rnode->l;
      else {

	 while (Get_STACK_Count (stack) > 0) {
	    rnode = (radix_node_t *) Pop (stack);
	    if (rnode == NULL) {printf ("\ndone"); exit (0);}
	    if (rnode == rm->radix->head) {
	      Destroy_STACK (stack);
	      return (0);
	    }

	    if (rnode->bit) {
	      route_head_t *route_head;
	      current = rnode;
	      route_head = (route_head_t *) rnode->data;
	      /*return (rnode);*/
	      call_fn (route_head);
	    }
	    if (rnode->r != NULL) {
	       rnode = rnode->r;
	       break;
	    }
	 }
      }
   }
   
   /* should NEVER get here */
   ASSERT (1);
}





route_head_t *rm_iter_reentrant (route_manager_t *rm,
				 radix_tree_iter_t **rtree_iter)
{
   STACK *stack;
   radix_node_t *rnode;
   int start = 0;

   if (*rtree_iter == NULL) {
      start = 1;
      *rtree_iter = New (radix_tree_iter_t);
      (*rtree_iter)->stack = Create_STACK (2000);
      Set_STACK_Dynamic ((*rtree_iter)->stack, True);
      (*rtree_iter)->rnode_current = rm->radix->head;
   }

   stack = (*rtree_iter)->stack;
   rnode = (*rtree_iter)->rnode_current;

   if (start == 0)
      goto renter;

   while (1) {
      if ((rnode == rm->radix->head) && (start == 0)) {
	 /*printf("\nBack at the beginning");*/
	 return (NULL);
      }

      Push (stack, rnode);
      start = 0;
      
      if (rnode->l != NULL)
	 rnode = rnode->l;
      else {

	 while (Get_STACK_Count (stack) > 0) {
	    rnode = (radix_node_t *) Pop (stack);
	    if (rnode == NULL) {printf ("\ndone"); exit (0);}
	    if (rnode == rm->radix->head) {
	      Destroy_STACK ((*rtree_iter)->stack);
	      Delete (*rtree_iter);
	      return (NULL);
	    }

	    if (rnode->bit) {
	       /*print_prefix (rnode->prefix);*/

	       (*rtree_iter)->rnode_current = rnode;
	       return (rnode->data);
	    }
renter:
	    if (rnode->r != NULL) {
	       rnode = rnode->r;
	       break;
	    }
	 }
      }
   }
   
   /* should NEVER get here */
   ASSERT (1);
}


int rm_close (route_manager_t *route_manager) {

  pthread_mutex_unlock (&route_manager->mutex_lock);
}


int rm_open (route_manager_t *route_manager) {

   if (pthread_mutex_trylock (&route_manager->mutex_lock) < 0) {
      trace (TR_DEBUG, route_manager->trace, "Going to block in rm_open");
      pthread_mutex_lock (&route_manager->mutex_lock);
   }
}


int rm_delete_route_node (route_manager_t *rm, route_node_t *route_node) 
{
  route_head_t *route_head;
  route_node_t *best, *tmp;

  route_head = route_node->route_head;

  /* yikes - we are the active route */
  if (route_head->active = route_node) {
    route_head->active = NULL;
    /* delete here */
    if (route_node->protocol_mask == 0) {
      Delete_Route_Node (route_node);
    }
  }

  /* find new active route */
  best = NULL;
  LL_Iterate (route_head->ll_route_nodes, tmp) {
    if ((best == NULL) || 
	((tmp->pref < best->pref) && 
	 (tmp->pref2 < best->pref2))) {
      best = tmp;
    }
  }

  route_head->active = best;
}



u_int rib_hash_fn (prefix_t *prefix, u_int size) {
  u_int tmp;
  tmp = prefix->dest[0] + prefix->dest[1] +
    prefix->dest[2] + prefix->dest[3];
  tmp = tmp % size;
  return (tmp);
}

u_int rib_lookup_fn (prefix_t *tmp1, prefix_t *tmp2) {
  u_int tmp;
  tmp = prefix_compare (tmp1, tmp2);
  return (tmp);
}

