/*
 * $Id: port_server.c,v 1.1.1.1 1996/04/09 15:50:37 labovit Exp $
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <netdb.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <mrt.h>
#include <timer.h>
#include <port_serv_lib.h>

#include "port_server.h"

Master_Struct Master;

main ()
{
   init ();
   mainloop ();
   
}


/*-----------------------------------------------------------
 *  Name: 	mainloop
 *  Created:	Tue Dec 20 14:54:54 1994
 *  Author: 	Craig Labovitz   <labovit@snoopy.merit.net>
 *  DESCR:  	
 */

void mainloop ()
{
   fd_set fdvar_read, fdvar_write, fdvar_excep;

   while (1) {
      fdvar_read = Master.fdvar_read;
      fdvar_write = Master.fdvar_write;
      fdvar_excep = Master.fdvar_except;

      if (select (FD_SETSIZE, &fdvar_read, &fdvar_write, 
                  &fdvar_excep, NULL) > 0)  {
	 
	 if (FD_ISSET (Master.sockfd, &fdvar_read)) 
	    process_portserv_port ();
	 else
	    process_client_ports (fdvar_read);
      }
   }
}

/*-----------------------------------------------------------
 *  Name: 	process_portserv_port
 *  Created:	Tue Dec 20 13:21:16 1994
 *  Author: 	Craig Labovitz   <labovit@snoopy.merit.net>
 *  DESCR:  	
 */

int process_portserv_port ()
{
   struct sockaddr addr;
   int len;
   int fd;

   len = sizeof (addr);

   if ( (fd = accept (Master.sockfd, &addr, &len)) < 0) {
      perror ("\nPortserv_Port Accept failed");
      return (-1);
   }

   /* try to resgister client. If succellful, return 1 on
      socket to client. Else return 0 if first byte, error
      code in second byte */
   if (register_client (fd) < 0) { 
      tell_client (fd, 0, errno);
      close (fd);
      return (-1);
   }

   tell_client (fd, 1, 0);
      
   return (1);
}


/*-----------------------------------------------------------
 *  Name: 	process_client_ports
 *  Created:	Tue Dec 20 13:21:29 1994
 *  Author: 	Craig Labovitz   <labovit@snoopy.merit.net>
 *  DESCR:  	we have a connection on a port registered by a client
 *		find the client, and send the socket descriptor to the
 *		client. Return -1 on error, 1 otherwise.
 */

int process_client_ports (fd_set fdvar_read)  {
   PortServ_Port *p_port;
   PortServ_Client *p_client;
   struct sockaddr_in addr;
   int len;
   int fd;

   LL_Iterate (Master.ll_ports, p_port) {
      if (FD_ISSET (p_port->sockfd, &fdvar_read)) 
	 break;
   }

   if (p_port == NULL) {
      printf ("\nPort not found");
      return (-1);
   }

   len = sizeof (addr);
	 
   if ((fd = 
	accept (p_port->sockfd, (struct sockaddr *) &addr, &len)) < 0)
      return (-1);

   printf("\nConnection to port %d from %s", 
	  p_port->port, 
	  my_inet_ntoa ((u_char *) &addr.sin_addr, 32));

   /* find client who wants this connection */
   LL_Iterate (p_port->ll_clients, p_client) {
      if (!memcmp (&p_client->address, &addr.sin_addr, 4)) {
	 printf("\nMATCHES");
	 
	 Master.last_client = p_client;
	 Master.last_fd = fd;
	 if (sendfile (p_client->sockfd, fd) < 0)
	    perror ("\nFailed to send descriptro");
	 return (1);
      }
   }

   /* Don't know what to do with connection */
   printf("\nNo client wants this connection");
   close (fd);
   return (-1);
}




/*-----------------------------------------------------------
 *  Name: 	kill_port_server
 *  Created:	Tue Dec 13 22:28:34 1994
 *  Author: 	Craig Labovitz   <labovit@snoopy.merit.edu>
 *  DESCR:  	clean up before exiting
 */

void kill_port_server ()
{

   close (Master.sockfd);
   unlink (DEFAULT_SERVICE_NAME);
}


/*-----------------------------------------------------------
 *  Name: 	init
 *  Created:	Tue Dec 13 22:28:14 1994
 *  Author: 	Craig Labovitz   <labovit@snoopy.merit.edu>
 *  DESCR:  	
 */

void init () {
   struct sockaddr sockname;

   /* maybe garbage leftover from earlier incarnartion */
   unlink (DEFAULT_SERVICE_NAME);

   /* handle broken client process connections */
   signal (SIGPIPE, port_serv_broken_pipe);

   Master.sockfd = socket (AF_UNIX, SOCK_STREAM, 0);
   strcpy (sockname.sa_data, DEFAULT_SERVICE_NAME);

   if (bind (Master.sockfd, &sockname, 
	     strlen (DEFAULT_SERVICE_NAME) + 2) == -1) {
      perror ("\nBind failed for port_server");
      exit ();
   }

 
   Master.ll_clients = LL_Create (NULL);
   Master.ll_ports = LL_Create (LL_FindFunction, _ps_port_find, NULL);

   listen (Master.sockfd, 10);
   signal (SIGQUIT, kill_port_server);
   signal (SIGINT, kill_port_server);

   FD_ZERO(&Master.fdvar_write);
   FD_ZERO(&Master.fdvar_read);
   FD_ZERO(&Master.fdvar_except);

   FD_SET (Master.sockfd, &Master.fdvar_read);
}




/*-----------------------------------------------------------
 *  Name: 	register_client
 *  Created:	Tue Dec 20 13:22:56 1994
 *  Author: 	Craig Labovitz   <labovit@snoopy.merit.net>
 *  DESCR:  	Register client process which provides long port
 *		and long ip addr on socket. Add entry to Master
 *		client and port lists as needed.
 */

int register_client (int fd) {
   u_char buffer[MAXLINE];
   long port;
   long addr;
   u_char *cp = buffer;
   PortServ_Port *p_port;
   PortServ_Client *p_client;

   read(fd, cp, 8);
   
   UTIL_GET_LONG (port, cp); /* port */
   UTIL_GET_LONG (addr, cp); /* ip address */
   
   if ((p_port = LL_Find (Master.ll_ports, &port)) == NULL) 
      p_port = (PortServ_Port *) New_Port (port);
   
   /* registration failed */
   if (p_port == NULL) {
      close (fd);
      return (-1);
   }

   if (find_client (p_port->ll_clients, addr, port) != NULL) {
      printf ("\nAnother client already registered at this address");
      close (fd);
      return (-1);
   }

   p_client = (PortServ_Client *) New_Client (fd, port, addr, p_port);
       
   LL_Add (p_port->ll_clients, p_client);
   LL_Add (Master.ll_ports, p_port);
   LL_Add (Master.ll_clients, p_client);

   FD_SET (p_port->sockfd, &Master.fdvar_read);
   printf("\nRegistered %d %s", port, my_inet_ntoa ((u_char *) &addr, 32));
   return (1);
}



/*-----------------------------------------------------------
 *  Name: 	New_Client
 *  Created:	Tue Dec 20 13:24:17 1994
 *  Author: 	Craig Labovitz   <labovit@snoopy.merit.net>
 *  DESCR:  	
 */

PortServ_Client *New_Client (int sockfd, int port, long addr, 
			     PortServ_Port *p_port) {

   PortServ_Client *p_client = New (PortServ_Client);
   
   p_client->sockfd = sockfd;
   p_client->port = port;
   p_client->address = addr;
   p_client->p_port = p_port;

   return (p_client);
}


/*-----------------------------------------------------------
 *  Name: 	New_Port
 *  Created:	Tue Dec 20 13:24:05 1994
 *  Author: 	Craig Labovitz   <labovit@snoopy.merit.net>
 *  DESCR:  	
 */

PortServ_Port *New_Port (long port) {
   PortServ_Port *p_port;
   struct sockaddr_in serv_addr;
   int sockfd;


   serv_addr.sin_port = port;
   serv_addr.sin_family = AF_INET;
   serv_addr.sin_addr.s_addr = INADDR_ANY;

   if ( (sockfd = socket (AF_INET, SOCK_STREAM, 0)) < 0) {
      perror ("\nCould not get socket");
      return (NULL);
   }

   if (bind (sockfd, (struct sockaddr *) &serv_addr, sizeof (serv_addr)) < 0) {
      perror ("\nCould not bind to port");
      return (NULL);
   }
   
   listen (sockfd, 5);

   p_port = New (PortServ_Port);
   p_port->ll_clients = LL_Create (NULL);
   
   p_port->port = port;
   p_port->sockfd = sockfd;

   return (p_port);
}


/*-----------------------------------------------------------
 *  Name: 	find_client
 *  Created:	Tue Dec 20 13:24:31 1994
 *  Author: 	Craig Labovitz   <labovit@snoopy.merit.net>
 *  DESCR:  	given a linked list of PortServ_Clients, find client
 *		who matches addr and port
 */

PortServ_Client *find_client (LINKED_LIST *ll_clients, long addr, long port)
{
   PortServ_Client *p_client;

   LL_Iterate (ll_clients, p_client) {
      if ((p_client->address == addr) &&
	  (p_client->port == port))
	 return (p_client);
   }

   return (NULL);
}

int _ps_port_find (PortServ_Port *ps_port, int *port) { return (ps_port->port == *port);}


/*-----------------------------------------------------------
 *  Name: 	port_serv_broken_pipe
 *  Created:	Tue Dec 20 14:24:21 1994
 *  Author: 	Craig Labovitz   <labovit@snoopy.merit.net>
 *  DESCR:  	
 */

void port_serv_broken_pipe ()
{
   PortServ_Port *p_port;

   if (Master.last_client == NULL) {
      printf ("\nError -- unknown client has died!");
      return;
   }
   
   printf("\nClient has died");
   p_port = Master.last_client->p_port;

   LL_Remove (Master.ll_clients, Master.last_client);
   LL_Remove (p_port->ll_clients, Master.last_client);

   if (LL_GetCount (p_port->ll_clients) <= 0) {
      printf ("\nNoone left who wants this port");
      LL_Remove (Master.ll_ports, p_port);
      destroy_port (p_port);
   }
   close (Master.last_fd);

   /* handle broken client process connections */
   signal (SIGPIPE, port_serv_broken_pipe);   

   mainloop ();
}


void destroy_port (PortServ_Port *p_port) {
   FD_CLR (p_port->sockfd, &Master.fdvar_read);

   close (p_port->sockfd);
   /* free linked list */
}




/*-----------------------------------------------------------
 *  Name: 	tell_client
 *  Created:	Tue Dec 20 16:04:43 1994
 *  Author: 	Craig Labovitz   <labovit@snoopy.merit.net>
 *  DESCR:  	
 */

void tell_client (int sockfd, int status, int error_code) {
   u_char buffer[100];
   u_char *cp = buffer;

   UTIL_PUT_BYTE (status, cp);
   UTIL_PUT_BYTE (error_code, cp);

   write (sockfd, buffer, 2);
}
