/*
 * $Id: main.c,v 1.3 1996/12/23 14:38:20 masaki Exp $
 */

/* msgserver: Listens on a well-known message key (mailbox) for
   requests.  Allows servers (bgp process, mrs process, etc.) to
   register their client-ID and message key, so other servers can send
   them messages.  */

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <hash.h>
#include <io.h>
#include <version.h>
#include "msgserver.h"

#define DEBUG 1

void shutdown_msg(int arg);

static HASH_TABLE *inithash(void);
static int initmsgq(int force);
static int mainloop(HASH_TABLE *mqhash);
static void delete_key(MQINFO_Struct *info);

static int mqid;
HASH_TABLE *mqhash;

int
main(int argc, char **argv)
{
  int arg;
  int force_queue = 0;
  int errors = 0;

  while((arg = getopt(argc, argv, "fh")) != EOF) {
    switch(arg) {
    case 'f':
      force_queue = 1;
      break;
    case 'h':
      errors++;
      break;
    case '?':
      errors++;
      break;
    }
  }

  if(errors) {
    fprintf(stderr, "msgserver: usage: msgserver [-f]\n");
    printf ("\nMRT version (%s) compiled on %s\n\n",
	    MRT_VERSION, __DATE__);
    exit(1);
  }

  /* create & initialize hash table */
  mqhash = inithash();

  /* create & initialize message queue */
  mqid = initmsgq(force_queue);

#ifdef SOLARIS
  sigset(SIGINT, shutdown_msg);
  sigset(SIGTERM, shutdown_msg);
#else
  signal(SIGINT, shutdown_msg);
  signal(SIGTERM, shutdown_msg);
#endif /* SOLARIS */

  /* listen for messages */
  mainloop(mqhash);

  /* shut down cleanly */
  shutdown_msg(0);
}


/* Initialize the hash table for MQINFO structs. */
static HASH_TABLE *
inithash(void)
{
  return HASH_Create(CLHASH_SIZE, HASH_EmbeddedKey, True, HASH_KeyOffset,
		     offsetof(MQINFO_Struct, client), NULL);
}


/* Initialize our message queue. */
static int
initmsgq(int force)
{
  int id;
  int flags = 0;

  flags = (IPC_CREAT | 0622);
  if(!force)
    flags |= IPC_EXCL;

  if((id = msgget(MSGSERVER_KEY, flags)) < 0) {
    perror("msgget");
    exit(-1);
  } else {
#ifdef DEBUG
    fprintf(stderr, "initialized message queue: key 0x%x, qid %d\n",
	    MSGSERVER_KEY, id);
#endif
    return id;
  }
}


/* Main processing loop.  Read a message, take appropriate actions. */
static int
mainloop(HASH_TABLE *mqhash)
{
  ARB_MSG_Struct msg;

  while(1) {
#ifdef DEBUG
    fprintf(stderr, "listening...\n");
#endif

    if(msgrcv(mqid, (struct msgbuf *)&msg, sizeof(ARB_MSG_Struct), 0L, 0) < 0) {
      perror("msgrcv");
      continue;
    }

    switch(msg.type) {
    case MSG_SETMBOX:
#ifdef DEBUG
      fprintf(stderr,
	      "received SETMBOX message from 0x%x: client=\"%s\", key=0x%x\n",
	      msg.sender, msg.mqinfo.client, msg.mqinfo.key);
#endif
      handle_setmbox(mqhash, &msg);
      break;

    case MSG_GETMBOX:
#ifdef DEBUG
      fprintf(stderr,
	      "received GETMBOX message from 0x%x: client=\"%s\"\n",
	      msg.sender, msg.mqinfo.client);
#endif
      handle_getmbox(mqhash, &msg);
      break;

    case MSG_CLRMBOX:
#ifdef DEBUG
      fprintf(stderr, "received CLRMBOX message from 0x%x: client=\"%s\"\n",
	      msg.sender, msg.mqinfo.client);
#endif
      handle_clrmbox(mqhash, &msg);
      break;

    case MSG_DUMP:
#ifdef DEBUG
      fprintf(stderr, "received DUMP message from 0x%x\n", msg.sender);
#endif
      handle_dump(mqhash, &msg);
      break;

    case MSG_SHUTDOWN:
#ifdef DEBUG
      fprintf(stderr, "received SHUTDOWN message from 0x%x\n", msg.sender);
#endif
      return 0;
      /* NOTREACHED */
      break;

    default:
      fprintf(stderr, "received invalid message: type %d\n",
	      (int) msg.type);
      break;
    }
  }
}


void
shutdown_msg(int arg)
{
  
#ifdef DEBUG
  fprintf(stderr, "Going down on signal %d...\n", arg);
#endif
  msgctl(mqid, IPC_RMID, (struct msqid_ds *) NULL);
  printf ("Deleting key  0x%x\n", mqid);

  HASH_Process (mqhash, (HASH_ProcessProc)delete_key);

  exit(0);
}


static void
delete_key (MQINFO_Struct *info)
{
  int key = msgget (info->key, 0);

  printf ("Deleting key  0x%x\n", info->key);
  if (msgctl(key, IPC_RMID, (struct msqid_ds *) NULL) == -1) {
    perror ("Could not delete key");
  }

}
