/*
 * $Id: ripng_proto.c,v 1.24 1997/03/21 17:24:41 masaki Exp $
 */

#include <config.h>
#ifdef HAVE_IPV6
#include <sys/types.h>
#include <sys/param.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/uio.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <netdb.h>

#ifdef NRL_IPV6
/*
 * NRL's IPV6 use IPPROTO_IP instead of IPPROTO_IPV6 for socket options
 */
#undef IPPROTO_IPV6
#define IPPROTO_IPV6 IPPROTO_IP
#endif /* NRL_IPV6 */

#include <mrt.h>
#include <trace.h>
#include <timer.h>
#include <select.h>
#include <interface.h>
#include <rib.h>
#include <user.h>
#include <ripng.h>
#include <api6.h>


struct in6_addr all_rip_routers;

/*
 * called from select on socket
 */
static void
ripng_schedule_receive_update ()
{
    schedule_event (RIPNG->schedule, ripng_receive_update, 0, NULL);
}

/*
 * called from timer
 */

/* 30 timer processing -- announce routing table */
static void
ripng_schedule_timer ()
{
    schedule_event (RIPNG->schedule, ripng_timer_update, 0, NULL);
}

/* holddown & expire */
static void
ripng_schedule_age ()
{
    schedule_event (RIPNG->schedule, ripng_timeout_routes, 0, NULL);
}

/* flash update */
static void
ripng_schedule_flash ()
{
    schedule_event (RIPNG->schedule, ripng_flash_update, 0, NULL);
}

static int flash_update_waiting = 0;

void
ripng_timer_update ()
{
    trace (TR_TRACE, RIPNG->trace, "RIPNG timer (update) fired\n");
    pthread_mutex_lock (&RIPNG->flash_update_mutex_lock);
    if (flash_update_waiting)
	flash_update_waiting = 0;	/* clear flash update */
    pthread_mutex_unlock (&RIPNG->flash_update_mutex_lock);
    ripng_output_processing (0);
}

void
ripng_flash_update ()
{
    trace (TR_TRACE, RIPNG->trace, "RIPNG timer (flash update) fired\n");
    pthread_mutex_lock (&RIPNG->flash_update_mutex_lock);
    if (flash_update_waiting) {
	flash_update_waiting = 0;
	pthread_mutex_unlock (&RIPNG->flash_update_mutex_lock);
	ripng_output_processing (1);
    }
    else {
	pthread_mutex_unlock (&RIPNG->flash_update_mutex_lock);
    }
}

void
ripng_set_flash_update ()
{
    time_t t;

    pthread_mutex_lock (&RIPNG->flash_update_mutex_lock);
    if (flash_update_waiting) {
	pthread_mutex_unlock (&RIPNG->flash_update_mutex_lock);
	return;
    }
    t = RIPNG_FLASH_DELAY_MIN +
	rand () % (RIPNG_FLASH_DELAY_MAX - RIPNG_FLASH_DELAY_MIN + 1);
    Timer_Set_Time (RIPNG->flash, t);
    flash_update_waiting++;
    pthread_mutex_unlock (&RIPNG->flash_update_mutex_lock);
}

static u_int
ripng_hash_fn (prefix_t * prefix, u_int size)
{
    u_int tmp, buff[4];
    IN6_ADDR_COPY ((struct in6_addr *) buff, prefix_toaddr6 (prefix));
    netmasking (prefix->family, (char *) buff, prefix->bitlen);
    /* Pedro's suggestion */
    tmp = buff[0] ^ buff[1] ^ buff[2] ^ buff[3];
    tmp ^= (tmp >> 16);
    tmp = tmp % size;
    return (tmp);
}

static u_int
ripng_lookup_fn (prefix_t * tmp1, prefix_t * tmp2)
{
    return (prefix_compare (tmp1, tmp2));
}

/*
 * calling sequence: init_ripng and then start_ripng
 */

/*
 * initialize ripng stuff
 */
int
ripng_init (trace_t * trace)
{
    ripng_route_t tmp;

    /* initialize multicast address to send and join
       defined in the spec */
    all_rip_routers.s6_addr[0] = 0xff;
    all_rip_routers.s6_addr[1] = 0x02;
    all_rip_routers.s6_addr[15] = 0x09;

    RIPNG = New (ripng_t);
    RIPNG->trace = trace_copy (trace);

    RIPNG->timer = New_Timer (ripng_schedule_timer, RIPNG_UPDATE_INTERVAL,
			      "RIPNG update timer", NULL);
    timer_set_jitter (RIPNG->timer, RIPNG_UPDATE_JITTER);

    RIPNG->age = New_Timer (ripng_schedule_age, 0, "RIPNG aging timer", NULL);
    timer_set_flags (RIPNG->age, TIMER_ONE_SHOT);

    RIPNG->flash = New_Timer (ripng_schedule_flash, 0, "RIPNG flash timer",
			      NULL);
    timer_set_flags (RIPNG->flash, TIMER_ONE_SHOT);

    RIPNG->schedule = New_Schedule ("RIPNG", RIPNG->trace);

    RIPNG->hash = HASH_Create (100,
			    HASH_KeyOffset, HASH_Offset (&tmp, &tmp.prefix),
			       HASH_LookupFunction, ripng_lookup_fn,
			       HASH_HashFunction, ripng_hash_fn,
			       NULL);

    pthread_mutex_init (&RIPNG->send_mutex_lock, NULL);
    pthread_mutex_init (&RIPNG->route_table_mutex_lock, NULL);
    pthread_mutex_init (&RIPNG->flash_update_mutex_lock, NULL);

    return (1);
}


/* 
 * the thread is alive! Start the timers and wait for events to process
 */
static void
ripng_start_main ()
{
    trace (TR_TRACE, RIPNG->trace, "RIPNG thread created\n");

    if (ripng_init_listen () < 0) {
	trace (TR_ERROR, RIPNG->trace, "RIPNG aborted due to error(s)\n");
	return;
    }
    Timer_Turn_ON (RIPNG->timer);
    Timer_Set_Time (RIPNG->age, RIPNG_TIMEOUT_INTERVAL);

    /* send initial request to routers */
    ripng_send_request (NULL);

#ifndef HAVE_LIBPTHREAD
    return;
#else
    while (1) {
	schedule_wait_for_event (RIPNG->schedule);
    }
#endif /* HAVE_LIBPTHREAD */
}


/* 
 * just call thread create routine
 */
void
ripng_start ()
{
    mrt_thread_create ("RIPNG thread", RIPNG->schedule,
		       ripng_start_main, NULL);
}


/* 
 * start listening for broadcast ripng updates and request
 */
int
ripng_init_listen ()
{
    struct sockaddr_in6 ripng;
    int one = 1, zero = 0, ttl = 255;
    int size, sockfd;
    struct ipv6_mreq mreq;
    int succeed = 0;
    char tmp6[INET6_ADDRSTRLEN];
    interface_t *interface;

    if ((sockfd = socket (PF_INET6, SOCK_DGRAM, 0)) < 0) {
	trace (TR_ERROR, RIPNG->trace, "RIPNG socket: %s\n",
	       strerror (errno));
	return (-1);
    }

#ifdef INRIA_IPV6
    if (setsockopt (sockfd, SOL_SOCKET, SO_REUSEADDR, (char *) &one,
		    sizeof (one)) < 0) {
	trace (TR_ERROR, RIPNG->trace,
	       "RIPNG setsockopt SO_REUSEADDR: %s\n", strerror (errno));
    }
#endif /* INRIA_IPV6 */

#ifdef SO_RCVBUF
    size = RIPNG_MIN_BUFSIZE;
    if (setsockopt (sockfd, SOL_SOCKET, SO_RCVBUF, (char *) &size,
		    sizeof (size)) < 0) {
	trace (TR_ERROR, RIPNG->trace,
	       "RIPNG setsockopt SO_RCVBUF (%s): %s\n", size,
	       strerror (errno));
    }
#endif

#ifdef SCM_RIGHTS
#ifdef IPV6_PKTINFO
    /*
     *  Receive destination address and interface from socket
     */
    if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_PKTINFO, &one,
		    sizeof (one)) < 0) {
	trace (TR_ERROR, RIPNG->trace,
	       "RIPNG setsockopt IPV6_PKTINFO: %s\n", strerror (errno));
    }
#endif /* IPV6_PKTINFO */
#ifdef IP_RECVINTERFACE
    if (setsockopt (sockfd, IPPROTO_IPV6, IP_RECVINTERFACE, &one,
		    sizeof (one)) < 0) {
	trace (TR_ERROR, RIPNG->trace,
	       "RIPNG setsockopt IP_RECVINTERFACE: %s\n", strerror (errno));
    }
#endif /* IP_RECVINTERFACE */
#if !defined(__linux__) && !defined(NRL_IPV6)	/* defined, for IPv4 only */
#ifdef IP_RECVDSTADDR
    if (setsockopt (sockfd, IPPROTO_IPV6, IP_RECVDSTADDR, &one,
		    sizeof (one)) < 0) {
	trace (TR_ERROR, RIPNG->trace,
	       "RIPNG setsockopt IP_RECVDSTADDR: %s\n", strerror (errno));
    }
#endif /* IP_RECVDSTADDR */
#endif /* !defined(__linux__) && !defined(NRL_IPV6) */
#ifdef IPV6_HOPLIMIT
    if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_HOPLIMIT, &one,
		    sizeof (one)) < 0) {
	trace (TR_ERROR, RIPNG->trace,
	       "RIPNG setsockopt IPV6_HOPLIMIT: %s\n", strerror (errno));
    }
#endif /* IPV6_HOPLIMIT */
#endif /* SCM_RIGHTS */

    /* I'm not sure that unicast ttl should be 255 when responding 
       routing table request, but anyway ... */
    if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_UNICAST_HOPS, (char *) &ttl,
		    sizeof (ttl)) < 0) {
	trace (TR_ERROR, RIPNG->trace,
	       "RIPNG setsockopt IPV6_UNICAST_HOPS: %s\n", strerror (errno));
    }

#ifdef NRL_IPV6
/*
 * On NRL's IPV6, IPV6_MULTICAST_LOOP and IPV6_MULTICAST_HOPS
 * require a pointer to one byte
 */
    {
	u_char zero = 0, ttl = 255;
#endif /* NRL_IPV6 */
	if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_MULTICAST_LOOP,
			(char *) &zero, sizeof (zero)) < 0) {
	    trace (TR_ERROR, RIPNG->trace,
		   "RIPNG setsockopt IPV6_MULTICAST_LOOP: %s\n",
		   strerror (errno));
	}

	if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_MULTICAST_HOPS,
			(char *) &ttl, sizeof (ttl)) < 0) {
	    trace (TR_ERROR, RIPNG->trace,
		   "RIPNG setsockopt IPV6_MULTICAST_HOPS: %s\n",
		   strerror (errno));
	}
#ifdef NRL_IPV6
    }
#endif /* NRL_IPV6 */

    /* 
     * Join the specified multicast address
     */

    memset (&mreq, 0, sizeof (mreq));
    IN6_ADDR_COPY (&mreq.ipv6mr_multiaddr, &all_rip_routers);

    LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
	/* if (!BIT_TEST (interface->flags, IFF_UP)) continue; */
	if (!BIT_TEST (interface->bit, RIPNG->interface_mask))
	    continue;
	if (!BIT_TEST (interface->flags, IFF_MULTICAST))
	    continue;
	if (interface->primary6 == NULL)
	    continue;

#ifdef WIDE_IPV6
	/* On WIDE IPv6, it is required to modify the destination address
	   as to include the outgoing interface's index */
	/* This is unnessesary when responding to a unicast address which
	   includes an index */

	mreq.ipv6mr_multiaddr.s6_addr8[2] = (interface->index >> 8) & 0xff;
	mreq.ipv6mr_multiaddr.s6_addr8[3] = interface->index & 0xff;
#endif /* WIDE_IPV6 */

#if defined(INRIA_IPV6) || defined (WIDE_IPV6) || defined(__linux__)
	mreq.ipv6mr_interface = interface->index;
#else /* at least, NRL_IPV6 */
	IN6_ADDR_COPY (&mreq.ipv6mr_interface,
		       prefix_toaddr6 (interface->primary6->prefix));
#endif /* INRIA_IPV6 || WIDE_IPV6 || __linux__ */

	if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_ADD_MEMBERSHIP,
			(char *) &mreq, sizeof (mreq)) >= 0) {
	    trace (TR_TRACE, RIPNG->trace,
		   "RIPNG IPV6_ADD_MEMBERSHIP %s on %s\n",
	    inet_ntop (AF_INET6, &mreq.ipv6mr_multiaddr, tmp6, sizeof tmp6),
		   interface->name);
	    succeed++;
	}
	else {
	    trace (TR_ERROR, RIPNG->trace,
		   "RIPNG setsockopt IPV6_ADD_MEMBERSHIP %s for %s: %s\n",
	    inet_ntop (AF_INET6, &mreq.ipv6mr_multiaddr, tmp6, sizeof tmp6),
		   interface->name, strerror (errno));
	}
    }
    if (!succeed) {
	close (sockfd);
	return (-1);
    }

    memset (&ripng, 0, sizeof (ripng));
    ripng.sin6_port = htons (RIPNG_DEFAULT_PORT);
    ripng.sin6_family = AF_INET6;
    ripng.sin6_flowinfo = htonl (RIPNG_IPV6_PRIORITY);
#ifdef SIN6_LEN
    ripng.sin6_len = sizeof (struct sockaddr_in6);
#endif /* SIN6_LEN */

    if (bind (sockfd, (struct sockaddr *) &ripng, sizeof (ripng)) < 0) {
	trace (TR_ERROR, RIPNG->trace, "RIPNG bind to port %d: %s\n",
	       RIPNG_DEFAULT_PORT, strerror (errno));
	close (sockfd);
	return (-1);
    }

    if (select_add_fd (sockfd, 1, ripng_schedule_receive_update, NULL) < 0) {
	trace (TR_ERROR, RIPNG->trace, "RIPNG select_add_fd failed\n");
	close (sockfd);
	return (-1);
    }

    trace (TR_TRACE, RIPNG->trace, "RIPNG listening for updates on port %d\n",
	   ntohs (ripng.sin6_port));
    RIPNG->sockfd = sockfd;
    return (1);
}


/*
 * 1) timoute routes (change metric to 16 and set change bit)
 * 2) garbage collect (delete route from hash, free memory and notify
 *    routing table 
 * Don't need to explicitly do outbound processing, as this will be picked up
 * by the regular 30 second timer
 */
void
ripng_timeout_routes ()
{
    ripng_route_t *route;
    char tmp1[INET6_ADDRSTRLEN];
    char tmp2[INET6_ADDRSTRLEN];
    LINKED_LIST *ll_delete = NULL;
    time_t now, t;
    time_t nexttime = 0;
    int changed = 0;

    trace (TR_TRACE, RIPNG->trace, "RIPNG timer (age) fired\n");

    time (&now);
    nexttime = now + RIPNG_TIMEOUT_INTERVAL;

    pthread_mutex_lock (&RIPNG->route_table_mutex_lock);

    HASH_Iterate (RIPNG->hash, route) {
	/* don't time out routes from other protocols, 
	   like static, kernel and interface routes */
	if (route->attr->type != PROTO_RIPNG)
	    continue;

	/* garbage collect and delete route */
	if (BIT_TEST (route->flags, RT_RIPNG_DELETE)) {
	    if (now - route->delete >= RIPNG_GARBAGE_INTERVAL) {
		trace (TR_TRACE, RIPNG->trace,
		       "RIPNG deleted %s/%d via %s (garbage collection)\n",
		       prefix_toa2 (route->prefix, tmp1),
		       route->prefix->bitlen,
		       prefix_toa2 (route->attr->gateway->prefix, tmp2));
		if (ll_delete == NULL)
		    ll_delete = LL_Create (0);
		LL_Add (ll_delete, route);
		RIPNG->update_call_fn (KERNEL_UPDATE_DEL,
				       route->prefix, route->attr);
	    }
	    else {
		/* see the earliest next time to be checked */
		if ((t = route->delete + RIPNG_GARBAGE_INTERVAL) < nexttime)
		    nexttime = t;
	    }
	}
	/* timeout route -- set metric to 16 and set change flag */
	else if (now - route->time >= RIPNG_TIMEOUT_INTERVAL) {
	    trace (TR_TRACE, RIPNG->trace,
		   "RIPNG timing out %s/%d via %s\n",
		   prefix_toa2 (route->prefix, tmp1), route->prefix->bitlen,
		   prefix_toa2 (route->attr->gateway->prefix, tmp2));
	    route->attr->metric = RIPNG_INFINITY;
	    route->flags |= RT_RIPNG_DELETE;
	    route->flags |= RT_RIPNG_CHANGE;
	    route->delete = now;
	    RIPNG->update_call_fn (KERNEL_UPDATE_HOLD,
				   route->prefix, route->attr);
	    changed++;

	    /* see the earliest next time to be checked */
	    if ((t = route->delete + RIPNG_GARBAGE_INTERVAL) < nexttime)
		nexttime = t;
	}
	else {			/* live routes */
	    /* see the earliest next time to be checked */
	    if ((t = route->time + RIPNG_TIMEOUT_INTERVAL) < nexttime)
		nexttime = t;
	}
    }

    /* we have to use a linked list cause we can't delete hash items
     * while iterating through hash table */
    if (ll_delete != NULL) {
	LL_Process (ll_delete, (void *) ripng_delete_route);
	LL_Destroy (ll_delete);
    }

    pthread_mutex_unlock (&RIPNG->route_table_mutex_lock);

    if (changed)
	ripng_set_flash_update ();

    if ((t = nexttime - time (NULL)) <= 0)
	t = 5;			/* don't want so strict? */
    Timer_Set_Time (RIPNG->age, t);
}


/* 
 * use multicast for multicast-capable interfaces 
 * use unicast for p-to-p non-multicast-capable interfaces (non standard way)
 * if ll_prefixes is NULL, send a request for complete routing table 
 *    ll_prefixes ... not implemented yet and usual daemon doesn't need it
 */
void
ripng_send_request (LINKED_LIST * ll_prefixes)
{
#define RIPNG_HDRLEN 4
#define RIPNG_RTELEN 20
    char buffer[RIPNG_HDRLEN + RIPNG_RTELEN];
    u_char *cp, *start;
    interface_t *interface;
    struct sockaddr_in6 ripng;
    char tmp6[INET6_ADDRSTRLEN];

    assert (ll_prefixes == NULL);	/* XXX not yet implemented */
    start = buffer;

    memset (&ripng, 0, sizeof (ripng));
    ripng.sin6_port = htons (RIPNG_DEFAULT_PORT);
    ripng.sin6_family = AF_INET6;
    ripng.sin6_flowinfo = htonl (RIPNG_IPV6_PRIORITY);
#ifdef SIN6_LEN
    ripng.sin6_len = sizeof (struct sockaddr_in6);
#endif /* SIN6_LEN */

    cp = buffer;
    memset (buffer, 0, sizeof (buffer));

    UTIL_PUT_BYTE (RIPNG_REQUEST, cp);
    UTIL_PUT_BYTE (RIPNG_VERSION, cp);

    UTIL_PUT_SHORT (0, cp);	/* must be 0 */
    memset (cp, 0, 16);
    cp += 16;

    UTIL_PUT_SHORT (0, cp);	/* must be 0 */
    UTIL_PUT_BYTE (0, cp);	/* length 0 or 16?? spec is not clear */
    UTIL_PUT_BYTE (RIPNG_INFINITY, cp);		/* infinity metric */

    LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {

	if (!BIT_TEST (interface->bit, RIPNG->interface_mask))
	    continue;
	if (!BIT_TEST (interface->flags, IFF_UP))
	    continue;
	if (interface->primary6 == NULL)
	    continue;

	if (!BIT_TEST (interface->flags, IFF_MULTICAST)) {
	    struct in6_addr *dest;
	    /*
	     * The RIPng draft onlys say about multicast,
	     * so this way may work but non standard
	     */
	    if (!BIT_TEST (interface->flags, IFF_POINTOPOINT))
		continue;

	    if ((dest = prefix_toaddr6 (
			       interface->link_local->broadcast)) == NULL) {
		/* shouldn't use global address, though */
		if ((dest = prefix_toaddr6 (
				    interface->primary->broadcast)) == NULL)
		    continue;
	    }
	    IN6_ADDR_COPY (&ripng.sin6_addr, dest);
	}
	else {
	    IN6_ADDR_COPY (&ripng.sin6_addr, &all_rip_routers);
	}

	trace (TR_TRACE, RIPNG->trace, "send RIPNG request to %s on %s\n",
	       inet_ntop (AF_INET6, &ripng.sin6_addr, tmp6, sizeof tmp6),
	       interface->name);

	ripng_sendmsgto (RIPNG->sockfd, buffer, (cp - start), 0,
			 &ripng, sizeof (ripng), interface);
    }
}


int
ripng_sendmsgto (int sockfd, char *buffer, int buflen, int flag,
	    struct sockaddr_in6 *addr, int addrlen, interface_t * interface)
{
    int is_mc = IN6_IS_ADDR_MULTICAST (&addr->sin6_addr);
    int is_ll = IN6_IS_ADDR_LINKLOCAL (&addr->sin6_addr);
    int rc;
    char tmp6[INET6_ADDRSTRLEN];

    pthread_mutex_lock (&RIPNG->send_mutex_lock);

    if (!is_ll && !is_mc) {
	if ((rc = sendto (sockfd, buffer, buflen, flag,
			  (struct sockaddr *) addr, addrlen)) < 0) {
	    trace (TR_ERROR, RIPNG->trace, "RIPNG sendto: %s\n",
		   strerror (errno));
	}
    }
    else {

#ifndef IPV6_PKTINFO

#ifdef __linux__
	Linux IPv6 should have IPV6_PKTINFO ! !!
#endif

#ifdef INRIA_IPV6

	/* On INRIA IPV6, it is required to create a new socket 
	   in order to bind it with a link-local address. */

	struct sockaddr_in6 sin6;
	int one = 1, zero = 0, ttl = 255;

	if ((sockfd = socket (PF_INET6, SOCK_DGRAM, 0)) < 0) {
	    trace (TR_ERROR, RIPNG->trace, "RIPNG socket: %s\n",
		   strerror (errno));
	    pthread_mutex_unlock (&RIPNG->send_mutex_lock);
	    return (-1);
	}

	if (setsockopt (sockfd, SOL_SOCKET, SO_REUSEADDR, (char *) &one,
			sizeof (one)) < 0) {
	    trace (TR_ERROR, RIPNG->trace,
		   "RIPNG setsockopt SO_REUSEADDR: %s\n", strerror (errno));
	}

	if (is_mc) {
	    assert (BIT_TEST (interface->flags, IFF_MULTICAST));

	    if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_MULTICAST_LOOP,
			    (char *) &zero, sizeof (zero)) < 0) {
		trace (TR_ERROR, RIPNG->trace,
		       "RIPNG setsockopt IPV6_MULTICAST_LOOP: %s\n",
		       strerror (errno));
	    }

	    if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_MULTICAST_HOPS,
			    (char *) &ttl, sizeof (ttl)) < 0) {
		trace (TR_ERROR, RIPNG->trace,
		       "RIPNG setsockopt IPV6_MULTICAST_HOPS: %s\n",
		       strerror (errno));
	    }
	}
	else {

	    /* I'm not sure that unicast ttl should be 255 when responding 
	       routing table request, but anyway ... */
	    if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_UNICAST_HOPS,
			    (char *) &ttl, sizeof (ttl)) < 0) {
		trace (TR_ERROR, RIPNG->trace,
		       "RIPNG setsockopt IPV6_UNICAST_HOPS: %s\n",
		       strerror (errno));
	    }
	}

	memset (&sin6, 0, sizeof (sin6));
	sin6.sin6_port = htons (RIPNG_DEFAULT_PORT);
	sin6.sin6_family = AF_INET6;
#ifdef SIN6_LEN
	sin6.sin6_len = sizeof (sin6);
#endif
	sin6.sin6_flowinfo = htonl (RIPNG_IPV6_PRIORITY);
	sin6.sin6_port = htons (RIPNG_DEFAULT_PORT);

	/* try to specify the source address in case of sending to multicast
	   and link-local addresses */

	if (!interface->link_local) {
	    trace (TR_ERROR, RIPNG->trace,
		   "RIPNG No link-local address for %s\n", interface->name);
	}
	else {
	    IN6_ADDR_COPY (&sin6.sin6_addr,
			   prefix_toaddr6 (interface->link_local->prefix));
	    trace (TR_PACKET, RIPNG->trace,
		   "RIPNG source address should be %s\n",
		   prefix_toa2 (interface->link_local->prefix, tmp6));
	}

	if (bind (sockfd, (struct sockaddr *) &sin6, sizeof (sin6)) < 0) {
	    trace (TR_ERROR, RIPNG->trace,
		   "RIPNG bind to port %d: %s\n",
		   ntohs (sin6.sin6_port), strerror (errno));
	    close (sockfd);
	    pthread_mutex_unlock (&RIPNG->send_mutex_lock);
	    return (-1);
	}

#endif /* INRIA_IPV6 */

#ifdef WIDE_IPV6

	struct sockaddr_in6 sin6;

	if (is_mc) {

	    /* On WIDE IPv6, it is required to modify the destination address
	       as to include the outgoing interface's index */
	    /* This is unnessesary when responding to a unicast address which
	       includes an index */

	    memcpy (&sin6, addr, sizeof (sin6));
	    addr = &sin6;
	    addr->sin6_addr.s6_addr8[2] = (interface->index >> 8) & 0xff;
	    addr->sin6_addr.s6_addr8[3] = interface->index & 0xff;
	}

#else /* WIDE_IPV6 */
	if (is_mc) {
	    /*
	     * choose outgoing intertface for multicast
	     */
	    if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_MULTICAST_IF,
#ifdef INRIA_IPV6
	    /* This way conforms with the draft */
			    &interface->index,
			    sizeof (interface->index)) < 0) {
#else
	    /* This way is the same with one used in IPv4 */
			    prefix_tochar (interface->primary6->prefix),
			    sizeof (struct in6_addr)) < 0) {
#endif /* INRIA_IPV6 */
		trace (TR_ERROR, RIPNG->trace,
		       "RIPNG setsockopt IPV6_MULTICAST_IF for %s: %s\n",
		       interface->name, strerror (errno));
	    }
	}
#endif /* WIDE_IPV6 */

	if ((rc = sendto (sockfd, buffer, buflen, flag,
			  (struct sockaddr *) addr, addrlen)) < 0) {
	    trace (TR_ERROR, RIPNG->trace, "RIPNG sendto: %s\n",
		   strerror (errno));
	}

#ifdef INRIA_IPV6
	close (sockfd);
#endif /* INRIA_IPV6 */

#else /* IPV6_PKTINFO */

	struct msghdr mhdr;
	struct cmsghdr *cmsg;
	struct in6_pktinfo *pkt_info;
	struct iovec iov;
	char chdr[sizeof (struct cmsghdr) + sizeof (struct in6_pktinfo)];

	mhdr.msg_name = (struct sockaddr_in *) addr;
	mhdr.msg_namelen = addrlen;
	mhdr.msg_iov = &iov;
	mhdr.msg_iovlen = 1;
	mhdr.msg_controllen = sizeof (chdr);
	iov.iov_base = buffer;
	iov.iov_len = buflen;
	cmsg = (struct cmsghdr *) chdr;
	cmsg->cmsg_len = sizeof (chdr);
	cmsg->cmsg_level = IPPROTO_IPV6;
	cmsg->cmsg_type = IPV6_PKTINFO;
	mhdr.msg_control = (void *) cmsg;
	pkt_info = (struct in6_pktinfo *) cmsg->cmsg_data;
	memset (pkt_info, 0, sizeof (struct in6_pktinfo));
	if (!BIT_TEST (interface->flags, IFF_USERDEF)) {
	    pkt_info->ipi6_ifindex = interface->index;
	}
	if (!interface->link_local) {
	    trace (TR_ERROR, RIPNG->trace,
		   "RIPNG No link-local address for %s\n", interface->name);
	}
	else {
	    trace (TR_PACKET, RIPNG->trace,
		   "RIPNG source address should be %s\n",
		   prefix_toa2 (interface->link_local->prefix, tmp6));
	    IN6_ADDR_COPY (&pkt_info->ipi6_addr,
			   prefix_toaddr6 (interface->link_local->prefix));
	}
	if ((rc = sendmsg (sockfd, &mhdr, flag)) < 0) {
	    trace (TR_ERROR, RIPNG->trace, "RIPNG sendmsg: %s\n",
		   strerror (errno));
	}
#endif /* IPV6_PKTINFO */
    }
    trace (TR_PACKET, RIPNG->trace, "send RIPNG %d bytes to %s on %s\n",
	   buflen, inet_ntop (AF_INET6, &addr->sin6_addr, tmp6, sizeof tmp6),
	   interface->name);
    pthread_mutex_unlock (&RIPNG->send_mutex_lock);
    return (rc);
}


/*
 * given an interface, broadcast ripng routes (according to policy)
 * we do this every RIPNG_UPDATE_INTERVAL
 * if change flag is set (==1) then only send routes marked as
 * changed
 */
void
ripng_send_routes (interface_t * interface, struct sockaddr_in6 *host,
		   int change)
{
    ripng_route_t *route;
    char *buffer, *cp;
    struct sockaddr_in6 ripng;
    int routes = 0;		/* count of routes */
    int pdulen, metric;
    char tag[MAXLINE];
    char tmp6[INET6_ADDRSTRLEN];
    LINKED_LIST *ll_buffer;
    struct message {
	u_long len;
	char *buffer;
    } *message;

    assert (interface);

    if (host) {
	memcpy (&ripng, host, sizeof (ripng));
    }
    else {
	memset (&ripng, 0, sizeof (ripng));
	ripng.sin6_port = htons (RIPNG_DEFAULT_PORT);
	ripng.sin6_family = AF_INET6;
	ripng.sin6_flowinfo = htonl (RIPNG_IPV6_PRIORITY);
#ifdef SIN6_LEN
	ripng.sin6_len = sizeof (struct sockaddr_in6);
#endif /* SIN6_LEN */

	if (BIT_TEST (interface->flags, IFF_MULTICAST)) {
	    IN6_ADDR_COPY (&ripng.sin6_addr, &all_rip_routers);
	}
	else if (BIT_TEST (interface->flags, IFF_POINTOPOINT)) {
	    struct in6_addr *dest;

	    if ((dest = prefix_toaddr6 (
			       interface->link_local->broadcast)) == NULL) {
		/* shouldn't use global address, though */
		if ((dest = prefix_toaddr6 (
				  interface->primary->broadcast)) == NULL) {
		    return;
		}
	    }
	    IN6_ADDR_COPY (&ripng.sin6_addr, dest);
	}
	else {
	    return;
	}
    }

#define IPV6_HDRLEN 40
#define UDP_HDRLEN 8
    pdulen = interface->mtu - IPV6_HDRLEN - UDP_HDRLEN;
    assert (pdulen >= RIPNG_HDRLEN + RIPNG_RTELEN);
    buffer = NewArray (char, pdulen);

    cp = buffer;
    memset (buffer, 0, pdulen);

    UTIL_PUT_BYTE (RIPNG_RESPONSE, cp);
    UTIL_PUT_BYTE (RIPNG_VERSION, cp);
    UTIL_PUT_SHORT (0, cp);

    trace (TR_TRACE, RIPNG->trace, "send RIPNG response to %s on %s\n",
	   inet_ntop (AF_INET6, &ripng.sin6_addr, tmp6, sizeof tmp6),
	   interface->name);

    /* make a list first and then send them to avoid waiting on sendto() 
       with locking the route table */
    ll_buffer = LL_Create (0);

    /* on flashing update, locking should be done outer side
       because of keeping change flags and reset them after sending out */
#ifdef HAVE_LIBPTHREAD
    if (change == 0)
	pthread_mutex_lock (&RIPNG->route_table_mutex_lock);
#endif /* HAVE_LIBPTHREAD */

    HASH_Iterate (RIPNG->hash, route) {
	prefix_t *prefix = route->prefix;

	if ((metric = ripng_policy (prefix, route->attr, interface)) < 0)
	    continue;

	/* doing ouput processing and only sending changed routes */
	if (change && (!BIT_TEST (route->flags, RT_RIPNG_CHANGE))) {
	    trace (TR_PACKET, RIPNG->trace,
		   "  x %s/%d metric %d (unchange)\n",
		   prefix_toa2 (prefix, tmp6), prefix->bitlen,
		   route->attr->metric);
	    continue;
	}

	routes++;

	if (route->attr->tag) {
	    sprintf (tag, " tag 0x%lx", route->attr->tag);
	}
	else {
	    tag[0] = '\0';
	}
	trace (TR_PACKET, RIPNG->trace, "  o %s/%d metric %d%s\n",
	       prefix_toa2 (prefix, tmp6), prefix->bitlen, metric, tag);

	IN6_ADDR_COPY ((struct in6_addr *) cp, prefix_toaddr6 (prefix));
	cp += 16;

	UTIL_PUT_SHORT (htons (route->attr->tag), cp);
	UTIL_PUT_BYTE (prefix->bitlen, cp);
	UTIL_PUT_BYTE (metric, cp);

	/* 
	 * see if we have filled up the buffer. If so, send packet and 
	 * create new buffer 
	 */
	if (cp - buffer > pdulen - RIPNG_RTELEN) {
	    message = New (struct message);
	    message->len = cp - buffer;
	    message->buffer = buffer;
	    LL_Add (ll_buffer, message);
	    buffer = NewArray (char, pdulen);
	    cp = buffer;
	    memset (buffer, 0, pdulen);
	    UTIL_PUT_BYTE (RIPNG_RESPONSE, cp);
	    UTIL_PUT_BYTE (RIPNG_VERSION, cp);
	    UTIL_PUT_SHORT (0, cp);
	    routes = 0;
	}
    }
#ifdef HAVE_LIBPTHREAD
    if (change == 0)
	pthread_mutex_unlock (&RIPNG->route_table_mutex_lock);
#endif /* HAVE_LIBPTHREAD */

    /* okay, send packet (assuming we have one) */
    if (routes > 0) {
	message = New (struct message);
	message->len = cp - buffer;
	message->buffer = buffer;
	LL_Add (ll_buffer, message);
    }

    LL_Iterate (ll_buffer, message) {
	ripng_sendmsgto (RIPNG->sockfd, message->buffer, message->len, 0,
			 &ripng, sizeof (ripng), interface);
	Delete (message->buffer);
	Delete (message);
    }

    LL_Destroy (ll_buffer);
/*  Delete (buffer); */
}


/* 
 * announce all change routes to our peers
 */
void
ripng_output_processing (int change)
{
    interface_t *interface;
    ripng_route_t *route;

    /* on flushing changes, this keeps locking the route table for a long time
       do we need to change our code ? XXX */

#ifdef HAVE_LIBPTHREAD
    if (change)
	pthread_mutex_lock (&RIPNG->route_table_mutex_lock);
#endif /* HAVE_LIBPTHREAD */

    /* announce routes */
    LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
	/* check if ripng is configured for this interface */
	if (BIT_TEST (interface->bit, RIPNG->interface_mask) &&
	    BIT_TEST (interface->flags, IFF_UP))
	    ripng_send_routes (interface, NULL, change);
    }
    /* clearing change flag */
    HASH_Iterate (RIPNG->hash, route) {
	route->flags &= ~(RT_RIPNG_CHANGE);
    }

#ifdef HAVE_LIBPTHREAD
    if (change)
	pthread_mutex_unlock (&RIPNG->route_table_mutex_lock);
#endif /* HAVE_LIBPTHREAD */
}


/* 
 * read and process and RIPNG update packet recieved on our interface
 */
void
ripng_receive_update ()
{
    u_char *ptr, *cp;
    int n, size, fromlen;
    u_char command, version;
    u_short zero;
    struct sockaddr_in6 from;
    prefix_t *prefix = NULL;
    gateway_t *gateway = NULL;
    char tmp6[INET6_ADDRSTRLEN];
    interface_t *interface;

    size = RIPNG_MIN_BUFSIZE;
    fromlen = sizeof (from);
    cp = ptr = NewArray (char, size);

    if ((n = ripng_recvmsgfrom (RIPNG->sockfd, ptr, size, O_NONBLOCK,
				&from, &fromlen, &interface)) < 0) {
	goto ignore;
    }

    assert (interface);

    if (fromlen != sizeof (from) || from.sin6_family != AF_INET6) {
	trace (TR_ERROR, RIPNG->trace, "RIPNG Error reading on socket\n");
	goto ignore;
    }

    /*
     * get command first for later check
     */
    UTIL_GET_BYTE (command, cp);
    UTIL_GET_BYTE (version, cp);
    UTIL_GET_SHORT (zero, cp);
    zero = ntohs (zero);

    prefix = New_Prefix (AF_INET6, (char *) &from.sin6_addr, 128);

    /*
     * common check for all commands
     */
    if (version != RIPNG_VERSION) {
	trace (TR_WARN, RIPNG->trace,
	       "RIPNG unsupported version %d from %s on %s\n",
	       version, prefix_toa2 (prefix, tmp6), interface->name);
/* XXX  goto ignore; */
    }

    if (zero) {
	trace (TR_WARN, RIPNG->trace,
	       "RIPNG non-zero pad field (value 0x%x) from %s on %s\n",
	       zero, prefix_toa2 (prefix, tmp6), interface->name);
/* XXX  goto ignore; */
    }

    if (command != RIPNG_RESPONSE && command != RIPNG_REQUEST) {
	trace (TR_WARN, RIPNG->trace,
	       "RIPNG unsupported command %d from %s on %s, ignore!\n",
	       command, prefix_toa2 (prefix, tmp6), interface->name);
	goto ignore;
    }

    if (command == RIPNG_RESPONSE) {

	/* register the gateway */
	gateway = add_gateway (prefix, 0, interface);

	/* don't listen to things broadcast from our own interface */
	if (find_interface_local (prefix)) {
	    trace (TR_PACKET, RIPNG->trace,
		   "recv RIPNG ignore own response from %s on %s, ignore!\n",
		   prefix_toa2 (prefix, tmp6), interface->name);
	    goto ignore;
	}

	if (IN6_IS_ADDR_V4MAPPED (&from.sin6_addr)) {
	    trace (TR_WARN, RIPNG->trace, "RIPNG received response "
		   "with ipv4 source from %s on %s, ignore!\n",
		   prefix_toa2 (prefix, tmp6), interface->name);
	    goto ignore;
	}

	if (!IN6_IS_ADDR_LINKLOCAL (&from.sin6_addr)) {
	    trace (TR_WARN, RIPNG->trace,
		   "RIPNG received response with non link-local source "
		   "from %s on %s, but continue\n",
		   prefix_toa2 (prefix, tmp6), interface->name);
	}

	/* check if this interface is configured for RIPNG */
	if (!BIT_TEST (interface->bit, RIPNG->interface_mask)) {
	    trace (TR_PACKET, RIPNG->trace, "RIPNG received response "
		   "from %s on unconfigured interface %s, ignore!\n",
		   prefix_toa2 (gateway->prefix, tmp6), interface->name);
	    goto ignore;
	}

	if (interface->mtu < n + IPV6_HDRLEN + UDP_HDRLEN) {
	    trace (TR_WARN, RIPNG->trace,
		 "RIPNG received packet size %d (+%d hdrs) exceeds mtu %d, "
		   "from %s on %s\n",
		   n, IPV6_HDRLEN + UDP_HDRLEN, interface->mtu,
		   prefix_toa2 (gateway->prefix, tmp6), interface->name);
	}
	else {
	    trace (TR_TRACE, RIPNG->trace,
		   "recv RIPNG response %d bytes from %s on %s\n",
		   n, prefix_toa2 (gateway->prefix, tmp6), interface->name);
	}

	/*
	 * ID also suggests to check if the hop count is 255
	 */

	if (ntohs (from.sin6_port) != RIPNG_DEFAULT_PORT) {
	    trace (NORM, RIPNG->trace,
		   "RIPNG non-standard source port %d from %s on %s, "
		   "ignored!\n", ntohs (from.sin6_port),
		   prefix_toa2 (gateway->prefix, tmp6), interface->name);
	    goto ignore;
	}

	if (n - (cp - ptr)) {
	    LINKED_LIST *ll_prefixes, *ll_attr;

	    ll_prefixes = LL_Create (LL_DestroyFunction, Deref_Prefix, NULL);
	    ll_attr = LL_Create (LL_DestroyFunction, ripng_deref_attr, NULL);

	    /* munge the ripng packet 
	       and return list of prefixes and attributes */

	    ripng_process_packet_response (gateway, cp, n - (cp - ptr),
					   ll_prefixes, ll_attr);

	    /* update our tables */
	    ripng_process_update (ll_prefixes, ll_attr);

	    LL_Destroy (ll_prefixes);
	    LL_Destroy (ll_attr);
	}
    }
    else if (command == RIPNG_REQUEST) {

	/* register the gateway */
	if (ntohs (from.sin6_port) == RIPNG_DEFAULT_PORT)
	    gateway = add_gateway (prefix, 0, interface);

	/* don't listen to things broadcast from our own interface
	   except for a query */
	if (find_interface_local (prefix) &&
	    ntohs (from.sin6_port) == RIPNG_DEFAULT_PORT) {
	    goto ignore;
	}

	trace (TR_TRACE, RIPNG->trace,
	       "recv RIPNG request %d bytes port %d from %s on %s\n",
	       n, ntohs (from.sin6_port), prefix_toa2 (prefix, tmp6),
	       interface->name);

#ifdef notdef
	/* request from over a router will be rejected for now */
	if (!IN6_IS_ADDR_LINKLOCAL (&from.sin6_addr)) {
	    trace (TR_WARN, RIPNG->trace,
		   "RIPNG received request "
		   "with non link-local source from %s on %s, ignore!\n",
		   prefix_toa2 (prefix, tmp6), interface->name);
	    goto ignore;
	}
#endif

	if (n - (cp - ptr) > 0) {
	    /* more than 1 entry */
	    if (ripng_process_packet_request (&from, cp, n - (cp - ptr),
					      interface) >= 0) {
		cp = ptr;
		UTIL_PUT_BYTE (RIPNG_RESPONSE, cp);
		UTIL_PUT_BYTE (RIPNG_VERSION, cp);
		UTIL_PUT_SHORT (0, cp);

		ripng_sendmsgto (RIPNG->sockfd, ptr, n, 0, &from,
				 sizeof (from), interface);
	    }
	    else {
		ripng_send_routes (interface, &from, 0);
	    }
	    trace (TR_PACKET, RIPNG->trace,
		   "recv RIPNG request answered to %s port %d on %s\n",
		   prefix_toa2 (prefix, tmp6), ntohs (from.sin6_port),
		   interface->name);
	}
	else {
	    trace (TR_WARN, RIPNG->trace,
		 "recv RIPNG request no entry from %s on %s, discard it!\n",
		   prefix_toa2 (prefix, tmp6), interface->name);
	}
    }

  ignore:
    Delete (ptr);
    select_enable_fd (RIPNG->sockfd);
    if (prefix)
	Deref_Prefix (prefix);
}


int
ripng_recvmsgfrom (int sockfd, char *buffer, int buflen, int flag,
		struct sockaddr_in6 *from, int *fromlen, interface_t ** ifp)
{
    int rc;
    prefix_t *prefix;
    interface_t *interface = NULL;
    int index = 0;
    char tmp6[INET6_ADDRSTRLEN];

#ifndef SCM_RIGHTS
/* Solaris has IP_RECVDSTADDR, but differs. */
    if ((rc = recvfrom (sockfd, buffer, buflen, flag, 
                       (struct sockaddr *) from, fromlen)) <= 0) {
	trace (TR_ERROR, RIPNG->trace, "RIPNG recvfrom: %s\n",
	       strerror (errno));
	return (-1);
    }
#else /* SCM_RIGHTS */
    struct msghdr mhdr;
    struct cmsghdr *cmsg;
#ifdef IPV6_PKTINFO
    struct in6_pktinfo *pkt_info;
#endif
    struct iovec iov;
    char chdr[BUFSIZ];

    mhdr.msg_name = (void *) from;
    mhdr.msg_namelen = *fromlen;
    memset (mhdr.msg_name, 0, mhdr.msg_namelen);
    mhdr.msg_iov = &iov;
    mhdr.msg_iovlen = 1;
    mhdr.msg_control = (void *) chdr;
    mhdr.msg_controllen = sizeof (chdr);
    memset (mhdr.msg_control, 0, mhdr.msg_controllen);
    iov.iov_base = buffer;
    assert (buflen >= INTERFACE_MASTER->max_mtu);
    iov.iov_len = buflen;

    if ((rc = recvmsg (sockfd, &mhdr, flag)) < 0) {
	trace (TR_ERROR, RIPNG->trace, "RIPNG recvmsg: %s\n",
	       strerror (errno));
	return (-1);
    }
    else {
	for (cmsg = CMSG_FIRSTHDR (&mhdr);
	     cmsg && cmsg->cmsg_len >= sizeof (struct cmsghdr);
	     cmsg = CMSG_NXTHDR (&mhdr, cmsg)) {
#ifdef notdef
	    if (cmsg->cmsg_level != IPPROTO_IPV6)
		continue;
#endif
	    switch (cmsg->cmsg_type) {
#ifdef IPV6_PKTINFO
	    case IPV6_PKTINFO:
		pkt_info = (struct in6_pktinfo *) cmsg->cmsg_data;
		index = pkt_info->ipi6_ifindex;
		trace (TR_PACKET, RIPNG->trace,
		       "recv RIPNG IPV6_PKTINFO index = %d\n", index);
		break;
#endif /* IPV6_PKTINFO */
#ifdef IP_RECVINTERFACE
	    case IP_RECVINTERFACE:
		index = *(u_short *) CMSG_DATA (cmsg);
		trace (TR_PACKET, RIPNG->trace,
		       "recv RIPNG IP_RECVINTERFACE index = %d\n", index);
		break;
#endif /* IP_RECVINTERFACE */
#ifdef IP_RECVDSTADDR
		/*
		 * XXX destination address of packet should be checked 
		 * RIPNG request may not be addressed with multicast addess
		 * it may be possible later
		 */
	    case IP_RECVDSTADDR:{
		    struct in6_addr dstaddr;
		    IN6_ADDR_COPY (&dstaddr,
				   (struct in6_addr *) CMSG_DATA (cmsg));
		    trace (TR_PACKET, RIPNG->trace,
			   "recv RIPNG IP_RECVDSTADDR dest = %s\n",
			 inet_ntop (AF_INET6, &dstaddr, tmp6, sizeof tmp6));
		}
		break;
#endif /* IP_RECVDSTADDR */
#ifdef IPV6_HOPLIMIT
		/*
		 * XXX it may be possible later checking hop limit
		 */
	    case IPV6_HOPLIMIT:{
		    int limit;
		    limit = *(u_char *) CMSG_DATA (cmsg);
		    trace (TR_PACKET, RIPNG->trace,
			   "recv RIPNG IPV6_HOPLIMIT = %d\n", limit);
		}
		break;
#endif /* IPV6_HOPLIMIT */
	    default:
		break;
	    }
	}
    }
#endif /* SCM_RIGHTS */

    prefix = New_Prefix (AF_INET6, (char *) &from->sin6_addr, 128);

#ifdef WIDE_IPV6
    assert (index == 0);

    index = (from->sin6_addr.s6_addr8[2] << 8) + from->sin6_addr.s6_addr8[3];

#endif /* WIDE_IPV6 */

    if (index > 0) {
	if ((interface = find_interface_flags (prefix, IFF_USERDEF))) {
	    trace (TR_PACKET, RIPNG->trace,
		   "recv RIPNG FIND TUNNEL %s index = %d\n",
		   interface->name, index);
	}
	else {
	    interface = find_interface_byindex (index);
	    if (interface == NULL) {
		trace (TR_PACKET, RIPNG->trace,
		       "RIPNG received response "
		       "on unknown interface from %s, ignore!\n",
		       prefix_toa2 (prefix, tmp6));
		Deref_Prefix (prefix);
		return (-1);
	    }
	}
    }
    else {
	gateway_t *gateway;
	/*
	 * if the system doesn't supply the incoming interface index,
	 * try to find among configured interfaces and gateways registered,
	 */
	if ((interface = find_interface (prefix))) {
	    trace (TR_PACKET, RIPNG->trace,
		   "RIPNG guesses %s on %s (find_interface)\n",
		   prefix_toa2 (prefix, tmp6), interface->name);
	}
	else if ((gateway = find_gateway (prefix, 0)) &&
		 gateway->interface) {

	    interface = gateway->interface;
	    trace (TR_PACKET, RIPNG->trace,
		   "RIPNG guesses %s on %s (find_gateway)\n",
		   prefix_toa2 (prefix, tmp6), interface->name);
	}
	else {
	    interface = INTERFACE_MASTER->default_interface;
	    trace (TR_PACKET, RIPNG->trace,
		   "RIPNG guesses %s on %s (default)\n",
		   prefix_toa2 (prefix, tmp6), interface->name);
	}
    }
    *ifp = interface;
    Deref_Prefix (prefix);
    return (rc);
}


/* 
 * process linked list of routes (prefix & attribute combinations)
 * check if 1) just updating time on already received route
 *          2) or a change (different attributes from same gateway,
 *             or better metric and attributes from new gateway
 */
int
ripng_process_update (LINKED_LIST * ll_prefixes,
		      LINKED_LIST * ll_attr)
{
    prefix_t *prefix;
    ripng_attr_t *ripng_attr;
    ripng_route_t *route;
    int metric, change = 0;
    char nexthop[MAXLINE], tag[MAXLINE];
    char tmp6[INET6_ADDRSTRLEN];
    time_t now;

    time (&now);
    pthread_mutex_lock (&RIPNG->route_table_mutex_lock);

    ripng_attr = LL_GetHead (ll_attr);
    LL_Iterate (ll_prefixes, prefix) {
	assert (ripng_attr->ref_count > 0);
	assert (prefix->ref_count > 0);

	/* do checks here -- if fail, log and ignore
	 * 1) destination prefix valid (i.e. not multicast)
	 * 2) prefix length valid 
	 * 3) metric valid (between 1 and 16 inclusive)
	 */

	/* see if valid and do policy to add cost to the metric */
	if ((metric = ripng_policy (prefix, ripng_attr, NULL)) < 0) {
	    continue;
	}
	ripng_attr->metric = metric;

#ifdef notdef
	IN6_ADDR_COPY (&in6, prefix_toaddr6 (prefix));
	netmasking (prefix->family, addr, prefix->bitlen);
	if (IN6_ADDR_COMP (&in6, prefix_toaddr6 (prefix)) != 0) {
	    trace (TR_PACKET, RIPNG->trace,
		   "RIPNG prefix %s will be truncated by prefixlen %d\n",
		   inet_ntop (AF_INET6, &in6, tmp6, sizeof tmp6),
		   prefix->bitlen);
	}
#endif
	if (IN6_ADDR_COMP (prefix_toaddr6 (ripng_attr->nexthop),
		       prefix_toaddr6 (ripng_attr->gateway->prefix)) != 0) {
	    sprintf (nexthop, " -> %s",
		     prefix_toa2 (ripng_attr->nexthop, tmp6));
	}
	else {
	    nexthop[0] = '\0';
	}
	if (ripng_attr->tag) {
	    sprintf (tag, " tag 0x%lx", ripng_attr->tag);
	}
	else {
	    tag[0] = '\0';
	}

	/* new route */
	if ((route = HASH_Lookup (RIPNG->hash, prefix)) == NULL) {

	    if (ripng_attr->metric == RIPNG_INFINITY) {
		trace (TR_PACKET, RIPNG->trace,
		       "  x %s/%d metric %d (infinity)\n",
		       prefix_toa2 (prefix, tmp6), prefix->bitlen,
		       ripng_attr->metric);
		goto next;
	    }

	    trace (TR_TRACE, RIPNG->trace,
		   "  o %s/%d metric %d%s (new)\n",
		   prefix_toa2 (prefix, tmp6), prefix->bitlen,
		   ripng_attr->metric, tag);

	    route = ripng_new_route (prefix, ripng_attr);
	    route->flags |= RT_RIPNG_CHANGE;
	    change++;

	    /* add route to rib */
	    assert (RIPNG->update_call_fn);
	    RIPNG->update_call_fn (KERNEL_UPDATE_ADD, route->prefix,
				   route->attr);
	}
	/* exiting route */
	else {

	    /* we already have a route for this prefix. Two cases:
	     *  1) from same gateway -- just update time, delete, or implicit withdraw.
	     *     if tag changed it should be updated and propagated -- masaki
	     *  2) from different gateway -- if metric better use this, otherwise
	     *     just ignore it. We'll hear about it again in 30 seconds...
	     */

	    if (route->attr->type == PROTO_CONNECTED) {
		trace (TR_PACKET, RIPNG->trace,
		       "  x %s/%d metric %d%s (direct)\n",
		       prefix_toa2 (prefix, tmp6), prefix->bitlen,
		       route->attr->metric, tag);
		goto next;
	    }

	    /* shoud check the preferences */
	    if (route->attr->type == PROTO_STATIC) {
		trace (TR_PACKET, RIPNG->trace,
		       "  x %s/%d metric %d%s (static)\n",
		       prefix_toa2 (prefix, tmp6), prefix->bitlen,
		       route->attr->metric, tag);
		goto next;
	    }

	    /* kernel route always overriden -- use pref instead */
	    if (route->attr->type == PROTO_KERNEL &&
		    ripng_attr->metric < RIPNG_INFINITY) {
		trace (TR_PACKET, RIPNG->trace,
		       "  o %s/%d%s metric %d%s (kernel)\n",
		       prefix_toa2 (prefix, tmp6), prefix->bitlen,
		       nexthop, route->attr->metric, tag);
		goto overwrite;
	    }

	    if (route->attr->gateway == ripng_attr->gateway) {

		/* case i: both infinity */
		if (ripng_attr->metric == RIPNG_INFINITY &&
		    route->attr->metric == RIPNG_INFINITY) {

		    trace (TR_PACKET, RIPNG->trace,
			   "  x %s/%d metric %d (deleting)\n",
			   prefix_toa2 (prefix, tmp6), prefix->bitlen,
			   ripng_attr->metric);

		    /* it is under deletion process, don't touch. 
		       ignoring it */
		    assert (route->flags & RT_RIPNG_DELETE);
		    /* should I propagate it? */
		    /* route->flags |= RT_RIPNG_CHANGE; */
		    /* change++; */
		}
		/* case ii: only new is infinity */
		else if (ripng_attr->metric == RIPNG_INFINITY) {

		    trace (TR_TRACE, RIPNG->trace,
			   "  o %s/%d%s metric %d%s (shift to delete)\n",
			   prefix_toa2 (prefix, tmp6), prefix->bitlen,
			   nexthop, ripng_attr->metric, tag);

		    /* shift it to deletion process */
		    route->flags |= RT_RIPNG_DELETE;
		    route->flags |= RT_RIPNG_CHANGE;

		    /* MRT doesn't hold another time, 
		       so this is ad hoc solution.
		       Someone may be confused the time changed and
		       strictry saying I need to start the timer */
		    route->time = now - RIPNG_TIMEOUT_INTERVAL;

		    change++;
		    ripng_deref_attr (route->attr);
		    route->attr = ripng_ref_attr (ripng_attr);
		    RIPNG->update_call_fn (KERNEL_UPDATE_HOLD,
					   route->prefix, route->attr);
		}
		/* case iii: new is not infinity (old may be infinity) */
		else {

		    if (route->attr->metric != ripng_attr->metric ||
		    /* ID says nothing about tag but I think it should be */
			route->attr->tag != ripng_attr->tag ||
			!prefix_compare (route->attr->nexthop,
					 ripng_attr->nexthop)) {

			trace (TR_TRACE, RIPNG->trace,
			       "  o %s/%d%s metric %d%s (change)\n",
			       prefix_toa2 (prefix, tmp6), prefix->bitlen,
			       nexthop, ripng_attr->metric, tag);

			if (!prefix_compare (route->attr->nexthop,
					     ripng_attr->nexthop)) {
			    /* first add the route with new next hop, 
			       then delete old one */
			    /* the following two calls should be atomic? 
			       -- masaki */
			    RIPNG->update_call_fn (KERNEL_UPDATE_DEL,
						route->prefix, route->attr);
			    RIPNG->update_call_fn (KERNEL_UPDATE_ADD,
						 route->prefix, ripng_attr);
			}
			else {
			    /* the same nexthop but stop deleting process */
			    if (BIT_TEST (route->flags, RT_RIPNG_DELETE)) {
				RIPNG->update_call_fn (KERNEL_UPDATE_UNHOLD,
						 route->prefix, ripng_attr);
			    }
			}

			route->flags &= ~(RT_RIPNG_DELETE);
			route->flags |= RT_RIPNG_CHANGE;
			change++;
			ripng_deref_attr (route->attr);
			route->attr = ripng_ref_attr (ripng_attr);
		    }
		    else {
			trace (TR_PACKET, RIPNG->trace,
			       "  x %s/%d metric %d (same)\n",
			       prefix_toa2 (prefix, tmp6),
			       prefix->bitlen, route->attr->metric);
		    }
		    route->time = now;
		}
	    }
	    /* from a gateway different for the existing one */
	    else {

		if (		/* better metric */
		       route->attr->metric > ripng_attr->metric ||
		       (route->attr->metric != RIPNG_INFINITY &&
		/* suggested heuristic in case of the same metric */
			(route->attr->metric == ripng_attr->metric &&
			 (now - route->time) >=
			 RIPNG_TIMEOUT_INTERVAL / 2))) {

		    trace (TR_TRACE, RIPNG->trace,
			   "  o %s/%d%s metric %d%s (change)\n",
			   prefix_toa2 (prefix, tmp6), prefix->bitlen,
			   nexthop, ripng_attr->metric, tag);
		    if (!prefix_compare (route->attr->nexthop,
					 ripng_attr->nexthop)) {
			/* add new one then remove old one */
			RIPNG->update_call_fn (KERNEL_UPDATE_DEL,
					       route->prefix, route->attr);

			/* don't remove kernel routes. 
			   remove RIPng routes only. */
		      overwrite:
			RIPNG->update_call_fn (KERNEL_UPDATE_ADD,
					       route->prefix, ripng_attr);
		    }
		    else {
			/* the same nexthop but metric changed from infinity */
			if (BIT_TEST (route->flags, RT_RIPNG_DELETE)) {
			    RIPNG->update_call_fn (KERNEL_UPDATE_UNHOLD,
						 route->prefix, ripng_attr);
			}
		    }

		    route->time = now;
		    route->flags &= ~(RT_RIPNG_DELETE);
		    route->flags |= RT_RIPNG_CHANGE;
		    change++;
		    ripng_deref_attr (route->attr);
		    route->attr = ripng_ref_attr (ripng_attr);
		}
		else {
		    trace (TR_PACKET, RIPNG->trace,
			   "  x %s/%d metric %d (>= existing %d)\n",
			   prefix_toa2 (prefix, tmp6), prefix->bitlen,
			   ripng_attr->metric, route->attr->metric);
		}
	    }
	}
      next:
	ripng_attr = LL_GetNext (ll_attr, ripng_attr);
    }

    pthread_mutex_unlock (&RIPNG->route_table_mutex_lock);

    if (change)
	ripng_set_flash_update ();
    return (1);
}


/* 
 * run policy on a ripng route 
 * returns -1 when policy rejects it
 * otherwise returns new metric
 */
int
ripng_policy (prefix_t * prefix, ripng_attr_t * attr, interface_t * out)
{
    int cost, num;
    struct in6_addr *addr = (struct in6_addr *) prefix_tochar (prefix);
    char tmp6[INET6_ADDRSTRLEN];

    if (out == NULL) {
	/* input policy processing */

	/* see if the prefix length valid */
	if (prefix->bitlen > 128) {
	    trace (TR_PACKET, RIPNG->trace,
		   "  x %s/%d metric %d (invalid prefix length)\n",
		   prefix_toa2 (prefix, tmp6), prefix->bitlen, attr->metric);
	    return (-1);
	}

	/* see if the destination prefix valid */
	if (IN6_IS_ADDR_MULTICAST (addr) ||
	    (IN6_IS_ADDR_V4COMPAT (addr) && prefix->bitlen > 0) ||
	    IN6_IS_ADDR_LINKLOCAL (addr)) {
	    trace (TR_PACKET, RIPNG->trace,
		   "  x %s/%d metric %d (non global)\n",
		   prefix_toa2 (prefix, tmp6), prefix->bitlen, attr->metric);
	    return (-1);
	}

	if (attr->metric < 1 || attr->metric > RIPNG_INFINITY) {
	    trace (TR_PACKET, RIPNG->trace,
		   "  x %s/%d metric %d (invalid metric)\n",
		   prefix_toa2 (prefix, tmp6), prefix->bitlen, attr->metric);
	    return (-1);
	}

	assert (attr->gateway);
	assert (attr->gateway->interface);
	/* check distribute-list for in */
	if ((num = attr->gateway->interface->dlist_in[PROTO_RIPNG])) {
	    if (apply_access_list (num, prefix) == 0) {
		trace (TR_PACKET, RIPNG->trace,
		       "  x %s/%d metric %d (a-list %d)\n",
		       prefix_toa2 (prefix, tmp6), prefix->bitlen,
		       attr->metric, num);
		return (-1);
	    }
	}
	cost = attr->metric + attr->gateway->interface->metric_in;
    }
    else {
	/* output policy processing */
	if (IN6_IS_ADDR_LINKLOCAL (addr) ||
	    IN6_IS_ADDR_MULTICAST (addr) ||
	    (IN6_IS_ADDR_V4COMPAT (addr) && prefix->bitlen > 0)) {
	    trace (TR_PACKET, RIPNG->trace,
		   "  x %s/%d metric %d (non global)\n",
		   prefix_toa2 (prefix, tmp6), prefix->bitlen, attr->metric);
	    return (-1);
	}
#ifdef notdef
	if (attr->type == PROTO_KERNEL) {
	    trace (TR_PACKET, RIPNG->trace,
		   "  x %s/%d metric %d (kernel route)\n",
		   prefix_toa2 (prefix, tmp6), prefix->bitlen, attr->metric);
	    return (-1);
	}
#endif
	/* split horizon (without poisoned reverse) */
	/* do not send to the same interface on which we got the route */
	if (attr->type == PROTO_RIPNG &&
	/* multicast capable and p-to-p interfaces */
	/* I'm not sure how NBMA will be supported */
	    ((BIT_TEST (out->flags, IFF_MULTICAST) ||
	      BIT_TEST (out->flags, IFF_POINTOPOINT)) &&
	     attr->gateway->interface == out)) {
	    /* No split horizon on NBMA which has not yet supported, though */
#ifdef RIPNG_POISONED_REVERSE
	    trace (TR_PACKET, RIPNG->trace,
		   "  o %s/%d metric %d (posoned reverse)\n",
		prefix_toa2 (prefix, tmp6), prefix->bitlen, RIPNG_INFINITY);
	    return (RIPNG_INFINITY);
#else
	    trace (TR_PACKET, RIPNG->trace,
		   "  x %s/%d metric %d (split horion)\n",
		   prefix_toa2 (prefix, tmp6), prefix->bitlen, attr->metric);
	    return (-1);
#endif /* RIPNG_POISONED_REVERSE */
	}
	if (attr->type == PROTO_CONNECTED && attr->gateway->interface == out) {
	    trace (TR_PACKET, RIPNG->trace,
		   "  x %s/%d metric %d (direct)\n",
		   prefix_toa2 (prefix, tmp6), prefix->bitlen, attr->metric);
	    return (-1);
	}
	/* check distribute-list for out */
	if ((num = out->dlist_out[PROTO_RIPNG])) {
	    if (apply_access_list (num, prefix) == 0) {
		trace (TR_PACKET, RIPNG->trace,
		       "  x %s/%d metric %d (a-list %d)\n",
		       prefix_toa2 (prefix, tmp6), prefix->bitlen,
		       attr->metric, num);
		return (-1);
	    }
	}
	cost = attr->metric + out->metric_out;
    }

    if (cost > RIPNG_INFINITY)
	cost = RIPNG_INFINITY;

    return (cost);
}


/*
 * ripng_update_route()
 * will be called from rib, that is, they are called from
 * other threads. Even though RIPng runs with a single thread,
 * locking on the route table and flash update flag have to be
 * protected by locking.
 */
int
ripng_update_route (prefix_t * prefix, generic_attr_t * new,
		    generic_attr_t * old)
{
    ripng_attr_t *ripng_attr;
    ripng_route_t *route;
    char tmp6[INET6_ADDRSTRLEN], tmp7[INET6_ADDRSTRLEN];

    pthread_mutex_lock (&RIPNG->route_table_mutex_lock);

    if (old) {

	trace (TR_TRACE, RIPNG->trace,
	       "RIPNG delete %s/%d nh %s proto %s\n",
	       prefix_toa2 (prefix, tmp6), prefix->bitlen,
	       prefix_toa2 (old->nexthop, tmp7),
	       proto2string (old->type));

	if ((route = HASH_Lookup (RIPNG->hash, prefix)) == NULL) {
	    pthread_mutex_unlock (&RIPNG->route_table_mutex_lock);
	    trace (TR_TRACE, RIPNG->trace, "RIPNG no entry!\n");
	    return (-1);
	}

	if (route->attr->type == old->type)
	    ripng_delete_route (route);
	else
	    trace (TR_TRACE, RIPNG->trace, 
                   "RIPNG entry was learned from proto %s\n", 
                   proto2string (route->attr->type));
    }

    if (new) {

	trace (TR_TRACE, RIPNG->trace,
	     "RIPNG add %s/%d nh %s proto %s\n", prefix_toa2 (prefix, tmp6),
	       prefix->bitlen, prefix_toa2 (new->nexthop, tmp7),
	       proto2string (new->type));

	if ((route = HASH_Lookup (RIPNG->hash, prefix)) == NULL) {
	    /* new route */
	    ripng_attr = ripng_new_attr (1 /* XXX */ );
	    ripng_attr->type = new->type;
	    ripng_attr->nexthop = Ref_Prefix (new->nexthop);
	    ripng_attr->gateway = new->gateway;
	    ripng_attr->tag = new->tag;

	    route = ripng_new_route (prefix, ripng_attr);
	    route->flags |= RT_RIPNG_CHANGE;
	    ripng_deref_attr (ripng_attr);

	    if (new->type != PROTO_STATIC &&
		new->type != PROTO_CONNECTED &&
		new->type != PROTO_KERNEL) {
		ripng_set_flash_update ();
	    }
	}
	else {
	    assert (old == NULL);
	    /* XXX should overwrite by comparing preferences */
/*	    trace (TR_ERROR, RIPNG->trace, "RIPNG ERROR entry exist!\n"); */
	    pthread_mutex_unlock (&RIPNG->route_table_mutex_lock);
	    return (-1);
	}
    }

    pthread_mutex_unlock (&RIPNG->route_table_mutex_lock);
    return (1);
}


/* 
 * change/set ripng attributes.
 */
void
ripng_set (int first,...)
{
    va_list ap;
    enum RIPNG_ATTR attr;

    /* Process the Arguments */
    va_start (ap, first);

    for (attr = (enum RIPNG_ATTR) first;
	 attr;
	 attr = va_arg (ap, enum RIPNG_ATTR)) {

	switch (attr) {
	case RIPNG_RT_UPDATE_FN:
	    RIPNG->update_call_fn = va_arg (ap, void *);
	    break;
	case RIPNG_TRACE_STRUCT:
	    RIPNG->trace = va_arg (ap, trace_t *);
	    break;
	default:
	    assert (0);
	    break;
	}
    }
    va_end (ap);
}


/* 
 * create/allocate new ripng_route_t structure and insert
 * it in the RIPNG prefix route hash table
 */
ripng_route_t *
ripng_new_route (prefix_t * prefix, ripng_attr_t * attr)
{
    ripng_route_t *route;

    route = New (ripng_route_t);
    route->prefix = Ref_Prefix (prefix);
    route->attr = ripng_ref_attr (attr);
    route->time = time (NULL);
    route->delete = 0;
    route->flags |= RT_RIPNG_CHANGE;

    HASH_Insert (RIPNG->hash, route);
    return (route);
}


/* 
 * free route memory and remove from hash
 */
void
ripng_delete_route (ripng_route_t * route)
{
    HASH_Remove (RIPNG->hash, route);
    Deref_Prefix (route->prefix);
    ripng_deref_attr (route->attr);
    Delete (route);
}


ripng_attr_t *
ripng_new_attr (int metric)
{
    ripng_attr_t *tmp;

    tmp = New (ripng_attr_t);
    tmp->type = PROTO_RIPNG;
    tmp->metric = metric;
    tmp->ref_count = 1;

    return (tmp);
}


ripng_attr_t *
ripng_ref_attr (ripng_attr_t * attr)
{
    assert (attr->ref_count > 0);
    attr->ref_count++;
    return (attr);
}


void
ripng_deref_attr (ripng_attr_t * attr)
{
    assert (attr->ref_count > 0);
    if (--attr->ref_count <= 0) {
	Deref_Prefix (attr->nexthop);
	Delete (attr);
    }
}


/* 
 * dump various ripng stats to a socket
 * usually called by UII (user interactive interface 
 */
void
ripng_show (uii_connection_t * uii)
{
    uii_send_data (uii, "\r\nRouting Protocol is \"ripng\" (Using IPV6)\r\n");

    if (RIPNG->sockfd < 0)
	uii_send_data (uii, "Not listening for RIPNG announcements\r\n",
		       RIPNG_DEFAULT_PORT, RIPNG->sockfd);
    else
	uii_send_data (uii, "Listening on port %d (socket %d)\r\n",
		       RIPNG_DEFAULT_PORT, RIPNG->sockfd);

    uii_send_data (uii,
      "Sending updates every %d seconds +/- %d, next due in %d seconds\r\n",
		   RIPNG_UPDATE_INTERVAL, RIPNG_UPDATE_JITTER,
		   RIPNG->timer->time_next_fire - time (NULL));
    uii_send_data (uii, "Triggered update and split horizon "
		   "(no poisoned reverse) implemented\r\n");
    uii_send_data (uii,
	     "Invalid after %d seconds, hold down %d, flushed after %d\r\n",
    RIPNG_TIMEOUT_INTERVAL, RIPNG_TIMEOUT_INTERVAL, RIPNG_GARBAGE_INTERVAL);
}


/*
 * dump RIPNG routing table to socket. Usually called by user
 * interactive interface
 */
void
ripng_show_routing_table (uii_connection_t * uii)
{
    ripng_route_t *route;
    time_t now;
    interface_t *interface;
    char tmp[MAXLINE], tmp6[INET6_ADDRSTRLEN], *name = NULL;

    if (parse_line (uii->cp, "%s", tmp) >= 1) {
	if ((interface = find_interface_byname (tmp)) == NULL) {
	    uii_send_data (uii, "no such interface: %s\r\n", tmp);
	    return;
	}
	name = tmp;
    }

    time (&now);
    uii_send_data (uii, "%c %-32s %-5s %-26s %-5s %-4s\r\n",
		   'P', "Prefix", "IF", "Next Hop", "Metric", "Time");

    pthread_mutex_lock (&RIPNG->route_table_mutex_lock);

    HASH_Iterate (RIPNG->hash, route) {
	char *ifname = (route->attr->gateway->interface) ?
	route->attr->gateway->interface->name : "?";

	if (name && strcmp (ifname, name))
	    continue;

	sprintf (tmp, "%s/%d", prefix_toa2 (route->prefix, tmp6),
		 route->prefix->bitlen);
	uii_send_data (uii, "%c %-32s %-5s %-26s %5d %4d\r\n",
		       proto2string (route->attr->type)[0],
		       tmp, ifname,
		       prefix_toa2 (route->attr->nexthop, tmp6),
		       route->attr->metric,
		       (route->attr->type == PROTO_STATIC ||
	     route->attr->type == PROTO_CONNECTED) ? 0 : now - route->time);
    }
    pthread_mutex_unlock (&RIPNG->route_table_mutex_lock);
}


#else /* NO HAVE_IPV6 */

/* just a null routine to give compilers something to stick in the
 * library so they don't complain about lack of table of contents
 */

void
ripng_stub ()
{
    while (1);
}

#endif /* HAVE_IPV6 */
