/* 
 * $Id: route_atob.c,v 1.7 1997/03/21 10:33:09 masaki Exp $
 */

#include <version.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <netdb.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <mrt.h>
#include <array.h>
#include <hash.h>
#include <lbgp.h>
#include <bgp.h>
#include <io.h>

static void read_ascii (char *name, io_t *io);
static void skip (FILE *fd);


main (int argc, char *argv[])
{
  char c;
  extern char *optarg;	/* getopt stuff */
  extern int optind;	/* getopt stuff */
  char *usage ="Usage: route_atob [-i ascii_data_file] [-(o|w) binary_out_file]\n";
  char *filename = "stdin";
  int errors = 0;
  io_t *IO;
  trace_t *default_trace;

  default_trace = New_Trace ();
  IO = (io_t *) New_IO (default_trace);
  io_set (IO, IO_OUTFILE, "stdout", NULL);

  while ((c = getopt(argc, argv, "i:f:o:w:v")) != -1)
      switch (c) {
      case 'o':	
	if (io_set(IO, IO_OUTFILE, (char *) (optarg), NULL) < 0) {
	  printf ("Error setting outfile\n");
	  errors++;
	}
	break;
      case 'w':
	if (io_set(IO, IO_OUTMSGQ, (char *) (optarg), NULL) < 0) {
          printf ("Error setting output %s\n", optarg);
	  errors++;
        } 
	break;
      case 'i':	
      case 'f':	
	 filename = (char *) optarg;
	 break;
      case 'v':
        set_trace (default_trace, TRACE_FLAGS, TR_ALL, 
                  TRACE_LOGFILE, "stdout", NULL);
        break;
      default:
	errors++;
      }

  if (errors) {
    fprintf(stderr, usage);
    printf ("\nMRT version (%s) compiled on %s\n\n",
	    MRT_VERSION, __DATE__);
    exit (0);
  }

  init_mrt (default_trace);
  read_ascii (filename, IO);
}



static void read_ascii (char *name, io_t *IO) {

  FILE *fd;
  char line[MAXLINE], *ret;
  char data[MAXLINE];
  bgp_attr_t *attr;
  int announce_flag;
  LINKED_LIST *ll_ann, *ll_with;
  prefix_t *prefix;
  short size;
  u_char *cp;
  int line_num = 0;
  int have_data = 0;


  if (!strcmp (name, "stdin")) 
    fd = stdin;
  else
    fd = fopen (name, "r");

  if (fd == NULL) {
    perror ("Error opening input");
    exit (1);
  }

  announce_flag = 0;
  attr = New (bgp_attr_t);
  attr->ref_count = 1;
  ll_ann = LL_Create (LL_DestroyFunction, Deref_Prefix, 0);
  ll_with = LL_Create (LL_DestroyFunction, Deref_Prefix, 0);

  line_num++;
  if (fgets (line, MAXLINE-1, fd) == NULL)
    return;
  while (line[0] == '\n') {
    line_num++;
    if (fgets (line, MAXLINE-1, fd) == NULL)
      return;
  }
  for (;;) {
    trace (NORM, MRT->trace, "%s", line);

    /* type BGP/UPDATE */
    if (sscanf (line, "TYPE: %s", data) >= 1) {
      if (strcmp ("BGP/UPDATE", data) != 0) {
        trace (ERROR, MRT->trace, "Unknown message type %s line %d\n", 
          data, line_num);
	skip (fd);  /* ignore rest of message until blank line */
      }
      else {
        have_data++;
      }
    }

    /* announce/withdraw */
    else if (!strncmp ("ANNOUNCE:", line, 6)) {
      announce_flag = 1;
    }
    else if (!strncmp (line, "WITHDRAW:", 6)) {
      announce_flag = 0;
    }

    /* aspath */
    else if (!strncmp (line, "ASPATH: ", 8)) {
      if ((attr->aspath = aspth_from_string (line+8)) == NULL) 
        trace (ERROR, MRT->trace, "Unrecognized ASPATH line %d\n", line_num);
    }

    /* next hop */
    else if (sscanf (line, "NEXT_HOP: %s", data) >= 1) {
      if ((attr->nexthop = ascii2prefix (AF_INET, data)) == NULL)
        trace (ERROR, MRT->trace, "Unrecognized NEXT_HOP line %d\n", line_num);
    }

    /* a prefix -- line begins with a space */
    else if ((line[0] == ' ') || (line[0] == '\t')) {
      line[strlen(line) - 1] = '\0';
      if ((prefix = ascii2prefix (AF_INET, (line+2))) == NULL) {
        trace (ERROR, MRT->trace, "Unrecognized prefix line %d\n", line_num);
      }
      else {
        if (announce_flag == 1) {
	  LL_Add (ll_ann, prefix);
        }
        else {
	  LL_Add (ll_with, prefix);
        }
      }
    }

    /* try to get next line */
    line_num++;
    if ((ret = fgets(line, MAXLINE-1, fd)) == NULL || line[0] == '\n') {
    /* eof or newline */
      if (have_data) {
        cp = bgp_create_update_msg (&size, attr, ll_ann, ll_with);
        io_write (IO, 0, MSG_PROTOCOL_BGP, MSG_BGP_UPDATE, size, (char *)cp);
        Delete (cp);
        have_data = 0;
        trace (NORM, MRT->trace, "***\n");
      }
      trace (NORM, MRT->trace, "\n");
      bgp_delete_attr (attr);
      if (ret == NULL) {
        LL_Destroy (ll_ann);
        LL_Destroy (ll_with);
        return;
      }
      while (line[0] == '\n') {
        line_num++;
        if (fgets (line, MAXLINE-1, fd) == NULL)
          return;
      }
      announce_flag = 0;
      LL_Clear (ll_ann);
      LL_Clear (ll_with);
      attr = New (bgp_attr_t);
      attr->ref_count = 1;
    }
  };
}



/* skip
 */
static void skip (FILE *fd) {
  char line[MAXLINE];

 while (fgets (line, MAXLINE-1, fd) != NULL) {
   if (line[0] == '\n') return;
 }
 return;
}    

   
  
