/* 
 * $Id: bgpsim.c,v 1.11 1997/03/21 10:32:49 masaki Exp $
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include <mrt.h>
#include <select.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <timer.h>
#include <bgp.h>
#include <io.h>
#include <lbgp.h>
#include "bgpsim.h"

simulation_t SIMULATION;
bgp_t *BGP;
io_t *IO;
trace_t *default_trace = NULL;

int init_gateways ();
int init_schedules ();
void err_quit ();
void start_config (uii_connection_t *uii);
int show_bgp ();
int show_bgp_routing_table();
int show_simulation();
int simulate ();
int start_simulate ();
int init_config ();
int simulate_add_routes (network_t *network, int change); 
int stop_simulation (uii_connection_t * uii);
int stop_simulation_schedule (uii_connection_t *uii);

/* main
 */
int main (int argc, char *argv[])
{
  char c;
  extern char *optarg;	/* getopt stuff */
  extern int optind;	/* getopt stuff */
  int errors = 0;

  char *usage ="Usage: bgpsim [-f file] [-v]\n";
  char *config_file = "./bgpsim.conf";
  trace_t *default_trace = NULL;
  
  /* init simulation */
  SIMULATION.ll_networks = LL_Create (0);

  while ((c = getopt(argc, argv, "vf:")) != -1) 
     switch (c) {
     case 'v': /* verbose */
       default_trace = New_Trace ();
       set_trace (default_trace, TRACE_FLAGS, TR_ALL, 
		  TRACE_LOGFILE, "stdout",
		  NULL);
       break;
     case 'f': /* config file */
       config_file = (char *) strdup ((char *) (optarg));
       break;
     case 'h':
     default:
       errors++;
       break;
     }

   if (errors) {
     fprintf(stderr, usage);
     printf ("\nMRT version (%s) compiled on %s\n\n",
	     MRT_VERSION, __DATE__);
     exit (0);
   }

   init_mrt (default_trace);
   init_BGP (default_trace);

   set_BGP (NULL);

   set_uii (UII, UII_PROMPT, 1, "bgpsim> ");
   set_uii (UII, UII_PROMPT, 2, "bgpsim-config> ");

   /*   signal (SIGINT, sbgp_quit);*/

   uii_add_command (1, "config", (void *) start_config);
   uii_add_command (1, "show bgp", (void *) show_bgp);
   uii_add_command (1, "show bgp routes", (void *) show_bgp_routing_table);
   uii_add_command (1, "show ip", (void *) show_bgp_routing_table);
   uii_add_command (1, "show simulation", (void *) show_simulation);
   uii_add_command (1, "stop", (void *) stop_simulation_schedule);

   init_config ();
   if (config_from_file (config_file) == -1) {
     perror ("Could not open file");
     exit (0);
   }

   listen_uii ();
   start_bgp ();
   start_simulation ();

#ifndef HAVE_LIBPTHREAD
   while (1) {
     process_all_schedules();
     mrt_select ();
   }
#endif /* HAVE_LIBPTHREAD */
   
   /* loop */
   while (1); 

}


int stop_simulation (uii_connection_t * uii) {
  network_t *network;

  SIMULATION.on_off = 0;

  uii_send_data (uii, "\r\nStopping Simulation...\r\n");

  LL_Iterate (SIMULATION.ll_networks, network) {
    Timer_Turn_OFF (network->timer_stability);
    Timer_Turn_OFF (network->timer_change);
    uii_send_data (uii, " Simulation network %d OFF\r\n", network->num);
  }
  
  return (1);
}


int show_simulation (uii_connection_t * uii) {
  char tmp[MAXLINE];
  u_int hours, minutes, seconds;
  
  seconds = time (NULL) - SIMULATION.time_start;
  minutes = seconds / 60;
  hours = minutes /60;
  minutes = minutes - hours*60;
  seconds = seconds - hours*60*60 - minutes*60;

  sprintf (tmp, "%02d:%02d:%02d", hours, minutes, seconds);

  if (SIMULATION.on_off == 1) 
    uii_send_data (uii, "BGP Simulation\t\t[Status: RUNNING]\r\n");
  else
    uii_send_data (uii, "BGP Simulation\t\t[Status: STOPPED]\r\n");
  uii_send_data (uii, "Time:%s\t\tEnd: Indefinite\r\n", tmp);
  uii_send_data (uii, "Packets:0\t\tEnd Total: Indefinite\r\n");


  return (1);
}



int start_simulation () {
  SIMULATION.schedule = (schedule_t *) New_Schedule ("BGP Simulation", default_trace);
  SIMULATION.trace = default_trace;
  SIMULATION.on_off = 1;
  SIMULATION.time_start = time (NULL);

  mrt_thread_create ("BGP Simulation", SIMULATION.schedule, 
		      (void *)  simulate, default_trace);
}


void network_stability (network_t *network) {
  bgp_route_t *route;

  if (network->flag == 1) {
    trace (NORM, SIMULATION.trace,
	   "Simlating network list %d announced\n",  network->num);
    simulate_add_routes (network, -1);
    network->flag = 0;
  }
  else {
    trace (NORM, SIMULATION.trace,
	 "Simlating network list %d withdrawn\n",  network->num);
    simulate_add_routes (network, 0);
    network->flag = 1;
  }
}


void network_change (network_t *network) {
  bgp_route_t *route;

  trace (NORM, SIMULATION.trace,
	 "Simlating network list %d change\n",  network->num);
  simulate_add_routes (network, 1);
  network->flag = 1;
}


int network_schedule_stability (mtimer_t *timer, network_t *network) {
  schedule_event (SIMULATION.schedule, (void *) network_stability, 1, (void *) network);
}

int network_schedule_change (mtimer_t *timer, network_t *network) {
  schedule_event (SIMULATION.schedule, (void *) network_change, 1, (void *) network);
}


int stop_simulation_schedule (uii_connection_t *uii) {
  schedule_event (SIMULATION.schedule, (void *) stop_simulation, 1, (void *) uii);
}

int simulate () {
  network_t *network;

  LL_Iterate (SIMULATION.ll_networks, network) {
    network->current = 1;
    simulate_add_routes (network, 0);

    if (network->stability > 0) {
      Timer_Turn_ON (network->timer_stability);
    }

    if (network->change_interval > 0) {
      Timer_Turn_ON (network->timer_change);
    }
  }

#ifdef HAVE_LIBPTHREAD
  while (1) 
      schedule_wait_for_event (SIMULATION.schedule);
   exit (0);
#else
   return (0);
#endif /* HAVE_LIBPTHREAD */
}



/* simulate_add_routes
 * if change == 1, use alternate aspath and nexthop 
 * if change == -1, delete the route
 */
int simulate_add_routes (network_t *network, int change) 
{
  bgp_attr_t *attr, *tmp_attr;
  u_long l, ll;
  u_char *c, *d;
  bgp_route_t *route;
  prefix_t *prefix;

  tmp_attr = New (bgp_attr_t);
  
  if (change == 1) {
    network->current++;

    if (network->current > network->max) {
      network->current = 1;
    }
  }

  if ((tmp_attr->nexthop = network->nexthops[network->current]) == NULL) 
    tmp_attr->nexthop =  network->nexthops[1];
  if ((tmp_attr->aspath = network->aspaths[network->current]) == NULL) 
    tmp_attr->aspath = network->aspaths[1];
  if ((tmp_attr->community = network->communities[network->current]) == NULL) 
    tmp_attr->community = network->communities[1];
  tmp_attr->local_pref = network->localprefs[network->current];
  tmp_attr->dpa = network->dpas[network->current];

  /* set optional attributes */
  if (tmp_attr->community != NULL) tmp_attr->attribs |= O_COMMUNITY;
  if (tmp_attr->dpa != NULL) tmp_attr->attribs |= O_DPA;
  if ( tmp_attr->local_pref > 0) tmp_attr->attribs |= O_LOCALPREF;
   
  /*attr = bgp_get_attr (tmp_attr);*/
    attr = tmp_attr;

    l = prefix_tolong (network->start_prefix);
    ll = prefix_tolong (network->end_prefix);
    c = (u_char *) &l;
    d = (u_char *) &ll;

    view_open (BGP->view);

    /* class A */
    /*if (c[0] <= 127) {*/
    if (IN_CLASSA (l)) {
      while ((c[0] <= d[0]) && (c[0] <= 126)){
	prefix = New_Prefix (AF_INET, c, 8);

	  if (change >= 0) 
	    route = (bgp_route_t *) bgp_add_route (BGP->view, prefix, attr, 0);
	  else {
	    route = (bgp_route_t *) view_find_bgp_active (BGP->view, prefix);
	    view_delete_bgp_route (BGP->view, route);
	  }
	  (c[0])++;
      }
    }
    else if (IN_CLASSB (l)) {
      while ((c[0] <= d[0]) && (c[0] >=128) && (c[0] <= 191)){
	while (c[1] <= 254) {
	  if ((c[0] == d[0]) && (c[1] > d[1])) {break;}
	  prefix = New_Prefix (AF_INET, c, 16);

	  if (change >= 0) 
	    route = (bgp_route_t *) bgp_add_route (BGP->view, prefix, attr, 0);
	  else {
	    route = (bgp_route_t *) view_find_bgp_active (BGP->view, prefix);
	    view_delete_bgp_route (BGP->view, route);
	  }
	  Delete_Prefix (prefix);
	  (c[1])++;
	}
	(c[1])= 0;
	(c[0])++;
      }
    }
    else if (IN_CLASSC (l)) {
      while ((c[0] <= d[0]) && (c[0] >=192) && (c[0] <= 223)){
	while (c[1] <= 254) {
	  prefix = New_Prefix (AF_INET, c, 24);

	  if (change >= 0) 
	    route = (bgp_route_t *) bgp_add_route (BGP->view, prefix, attr, 0);
	  else {
	    route = (bgp_route_t *) view_find_bgp_active (BGP->view, prefix);
	    view_delete_bgp_route (BGP->view, route);
	  }
	  (c[1])++;
	}
	(c[1])= 0;
	(c[0])++;
      }
    }
    bgp_process_changes (BGP->view);
    view_close (BGP->view);

    network->flag = 1;
}
