/*
 * $Id: packet.c,v 1.14 1997/03/21 09:12:16 masaki Exp $
 */

#include <config.h>

#ifdef HAVE_IPV6
#include <sys/types.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <signal.h>

#include <sys/socket.h>
#include <netinet/in.h>

#include <mrt.h>
#include <trace.h>
#include <ripng.h>
#include <interface.h>
#include <api6.h>

/* 
 * Given 1) the gateway we received the packet from, 
 *       2) a pointer to the packet, and
 *       3) the length (num bytes) of the packet, 
 * munge the packet making a linked list of prefix and attribute tuples
 */

void
ripng_process_packet_response (gateway_t * gateway, char *update, int bytes,
			   LINKED_LIST * ll_prefixes, LINKED_LIST * ll_attr)
{
    char *cp;
    prefix_t *prefix;
    prefix_t *nexthop = Ref_Prefix (gateway->prefix);
    ripng_attr_t *p_attr;
    struct in6_addr *addr6;
    char tmp6[INET6_ADDRSTRLEN], tmp7[INET6_ADDRSTRLEN];

#define RIPNG_RTELEN 20

    if ((bytes % RIPNG_RTELEN) != 0) {
	trace (NORM, RIPNG->trace, "RIPNG invalid RTE size %d\n", bytes);
    }

    cp = update;

    while (cp < update + bytes) {
	u_short tag;
	u_char prefixlen, metric;

	if ((update + bytes) - cp < RIPNG_RTELEN)
	    break;
	assert ((update + bytes) - cp >= RIPNG_RTELEN);
	addr6 = (struct in6_addr *) cp;
	cp += 16;

	UTIL_GET_SHORT (tag, cp);
	tag = ntohs (tag);
	UTIL_GET_BYTE (prefixlen, cp);
	UTIL_GET_BYTE (metric, cp);

#if 0
	trace (TR_PACKET, RIPNG->trace,
	       "  recv %s/%d metric %d tag %d\n",
	       inet_ntop (AF_INET6, addr6, tmp6, sizeof tmp6),
	       prefixlen, metric, tag);
#endif

#define RIPNG_NEXT_HOP 0xff
	if (metric == RIPNG_NEXT_HOP) {
	    if (IN6_IS_ADDR_UNSPECIFIED (addr6)) {
		trace (TR_PACKET, RIPNG->trace,
		       "RIPNG nexthop is the originator itself %s\n",
		       prefix_toa2 (gateway->prefix, tmp6));
		continue;
	    }
	    if (!IN6_IS_ADDR_LINKLOCAL (addr6)) {
		trace (NORM, RIPNG->trace,
		    "RIPNG nexthop %s but not link-local address from %s\n",
		       inet_ntop (AF_INET6, addr6, tmp6, sizeof tmp6),
		       prefix_toa2 (gateway->prefix, tmp7));
		continue;
	    }
	    trace (TR_PACKET, RIPNG->trace,
		   "  nexthop %s\n",
		   inet_ntop (AF_INET6, addr6, tmp6, sizeof tmp6));
	    if (prefixlen != 0 && tag != 0)
		trace (NORM, RIPNG->trace,
		       "RIPNG non-zero prefixlen (%d) or tag (%d) "
		       "in specifying nexthop %s from %s\n",
		       prefixlen, tag,
		       inet_ntop (AF_INET6, addr6, tmp6, sizeof tmp6),
		       prefix_toa2 (gateway->prefix, tmp7));
	    Deref_Prefix (nexthop);
	    nexthop = New_Prefix (AF_INET6, (char *) addr6, 128);
	}
	else {
	    prefix = New_Prefix (AF_INET6, (char *) addr6, prefixlen);
	    p_attr = ripng_new_attr (metric);
	    p_attr->gateway = gateway;
	    p_attr->nexthop = Ref_Prefix (nexthop);
	    p_attr->metric = metric;
	    p_attr->tag = tag;

	    LL_Add (ll_prefixes, prefix);
	    LL_Add (ll_attr, p_attr);
	}
    }
}


/* 
 * Given 1) the gateway we received the packet from, 
 *       2) a pointer to the packet, and
 *       3) the length (num bytes) of the packet, 
 * fill up metrics if exists
 *
 * return value >0: ordinal request (number of entries)
 *              -1: whole-table request
 */

int
ripng_process_packet_request (struct sockaddr_in6 *from, char *update,
			      int bytes, interface_t * interface)
{
    char *cp = update;
    prefix_t *prefix;
    ripng_route_t *route;
    struct in6_addr *addr6;
    int n = 0;
    char tmp6[INET6_ADDRSTRLEN];

    prefix = New_Prefix (AF_INET6, (char *) &from->sin6_addr, 128);

    /* it may be a little bit expensive to lock the route table
       we may not need to do it */
    pthread_mutex_lock (&RIPNG->route_table_mutex_lock);

    while (cp < update + bytes) {
	u_short tag;
	u_char prefixlen;
	char metric;

	assert ((update + bytes) - cp >= RIPNG_RTELEN);
	addr6 = (struct in6_addr *) cp;
	cp += 16;

	UTIL_GET_SHORT (tag, cp);
	tag = ntohs (tag);
	UTIL_GET_BYTE (prefixlen, cp);
	UTIL_GET_BYTE (metric, cp);

	if (cp == update + bytes && metric == RIPNG_INFINITY &&
	    prefixlen == 0 &&	/* tag == 0 && */
	    IN6_IS_ADDR_UNSPECIFIED (addr6)) {
	    trace (TR_PACKET, RIPNG->trace,
		   "  whole-table request, responding\n");
    	    pthread_mutex_unlock (&RIPNG->route_table_mutex_lock);
    	    Deref_Prefix (prefix);
	    return (-1);
	}

	prefix = Change_Prefix (AF_INET6, (char *) addr6, 128, prefix);

	if ((route = HASH_Lookup (RIPNG->hash, prefix))) {
	    tag = route->attr->tag;
	    if (ntohs (from->sin6_port) == RIPNG_DEFAULT_PORT
	    /* && IN6_IS_ADDR_LINKLOCAL (&from->sin6_addr) */ ) {
		/* router's query, apply policy for that interface */
		if ((metric = ripng_policy (prefix, route->attr,
					    interface)) < 0) {
		    metric = RIPNG_INFINITY;
		}
	    }
	    else {
		trace (TR_PACKET, RIPNG->trace, "  o %s/%d metric %d\n",
		       prefix_toa2 (prefix, tmp6), prefix->bitlen, metric);
	    }
	}
	else {
	    metric = RIPNG_INFINITY;
	    tag = 0;		/* ID doesn't specify -- masaki */
	    trace (TR_PACKET, RIPNG->trace, "  x %s/%d metric %d (no exist)\n",
		   prefix_toa2 (prefix, tmp6), prefix->bitlen, metric);
	}
	cp -= 4;
	UTIL_PUT_SHORT (htons (tag), cp);
	UTIL_PUT_BYTE (prefixlen, cp);
	UTIL_PUT_BYTE (metric, cp);
	n++;
    }

    pthread_mutex_unlock (&RIPNG->route_table_mutex_lock);
    Deref_Prefix (prefix);
    return (n);
}

#endif /* HAVE_IPV6 */
