/* 
 * $Id: bgp_util.c,v 1.12 1997/03/21 10:32:02 masaki Exp $
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <stdarg.h>
#include <netinet/in.h>
#include <unistd.h>
#include <netinet/tcp.h>
#include <netdb.h>
#include <stdlib.h>
#include <string.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <netdb.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <mrt.h>
#include <timer.h>
#include <trace.h>
#include <bgp.h>
#include <select.h>
#include <interface.h>
#include <view.h>

u_int aspath_hash_fn (aspath_t *aspath, u_int size);
void bgp_schedule_timer (mtimer_t *timer, bgp_peer_t *peer);
char *sbgp_states[] = {
   "NULL",
   "Idle",
   "Connect",
   "Active", 
   "Opensent", 
   "Openconfirm",
   "Established"
};

char *sbgp_pdus[] = {
   "NULL",
   "Open",
   "Update",
   "Notification",
   "Keepalive",
   "PACKET_MAX"
};

char *sbgp_events[] = {
   "NULL",
   "Start",
   "Stop",
   "Open",
   "Closed",
   "OpenFail",
   "Error",
   "ConnectRetry",
   "HoldTime",
   "KeepAlive",
   "RecvOpen",
   "RecvKeepAlive",
   "RecvUpdate",
   "RecvNotify"
};

   
char *s_bgp_error[] = {
   "UNUSED",
   "Message Header Error",
   "Open Message Error",
   "Update Message Error",
   "Hold mtimer_t Expired",
   "Finite State Machine Error",
   "Cease"
};

char *s_bgp_error_header[] = {
   "UNUSED",
   "Connection Not Synchronized",
   "Bad Message Length",
   "Bad Message Type"
};

char *s_bgp_error_open[] = {
   "UNUSED",
   "Unsupported Version Number",
   "Bad Peer AS",
   "Bad BGP Identifier",
   "Unsupported Authentication Code",
   "Authentication Failure",
   "Unacceptable Hold Time"
};

char *s_bgp_error_update[] = {
   "Unused",
   "Malformed Attribute List",
   "Unrecognized Well-known Attribute",
   "Missing Well-known Attribute",
   "Attribute Flags Error",
   "Attribute Length Error",
   "Invalid ORIGIN Attribute",
   "AS Routing Loop",
   "Invalid NEXT_HOP Attribute",
   "Optional Attribute Error",
   "Invalid Network Field",
   "Malformed AS_PATH"
};


/* bgp_notify_to_string
 * return human readable string of BGP notify error
 */
char *bgp_notify_to_string (u_char code, u_char subcode)
{
   static char tmp[MAXLINE];

   if (subcode > 0)
	 switch (code) {
	 case BGP_ERR_HEADER:
	    sprintf(tmp, "%s (%s)",  s_bgp_error[code], 
		   s_bgp_error_header[subcode]);
	    break;
	 case BGP_ERR_OPEN:
	    sprintf(tmp, "%s (%s)",  s_bgp_error[code], 
		   s_bgp_error_open[subcode]);
	    break;
	 case BGP_ERR_UPDATE:
	    sprintf(tmp, "%s (%s)",  s_bgp_error[code], 
		   s_bgp_error_update[subcode]);
	    break;
	 default:
	    sprintf(tmp, "%s (No Subcode)",  s_bgp_error[code]);
	 }
   else
      sprintf(tmp, "%s",  s_bgp_error[code]);

   return (tmp);
}



/* bgp_start_transport_connection
 * Starts tcp connection to peer. Returns 1 on sucess, -1 oltherwise
 */
int bgp_start_transport_connection (bgp_peer_t *peer)
{
   int optval = 1;
   int ret;

   /* okay, start a TCP connection */
   if ((peer->sockfd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
      trace (FATAL, peer->trace, "socket open failed (%s:%d)\n",
             __FILE__, __LINE__);

   /* transport connection succeeded */
   setsockopt (peer->sockfd, SOL_SOCKET, SO_REUSEADDR, 
		   (char *) &optval, sizeof (optval));

   if ((ret = connect (peer->sockfd, (struct sockaddr *) &peer->peer_addr,
                sizeof (struct sockaddr))) < 0) {
      return (-1);
   }


   return (1);
}


/* bgp_accept_connection
 *
 */
int bgp_accept_connection (int fd)
{
   int sockfd;
   int len;
   struct sockaddr_in addr;
   bgp_peer_t *peer;

   len = sizeof (addr);

   if ((sockfd = accept (fd, (struct sockaddr *) &addr, &len)) < 0) {
      trace (NORM, BGP->trace, "BGP accept failed\n");
      return (-1);
   }

   select_enable_fd (fd);

   /* scan through peers */
   LL_Iterate (BGP->ll_bgp_peers, peer) {
      if (!memcmp (&peer->peer_addr.sin_addr, &addr.sin_addr, 4)) {

	 trace (NORM, BGP->trace,
		"BGP Valid connection from %s\n", inet_ntoa (addr.sin_addr));
	 
	 if (peer->state == BGPSTATE_OPENCONFIRM) {
	    printf ("\nCOLLISION");
	    /*select_delete_fd (peer->sockfd);*/
	    /*bgp_sm_process_event (peer, peer->state, BGPEVENT_OPEN);*/
	    exit (0);
	 }
	 if ((peer->state = BGPSTATE_IDLE))
	    peer->state = BGPSTATE_ACTIVE;
	    
	 break;
      }
   }

   if (peer == NULL) {
     trace (NORM, BGP->trace, "BGP ERROR -- Unconfigured peer %s\n",
	     inet_ntoa (addr.sin_addr));
     bgp_send_notification (NULL, sockfd, &addr, 
			    BGP_ERR_OPEN, BGP_ERROPN_AUTH);
      return (-1);
   }

   if (peer->sockfd > 0)
      select_delete_fd (peer->sockfd);

   peer->sockfd = sockfd;

   /* schedule it !*/
   /* FIX ME */
   /*bgp_sm_process_event (peer, peer->state, BGPEVENT_OPEN);*/

   schedule_event (peer->schedule, (void *) bgp_sm_process_event,
		   3, peer, peer->state, BGPEVENT_OPEN);
   return (1);
}



/*-----------------------------------------------------------
 *  Name: 	bgp_release_resources
 *  Created:	Fri Dec 23 14:16:27 1994
 *  Author: 	Craig Labovitz   <labovit@snoopy.merit.net>
 *  DESCR:  	
 */

void bgp_release_resources (bgp_peer_t *peer) {

   select_delete_fd (peer->sockfd);
   close (peer->sockfd);
   
   Timer_Turn_OFF (peer->timer_ConnectRetry);
   Timer_Turn_OFF (peer->timer_KeepAlive);
   Timer_Turn_OFF (peer->timer_HoldTime);

   peer->read_ptr = peer->buffer;
   peer->start_ptr = peer->buffer;
   peer->room_in_buffer = BGPMAXPACKETSIZE;

   /* should flush input buffer here */
   /* delete sockfd stuff */
   clear_schedule (peer->schedule);
   /*trace (NORM, BGP->trace,
	  "Clearing scehdule for %s", prefix_toa (peer->gateway->prefix));*/

}


u_int aspath_lookup_fn (aspath_t *tmp1, aspath_t *tmp2) {
  u_int tmp;

  if ((tmp1 == NULL) || (tmp2 == NULL)) {return (-1);}

  tmp = compare_aspaths (tmp1, tmp2);
  return (tmp);
}

/* init_BP
 */
int init_BGP (trace_t* ltrace)
{
   bgp_attr_t attr;

   BGP = (bgp_t *) New (bgp_t);
   BGP->ll_bgp_peers = (LINKED_LIST *) LL_Create (0);

   BGP->lport = BGP_PORT;
   BGP->cport = BGP_PORT;

   BGP->accept_all_peers = 0;
   BGP->my_as = DEFAULT_MY_AS;
   memcpy (&BGP->my_id, 
      prefix_tochar (INTERFACE_MASTER->default_interface->primary->prefix), 4);
   BGP->trace = trace_copy (ltrace);
   BGP->view = BGP->views[0] = New_View (32);
   BGP->view->trace = ltrace; /* XXX why doesn't need trace_copy or ? */

   BGP->attr_hash = 
     HASH_Create (100, HASH_KeyOffset, HASH_Offset(&attr, &attr.aspath),
			    HASH_LookupFunction, aspath_lookup_fn,
			    HASH_HashFunction, aspath_hash_fn,
			    NULL);
   trace (NORM, BGP->trace, "BGP my AS %d ID %08x\n",
      BGP->my_as, BGP->my_id);
   
#if 0
   /* moved to mrt_init() */
   init_timer ();
   init_select ();
#endif

   return (1);
}


/* init_BGP_listen
 * Start BGP listening on port (usually 179)
 */
int init_BGP_listen () {
   struct sockaddr_in serv_addr;
   int optval = 1;

   /* check that NOT listening to portmaster! */
   serv_addr.sin_port = htons (BGP->lport);
   serv_addr.sin_family = AF_INET;
   serv_addr.sin_addr.s_addr = INADDR_ANY;

   if ( (BGP->sockfd = socket (AF_INET, SOCK_STREAM, 0)) < 0) {
      trace (NORM, BGP->trace, "ERROR -- Could not get socket (%s)\n",
	     strerror (errno));
      return (-1);
   }

   /* allow the reuse this port */
   if (setsockopt (BGP->sockfd, SOL_SOCKET, SO_REUSEADDR, 
		   (char *) &optval, sizeof (optval)) < 0) 
     trace (NORM, BGP->trace, "ERROR -- BGP setsockopt failed (%s)\n",
	    strerror (errno));

   if (bind (BGP->sockfd, 
	     (struct sockaddr *) &serv_addr, sizeof (serv_addr)) < 0) {
      trace (NORM, BGP->trace, "BGP ERROR -- Could not bind to port %d (%s)\n",
	     BGP->lport, strerror (errno));
      return (-1);
   }
   
   listen (BGP->sockfd, 5);

   trace (NORM, BGP->trace, "BGP listening for connections on port %d\n", 
	  BGP->lport);

   select_add_fd (BGP->sockfd, 1, (void *) bgp_accept_connection, 
		  (void *) BGP->sockfd);

   return (1);
}



/* Add_BGP_Peer
 * Initialize BGP peer and add to list of peers
 */
bgp_peer_t *Add_BGP_Peer (prefix_t *prefix, short AS, 
			  trace_t *trace) {
   char buf[MAXLINE], str[MAXLINE];
   bgp_peer_t *tmp = New (bgp_peer_t);

   tmp->trace = trace_copy (trace);
   tmp->view = 0;
   tmp->bit = 2 << (++BGP->num_peers);

   tmp->gateway = (gateway_t *) add_gateway (prefix, AS, NULL);

   memset ((char *) &tmp->peer_addr, 0, sizeof (struct sockaddr_in));
   tmp->peer_addr.sin_family 	= 	AF_INET;
   memcpy (&tmp->peer_addr.sin_addr, (char *) prefix_tochar (prefix), 
	   sizeof (struct in_addr));
   tmp->peer_addr.sin_port = htons (BGP->cport); /* usually 179 */

   tmp->read_ptr = tmp->buffer;
   tmp->room_in_buffer = BGPMAXPACKETSIZE;

   tmp->state = BGPSTATE_IDLE;
   
   tmp->Start_Interval = DEFAULT_START_INTERVAL;
   tmp->ConnetRetry_Interval = DEFAULT_CONNET_RETRY_INTERVAL;
   tmp->KeepAlive_Interval = DEFAULT_KEEPALIVE_INTERVAL;
   tmp->HoldTime_Interval = DEFAULT_HOLDTIME_INTERVAL; 

   /* create timers */
   tmp->timer_ConnectRetry  = New_Timer (bgp_schedule_timer,
					 tmp->ConnetRetry_Interval,
					 "ConnectRetry", tmp);
   tmp->timer_KeepAlive     = New_Timer (bgp_schedule_timer,
					 tmp->KeepAlive_Interval,
					 "KeepALive", tmp);
   tmp->timer_HoldTime      = New_Timer (bgp_schedule_timer,
					 tmp->HoldTime_Interval,
					 "HoldTime", tmp);
   tmp->timer_Start	    = New_Timer (bgp_schedule_timer,
					 tmp->Start_Interval,
					 "StartTime", tmp);
   LL_Add (BGP->ll_bgp_peers, tmp);

   tmp->schedule = (schedule_t *) 
     New_Schedule (gateway_toa (buf, tmp->gateway), tmp->trace);

   sprintf (str, "BGP %-15s", prefix_toa (tmp->gateway->prefix));
   mrt_thread_create (str, tmp->schedule, 
		      (void *)  bgp_start_peer_thread, tmp);

 
   return (tmp);
}



/*-----------------------------------------------------------
 *  Name: 	bgp_flush_socket
 *  Created:	Fri Jan 13 00:01:29 1995
 *  Author: 	Craig Labovitz   <labovit@snoopy.merit.net>
 *  DESCR:  	get any error messages waiting before tearing down
 *		connectionque
 */

void bgp_flush_socket (bgp_peer_t *peer)
{
   u_char *tmp;
   int type;
   int more_flag;

   while ((tmp = get_packet (peer, &more_flag)) != NULL) {
      printf("\nsomething");

      BGP_GET_HDRTYPE(type, tmp);
      if (type == BGP_NOTIFY) {
	 bgp_process_notify (peer, tmp);
      }
   }
   close (peer->sockfd);
}


/* bgp_peer_dead
 * peer has left established state, withdraw routes and
 * free resources
 */
void bgp_peer_dead (bgp_peer_t *peer) {
   char tmp[MAXLINE];

   trace (NORM, peer->trace,"\n ");
   trace (NORM, peer->trace,
	  "BGP Peer is now %s is dead -- closing connection\n", 
	  gateway_toa (tmp, peer->gateway));

   /*bgp_change_state (peer, BGPSTATE_IDLE, BGPEVENT_ERROR);   */
   /*view_delete_peer (BGP->view);*/
   bgp_release_resources (peer);

   if (BGP->peer_down_call_fn)
      BGP->peer_down_call_fn (peer);
   else
      default_bgp_peer_down_fd (peer);
}



/*-----------------------------------------------------------
 *  Name: 	bgp_die
 *  Created:	Tue Jan 24 09:48:34 1995
 *  Author: 	Craig Labovitz   <labovit@snoopy.merit.net>
 *  DESCR:  	Cleanly die on Control-C's and other aborts
 */

void bgp_die ()
{
   bgp_peer_t *peer; 

   exit (0);

   LL_Iterate (BGP->ll_bgp_peers, peer) 
      close (peer->sockfd);

   shutdown (BGP->sockfd, 2);
   close (BGP->sockfd);

   trace (NORM, BGP->trace,
	  "BGP terminated by user\n");
   exit (0);
}




/* set_BGP
 */
int set_BGP(int first, ...) 
{
   va_list    		ap;
   enum BGP_ATTR	attr;

   /* Process the Arguments */
   va_start(ap, first);
   for (attr = (enum BGP_ATTR)first; attr; attr = va_arg(ap, enum BGP_ATTR)) {
      switch (attr) {
      case BGP_MY_AS:
	 BGP->my_as = va_arg(ap, int);
	 break;
      case BGP_MY_ID:
	 BGP->my_id = va_arg(ap, int);
	 break;
      case BGP_PEER_DOWN_FN:
	 BGP->peer_down_call_fn = va_arg (ap, void*);
	 break;
      case BGP_PEER_ESTABLISHED_FN:
	 BGP->peer_established_call_fn = va_arg (ap, void*);
	 break;
      case BGP_RECV_UPDATE_FN:
	 BGP->update_call_fn = va_arg (ap, void*);
	 break;
      case BGP_ACCEPT_ALL_PEERS:
	 BGP->accept_all_peers = va_arg (ap, int);
	 break;
      case BGP_LPORT:
	 BGP->lport = va_arg (ap, int);
	 break;
      case BGP_CPORT:
	 BGP->cport = va_arg (ap, int);
	 break;
      case BGP_TRACE_STRUCT:
	 BGP->trace = va_arg (ap, trace_t *);
	 break;
      default:
	 trace (NORM, BGP->trace, "ERROR -- BGP unknown attribute set\n");
	 break;
      }
   }
   va_end(ap);

   return (1);
}


int default_bgp_peer_down_fd (bgp_peer_t *peer) {
   Timer_Turn_ON (peer->timer_Start);

   return (1);
}


/* find_bgp_peer
 * given a prefix, find the corresponding bgp peer
 * structure.
 */
bgp_peer_t *find_bgp_peer (gateway_t *gateway) {
   bgp_peer_t *tmp;
   
   LL_Iterate (BGP->ll_bgp_peers, tmp) {
     if (tmp->gateway == gateway)
	 return (tmp);
   }

   return (NULL);
}





/* start_bgp 
 * put all peers into start state, and begin listening on BGP
 * port (usually 179)
 */
int start_bgp () {
  bgp_peer_t *peer;

  LL_Iterate (BGP->ll_bgp_peers, peer) {
    schedule_event (peer->schedule, (void *) bgp_sm_process_event,
		    3, peer, peer->state, BGPEVENT_START);
  }

  init_BGP_listen ();

  return (1);
}


/* show_bgp_routing table
 * dump BGP routes to socket. usually called by UII
 */
int show_bgp_routing_table (uii_connection_t *uii)
{
  bgp_dump_view (uii, 0);
  return (1);
}


/* show_bgp
 * dump various BGP stats to a socket
 * usually called by UII (user interactive interface 
 */
int show_bgp (uii_connection_t * uii) {
  bgp_peer_t *peer;
  char tmp[MAXLINE];

  uii_send_data (uii, "Routing Protocol is \"BGP\"\r\n");
  uii_send_data (uii, "Local Router ID is %d\r\n", BGP->my_as);
  if (BGP->trace != NULL)
    uii_send_data (uii, "Trace flags 0x%x\r\n", BGP->trace->flags);
  uii_send_data (uii, "%d Attribute Blocks\r\n", HASH_GetCount (BGP->attr_hash));
  LL_Iterate (BGP->ll_bgp_peers, peer) {
    uii_send_data (uii, "\r\n  peer %s [%s] (BIT 0x%x) View %d",
		   gateway_toa (tmp, peer->gateway), 
		   sbgp_states[peer->state],
		   peer->bit, peer->view);
    if (peer->timer_KeepAlive->on != 1) 
      uii_send_data (uii, "\r\n    KeepAlive Off");
    else
      uii_send_data (uii, "\r\n    KeepAlive %d", 
		     peer->timer_KeepAlive->time_next_fire - time(NULL));
    if (peer->timer_Start->on != 1)
            uii_send_data (uii, "  Startmtimer_t Off");
    else 
      uii_send_data (uii, "  Startmtimer_t %d", 
		     peer->timer_Start->time_next_fire - time(NULL));
    if (peer->timer_HoldTime->on != 1)
      uii_send_data (uii, "  Holdmtimer_t Off\r\n");
    else 
      uii_send_data (uii, "  Holdmtimer_t %d\r\n", 
		     peer->timer_HoldTime->time_next_fire - time(NULL));
    
  }

  return (1);
}

/* bgp_debug
 * set the debug flags or file for BGP
 * called by config routines
 */
int bgp_debug (uii_connection_t *uii)
{
  char *token, *line;
  u_long flag = 0;

  line = uii->cp;

  /* get flag */
  if ((token = uii_parse_line (&line)) == NULL) {
      uii_send_data (uii, "Error:  debup trace_flag [file]\r\n");
      return (-1);
  }
  flag = trace_flag (token);
  set_trace (BGP->trace, TRACE_FLAGS, flag, NULL);

  /* get file name (OPTIONAL) */
  if ((token = (char *) uii_parse_line (&line)) == NULL) {return 1;}
  set_trace (BGP->trace, TRACE_LOGFILE, token, NULL);

  return (1);
}




/* bgp_dump_view
 * dump BGP routes to socket. usually called by UII
 */
int bgp_dump_view (uii_connection_t *uii, int view)
{
  bgp_route_head_t *rt_head;
  bgp_route_t *route;
  char *s_origin[] = {"i", "e", "?"};

  view_open (BGP->views[view]);

  uii_send_data (uii, "Status code: s suppressed, * valid, > best, i - internal\r\n");
  uii_send_data (uii, "Origin codes: i - IGP, e - EGP, ? - incomplete\r\n");
  uii_send_data (uii, "\r\n   %-17s%-17s Path\r\n",
		 "Network", "NextHop");


  HASH_Iterate (BGP->views[view]->hash, rt_head) {

    if (rt_head->state_bits & VRTS_DELETE) {continue;}

    LL_Iterate (rt_head->ll_routes, route) {

      uii_send_data (uii, "*");

      if (route == rt_head->active) 
	uii_send_data (uii, "> ");
      else
	uii_send_data (uii, "  ");

      uii_send_data (uii, "%-17s", prefix_toa (&rt_head->prefix));

      if (route->attr->nexthop != NULL)
	uii_send_data (uii, "%-17s", 
		       prefix_toa (route->attr->nexthop));
      else
	uii_send_data (uii, "%-17s", "0.0.0.0");

      if (route->attr->aspath != NULL) 
	uii_send_data (uii, "%s %s\r\n", 
		       aspath_toa (route->attr->aspath),
		       s_origin[route->attr->origin]);
      else
	uii_send_data (uii, "%s\r\n", s_origin[route->attr->origin]);


    }
  }

  view_close (BGP->views[view]);

  return (1);
}


/* bgp_get_attr
 * attribute are stored in global hash. This 
 * 1) fetches attr if exists, and deleted passed in memory
 * 2) or creates new attr using memory passed in 
 */
bgp_attr_t *bgp_get_attr (bgp_attr_t *attr) {
  bgp_attr_t *tmp;

  tmp = HASH_Lookup (BGP->attr_hash, attr);

  if (tmp == NULL) {
    HASH_Insert (BGP->attr_hash, attr);
    tmp = attr;
  }
  else {
    bgp_delete_attr (attr);
  }

  return (tmp);
}
