/*
 *  $Id: bgp_sm.c,v 1.3 1996/06/27 15:43:39 labovit Exp $
 */

#include <version.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <netdb.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <mrt.h>
#include <bgp_proto.h>
#include <timer.h>
#include <trace.h>
#include <select.h>
#include <bgp.h>


/* bgp_change_state
 * A convenience routine that does tracing on state
 * transition.
 */
void bgp_change_state (bgp_peer_t *peer, int state, int event) {
   char tmp[MAXLINE];

   trace (TR_STATE, peer->trace, "BGP <%s> %s -> %s (event %s)\n",
	  gateway_toa (tmp, peer->gateway), sbgp_states[peer->state], 
	  sbgp_states[state], sbgp_events[event]);
   peer->state = state;

   if ((state == BGPSTATE_ESTABLISHED) && (BGP->peer_established_call_fn))
      BGP->peer_established_call_fn (peer);   
}


void bgp_sm_state_idle (bgp_peer_t *peer, int event) {
   int ret;
   char tmp[MAXLINE];

   /* ignore all events but START event */
   if (event != BGPEVENT_START) {
      trace (NORM, peer->trace, "BGP <%s> Idle -> Idle (event %s IGNORED)\n",
	     gateway_toa (tmp, peer->gateway), sbgp_events[event]);
      trace (NORM, peer->trace, "\n");
      return;
   }


   Timer_Turn_ON (peer->timer_ConnectRetry);

   trace (TR_STATE | NORM, peer->trace, "\n");
   trace (TR_STATE | NORM, peer->trace,
	  "BGP Attemtping connection to %s\n",
	  gateway_toa (tmp, peer->gateway));
   trace (TR_STATE | NORM, peer->trace, "\n");

   ret = bgp_start_transport_connection(peer);

   bgp_change_state (peer, BGPSTATE_CONNECT, BGPEVENT_START);

   if (ret < 0) {
      trace (NORM, peer->trace,
             "BGP connection to %s failed (%s)\n",
             gateway_toa (tmp, peer->gateway), strerror (errno));
      trace (TR_STATE | NORM, peer->trace, "\n");
      bgp_sm_process_event (peer, peer->state, BGPEVENT_OPENFAIL);
   }
   else {
      bgp_sm_process_event (peer, peer->state, BGPEVENT_OPEN);
   }
}



void bgp_sm_state_active (bgp_peer_t *peer, int event) {
   int ret;
   char tmp[MAXLINE];

   switch (event) {
   case BGPEVENT_START:
      return; /* do nothing */
      break;
   case BGPEVENT_OPEN:
      Timer_Turn_OFF (peer->timer_ConnectRetry);
      select_add_fd (peer->sockfd, 1, bgp_schedule_socket, peer);
      bgp_send_open (peer);
      bgp_change_state(peer, BGPSTATE_OPENSENT, BGPEVENT_OPEN);
      break;
   case BGPEVENT_OPENFAIL:
      Timer_Reset_Time (peer->timer_ConnectRetry);
      break;
   case BGPEVENT_CONNRETRY:
      Timer_Reset_Time (peer->timer_ConnectRetry);
      ret = bgp_start_transport_connection(peer);
      if (ret < 0) {
	 trace (NORM, peer->trace,
		"Transport connection to %s failed (%s)\n",
		gateway_toa (tmp, peer->gateway), strerror (errno));
	 bgp_change_state(peer, BGPSTATE_OPENSENT, BGPEVENT_OPEN);
	 bgp_sm_process_event (peer, peer->state, BGPEVENT_OPENFAIL);
      }
      else {
	select_add_fd (peer->sockfd, 1, (void *) bgp_get_pdu, peer);
	 bgp_sm_process_event (peer, peer->state, BGPEVENT_OPEN);
      }
      break;
   default:
      bgp_change_state (peer, BGPSTATE_IDLE, event);
      bgp_peer_dead (peer);
      break;
   } 
}

void bgp_sm_state_connect (bgp_peer_t *peer, int event) {
   int ret;
   char tmp[MAXLINE];

   switch (event) {
   case BGPEVENT_START:
      return; /* do nothing */
      break;
   case BGPEVENT_OPEN:
     Timer_Turn_OFF (peer->timer_ConnectRetry);
      select_add_fd (peer->sockfd, 1, bgp_schedule_socket, peer);
      bgp_send_open (peer);
      bgp_change_state(peer, BGPSTATE_OPENSENT, BGPEVENT_OPEN);
      break;
   case BGPEVENT_OPENFAIL:
      Timer_Reset_Time (peer->timer_ConnectRetry);
      bgp_change_state(peer, BGPSTATE_ACTIVE, BGPEVENT_OPENFAIL);
      /*trace (NORM, peer->trace, "LOOPING\n");
      while (1) {sleep (10000); } FIX ME */
      break;
   case BGPEVENT_CONNRETRY:
      Timer_Reset_Time (peer->timer_ConnectRetry);
      ret = bgp_start_transport_connection(peer);
      if (ret < 0) {
	 trace (NORM, peer->trace,
		"Transport connection to %s failed (%s)\n",
		gateway_toa (tmp, peer->gateway), strerror (errno));
	 bgp_sm_process_event (peer, peer->state, BGPEVENT_OPENFAIL);
      }
      else {
	select_add_fd (peer->sockfd, 1, (void *) bgp_get_pdu, peer);
	 bgp_sm_process_event (peer, peer->state, BGPEVENT_OPEN);
      }
      break;
   default:
      bgp_change_state (peer, BGPSTATE_IDLE, event);
      bgp_peer_dead (peer);
      bgp_release_resources (peer);
      break;
   }
}


void bgp_sm_state_opensent (bgp_peer_t *peer, int event) {
   
   switch (event) {
   case BGPEVENT_START:
      return; /* do nothing */
      break;
   case BGPEVENT_CLOSED:
      /* close transport connection */
      Timer_Reset_Time (peer->timer_ConnectRetry);
      bgp_change_state (peer, BGPSTATE_ACTIVE, BGPEVENT_CLOSED);
      break;
   case BGPEVENT_ERROR:
      bgp_change_state (peer, BGPSTATE_IDLE, BGPEVENT_ERROR);
      bgp_peer_dead (peer);
      bgp_release_resources (peer);
      break;
   case BGPEVENT_RECVOPEN:
      /* if OPEN is good */
      if (bgp_process_open(peer, peer->packet) > 0) {
	 bgp_send_keepalive(peer);
	 bgp_change_state(peer, BGPSTATE_OPENCONFIRM, BGPEVENT_RECVOPEN);
      }
      /* OPEN was bad */
      else {
	 /* notification already sent */
	 bgp_change_state (peer, BGPSTATE_IDLE, event);
	 bgp_peer_dead (peer);
      }
      break;
   default:
      /* send notification */
      /* close transport */
      bgp_change_state (peer, BGPSTATE_IDLE, 0);
      bgp_peer_dead (peer);
   }
}


void bgp_sm_state_openconfirm (bgp_peer_t *peer, int event) {
   
   switch (event) {
   case BGPEVENT_START:
      return; /* do nothing */
      break;
   case BGPEVENT_CLOSED:
   case BGPEVENT_OPENFAIL:
      bgp_release_resources (peer);
      bgp_change_state (peer, BGPSTATE_IDLE, BGPEVENT_OPENFAIL);
      break;
   case BGPEVENT_KEEPALIVE:
      Timer_Reset_Time(peer->timer_KeepAlive);
      bgp_send_keepalive(peer);
      break;
   case BGPEVENT_RECVKEEPALIVE:
      Timer_Reset_Time(peer->timer_HoldTime);
      bgp_change_state (peer, BGPSTATE_ESTABLISHED, BGPEVENT_RECVKEEPALIVE);
      bgp_establish_peer (BGP->view, peer);
      break;
   case BGPEVENT_RECVNOTIFY:
      /* close transport connection */
      bgp_change_state (peer, BGPSTATE_IDLE, BGPEVENT_RECVNOTIFY);
      bgp_peer_dead (peer);
      break;
   default:
      /* send notification */
      bgp_change_state (peer, BGPSTATE_IDLE, event);
      break;
   }
}



void bgp_sm_state_established (bgp_peer_t *peer, int event) {
   
   switch (event) {
   case BGPEVENT_START:
      return; /* do nothing */
      break;
   case BGPEVENT_CLOSED:
   case BGPEVENT_ERROR:
      bgp_change_state (peer, BGPSTATE_IDLE, BGPEVENT_ERROR);
      view_delete_peer (BGP->view, peer->gateway);
      bgp_peer_dead (peer);
      break;
   case BGPEVENT_KEEPALIVE:
      Timer_Reset_Time(peer->timer_KeepAlive);
      bgp_send_keepalive(peer);
      break;
   case BGPEVENT_RECVKEEPALIVE:
      Timer_Reset_Time(peer->timer_HoldTime);
      break;
   case BGPEVENT_RECVUPDATE:
      Timer_Reset_Time(peer->timer_HoldTime);
      bgp_process_update (peer, peer->packet);
      /* check update PDU */
      break;
   case BGPEVENT_RECVNOTIFY:
      bgp_change_state (peer, BGPSTATE_IDLE, BGPEVENT_RECVNOTIFY);
      bgp_peer_dead (peer);
      break;
   default:
      /* send notification */
      bgp_change_state (peer, BGPSTATE_IDLE, event);
      bgp_peer_dead (peer);
      break;
   }
}


void bgp_sm_process_event (bgp_peer_t *peer, int state, int event)
{

   /*trace (TR_STATE, NULL, "TR_STATE In state %s event %s",
	  sbgp_states[state], sbgp_events[event]);*/

   switch (state) {
   case BGPSTATE_IDLE:
      bgp_sm_state_idle (peer, event);
      break;
   case BGPSTATE_ACTIVE:
      bgp_sm_state_active (peer, event);
      break;
   case BGPSTATE_CONNECT:
      bgp_sm_state_connect (peer, event);
      break;
   case BGPSTATE_OPENSENT:
      bgp_sm_state_opensent (peer, event);
      break;
   case BGPSTATE_OPENCONFIRM:
      bgp_sm_state_openconfirm (peer, event);
      break;
   case BGPSTATE_ESTABLISHED:
      bgp_sm_state_established (peer, event);
      break;
   default:
      trace(FATAL, BGP->trace, "UNKNOWN STATE %d (%s:%d)\n", peer->state,
	    __FILE__, __LINE__);
      break;
   }
}

