/*
 * $Id: trace.c,v 1.4 1996/12/23 14:37:54 masaki Exp $
 */


#include <stdio.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <unistd.h>
#include <syslog.h>
#include <string.h>
#include <mrt.h>
#include <trace.h>


/* internal routines */
static FILE *get_trace_fd (trace_t *trace_struct);
static char *_my_strftime(char *tmp, long in_time, char *fmt);

static int syslog_notify = 0;

int
init_trace (name, syslog_flag)
    const char *name;
    int syslog_flag;
{
    if (syslog_flag) {
	openlog (name, LOG_PID, LOG_DAEMON);
	syslog_notify = 1;
    }
}

/* trace
 */
int 
trace (int flag, trace_t *tr, ...)
{
   va_list args;
   char *format;
   char *ptime;
   int ret;
   char tmp[MAXLINE];

   if (tr == NULL) {
     return (0);
   }

   /* check that trace is not open -- blocks until it can get lock */
   if (tr->thread_id != pthread_self ())
       pthread_mutex_lock (&tr->mutex_lock);
   
   va_start (args, tr);
   format = va_arg(args, char *);

   if (syslog_notify && (BIT_TEST (flag, INFO|ERROR|FATAL))) {
     vsprintf (tmp, format, args);
     syslog (LOG_INFO, tmp);
   }

   /* nope, we don't log this message */
   if ((!BIT_TEST (tr->flags, flag)) || (tr->logfd == NULL)) {
     /* unlock if we are not locking it in trace_open */
     if (tr->thread_id != pthread_self ())
       pthread_mutex_unlock (&tr->mutex_lock);
     if (BIT_TEST (flag, FATAL))
       mrt_exit (1);
     return (0);
   }

   ptime = (char *) _my_strftime(tmp, 0, "%h %e %T");
   ret = fprintf(tr->logfd, "%s [%d] ", ptime,pthread_self ());
   vfprintf(tr->logfd, format, args);
   fflush (tr->logfd);

   /* unlock if we are not locking this in trace_open */
   if (tr->thread_id != pthread_self ())
      pthread_mutex_unlock (&tr->mutex_lock);

   /* check if errors */
   if (ret < 0) {
     perror ("fprintf failed:");
     mrt_exit (1);
   }
   if (BIT_TEST (flag, FATAL))
     mrt_exit (1);
   return (1);
}

/* New_Trace
 */ 
trace_t *New_Trace (void)
{
   trace_t		*tmp;

   tmp = New (trace_t);

   tmp->logfile_name = TR_DEFAULT_LOGFILE;
   tmp->flags = TR_DEFAULT_FLAGS;


   tmp->thread_id = -1;
   pthread_mutex_init (&tmp->mutex_lock, NULL);

   return (tmp);
}

/* trace_copy
 */
trace_t* trace_copy (trace_t *old) {
  trace_t *tmp;
  
  tmp = New (trace_t);

  if (old == NULL) 
    return (NULL);

  memcpy (&tmp->mutex_lock, &old->mutex_lock, sizeof (pthread_mutex_t));
  tmp->logfile_name = strdup (old->logfile_name);
  tmp->logfd = old->logfd;
  tmp->flags = old->flags;
  return (tmp);
}



/* get_trace_fd
 */
static
FILE *get_trace_fd (trace_t *tr)
{
   char *type;

   if (!tr)
      return (stdout);

   if (!strcasecmp (tr->logfile_name, "stdout")) {
     tr->logfd = (FILE *) stdout;
     return (tr->logfd);
   }

   if (tr->logfile_name) {
      if (tr->append_flag)
	 type = "w+";
      else
	 type = "w";
      if ((tr->logfd = fopen (tr->logfile_name, type))) {
	 return (tr->logfd);
      }
   }

   tr->logfd = NULL;
   return (NULL);
}



/* my_strftime
 * Given a time long and format, return string. 
 * If time <=0, use current time of day
 */
static 
char *_my_strftime(char *tmp, long in_time, char *fmt)
{
   time_t t;
   struct tm tms;

   if (in_time <= 0)
      t = time (NULL);
   else
      t = in_time;

#ifdef HAVE_LOCALTIME_R
   localtime_r(&t, &tms);
   strftime (tmp, MAXLINE, fmt, &tms);
#else
   strftime (tmp, MAXLINE, fmt, localtime(&t));
#endif /* HAVE_LOCALTIME_R */

   return (tmp);
}

/* set_trace
 */
int 
set_trace (trace_t *tmp, int first, ...)
{
   va_list    		ap;
   enum Trace_Attr	attr;

   if (tmp == NULL) return (-1);

   pthread_mutex_lock (&tmp->mutex_lock);

   /* Process the Arguments */
   va_start(ap, first);
   for (attr = (enum Trace_Attr)first; attr; 
	attr = va_arg(ap, enum Trace_Attr)) {
      switch (attr) {
      case TRACE_LOGFILE:
	 tmp->logfile_name = strdup (va_arg(ap, char *));
	 tmp->logfd = get_trace_fd (tmp);
	 break;
      case TRACE_FLAGS:
	 tmp->flags |= va_arg(ap, long);
	 break;
      default:
	 break;
      }
   }
   va_end(ap);

   pthread_mutex_unlock (&tmp->mutex_lock);
   return (1);
}

/* trace_generate_file_nam
 * what is this used for?
 */
char *trace_generate_file_name (char *name) {
   char *tmp;
   int size;

   size = strlen (name) + 10;
   tmp = (char *) NewArray (char, size);

   sprintf (tmp, "%s.%d", name, (int) getpid());

   return (tmp);
}



/* trace_open
 * open a trace for multi-line traces. Obtains
 * mutex lock and marks trace structure with 
 * thread_id
 */
int 
trace_open (trace_t *tr) {

  if (tr == NULL)
    return (-1);


   pthread_mutex_lock (&tr->mutex_lock);
   tr->thread_id = pthread_self ();


   return (1);
}


/* trace_close
 */
int 
trace_close (trace_t *tr) {

  if (tr == NULL) {return (-1);}

  if (pthread_self () != tr->thread_id) {
    printf ("REALLY BOGUS -- my ID did not open this trace\n");
    exit (0);
  }

  tr->thread_id = -1;
  pthread_mutex_unlock (&tr->mutex_lock);

  return (1);
}


/* trace_flag
 * convert ascii description of trace flag to flag bit
 */
u_int trace_flag (char *str) {
  
  if (!strcasecmp (str, "trace")) return (TRACE);
  if (!strcasecmp (str, "norm")) return (NORM);
  if (!strcasecmp (str, "parse")) return (TR_PARSE);
  if (!strcasecmp (str, "packet")) return (TR_PACKET);
  if (!strcasecmp (str, "state")) return (TR_STATE);
  if (!strcasecmp (str, "timer")) return (TR_TIMER);
  if (!strcasecmp (str, "all")) return (TR_ALL);

  return (0);
}
