/*
 * $Id: timer.c,v 1.13 1997/02/18 23:17:29 masaki Exp $
 */

#include <stdio.h>
#include <signal.h>
#include <errno.h>
#include <sys/time.h>
#include <stdarg.h>
#include <mrt.h>
#include <timer.h>
#include <version.h>


void timer_lock ();
void timer_unlock ();
static int timer_jitter (mtimer_t *timer);

/* #define DEBUG_TIMER	1 */
 
Timer_Master *TIMER_MASTER;

/* Timer_Compare
 * Compare two timers for use in sorting. A timer that is 
 * ON always comes before OFF timers
 */
int Timer_Compare (mtimer_t *t1, mtimer_t *t2) {

#ifdef DEBUG_TIMER
   printf("SORTING: Comparing %s:%d/%d and %s:%d/%d\n",
	  t1->name, t1->time_next_fire, t1->on,
	  t2->name, t2->time_next_fire, t2->on);
#endif
   
   if ((t1->on == 0) && (t2->on != 0))
      return (1);
   if ((t2->on == 0) && (t1->on != 0))
      return (-1);

   return (t1->time_next_fire - t2->time_next_fire);
}

 
sigset_t
block_signal (int sig)
{
   sigset_t new, old;

#ifdef HAVE_LIBPTHREAD
   sigemptyset(&new);
   sigaddset(&new, sig);
   pthread_sigmask(SIG_BLOCK, &new, &old);
#else
#ifdef HAVE_SIGSET
   sighold(sig);
   old = sig;
   /* in this case, old has a signal number being held. */
#else
   new = sigmask(sig);
   old = sigblock(new);
#endif
#endif
   return(old);
}


void
recover_signal(sigset_t old)
{
#ifdef HAVE_LIBPTHREAD
   pthread_sigmask(SIG_SETMASK, &old, NULL);
#else
#ifdef HAVE_SIGSET
   /* unable to recover the old value */
   sigrelse(old);
#else
   sigsetmask(old);
#endif
#endif
}


/* Timer_Master_Fire
 * Multiplex alarm signal to list of timers. Fire timers
 * if next fire time before current time
 */
void Timer_Master_Fire () {
   mtimer_t *p_timer;
   sigset_t old;
   int ret;
   time_t now, t = 0;
   int changed = 0;
   
   LINKED_LIST *ll_tmp;

   old = block_signal (SIGALRM);

#ifdef DEBUG_TIMER
   printf("Master TIMER FIRE!\n");
#endif

   assert (LL_GetCount (TIMER_MASTER->ll_timers));

   ll_tmp = LL_Create (0);

   timer_lock ();

   time (&now);
   LL_Iterate (TIMER_MASTER->ll_timers, p_timer) {

      if ((p_timer->on == 1) &&
	  (p_timer->time_next_fire <= now)) {

	 LL_Add (ll_tmp, p_timer);
	 changed++;

         if (p_timer->flags & TIMER_ONE_SHOT) {
             p_timer->time_next_fire = 0;
   	     p_timer->on = 0;
         }
	 else {
	    p_timer->time_interval = 
	      p_timer->time_interval_base + timer_jitter (p_timer);
	    assert (p_timer->time_interval > 0);
	    p_timer->time_next_fire += p_timer->time_interval;
	}
      }
      else break;
   }

   if (changed)
       LL_Sort (TIMER_MASTER->ll_timers);

   if (((p_timer = LL_GetHead (TIMER_MASTER->ll_timers)) != NULL) &&
       (p_timer->on == 1)) {
#ifdef DEBUG_TIMER
   printf("Head of list = %s:%d\n", p_timer->name, p_timer->time_next_fire);
#endif
      TIMER_MASTER->time_interval = p_timer->time_interval;
      TIMER_MASTER->time_next_fire = p_timer->time_next_fire;
   }
   else {
#ifdef DEBUG_TIMER
      if (p_timer)
         printf("Head of list = %s:%d but OFF\n", p_timer->name, 
		p_timer->time_next_fire);
#endif
      TIMER_MASTER->time_interval = 0;
      TIMER_MASTER->time_next_fire = 0;
   }

   if (TIMER_MASTER->time_next_fire) {
      t = TIMER_MASTER->time_next_fire - time(NULL);
      if (t <= 0) t = 1;
      alarm (t);
   }

   timer_unlock ();

   /* okay, finally schedule timer calls */
   LL_Iterate (ll_tmp, p_timer) {
     
     p_timer->call_fn(p_timer, p_timer->arg);
    
     }
		  
   LL_Destroy (ll_tmp);

#ifdef DEBUG_TIMER
   printf("Master TIMER set to %d\n", t);
#endif

   recover_signal (old);
}


/* init_timer
 * Allocate memory and initialize global TIMER_MASTER variable
 */
void init_timer (trace_t *trace) {

  assert (TIMER_MASTER == NULL);

  TIMER_MASTER = New (Timer_Master);
  
  TIMER_MASTER->ll_timers = LL_Create (LL_CompareFunction, Timer_Compare,
			       LL_AutoSort, True,
			       NULL);
  TIMER_MASTER->time_interval = 0;
  TIMER_MASTER->time_next_fire = 0;
  TIMER_MASTER->trace = trace_copy (trace);

#ifdef HAVE_SIGACTION /* POSIX actually */
  {
       struct sigaction act;

       memset (&act, 0, sizeof (act));
       act.sa_handler = Timer_Master_Fire;
#ifdef SA_RESTART
       act.sa_flags = SA_RESTART; 
#endif
       sigaction (SIGALRM, &act, NULL);
  }
#else
#ifdef HAVE_SIGSET
   sigset (SIGALRM, Timer_Master_Fire);
#else
   signal (SIGALRM, Timer_Master_Fire);
#endif /* HAVE_SIGSET */
#endif /* POSIX signals */

   pthread_mutex_init (&TIMER_MASTER->mutex_lock, NULL);
}


/* Destroy_Timer_Master
 */
void Destroy_Timer_Master (Timer_Master *timer) {
   LL_Destroy (timer->ll_timers);
   Destroy (timer);
}


/* Timer_Turn_OFF
 * Turn a timer off and update the TIMER_MASTER info
 */
void Timer_Turn_OFF (mtimer_t *timer) {
   if (timer == NULL) return;
   
   timer->time_next_fire = 0;
   timer->on = 0;
   
   Timer_Master_Update (timer);
}


/* Timer_Turn_ON
 * Turn a timer ON and update TIMER_MASTER info
 */
void Timer_Turn_ON (mtimer_t *timer) {
   timer->time_interval = timer->time_interval_base + timer_jitter (timer);
   assert (timer->time_interval > 0);
   timer->time_next_fire = time(NULL) + timer->time_interval;
   timer->on = 1;

   Timer_Master_Update (timer);
}


/* Timer_Reset_Time
 * Reset timer to fire in timer->time_interval and 
 * notify TIMER_MASTER of change
 */

void Timer_Reset_Time (mtimer_t *timer) {
   timer->time_interval = timer->time_interval_base + timer_jitter (timer);
   assert (timer->time_interval > 0);
   timer->time_next_fire = time(NULL) + timer->time_interval;
   timer->on = 1;

   if (timer->time_interval <= 0) 
      timer->on = 0;

   Timer_Master_Update (timer);
}


/* Timer_Set_Time
 * Change intreval of timer and notify 
 * TIMER_MASTER of new info.
 */
void Timer_Set_Time (mtimer_t *timer, int interval) {
   timer->time_interval_base = interval;
   timer->time_interval = timer->time_interval_base + timer_jitter (timer);
   assert (timer->time_interval > 0);
   timer->time_next_fire = time(NULL) + timer->time_interval;
   timer->on = 1;

   if (timer->time_interval <= 0) 
      timer->on = 0;

   Timer_Master_Update (timer);
} 

void timer_set_flags (mtimer_t *timer, int flags) {

   timer->flags = flags;
} 


int timer_set_jitter (mtimer_t *timer, int jitter) {
  timer->jitter = jitter;
  assert (jitter >= 0);
}


/* Timer_Master_Update
 * Update global TIMER_MASTER structure with new, 
 * or changed timer. Resort timer list and
 * change next_fire if timer fire before, and
 * set alarm.
 */
void Timer_Master_Update (mtimer_t *p_timer) {
   timer_lock ();

   LL_ReSort (TIMER_MASTER->ll_timers, p_timer);

   if ((p_timer = LL_GetHead (TIMER_MASTER->ll_timers)) == NULL) {
      timer_unlock ();
      return;
   }
   
   if (p_timer->on == 0) {
      if (TIMER_MASTER->time_next_fire) {
         TIMER_MASTER->time_interval = 0;
         TIMER_MASTER->time_next_fire = 0;
         alarm (0);
#ifdef DEBUG_TIME
         printf("Master mtimer_t has been updated OFF\n");
#endif
      }
      timer_unlock ();
      return;
   }

   if ((TIMER_MASTER->time_next_fire == 0) ||
       (TIMER_MASTER->time_next_fire >  p_timer->time_next_fire)) {
      time_t t;
      
      TIMER_MASTER->time_next_fire = p_timer->time_next_fire;
      TIMER_MASTER->time_interval = p_timer->time_interval;

      t = TIMER_MASTER->time_next_fire - time (NULL);
      if (t <= 0) t = 1;
      alarm (t);
#ifdef DEBUG_TIMER
      printf("Master mtimer_t has been updated to %d\n", 
	     TIMER_MASTER->time_interval);
#endif
   }

   timer_unlock ();
}



/* New_Timer
 * Allocate memory and initialize a mtimer_t structure. 
 * Also add new mtimer_t info to global TIMER_MASTER
 * NOTE: timers are created OFF!
 */
mtimer_t *New_Timer (void (*call_fn)(), int interval, char *name, void *arg) {
   mtimer_t *tmp;

   tmp = New (mtimer_t);
   
   tmp->call_fn = call_fn;
   tmp->arg = arg;
   tmp->time_interval_base = interval;
   tmp->time_interval = 0;
   tmp->time_next_fire = 0; /* later calculate */
   tmp->jitter = 0;
   tmp->on = 0;
   if (name != NULL)
      tmp->name = (char *) strdup (name);
   else
      tmp->name = "unknown";

   timer_lock ();

   LL_Add (TIMER_MASTER->ll_timers, tmp);

   timer_unlock ();

   Timer_Master_Update (tmp);


   return (tmp);
}



void timer_lock () {
   /* printf ("LOCKING TIMER\n");*/


  pthread_mutex_lock (&TIMER_MASTER->mutex_lock);

}

void timer_unlock () {
   /* printf ("unLOCKING TIMER\n");*/


  pthread_mutex_unlock (&TIMER_MASTER->mutex_lock);
}


/* timer_jitter
 * return some random number based on timer jitter value and srand
 */
int static timer_jitter (mtimer_t *timer) {
  int val;
  
  if (timer->jitter <= 0) {val =  0;}
  else {
    val = (rand() % (timer->jitter * 2 + 1)) - timer->jitter;
  }
  
  return (val);
}
