/*
 * $Id: ripng_proto.c,v 1.22 1997/02/18 23:17:20 masaki Exp $
 */


#include <mrt.h>

#ifdef HAVE_IPV6 
#include <stdarg.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <net/if.h>
#include <netdb.h>
#include <stdlib.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <string.h>
#include <netdb.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <trace.h>
#include <timer.h>
#include <select.h>
#include <interface.h>
#include <rib.h>
/*#include <route_manager.h>*/
#include <ripng.h>
#include <sys/uio.h>
#ifdef __linux__
#include <netinet6/in6.h>
#include <netinet6/ipv6.h>
#else
#ifdef CMSG_FIRSTHDR /* BSD ... maybe bad way */
#include <machine/param.h>    
#endif
#endif

void	ripng_send_routes (interface_t *interface, struct sockaddr_in6 *host,
            int change);
void	ripng_timeout_routes (void);
void	ripng_receive_update (void);
int	ripng_update_routing_table (void);
int	init_ripng_listen (void);
int	ripng_send_request (LINKED_LIST *ll_prefixes);
int	ripng_process_update (LINKED_LIST *ll_prefixes, LINKED_LIST *ll_attr);
int	ripng_process_packet_response (gateway_t *gateway, char *update, 
            int bytes, LINKED_LIST *ll_prefixes, LINKED_LIST *ll_attr);
int 	ripng_process_packet_request (struct sockaddr_in6 *from, char *update, 
            int bytes, interface_t *interface);
void	show_ripng ();
int ripng_policy (prefix_t *prefix, ripng_attr_t *attr, interface_t *out);
void	ripng_output_processing (int change);
void	ripng_timer_update ();
void	ripng_flash_update ();
void	ripng_set_flash_update ();

struct in6_addr all_rip_routers;

/* ripng_schedule_receive_update
 */
void ripng_schedule_receive_update () {
  schedule_event (RIPNG->schedule, ripng_receive_update, 0, NULL);
}

/* ripng_schedule_timer
 * 30 timer processing -- announce routing table
 */
void ripng_schedule_timer () {
  schedule_event (RIPNG->schedule, ripng_timer_update, 0, NULL);
}

/* ripng_schedule_age
 * holddown & expire
 */
void ripng_schedule_age () {
  schedule_event (RIPNG->schedule, ripng_timeout_routes, 0, NULL);
}

/* ripng_schedule_flash
 * flash update
 */
void ripng_schedule_flash () {
  schedule_event (RIPNG->schedule, ripng_flash_update, 0, NULL);
}


/* fix me ! Someone please write a more reasonable hash function! */
u_int ripng_hash_fn (prefix_t *prefix6, u_int size) {
  u_int tmp, buff[4];
#ifdef notdef
  u_char *cp = prefix_tochar (prefix6);

  tmp = cp[0] + cp[1] + 
    cp[2] + cp[3] + cp[4] + cp[5] + cp[6] + cp[7] + cp [8]
    + cp[9] + cp[10] + cp[11] + cp[12] +cp[13] + cp[14] + cp[15]; /* fix */
#else
  memcpy (buff, prefix_tochar (prefix6), sizeof (buff));
  netmasking (prefix6->family, (char *)buff, prefix6->bitlen);
  /* Pedro's suggestion */
  tmp = buff[0] ^ buff[1] ^ buff[2] ^ buff[3];
  tmp ^= (tmp >> 16);
#endif
  tmp = tmp % size;
  return (tmp);
}

u_int ripng_lookup_fn (prefix_t *tmp1, prefix_t *tmp2) {
  return (prefix_compare (tmp1, tmp2));
}


/*
 * init_ripng
 */
int init_ripng (trace_t *trace) 
{
   ripng_route_t tmp;

   all_rip_routers.s6_addr[0] = 0xff;
   all_rip_routers.s6_addr[1] = 0x02;
   all_rip_routers.s6_addr[15] = 0x09;

   RIPNG = (ripng_t *) New(ripng_t);
   RIPNG->trace = trace_copy (trace);

   RIPNG->timer = (mtimer_t *)
     New_Timer (ripng_schedule_timer, RIPNG_UPDATE_INTERVAL, 
		"RIPNG update timer", NULL);
   timer_set_jitter (RIPNG->timer, RIPNG_UPDATE_JITTER);

   RIPNG->age = (mtimer_t *)
     New_Timer (ripng_schedule_age, 0, "RIPNG aging timer", NULL);
   timer_set_flags (RIPNG->age, TIMER_ONE_SHOT);

   RIPNG->flash = (mtimer_t *)
     New_Timer (ripng_schedule_flash, 0, "RIPNG flash timer", NULL);
   timer_set_flags (RIPNG->flash, TIMER_ONE_SHOT);

   RIPNG->schedule = New_Schedule ("RIPNG", RIPNG->trace);

   RIPNG->hash = HASH_Create (100, 
			    HASH_KeyOffset, HASH_Offset(&tmp, &tmp.prefix6),
			    HASH_LookupFunction, ripng_lookup_fn,
			    HASH_HashFunction, ripng_hash_fn,
			    NULL);

   pthread_mutex_init (&RIPNG->send_mutex_lock, NULL);

   return (1);
}


/* start_ripng_thread 
 * the thread is alive! Start the timers and wait for events to process
 */
void start_ripng_main () {
  trace (TRACE, RIPNG->trace, "RIPNG thread created\n");
  
  if (init_ripng_listen () < 0) {
    trace (NORM|INFO, RIPNG->trace, "RIPNG aborted due to error(s)\n");
    return;
  }
  Timer_Turn_ON (RIPNG->timer);
  Timer_Set_Time (RIPNG->age, RIPNG_TIMEOUT_INTERVAL);

  /* send initial request to routers */
  ripng_send_request (NULL);

#ifndef HAVE_LIBPTHREAD
 return;
#endif /* THREADS */

  while (1) {
    schedule_wait_for_event (RIPNG->schedule);
  }
}


/* start_ripng
 * just call thread create routine
 */
int start_ripng () {
  pthread_t thread;

  mrt_thread_create ("RIPNG thread", RIPNG->schedule, 
		     (void *) start_ripng_main, NULL);
  return (1);
}


/* init_ripng_listen
 * start listening for broadcast ripng updates
 */
int init_ripng_listen () {
   struct sockaddr_in6 serv_addr, *src;
   int one = 1;
   int ttl = 255;
   interface_t *interface;
   int sockfd;

   memset (&serv_addr, 0, sizeof (serv_addr));
   serv_addr.sin6_port = htons (RIPNG_DEFAULT_PORT);
   serv_addr.sin6_family = AF_INET6;
   serv_addr.sin6_flowinfo = 0;
#ifdef SIN6_LEN
   serv_addr.sin6_len = sizeof (struct sockaddr_in6);
#endif /* SIN6_LEN */

   if ((sockfd = socket (PF_INET6, SOCK_DGRAM, 0)) < 0) {
      trace (ERROR, RIPNG->trace, "ERROR -- Could not get socket (%s)\n",
	     strerror (errno));
      return (-1);
   }

#ifdef SCM_RIGHTS
#ifdef IPV6_RXINFO
  /*
   *  Receive destination address and interface from socket
   */
  if (setsockopt (sockfd, SOL_IPV6, IPV6_RXINFO, &one, sizeof(one)) < 0) {
    trace (ERROR, RIPNG->trace, 
         "RIPNG setsockopt IPV6_RXINFO failed: %s\n", strerror (errno));
  }
#endif
#ifdef IP_RECVINTERFACE
        if (setsockopt (sockfd, IPPROTO_IPV6, IP_RECVINTERFACE, &one, 
	        sizeof (one)) < 0) {
            trace (ERROR, RIPNG->trace, 
	          "RIPNG setsockopt IP_RECVINTERFACE failed: %s\n",
	          strerror (errno));
        }
#endif
#ifdef IP_RECVDSTADDR
        if (setsockopt (sockfd, IPPROTO_IPV6, IP_RECVDSTADDR, &one, 
	        sizeof (one)) < 0) {
            trace (ERROR, RIPNG->trace, 
	          "RIPNG setsockopt IP_RECVDSTADDR failed: %s\n",
	          strerror (errno));
        }
#endif
#endif /* SCM_RIGHTS */

  /* I'm not sure that unicast ttl should be 255 when responding 
     routing table request, but anyway ... */
  if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_UNICAST_HOPS, (void *)&ttl, sizeof(ttl)) < 0) {
    trace (ERROR, RIPNG->trace, 
         "RIPNG setsockopt IPV6_UNICAST_HOPS failed %s\n", strerror (errno));
  }
  if (setsockopt(sockfd, IPPROTO_IPV6, IPV6_MULTICAST_HOPS, (void *)&ttl, sizeof(ttl)) < 0) {
    trace (ERROR, RIPNG->trace, 
         "RIPNG setsockopt IPV6_MULTICAST_HOPS failed %s\n", strerror (errno));
  }

  /* I'd like to set priority of outgoing packets to 7 the ipng draft
     suggests ... but no way for now */

   if (bind (sockfd, (struct sockaddr *) &serv_addr, sizeof (serv_addr)) < 0) {
      trace (ERROR, RIPNG->trace, "ERROR -- Could not bind to port %d (%s)\n",
	     RIPNG_DEFAULT_PORT, strerror (errno));
      close (sockfd);
      return (-1);
   }

   if (select_add_fd (sockfd, 1, ripng_schedule_receive_update, NULL) < 0) {
      trace (ERROR, RIPNG->trace, "ERROR -- select_add_fd failed\n");
      close (sockfd);
      return (-1);
   }
   if (ipv6_join_multicast (sockfd, all_rip_routers.s6_addr) < 0) {
      close (sockfd);
      return (-1);
   }

   trace (NORM, RIPNG->trace, "RIPNG listening for updates on port %d\n", 
	  RIPNG_DEFAULT_PORT);
   RIPNG->sockfd = sockfd;
   return (1);
}


/* ripng_timeout_routes
 * 1) timoute routes (change metric to 16 and set change bit)
 * 2) garbage collect (delete route from hash, free memory and notify
 *    routing table 
 * Don't need to explicitly do outbound processing, as this will be picked up
 * by the regular 30 second timer
 */
void ripng_timeout_routes () {
  ripng_route_t *route;
  char tmp1[MAXLINE];
  char tmp2[MAXLINE];
  LINKED_LIST *ll_delete = NULL;
  time_t now, t;
  time_t nexttime = 0;
  int changed = 0;

  trace (TRACE, RIPNG->trace, "RIPNG timer (age) fired\n");

  time (&now);
  nexttime = now + RIPNG_TIMEOUT_INTERVAL;
  HASH_Iterate (RIPNG->hash, route) {
    /* don't time out routes from other protocols, like static and 
     * interface routes */
    if (route->attr->type != PROTO_RIPNG &&
        route->attr->type != PROTO_KERNEL) continue; 

    /* garbage collect and delete route */
    if (now - route->time >= RIPNG_GARBAGE_INTERVAL) {

      if (!BIT_TEST (route->flags, RT_RIPNG_DELETE) ||
          route->attr->metric != RIPNG_INFINITY) {
	/* this situation occurs when mrt stops working due to debug
           output freeze or something like that and when system clock jumps */
         /* fake its time received, and shift it timeout stage */
         trace (NORM, RIPNG->trace, 
	     "RIPNG timeout routine detected warp route %s/%d via %s\n",
	     prefix_toa (route->prefix6), route->prefix6->bitlen,
             (route->attr->type == PROTO_KERNEL)?"kernel":
	     prefix_toa2 (route->attr->gateway->prefix, tmp1));
         route->time = now - RIPNG_TIMEOUT_INTERVAL;
	 goto timeout;
      }

      trace (NORM, RIPNG->trace, 
	     "RIPNG deleted %s/%d via %s (garbage collection)\n",
	     prefix_toa (route->prefix6), route->prefix6->bitlen,
             (route->attr->type == PROTO_KERNEL)?"kernel":
	     prefix_toa2 (route->attr->gateway->prefix, tmp1));
      if (ll_delete == NULL)
	ll_delete = LL_Create (0);
      LL_Add (ll_delete, route);
    }
    /* timeout route -- set metric to 16 and set change flag */
    else if (now - route->time >= RIPNG_TIMEOUT_INTERVAL) {
timeout:
      trace (NORM, RIPNG->trace, 
	     "RIPNG timing out %s/%d via %s\n",
	     prefix_toa (route->prefix6), route->prefix6->bitlen,
             (route->attr->type == PROTO_KERNEL)?"kernel":
	     prefix_toa2 (route->attr->gateway->prefix, tmp1));
      route->attr->metric = RIPNG_INFINITY;
      route->flags |= RT_RIPNG_DELETE;
      RIPNG->update_call_fn (KERNEL_UPDATE_HOLD, route->prefix6, route->attr);
      if (route->attr->type != PROTO_KERNEL) {
         route->flags |= RT_RIPNG_CHANGE;
         changed++;
      }

      /* see the earliest next time to be checked */
      if ((t = route->time + RIPNG_GARBAGE_INTERVAL) < nexttime)
	nexttime = t;
    }
    else { /* live routes */
      /* see the earliest next time to be checked */
      if ((t = route->time + RIPNG_TIMEOUT_INTERVAL) < nexttime)
	nexttime = t;
    }
  }

  /* we have to use a linked list cause we can't delete hash items
   * while iterating through hash table */
  if (ll_delete != NULL) {
    LL_Process (ll_delete, (void *) Delete_Ripng_Route);
    LL_Destroy (ll_delete);
  }

  if (changed)
    ripng_set_flash_update ();

  if ((t = nexttime - time (NULL)) <= 0)
    t = 1;
  Timer_Set_Time (RIPNG->age, t);
}


#ifndef IPV6_TXINFO
int ipv6_choose_outgoing (interface_t *interface) {

   assert (interface);
   assert (BIT_TEST (interface->flags, IFF_MULTICAST));
   assert (interface->primary6);

   if (setsockopt (RIPNG->sockfd, IPPROTO_IPV6, IPV6_MULTICAST_IF,
       prefix_tochar (interface->primary6->prefix), 
       sizeof (struct in6_addr)) < 0) {
     trace (ERROR, RIPNG->trace, 
        "RIPNG setsockopt IPV6_MULTICAST_IF failed for %s: %s\n",
         interface->name, strerror (errno));
     return (0);
  }
  return (1);
}
#endif


/* 
 * use multicast for multicast-capable interfaces 
 * use unicast for p-to-p non-multicast-capable interfaces (non standard way)
 * if ll_prefixes is NULL, send a request for complete routing table 
 *    ll_prefixes ... not implemented yet and usual daemon doesn't need it
 */
int ripng_send_request (LINKED_LIST *ll_prefixes)
{
#define RIPNG_HDRLEN 4
#define RIPNG_RTELEN 20
   char buffer[RIPNG_HDRLEN+RIPNG_RTELEN];
   u_char *cp, *start;
   interface_t *interface;
   struct sockaddr_in6 serv_addr;

   start = buffer;

   memset (&serv_addr, 0, sizeof (serv_addr));
   serv_addr.sin6_port = htons (RIPNG_DEFAULT_PORT);
   serv_addr.sin6_family = AF_INET6;
   serv_addr.sin6_flowinfo = 0;
#ifdef SIN6_LEN
   serv_addr.sin6_len = sizeof (struct sockaddr_in6);
#endif /* SIN6_LEN */

   cp = buffer; memset (buffer, 0, sizeof (buffer));

   UTIL_PUT_BYTE (RIPNG_REQUEST, cp);
   UTIL_PUT_BYTE (RIPNG_VERSION, cp);
    
   UTIL_PUT_SHORT (0, cp); /* must be 0 */
   memset (cp, 0, 16);
   cp += 16;
    
   UTIL_PUT_SHORT (0, cp); /* must be 0 */
   UTIL_PUT_BYTE (0, cp); /* length 0 or 16?? spec is not clear */
   UTIL_PUT_BYTE (RIPNG_INFINITY, cp); /* infinity metric */

   LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {

     if (!BIT_TEST (interface->bit, RIPNG->interface_mask)) continue;
     if (!BIT_TEST (interface->flags, IFF_UP)) continue;
     if (interface->primary6 == NULL) continue;

     if (!BIT_TEST (interface->flags, IFF_MULTICAST)) {
        struct in6_addr *dest;
       /*
        * The RIPng draft only say about multicast,
        * so this way may work but non standard
        */
	if (!BIT_TEST (interface->flags, IFF_POINTOPOINT))
	   continue;

        if ((dest = (struct in6_addr *) 
              prefix_tochar (interface->link_local->broadcast)) == NULL) {
	   /* shouldn't use global address, though */
	   if ((dest = (struct in6_addr *) 
                 prefix_tochar (interface->primary->broadcast)) == NULL)
	      continue;
	}
	memcpy (&serv_addr.sin6_addr, dest, 16);
        pthread_mutex_lock (&RIPNG->send_mutex_lock);
     }
     else {
        memcpy (&serv_addr.sin6_addr, &all_rip_routers, 16);
        pthread_mutex_lock (&RIPNG->send_mutex_lock);
#ifndef IPV6_TXINFO
	ipv6_choose_outgoing (interface);
#endif
     }

     trace (NORM, RIPNG->trace, "send RIPNG initial request on %s to %s\n",
    	interface->name, inet_ntop (AF_INET6, 
	    &serv_addr.sin6_addr, NULL, 0));

     sendmsgto (RIPNG->sockfd, buffer, (cp - start), 0, 
            &serv_addr, sizeof (serv_addr), interface);

     pthread_mutex_unlock (&RIPNG->send_mutex_lock);
   }

   return (1);
}

int sendmsgto (int sockfd, char *buffer, int buflen, int flag, 
       struct sockaddr_in6 *addr, int addrlen, interface_t *interface) {
   int rc;

#ifdef IPV6_TXINFO
   struct msghdr mhdr;
   struct cmsghdr *cmsg;
   struct in6_pktinfo *pkt_info;
   struct iovec iov;
   char chdr[sizeof (struct cmsghdr) + sizeof (struct in6_pktinfo)];

   if (!ipv6_link_local_addr (&addr->sin6_addr) &&
       !ipv6_multicast_addr (&addr->sin6_addr)) {
#endif /* IPV6_TXINFO */

      if ((rc = sendto (sockfd, buffer, buflen, flag, 
            (struct sockaddr *) addr, addrlen)) < 0) {
         perror ("Error in sendto");
      }

#ifdef IPV6_TXINFO
   }
   else {

      mhdr.msg_name = (struct sockaddr_in *) addr;
      mhdr.msg_namelen = addrlen;
      mhdr.msg_iov = &iov;
      mhdr.msg_iovlen = 1;
      mhdr.msg_controllen = sizeof (chdr);
      iov.iov_base = buffer;
      iov.iov_len = buflen;
      cmsg = (struct cmsghdr *) chdr;
      cmsg->cmsg_len = sizeof (chdr);
      cmsg->cmsg_level = SOL_IPV6;
      cmsg->cmsg_type  = IPV6_TXINFO;
      mhdr.msg_control = (void *) cmsg;
      pkt_info = (struct in6_pktinfo *) cmsg->cmsg_data;
      memset (pkt_info, 0, sizeof (struct in6_pktinfo));
      if (!BIT_TEST (interface->flags, IFF_USERDEF)) {
         pkt_info->ipi6_ifindex = interface->index;
      }
      memcpy (&pkt_info->ipi6_addr, 
          prefix_tochar (interface->link_local->prefix), 16);
#if 0
printf("pkt_info->ipi6_ifindex = %d, pkt_info->ipi6_addr = %s\n",
   pkt_info->ipi6_ifindex,
   inet_ntop (AF_INET6, &pkt_info->ipi6_addr, NULL, 0));
#endif
      if ((rc = sendmsg (sockfd, &mhdr, flag)) < 0) {
         perror ("Error in sendmsg");
      }
   }
   #endif /* IPV6_TXINFO */
   return (rc);
}


/*
 * given an interface, broadcast ripng routes (according to policy)
 * we do this every RIPNG_UPDATE_INTERVAL
 * if change flag is set (==1) then only send routes marked as
 * changed
 */
void ripng_send_routes (interface_t *interface, struct sockaddr_in6 *host, 
      int change) {
  ripng_route_t *route;
  char *buffer, *cp;
  struct sockaddr_in6 serv_addr;
  int routes = 0; /* count of routes */
  int pdulen, metric;
  char tag[MAXLINE];

  assert (interface);

  if (host) {
    memcpy (&serv_addr, host, sizeof (serv_addr));
    pthread_mutex_lock (&RIPNG->send_mutex_lock);
  }
  else {
    memset (&serv_addr, 0, sizeof (serv_addr));
    serv_addr.sin6_port = htons (RIPNG_DEFAULT_PORT);
    serv_addr.sin6_family = AF_INET6;
    serv_addr.sin6_flowinfo = 0;
#ifdef SIN6_LEN
    serv_addr.sin6_len = sizeof (struct sockaddr_in6);
#endif /* SIN6_LEN */

    if (BIT_TEST (interface->flags, IFF_MULTICAST)) {
      pthread_mutex_lock (&RIPNG->send_mutex_lock);
#ifndef IPV6_TXINFO
      ipv6_choose_outgoing (interface);
#endif
      memcpy (&serv_addr.sin6_addr, &all_rip_routers, 16);
    }
    else if (BIT_TEST (interface->flags, IFF_POINTOPOINT)) {
      struct in6_addr *dest;

      if ((dest = (struct in6_addr *)
	   prefix_tochar (interface->link_local->broadcast)) == NULL) {
	 /* shouldn't use global address, though */
	 if ((dest = (struct in6_addr *) 
            prefix_tochar (interface->primary->broadcast)) == NULL) {
	       return;
	 }
      }
      pthread_mutex_lock (&RIPNG->send_mutex_lock);
      memcpy (&serv_addr.sin6_addr, dest, 16);
    }
    else {
      return;
    }
  }

#define IPV6_HDRLEN 40
#define UDP_HDRLEN 8
  pdulen = interface->mtu - IPV6_HDRLEN - UDP_HDRLEN;
  assert (pdulen >= RIPNG_HDRLEN+RIPNG_RTELEN);
  buffer = NewArray (char, pdulen);

  cp = buffer;
  memset (buffer, 0, pdulen);
  
  UTIL_PUT_BYTE (RIPNG_RESPONSE, cp);
  UTIL_PUT_BYTE (RIPNG_VERSION, cp);
  UTIL_PUT_SHORT (0, cp);

  trace (NORM, RIPNG->trace, "send RIPNG (via %s to %s)\n",
    interface->name, inet_ntop (AF_INET6, &serv_addr.sin6_addr, NULL, 0));

  HASH_Iterate (RIPNG->hash, route) {
    prefix_t *prefix6 = route->prefix6;

    if ((metric = ripng_policy (prefix6, route->attr, interface)) < 0)
        continue;

    /* doing ouput processing and only sending changed routes */
    if ((change == 1) && (!(route->flags & RT_RIPNG_CHANGE))) {
      trace (TR_PACKET, RIPNG->trace, 
          "  x %s/%d metric %d (unchange)\n",
           prefix_toa (prefix6), prefix6->bitlen, route->attr->metric);
       continue;
    }

#if 0
    /* unset the change flag */ /* -- this should be done later */
    route->flags &= ~(RT_RIPNG_CHANGE);
#endif
    routes++;

    if (route->attr->tag) {
      sprintf(tag, " tag %d", route->attr->tag);
    }
    else {
      tag[0] = '\0';
    }
    trace (TR_PACKET, RIPNG->trace, "  o %s/%d metric %d%s\n",
       prefix_toa (prefix6), prefix6->bitlen, metric, tag, route->flags);

    memcpy (cp, prefix_tochar (prefix6), 16);
    cp += 16;

    UTIL_PUT_SHORT (htons (route->attr->tag), cp);
    UTIL_PUT_BYTE (prefix6->bitlen, cp);
    UTIL_PUT_BYTE (metric, cp);

    /* see if we have filled up the buffer. If so, send packet and 
     * create new buffer 
     */
    if (cp - buffer > pdulen - RIPNG_RTELEN) {
      sendmsgto (RIPNG->sockfd, buffer, cp - buffer, 0, 
	  &serv_addr, sizeof (serv_addr), interface);
      cp = buffer;
      memset (buffer, 0, pdulen);
      UTIL_PUT_BYTE (RIPNG_RESPONSE, cp);
      UTIL_PUT_BYTE (RIPNG_VERSION, cp);
      UTIL_PUT_SHORT (0, cp);
      routes = 0;
    }
  }
  
  /* okay, send packet (assuming we have one) */
  if (routes > 0) {
    sendmsgto (RIPNG->sockfd, buffer, cp - buffer, 0, 
        &serv_addr, sizeof (serv_addr), interface);
  }
  Delete (buffer);
  pthread_mutex_unlock (&RIPNG->send_mutex_lock);
}

/* ripng_receive_update
 * read and process and RIPNG update packet recieved on
 * our interface
 */
void ripng_receive_update ()
{
   u_char *ptr;
   int n;
   struct sockaddr_in6 from;
   int fromlen;
   prefix_t *prefix6;
   gateway_t *gateway = NULL;
   char tmp[MAXLINE];
   interface_t *interface;
   u_char command, version;
   u_short zero;
   u_char *cp;

   fromlen = sizeof (from);
   cp = ptr = NewArray (char, INTERFACE_MASTER->max_mtu);

   if ((n = recvmsgfrom (RIPNG->sockfd, ptr, INTERFACE_MASTER->max_mtu, 
      O_NONBLOCK, (struct sockaddr *) &from, &fromlen, &interface)) < 0) {
     goto ignore;
   }

   if (fromlen != sizeof (from) || from.sin6_family != AF_INET6) {
     trace (ERROR, RIPNG->trace, "RIPNG Error reading on socket\n");
     goto ignore;
   }

   /*
    * get command first for later check
    */
   UTIL_GET_BYTE (command, cp);
   UTIL_GET_BYTE (version, cp);
   UTIL_GET_SHORT (zero, cp);
   zero = ntohs (zero);

   prefix6 = New_Prefix (AF_INET6, (char *)&from.sin6_addr, 128);

   /*
    * common check for all commands
    */
   if (version != RIPNG_VERSION) {
       trace (NORM, RIPNG->trace, "RIPNG version %d from %s is unsupported!\n", 
 	    version, prefix_toa (prefix6));
/* XXX        goto ignore; */
   }

   if (zero) {
       trace (NORM, RIPNG->trace, 
	   "RIPNG pad field from %s is not zero (%d)!\n", 
	   prefix_toa (prefix6), zero);   
/* XXX       goto ignore; */
   }

   if (command != RIPNG_RESPONSE && command != RIPNG_REQUEST) {
       trace (NORM, RIPNG->trace, "RIPNG command %d from %s is unsupported!\n", 
          command, prefix_toa (prefix6));
       goto ignore;
   }

   if (command == RIPNG_RESPONSE) {

     /* register the gateway */
     gateway = add_gateway (prefix6, 0, interface);

     /* don't listen to things broadcast from our own interface */
     if (find_interface_local (prefix6)) {
       trace (TR_PACKET, RIPNG->trace, 
          "recv RIPNG ignore own response from %s\n", 
          gateway_toa (tmp, gateway));
       goto ignore;
     }

     if (ipv6_ipv4_addr (&from.sin6_addr)) {
       trace (NORM, RIPNG->trace, 
	  "RIPNG received response with ipv4 source address %s, ignore\n",
          gateway_toa (tmp, gateway));
       goto ignore;
     }

     if (!ipv6_link_local_addr (&from.sin6_addr)) {
       trace (NORM, RIPNG->trace, 
"RIPNG received response with non link-local source address %s, but continue\n",
	   prefix_toa (prefix6));
     }

     /* check if this interface is configured for RIPNG */
     if (!(interface->bit & RIPNG->interface_mask)) {
        trace (TR_PACKET, RIPNG->trace, 
	     "RIPNG received response on unconfigured interface %s from %s\n",
	     interface->name, 
             gateway_toa (tmp, gateway));
        goto ignore;
     }

     if (interface->mtu < n + IPV6_HDRLEN + UDP_HDRLEN) {
       trace (NORM, RIPNG->trace, 
"RIPNG received packet size %d exceeds mtu %d - %d hdrs, on %s from %s\n",
	 n, interface->mtu, IPV6_HDRLEN + UDP_HDRLEN,
	 interface->name, gateway_toa (tmp, gateway));
     }

     trace (NORM, RIPNG->trace, 
         "recv RIPNG response from %s (%d bytes) on %s\n",
          gateway_toa (tmp, gateway), n, interface->name);

     /*
      * ID also suggests to check if the hop count is 255, 
      * but I don't know the way -- masaki
      */
     /*
      * That should be unnecessary. The check for link local source above
      * warranties that the packet has not been forwarded. If it was
      * your routers are broken big time -- pedro
      */


     if (ntohs (from.sin6_port) !=  RIPNG_DEFAULT_PORT) {
       trace (NORM, RIPNG->trace, 
	   "RIPNG source port %d from %s is ignored!\n", 
            ntohs(from.sin6_port), 
            gateway_toa (tmp, gateway));
       goto ignore;
     }

     if (n - (cp-ptr)) {
       LINKED_LIST *ll_prefixes, *ll_attr;

       ll_prefixes = LL_Create (LL_DestroyFunction, Deref_Prefix, NULL);
       ll_attr = LL_Create (LL_DestroyFunction, Deref_Ripng_Attr, NULL);

       /* munge the ripng packet and return list of prefixes and attributes */

       ripng_process_packet_response (gateway, cp, n-(cp-ptr), 
           ll_prefixes, ll_attr);
  
       /* update our tables */
       ripng_process_update (ll_prefixes, ll_attr);

      /*
       * I tried to implement reference counter in prefix_t and ripng_attr_t
       */
       LL_Destroy (ll_prefixes);
       LL_Destroy (ll_attr);
     }
   }
   else if (command == RIPNG_REQUEST) {

     /* register the gateway */
     if (ntohs (from.sin6_port) ==  RIPNG_DEFAULT_PORT)
        gateway = add_gateway (prefix6, 0, interface);

     /* don't listen to things broadcast from our own interface
        except for a query */
     if (find_interface_local (prefix6) &&
         ntohs (from.sin6_port) ==  RIPNG_DEFAULT_PORT) {
       goto ignore;
     }

#ifdef notdef
     /* request from over a router will be rejected for now */
     if (!ipv6_link_local_addr (&from.sin6_addr)) {
       trace (NORM, RIPNG->trace, 
"RIPNG received request with non link-local source address %s, ignore!\n",
	   prefix_toa (prefix6));
       goto ignore;
     }
#endif

     trace (NORM, RIPNG->trace, 
         "recv RIPNG request from %s (%d bytes) port %d\n",
	  (gateway)? gateway_toa (tmp, gateway): prefix_toa (prefix6), 
          n, ntohs (from.sin6_port));

     if (n - (cp-ptr) > 0) {
       /* more than 1 entry */
       if (ripng_process_packet_request (&from, cp, n-(cp-ptr), interface) 
	    >= 0) {
         cp = ptr;
         UTIL_PUT_BYTE (RIPNG_RESPONSE, cp);
         UTIL_PUT_BYTE (RIPNG_VERSION, cp);
         UTIL_PUT_SHORT (0, cp);

  	 pthread_mutex_lock (&RIPNG->send_mutex_lock);
         sendmsgto (RIPNG->sockfd, ptr, n, 0, &from,
	     sizeof (from), interface);
  	 pthread_mutex_unlock (&RIPNG->send_mutex_lock);
       }
       else {
         ripng_send_routes (interface, &from, 0); 
       }
       trace (TR_PACKET, RIPNG->trace, 
          "recv RIPNG request answered to %s port %d\n",
	  (gateway)? gateway_toa (tmp, gateway): prefix_toa (prefix6), 
          ntohs (from.sin6_port));
     }
     else {
       trace (NORM, RIPNG->trace, 
          "recv RIPNG request no entry, discard it!\n");
     }
   }

ignore:
   Delete (ptr);
   select_enable_fd (RIPNG->sockfd);
   Deref_Prefix (prefix6);
}


int recvmsgfrom (int sockfd, char *buffer, int buflen, int flag,
                 struct sockaddr *from, int *fromlen, interface_t **ifp) {
   int rc;
   prefix_t prefix, *prefix6 = &prefix;
   interface_t *interface = NULL;
   int index = -1;

#ifndef SCM_RIGHTS
/* Solaris has IP_RECVDSTADDR, but differs. */
   if ((rc = recvfrom (sockfd, buffer, buflen, flag, from, fromlen)) <= 0) {
     perror ("error in recvfrom");
     return (-1);
   }
#else /* SCM_RIGHTS */
   struct msghdr mhdr;
   struct cmsghdr *cmsg;
#ifdef IPV6_RXINFO
   struct in6_pktinfo *pkt_info;
#endif
   struct iovec iov;
   char chdr[BUFSIZ];

   mhdr.msg_name = (void *)from;
   mhdr.msg_namelen = *fromlen;
   memset (mhdr.msg_name, 0, mhdr.msg_namelen);
   mhdr.msg_iov = &iov;
   mhdr.msg_iovlen = 1;
   mhdr.msg_control = (void *) chdr;
   mhdr.msg_controllen = sizeof (chdr);
   memset (mhdr.msg_control, 0, mhdr.msg_controllen);
   iov.iov_base = buffer;
   assert (buflen >= INTERFACE_MASTER->max_mtu);
   iov.iov_len = buflen;

   if ((rc = recvmsg (sockfd, &mhdr, flag)) < 0) {
     perror ("error in recvmsg");
     return (-1);
   }
   else {
#if !defined(CMSG_FIRSTHDR) && defined(CMSG_FIRST) /* linux */
#define CMSG_FIRSTHDR(a) CMSG_FIRST(a)
#endif
     for (cmsg = CMSG_FIRSTHDR (&mhdr); 
          cmsg && cmsg->cmsg_len >= sizeof (struct cmsghdr);
          cmsg = CMSG_NXTHDR (&mhdr, cmsg)) {
#ifdef notdef
        if (cmsg->cmsg_level != IPPROTO_IPV6) continue;
#endif
        switch (cmsg->cmsg_type) {
#ifdef IPV6_RXINFO
          case IPV6_RXINFO:
             pkt_info = (struct in6_pktinfo *) cmsg->cmsg_data;
             index = pkt_info->ipi6_ifindex;
             trace (TR_PACKET, RIPNG->trace, 
                 "recv RIPNG IPV6_RXINFO index = %d\n", index);
             break;
#endif
#ifdef IP_RECVINTERFACE
          case IP_RECVINTERFACE: {
             u_short ui;
             memcpy (&ui, CMSG_DATA(cmsg), 2);
             index = ui;
             trace (TR_PACKET, RIPNG->trace, 
                 "recv RIPNG IP_RECVINTERFACE index = %d\n", ui);
             }
             break;
#endif
#ifdef IP_RECVDSTADDR
          /*
           * destination address of packet should be checked here
           */
          case IP_RECVDSTADDR: {
             struct in6_addr dstaddr;
             memcpy (&dstaddr, CMSG_DATA(cmsg), 16);
             trace (TR_PACKET, RIPNG->trace, 
                 "recv RIPNG IP_RECVDSTADDR dest = %s\n", 
                 inet_ntop (AF_INET6, &dstaddr, NULL, 0));
             }
             break;
#endif
          default:
        }
     }
  }
#endif /* SCM_RIGHTS */
  prefix6->family = AF_INET6;
  prefix6->bitlen = 128;
  prefix6->ref_count = 1; /* dummy for check */
  memcpy (prefix_tochar (prefix6), 
     &((struct sockaddr_in6 *)from)->sin6_addr, 16);
  if (index > 0) {
     if (interface = find_interface_flags (prefix6, IFF_USERDEF)) {
        trace (TR_PACKET, RIPNG->trace, 
           "recv RIPNG FIND TUNNEL %s index = %d\n", 
           interface->name, index);
     }
     else {
        interface = find_interface_byindex (index);
        if (interface == NULL) {
           trace (TR_PACKET, RIPNG->trace, 
              "RIPNG received response on unknown interface from %s, ignore!\n",
              prefix_toa (prefix6));
           return (-1);
        }
     }
  }
  else {
     gateway_t *gateway;
     /*
      * if the system doesn't supply the incoming interface index,
      * try to find among configured interfaces and gateways registered,
      */
     if (interface = find_interface (prefix6)) {
        trace (TR_PACKET, RIPNG->trace, 
	  "RIPNG guesses %s on %s (find_interface)\n", 
	  prefix_toa (prefix6), interface->name);
     }
     else if ((gateway = find_gateway (prefix6, 0)) &&
	gateway->interface) {

        interface = gateway->interface;
        trace (TR_PACKET, RIPNG->trace, 
	  "RIPNG guesses %s on %s (find_gateway)\n", 
	  prefix_toa (prefix6), interface->name);
     }
     else {
        interface = INTERFACE_MASTER->default_interface;
        trace (TR_PACKET, RIPNG->trace, 
	  "RIPNG guesses %s on %s (default)\n",
          prefix_toa (prefix6), interface->name);
     }
  }
  *ifp = interface;
  return (rc);
}


/* 
 * ripng_process_update
 *
 * process linked list of routes (prefix & attribute combinations)
 * check if 1) just updating time on already received route
 *	    2) or a change (different attributes from same gateway,
 *	       or better metric and attributes from new gateway
 */
int ripng_process_update (LINKED_LIST *ll_prefixes, 
			  LINKED_LIST *ll_attr) {
  prefix_t *prefix6;
  ripng_attr_t *ripng_attr;
  ripng_route_t *route;
  char tmp[MAXLINE], nexthop[MAXLINE], tag[MAXLINE];
  int metric, change = 0; 
  struct in6_addr tmp6;

  ripng_attr = LL_GetHead (ll_attr);
  LL_Iterate (ll_prefixes, prefix6) {
    assert (ripng_attr->ref_count > 0);
    assert (prefix6->ref_count > 0);

    /* do checks here -- if fail, log and ignore
     * 1) destination prefix valid (i.e. not multicast)
     * 2) prefix length valid 
     * 3) metric valid (between 1 and 16 inclusive)
     */

    /* see if valid and do policy to add cost to the metric */
     if ((metric = ripng_policy (prefix6, ripng_attr, NULL)) < 0) {
       continue;
     }
     ripng_attr->metric = metric;

#ifdef notdef
     memcpy (&tmp6, prefix_tochar (prefix6), 16);
     netmasking (prefix6->family, addr, prefix6->bitlen);
     if (memcmp (&tmp6, prefix_tochar (prefix6), 16) != 0) {
         trace (TR_PACKET, RIPNG->trace, 
             "RIPNG prefix %s will be truncated by prefixlen %d\n", 
	     inet_ntop (AF_INET6, &tmp6, NULL, 0), prefix6->bitlen);
     }
#endif
    if (memcmp (prefix_tochar (ripng_attr->nexthop), 
                prefix_tochar (ripng_attr->gateway->prefix), 16) != 0) {
      sprintf(nexthop, " -> %s", prefix_toa (ripng_attr->nexthop));
    }
    else {
      nexthop[0] = '\0';
    }
    if (ripng_attr->tag) {
      sprintf(tag, " tag %d", ripng_attr->tag);
    }
    else {
      tag[0] = '\0';
    }

    if ((route = HASH_Lookup (RIPNG->hash, prefix6)) == NULL) { /* new route */

      if (ripng_attr->metric == RIPNG_INFINITY) {
        trace (TR_PACKET, RIPNG->trace, "  x %s/%d metric %d (infinity)\n",
	      prefix_toa (prefix6), prefix6->bitlen, ripng_attr->metric);
        goto next;
      }

      trace (NORM, RIPNG->trace, 
          "  o %s/%d metric %d%s (new)\n",
	  prefix_toa (prefix6), prefix6->bitlen, ripng_attr->metric, tag);

      route = New_Ripng_Route (prefix6, ripng_attr);
      route->flags |= RT_RIPNG_CHANGE;
      change++;

      /* add route to rib */
      assert (RIPNG->update_call_fn != NULL);
      RIPNG->update_call_fn (KERNEL_UPDATE_ADD, route->prefix6, route->attr); 
    }
    else { /* exiting route */

   /* we already have a route for this prefix. Two cases:
    *  1) from same gateway -- just update time, delete, or implicit withdraw.
    *     if tag changed it should be updated and propagated -- masaki
    *  2) from different gateway -- if metric better use this, otherwise
    *     just ignore it. We'll hear about it again in 30 seconds...
    */

      if (route->attr->type == PROTO_CONNECTED) {
         trace (TR_PACKET, RIPNG->trace, "  x %s/%d metric %d (direct)\n",
	     prefix_toa (prefix6), prefix6->bitlen, route->attr->metric);
         goto next;
      }

      if (route->attr->gateway == ripng_attr->gateway) {

	  /* case i: both infinity */
	  if (ripng_attr->metric == RIPNG_INFINITY &&
	        route->attr->metric == RIPNG_INFINITY) {

              trace (TR_PACKET, RIPNG->trace, 
		"  x %s/%d metric %d (deleting)\n",
	        prefix_toa (prefix6), prefix6->bitlen, ripng_attr->metric);

	      /* it is under deletion process, don't touch. ignoring it */
	      assert (route->flags & RT_RIPNG_DELETE);
	      /* should I propagate it? */
              /* route->flags |= RT_RIPNG_CHANGE; */
              /* change++; */
	  }
	  /* case ii: only new is infinity */
	  else if (ripng_attr->metric == RIPNG_INFINITY) {

		  trace (NORM, RIPNG->trace, 
		    "  o %s/%d%s metric %d%s (shift to delete)\n",
		     prefix_toa (prefix6), prefix6->bitlen, nexthop,
		     ripng_attr->metric, tag);

		  /* shift it to deletion process */
                   route->flags |= RT_RIPNG_DELETE;
                   route->flags |= RT_RIPNG_CHANGE;

		  /* MRT doesn't hold another time, so this is ad hoc solution.
		     Someone may be confused the time changed and
		     strictry saying I need to start the timer */
	      	   route->time = time (NULL) - RIPNG_TIMEOUT_INTERVAL;

		   change++;
		   Deref_Ripng_Attr (route->attr);
		   route->attr = Ref_Ripng_Attr (ripng_attr);
		   RIPNG->update_call_fn (KERNEL_UPDATE_HOLD, 
		      route->prefix6, route->attr);
	  }
	  /* case iii: new is not infinity (old may be infinity) */
	  else {

		if (route->attr->metric != ripng_attr->metric ||
		/* ID says nothing about tag but I think it should be */
		   route->attr->tag != ripng_attr->tag ||
		  !prefix_compare (route->attr->nexthop, 
		       ripng_attr->nexthop)) {

                    trace (NORM, RIPNG->trace, 
	                "  o %s/%d%s metric %d%s (change)\n",
	                 prefix_toa (prefix6), prefix6->bitlen, nexthop,
	                 ripng_attr->metric, tag);

		    if (!prefix_compare (route->attr->nexthop, 
				ripng_attr->nexthop)) {
		        /* first add the route with new next hop, 
		           then delete old one */
		        /* the following two calls should be atomic? 
								-- masaki */
		        RIPNG->update_call_fn (KERNEL_UPDATE_DEL, 
			    route->prefix6, route->attr);
		        RIPNG->update_call_fn (KERNEL_UPDATE_ADD, 
			    route->prefix6, ripng_attr);
		    }
		    else {
			/* the same nexthop but stop deleting process */
			if (BIT_TEST (route->flags, RT_RIPNG_DELETE)) {
			   RIPNG->update_call_fn (KERNEL_UPDATE_UNHOLD,
			      route->prefix6, ripng_attr);
			}
		    }

	            route->flags &= ~(RT_RIPNG_DELETE);
                    route->flags |= RT_RIPNG_CHANGE;
                    change++;
		    Deref_Ripng_Attr (route->attr);
	            route->attr = Ref_Ripng_Attr (ripng_attr);
	        }
                else {
                    trace (TR_PACKET, RIPNG->trace, 
                        "  x %s/%d metric %d (same)\n", prefix_toa (prefix6), 
                        prefix6->bitlen, route->attr->metric);
                }
	        route->time = time (NULL); 
	  }
      }
      else { /* from a gateway different for the existing one */

	if (route->attr->metric > ripng_attr->metric /* better metric */ ||
		(route->attr->metric != RIPNG_INFINITY && (
		    /* suggested heuristic in case of the same metric */
                    route->attr->metric == ripng_attr->metric &&
		    (time(NULL) - route->time) >= RIPNG_TIMEOUT_INTERVAL/2) ||
                    /* kernel route always overriden -- use pref instead */
                    route->attr->type == PROTO_KERNEL)) {

            trace (NORM, RIPNG->trace, 
	        "  o %s/%d%s metric %d%s (change)\n",
	         prefix_toa (prefix6), prefix6->bitlen, nexthop,
	         ripng_attr->metric, tag);

	    if (!prefix_compare (route->attr->nexthop, ripng_attr->nexthop)) {
		/* add new one then remove old one */
		RIPNG->update_call_fn (KERNEL_UPDATE_DEL, 
		    route->prefix6, route->attr);
		RIPNG->update_call_fn (KERNEL_UPDATE_ADD, 
		    route->prefix6, ripng_attr);
	    }
	    else {
		/* the same nexthop but metric changed from infinity */
		if (BIT_TEST (route->flags, RT_RIPNG_DELETE)) {
		   RIPNG->update_call_fn (KERNEL_UPDATE_UNHOLD,
		      route->prefix6, ripng_attr);
		}
	    }

	    route->time = time (NULL); 
            route->flags &= ~(RT_RIPNG_DELETE);
            route->flags |= RT_RIPNG_CHANGE;
            change++;
	    Deref_Ripng_Attr (route->attr);
	    route->attr = Ref_Ripng_Attr (ripng_attr);
	}
        else {
            trace (TR_PACKET, RIPNG->trace, 
                "  x %s/%d metric %d (>= existing %d)\n",
	        prefix_toa (prefix6), prefix6->bitlen,  ripng_attr->metric,
                route->attr->metric);
        }
      }
    }
next:
    ripng_attr = LL_GetNext (ll_attr, ripng_attr);
  }
  
  if (change) 
      ripng_set_flash_update ();
  return (1);
}


/* set_ripng
 * change/set ripng attributes.
 */
int set_ripng (int first, ...) {
  va_list    		ap;
  enum RIPNG_ATTR	attr;

  /* Process the Arguments */
   va_start(ap, first);
   for (attr = (enum RIPNG_ATTR)first; attr; attr = va_arg(ap, enum RIPNG_ATTR)) {
      switch (attr) {
      case  RIPNG_RT_UPDATE_FN:
	 RIPNG->update_call_fn = va_arg (ap, void*);
	 break;
      case RIPNG_TRACE_STRUCT:
	 RIPNG->trace = va_arg (ap, trace_t *);
	 break;
      default:
	 trace (NORM, RIPNG->trace, "Unknown RIPNG attribute %s:(%d)", 
		__FILE__, __LINE__);
	 break;
      }
   }
   va_end(ap);
}

ripng_attr_t *Ref_Ripng_Attr (ripng_attr_t *attr) 
{
  assert (attr->ref_count > 0);
  attr->ref_count++;
  return (attr);
}

/* New_Ripng_Route
 * create/allocate new ripng_route_t structure and insert
 * it in the RIPNG prefix6 route hash table
 */
ripng_route_t *New_Ripng_Route (prefix_t *prefix6, ripng_attr_t *attr) 
{
  ripng_route_t *route;

  route = New (ripng_route_t);
  route->prefix6 = Ref_Prefix (prefix6);
  route->attr = Ref_Ripng_Attr (attr);
  route->time = time (NULL);
  route->flags |= RT_RIPNG_CHANGE;

  HASH_Insert (RIPNG->hash, route);
  return (route);
}


/* Delete_Ripng_Route
 * free route memory and remove from hash
 */
int Delete_Ripng_Route (ripng_route_t *route) {

  HASH_Remove (RIPNG->hash, route);
  RIPNG->update_call_fn (KERNEL_UPDATE_DEL, route->prefix6, route->attr);
  Deref_Prefix (route->prefix6);
  Deref_Ripng_Attr (route->attr);
  Delete (route);

  return (1);
}


ripng_attr_t *New_Ripng_Attr (int metric) {
   ripng_attr_t *tmp;

   tmp = New (ripng_attr_t);
   tmp->type = PROTO_RIPNG;
   tmp->metric = metric;
   tmp->ref_count = 1;

   return (tmp);
}


void Deref_Ripng_Attr (ripng_attr_t *tmp) {
  assert (tmp->ref_count > 0);
  if (--tmp->ref_count <= 0) {
    Deref_Prefix (tmp->nexthop);
    Delete (tmp);
  }
}


/* show_ripng 
 * dump various ripng stats to a socket
 * usually called by UII (user interactive interface 
 */
void show_ripng (uii_connection_t *uii) {
  uii_send_data (uii, "\r\nRouting Protocol is \"ripng\" (Using IPV6)\r\n");

  if (RIPNG->sockfd < 0) 
    uii_send_data (uii, "Not listening for RIPNG announcements\r\n",
		   RIPNG_DEFAULT_PORT, RIPNG->sockfd);
  else 
    uii_send_data (uii, "Listening on port %d (socket %d)\r\n",
		   RIPNG_DEFAULT_PORT, RIPNG->sockfd);

  uii_send_data (uii, 
    "Sending updates every %d seconds +/- %d, next due in %d seconds\r\n",
	 RIPNG_UPDATE_INTERVAL, RIPNG_UPDATE_JITTER,
	 RIPNG->timer->time_next_fire - time (NULL));
  uii_send_data (uii, 
    "Triggered update and split horizon (no poisoned reverse) implemented\r\n");
  uii_send_data (uii, 
    "Invalid after %d seconds, hold down %d, flushed after %d\r\n", 
    RIPNG_TIMEOUT_INTERVAL, RIPNG_TIMEOUT_INTERVAL, RIPNG_GARBAGE_INTERVAL);
}


/* show_ripng_routing_table 
 * dump RIPNG routing table to socket. Usually called by user
 * interactive interface
 */
void show_ripng_routing_table (uii_connection_t *uii) {
  ripng_route_t *route;
  char tmp1[MAXLINE], tmp2[MAXLINE];
  time_t now;
  interface_t *interface;
  char tmp[MAXLINE], *name = NULL;

  if (parse_line (uii->cp, "%s", tmp) >= 1) {
     if ((interface = find_interface_byname (tmp)) == NULL) {
	uii_send_data (uii, "no such interface: %s\r\n", tmp);
	return;
     }
     name = tmp;
  }

  time (&now);
  uii_send_data (uii, "%c %-32s %-5s %-26s %-6s %-4s\r\n", 
		 'P', "Prefix", "IF", "Next Hop", "Metric", "Time"); 

  HASH_Iterate (RIPNG->hash, route) {
    char *ifname = (route->attr->gateway->interface)?
	    route->attr->gateway->interface->name:"?";

    if (name && strcmp (ifname, name))
	continue;

    sprintf (tmp1, "%s/%d", prefix_toa (route->prefix6), 
                            route->prefix6->bitlen);
    uii_send_data (uii, "%c %-32s %-5s %-26s %6d %4d\r\n", 
	proto2string(route->attr->type)[0], 
	tmp1, ifname,
	prefix_toa2 (route->attr->nexthop, tmp2),
	route->attr->metric, 
	(route->attr->type == PROTO_STATIC ||
	 route->attr->type == PROTO_CONNECTED)?0:now - route->time);
  }
}


/* ripng_debug
 * set the debug flags or file for ripng
 * called by config routines
 */
int ripng_debug (uii_connection_t *uii)
{
  char *token, *line;
  u_long flag = 0;
  line = uii->cp;

  /* get flag */
  if ((token = (char *) uii_parse_line (&line)) == NULL) return (-1);
  flag = trace_flag (token);
  set_trace (RIPNG->trace, TRACE_FLAGS, flag, NULL);

  /* get file name (OPTIONAL) */
  if ((token = (char *) uii_parse_line (&line)) == NULL) return (1);
  set_trace (RIPNG->trace, TRACE_LOGFILE, token, NULL);

  return (1);
}


/* 
 * run policy on a ripng route 
 * returns -1 when policy rejects it
 * otherwise returns new metric
 */
int ripng_policy (prefix_t *prefix, ripng_attr_t *attr, interface_t *out) {

  int cost, num;
  struct in6_addr *addr = (struct in6_addr *)prefix_tochar (prefix);

  if (out == NULL) {
    /* input policy processing */

    /* see if the prefix length valid */
    if (prefix->bitlen > 128) {
      trace (TR_PACKET, RIPNG->trace,
          "  x %s/%d metric %d (invalid prefix length)\n",
          prefix_toa (prefix), prefix->bitlen, attr->metric);
      return (-1);
    }

    /* see if the destination prefix valid */
    if (ipv6_multicast_addr (addr) ||
       (ipv6_compat_addr (addr) && prefix->bitlen > 0) ||
        ipv6_link_local_addr (addr)) {
      trace (TR_PACKET, RIPNG->trace,
          "  x %s/%d metric %d (non global)\n",
          prefix_toa (prefix), prefix->bitlen, attr->metric);
       return (-1);
    }

    if (attr->metric < 1 || attr->metric > RIPNG_INFINITY) {
      trace (TR_PACKET, RIPNG->trace,
          "  x %s/%d metric %d (invalid metric)\n",
          prefix_toa (prefix), prefix->bitlen, attr->metric);
       return (-1);
    }

    assert (attr->gateway);
    assert (attr->gateway->interface);
    /* check distribute-list for in */
    if (num = attr->gateway->interface->dlist_in[PROTO_RIPNG]) {
      if (apply_access_list (num, prefix) == 0) {
        trace (TR_PACKET, RIPNG->trace, "  x %s/%d metric %d (a-list %d)\n",
            prefix_toa (prefix), prefix->bitlen, attr->metric, num);
        return (-1);
      }
    }
    cost = attr->metric + attr->gateway->interface->metric_in;
  }
  else {
    /* output policy processing */
    if (ipv6_link_local_addr (addr) ||
        ipv6_multicast_addr (addr) ||
       (ipv6_compat_addr (addr) && prefix->bitlen > 0)) {
          trace (TR_PACKET, RIPNG->trace,
              "  x %s/%d metric %d (non global)\n",
              prefix_toa (prefix), prefix->bitlen, attr->metric);
      return (-1);
    }
    if (attr->type == PROTO_KERNEL) {  
      trace (TR_PACKET, RIPNG->trace,
          "  x %s/%d metric %d (kernel route)\n",
            prefix_toa (prefix), prefix->bitlen, attr->metric);
      return (-1);
    }
    /* split horizon (without poisoned reverse) */
    /* do not send to the same interface on which we got the route */
    if (attr->type == PROTO_RIPNG &&
           /* multicast capable and p-to-p interfaces */
           /* I'm not sure how NBMA will be supported */
           ((BIT_TEST (out->flags, IFF_MULTICAST) ||
             BIT_TEST (out->flags, IFF_POINTOPOINT)) &&
             attr->gateway->interface == out)) {
        /* No split horizon on NBMA which has not yet supported, though */
#ifdef RIPNG_POISONED_REVERSE
      trace (TR_PACKET, RIPNG->trace, 
          "  o %s/%d metric %d (posoned reverse)\n",
           prefix_toa (prefix), prefix->bitlen, RIPNG_INFINITY);
      return (RIPNG_INFINITY);
#else
      trace (TR_PACKET, RIPNG->trace,
          "  x %s/%d metric %d (split horion)\n",
           prefix_toa (prefix), prefix->bitlen, attr->metric);
      return (-1);
#endif
    }
    if (attr->type == PROTO_CONNECTED && attr->gateway->interface == out) {
      trace (TR_PACKET, RIPNG->trace, 
          "  x %s/%d metric %d (direct)\n",
           prefix_toa (prefix), prefix->bitlen, attr->metric);
      return (-1);
    }
    /* check distribute-list for out */
    if (num = out->dlist_out[PROTO_RIPNG]) {
      if (apply_access_list (num, prefix) == 0) {
        trace (TR_PACKET, RIPNG->trace, "  x %s/%d metric %d (a-list %d)\n",
            prefix_toa (prefix), prefix->bitlen, attr->metric, num);
        return (-1);
      }
    }
    cost = attr->metric + out->metric_out;
  }
    
  if (cost > RIPNG_INFINITY) 
    cost = RIPNG_INFINITY;

  return (cost);
}


/* ripng_change_update
 * announce all change routes to our peers
 */
void ripng_output_processing (int change) {
  interface_t *interface;
  ripng_route_t *route;

  /* announce routes */
  LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
    /* check if ripng is configured for this interface */
    if (interface->bit & RIPNG->interface_mask &&
        BIT_TEST (interface->flags, IFF_UP))
      ripng_send_routes (interface, NULL, change); 
  }
  /* clearing change flag */
  HASH_Iterate (RIPNG->hash, route) {
    route->flags &= ~(RT_RIPNG_CHANGE);
  }
}


static int flash_update_ongoing = 0;

void ripng_timer_update () {
  trace (TRACE, RIPNG->trace, "RIPNG timer (update) fired\n");
  if (flash_update_ongoing)
      flash_update_ongoing = 0; /* clear flash update */
  ripng_output_processing (0);
}

void ripng_flash_update () {
  trace (TRACE, RIPNG->trace, "RIPNG timer (flash update) fired\n");
  if (flash_update_ongoing) {
    ripng_output_processing (1);
    flash_update_ongoing = 0;
  }
}

void ripng_set_flash_update () {
  time_t t;

  if (flash_update_ongoing)
    return;
/* These values come from gated */
#define RIPNG_FLASH_DELAY_MIN 1
#define RIPNG_FLASH_DELAY_MAX 5
  t = RIPNG_FLASH_DELAY_MIN + 
	rand () % (RIPNG_FLASH_DELAY_MAX - RIPNG_FLASH_DELAY_MIN + 1);
  Timer_Set_Time (RIPNG->flash, t);
  flash_update_ongoing++;
}


/* 
 * Join the specified multicast address
 */
int ipv6_join_multicast (int sockfd, struct in6_addr *mc_addr) {
  struct ipv6_mreq mreq;
  interface_t *interface;
  int succeed = -1;

  memset (&mreq, 0, sizeof (mreq));
  memcpy (mreq.ipv6mr_multiaddr.s6_addr, mc_addr, 16);

  LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
    if (interface->primary6 == NULL) continue;
    if (!BIT_TEST (interface->flags, IFF_MULTICAST)) continue;

#ifdef HAVE_IPV6MR_IFINDEX
    mreq.ipv6mr_ifindex = interface->index;
#else
    memcpy (&mreq.ipv6mr_interface, 
       prefix_tochar (interface->primary6->prefix),
       sizeof (mreq.ipv6mr_interface));
#endif

    if (setsockopt (sockfd, IPPROTO_IPV6, IPV6_ADD_MEMBERSHIP, 
		    (void *)&mreq, sizeof (mreq)) >= 0) {
       succeed = 1;
    }
    else {
      trace (NORM, RIPNG->trace, 
	      "RIPNG setsockopt IPV6_ADD_MEMBERSHIP failed %s for %s : %s\n",
	     inet_ntop (AF_INET6, mc_addr, NULL, 0),
	     interface->name, strerror (errno));
    }
  }
  return (succeed);
}

#else /* NO HAVE_IPV6 */

/* just a null routine to give compilers something to stick in the
 * library so they don't complain about lack of table of contents
 */

void ripng_stub () {
  while (1) ;
}

#endif /* HAVE_IPV6 */
 
