/*
 * $Id: bgp_attr.c,v 1.13 1997/02/18 23:17:05 masaki Exp $
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <netdb.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <array.h>
#include <New.h>
#include <hash.h>
#include <mrt.h>
#include <io.h>
#include <lbgp.h>
#include <bgp_proto.h>
#include <aspath.h>

u_char *bgp_add_attributes (u_char *cp, short *size, 
			    bgp_attr_t *attr, short as);

bgp_attr_t *bgp_munge_attributes (int attrlen, u_char *cp);

int bgp_get_announced_prefixes (u_char *cp, u_char *cpend,
				bgp_attr_t **attr, 
				LINKED_LIST **ll_ann_prefixes);

int bgp_get_withdrawn_prefixes (u_char *cp, int *unreachlen, 
			       LINKED_LIST **ll_prefixes);

int bgp_process_update_packet (u_char *value, int length,
			       gateway_t *gateway,
			       bgp_attr_t **attr,
			       LINKED_LIST **ll_with_prefixes,
			       LINKED_LIST **ll_ann_prefixes);

short bgp_get_home_AS(bgp_attr_t *bgp_attr);

/* bgp_process_update_msg
 * Process an MRT BGP update message. Fill in pp_attr
 * and ll_ann_prefixes and ll_with_prefixes.
 */
int bgp_process_update_msg (u_char *value, int length,
			    gateway_t **gateway_to,
			    bgp_attr_t **pp_attr,
			    LINKED_LIST **ll_with_prefixes,
			    LINKED_LIST **ll_ann_prefixes)
{
   u_short AS;
   long addr;
   int ret;
   prefix_t *prefix;
   gateway_t *gateway_from;

   /* from */
   BGP_GET_SHORT (AS, value);
   BGP_GET_LONG (addr, value);
   prefix = New_Prefix (AF_INET, (u_char *) &addr, 32);
   gateway_from = (gateway_t *) add_gateway (prefix, AS, NULL);
   Deref_Prefix (prefix);

   /* to */
   BGP_GET_SHORT (AS, value);
   BGP_GET_LONG (addr, value);
   prefix = New_Prefix (AF_INET, (u_char *) &addr, 32);
   *gateway_to = (gateway_t *) add_gateway (prefix, AS, NULL);
   Deref_Prefix (prefix);

   ret = bgp_process_update_packet (value, length - 12,  gateway_from,
				    pp_attr,
				    ll_with_prefixes, ll_ann_prefixes);

   /* in case of withdraw only */
   if (*pp_attr == NULL) {
      *pp_attr = New (bgp_attr_t);
      (*pp_attr)->ref_count = 1;
      (*pp_attr)->gateway = gateway_from;
   }
   

   return (ret);
}

/* bgp_process_update
 * Given a pointer (and size) of BGP packet in memory, 
 * return linked lists of withdrawn prefixes, announced prefixes
 * and a BGP attribute structure
 */
int bgp_process_update_packet (u_char *value, int length,
				gateway_t *gateway,
				bgp_attr_t **pp_attr,
				LINKED_LIST **ll_with_prefixes,
				LINKED_LIST **ll_ann_prefixes)
{
   u_char *cp, *cpend;
   int offset;

   cp = value;
   cpend = value + length;

   bgp_get_withdrawn_prefixes (cp, &offset, ll_with_prefixes);

   cp += 2+offset; 
   
   bgp_get_announced_prefixes (cp, cpend, pp_attr, ll_ann_prefixes);

   if (*pp_attr)
      (*pp_attr)->gateway = gateway;

   return (1);
}


/* bgp_get_announced_prefixes
 */
int bgp_get_announced_prefixes (u_char *cp, u_char *cpend,
				bgp_attr_t **pp_attr, 
				LINKED_LIST **ll_ann_prefixes)
{
   u_char dest[4]; 
   int bitlen;
   int attrlen;
   u_char *attrp;
   int routelen;

   *ll_ann_prefixes = NULL;
   *pp_attr = NULL;

   BGP_GET_UPDATE(attrlen, attrp, cp);
   routelen = cpend - cp;

   if (routelen > 0) {
      *ll_ann_prefixes = LL_Create (LL_DestroyFunction, Deref_Prefix, NULL);
      *pp_attr = (bgp_attr_t *) bgp_munge_attributes (attrlen, attrp);

      while (cp < cpend) {
	 prefix_t *p_prefix;

	 BGP_GET_BITCOUNT(bitlen, cp);
	 BGP_GET_PREFIX(bitlen, dest, cp);

	 p_prefix = (prefix_t *) New_Prefix (AF_INET, dest, bitlen);

	 LL_Add ((*ll_ann_prefixes), p_prefix);
      }
   }

   return (1);
}





/*lbgp_get_withdrawn_prefixes
 */
int bgp_get_withdrawn_prefixes (u_char *cp, int *unreachlen, 
			       LINKED_LIST **ll_prefixes)
{
   u_char *cpendu;
   int bitlen;
   u_char dest[4]; 

   *ll_prefixes = NULL;
   *unreachlen = 0;		/* end of unreachable field */

   /* end of widthdraw prefixes portion of pdu */
   BGP_GET_V4UPDATE_UNREACH(*unreachlen, cp);
   cpendu = cp + *unreachlen;	

   
   /* Withdraws */
   if (*unreachlen > 0) {
      prefix_t *p_prefix;

      *ll_prefixes = LL_Create (LL_DestroyFunction, Deref_Prefix, NULL);

      while (cp < cpendu) {
	 BGP_GET_BITCOUNT(bitlen, cp);
	 BGP_GET_PREFIX(bitlen, dest, cp);

	 p_prefix = (prefix_t *) New_Prefix (AF_INET, dest, bitlen);
	 LL_Add ((*ll_prefixes), p_prefix);
      }
   }

   return (1);
}



dpa_t* New_DPA (short as, u_long value);


/* bgp_munge_attributes
 * Given a length and byte string from a BGP4 packet, return origin,
 * linked list of as_path_segments, and other attributes
 */
bgp_attr_t *bgp_munge_attributes (int attrlen, u_char *cp)
{
   bgp_attr_t *tmp;
   u_char *end;
   u_int flags;
   u_int code;
   u_int len = 0;
   u_long next_hop;
   u_short as;
   u_long value;
   
   end = cp + attrlen;
   tmp = New (bgp_attr_t);

   while (cp < end) {
      GET_PATH_ATTR(flags, code, len, cp);
      
      /*ASSERT ((code <= PA4_MAXTYPE));*/

      switch (code) {
       case PA4_TYPE_ORIGIN:
	 BGP_GET_BYTE(tmp->origin, cp);
	 break;
       case PA4_TYPE_ASPATH:
	 tmp->aspath = (aspath_t *) munge_aspath (len, &cp);
	 tmp->home_AS = bgp_get_home_AS (tmp);
	 break;
       case PA4_TYPE_COMMUNITY:
	 tmp->community = (community_t *) munge_community (len, &cp);
	 tmp->attribs |= O_COMMUNITY;
	 break;
      case PA4_TYPE_NEXTHOP:
	 BGP_GET_LONG ((long) next_hop, cp);
	 tmp->nexthop = New_Prefix (AF_INET, (u_char *) &next_hop, 32);
	 break;
      case PA4_TYPE_METRIC: /* multiexit */
	 BGP_GET_LONG ((long) tmp->multiexit, cp);
	 break;
      case PA4_TYPE_LOCALPREF:
	 tmp->attribs |= O_LOCALPREF;
	 BGP_GET_LONG ((long) tmp->local_pref, cp);
	 /*if ( tmp->local_pref > 100)
	    err_dump ("local pref > 100");*/
	 break;
      case PA4_TYPE_ATOMICAGG:
	 tmp->attribs |= O_ATOMICAGG;
	 tmp->atomic_agg = 1;
	 break;
      case PA4_TYPE_DPA:
	 tmp->attribs |= O_DPA;
	 BGP_GET_SHORT (as, cp);
	 BGP_GET_LONG (value, cp);
	 tmp->dpa = New_DPA (as, value);
	 break;
       default:
	 cp += len;
      }
   }

   return (tmp);
}



/* bgp_unmunge_path_attr
 */
u_char *bgp_unmunge_path_attr (bgp_attr_t* attr, short *len)
{
   static u_char buffer [BGPMAXPACKETSIZE];
   u_char *out;

   out = (u_char *) bgp_add_attributes (buffer, len, attr, 0);

   return (out);
}

 

/* bgp_create_update_msg
 * Create MRT update message. Take in pointer to
 * buffer of allocated memory, and attr and with and
 * ann prefixes. Return pointer to end of buffer
 * filled in with MRT update message. Set size to size
 * filled in.
 */
u_char *bgp_create_update_msg (short *size, 
			       bgp_attr_t *p_bgp_route_attr,
			       LINKED_LIST *ll_ann_prefixes,
			       LINKED_LIST *ll_with_prefixes)
{
   u_char *cp, *start_pdu;
   u_char *p_total_attrib_len;
   u_char *p_withdraw_len;
   short len;

   cp = start_pdu = NewArray (u_char, MAX_MSG_SIZE);

   /* from */
   if (p_bgp_route_attr->gateway == NULL) 
     cp += 6;
   else {
     BGP_PUT_SHORT (p_bgp_route_attr->gateway->AS, cp);
     memcpy (cp, (char *) prefix_tochar (p_bgp_route_attr->gateway->prefix), 4);
     cp +=4;
   }

   /* to */
   if (p_bgp_route_attr->gateway == NULL) 
     cp += 6;
   else {
     BGP_PUT_SHORT (p_bgp_route_attr->gateway->AS, cp);
     /* memcpy (p_bgp_route_attr->gateway->prefix->dest, cp, 4);*/
     cp +=4;
   }

   /* skip unfeasible route length for now */
   p_withdraw_len = (u_char *) cp;
   cp +=2; 

   len = 0;

   /* Now stick in WITHDRAW prefixes */
   if (ll_with_prefixes != NULL) {
      prefix_t *p_prefix;

      LL_Iterate (ll_with_prefixes, p_prefix) {
	 len += BGP_PREFIX_LEN(p_prefix->bitlen);
	 BGP_PUT_PREFIX (p_prefix->bitlen, prefix_tochar (p_prefix), cp);
      }
   }

   /* Now put in attribute length */
   BGP_PUT_SHORT (len, p_withdraw_len);

   /* skip total path attribute length for now */
   p_total_attrib_len = cp;
   cp += 2; 

   cp = (u_char *) bgp_add_attributes (cp, &len, p_bgp_route_attr, 0);

   /* Now put in attribute length */
   BGP_PUT_SHORT (len, p_total_attrib_len);

   /* Now stick in prefixes */
   {
      prefix_t *p_prefix;

      LL_Iterate (ll_ann_prefixes, p_prefix) {
	 BGP_PUT_PREFIX (p_prefix->bitlen, prefix_tochar (p_prefix), cp);
      }
   }

   *size = cp - start_pdu;
   return (start_pdu);
}

/* bgp_add_attributes 
 * Given a pointer to allocated memory (cp), fill in memory with BGP
 * attributes in BGP4 packet format. Set size to total length of 
 * attributes, and return pointer to end of attributes block
 * If as > 0, add as to end of aspath
 */
u_char *bgp_add_attributes (u_char *cp, short *size, 
			    bgp_attr_t *attr, short as)
{
   u_char *start = cp;
   int len = 0;
   
   /* origin */
   PATH_PUT_ATTR(PA_FLAG_TRANS, PA4_TYPE_ORIGIN, 1, cp);
   BGP_PUT_BYTE (attr->origin, cp);

   /* aspath */ 
   len =  aspath_length (attr->aspath, as);
   PATH_PUT_ATTR(PA_FLAG_TRANS, PA4_TYPE_ASPATH, len, cp);
   cp = (u_char *) unmunge_aspath (attr->aspath, cp, as);

   /* community */
   if (attr->community && BIT_TEST (attr->attribs, O_COMMUNITY)) {
     PATH_PUT_ATTR(PA_FLAG_OPT | PA_FLAG_TRANS, PA4_TYPE_COMMUNITY, 
						attr->community->len*4, cp);
     cp = (u_char *) unmunge_community (attr->community, cp);
   }

   /* next hop */
   if (attr->nexthop != NULL) {
     PATH_PUT_ATTR(PA_FLAG_TRANS, PA4_TYPE_NEXTHOP, 4, cp);
     BGP_PUT_LONG (prefix_tolong (attr->nexthop), cp);
   }

   /* non - mandatory attributes */
   if (BIT_TEST (attr->attribs, O_DPA)) {
     PATH_PUT_ATTR(PA_FLAG_OPT | PA_FLAG_TRANS, PA4_TYPE_DPA, 6, cp);
     BGP_PUT_SHORT (attr->dpa->AS, cp);
     BGP_PUT_LONG (attr->dpa->value, cp);
   }

   *size = cp - start;
   return (cp);
}


int attr_compare_fn (Route *a, Route *b) { return (a->attr - b->attr);}




/* bgp_print_attr
 * 
 */
void bgp_print_attr (bgp_attr_t *attr) 
{
   char tmp[MAXLINE];
   static char *s_origins[] = {"IGP", "EGP", "INCOMPLETE"};

   if (attr->gateway && (attr->gateway->AS > 0))
     printf ("\nFROM: %s", gateway_toa (tmp, attr->gateway));

   /* may be a withdraw attr with just peer info */
   if (attr->aspath == NULL)
     return;

   printf ("\nASPATH: %s", 
           aspath_toa (attr->aspath));
   /*ASSERT (((attr->origin >= 0) && (attr->origin <=2))); */
   printf ("\nORIGIN: %-12s",
	   s_origins[attr->origin]);
   printf("\nNEXT_HOP: %s", 
	  prefix_toa (attr->nexthop));
   printf("\nLOCAL_PREF: %d", (int) attr->local_pref);
   printf("\nMULTIEXIT: %d", (int) attr->multiexit);
}


/* bgp_trace_attr
 */
void bgp_trace_attr (char *mesg, trace_t *_trace,
		      bgp_attr_t *attr) 
{
   static char *s_origins[] = {"IGP", "EGP", "INCOMPLETE"};

   if (attr == NULL) return;

   if ((attr->gateway != NULL) && (attr->gateway->AS > 0))
     trace (TR_PACKET, _trace, "%s FROM: AS%d   %s\n", mesg,
	    attr->gateway->AS,
	    prefix_toa (attr->gateway->prefix));

   /* may be a withdraw attr with just peer info */
   if (attr->aspath == NULL)
      return;

   trace (TR_PACKET, _trace, "%s ASPATH: %s\n", mesg,
	  aspath_toa (attr->aspath));
   if (BIT_TEST (attr->attribs, O_COMMUNITY))
     trace(TR_PACKET, _trace, "%s COMMUNITY: %s\n", mesg,
	   community_toa (attr->community));
   /*ASSERT (((attr->origin >= 0) && (attr->origin <=2))); */ 
   trace (TR_PACKET, _trace, "%s ORIGIN: %-12s\n", mesg,
	  s_origins[attr->origin]);
   trace(TR_PACKET, _trace, "%s NEXT_HOP: %s\n", mesg,
	 prefix_toa (attr->nexthop));
   trace(TR_PACKET, _trace, "%s LOCAL_PREF: %d\n", mesg,
	 attr->local_pref);
   trace(TR_PACKET, _trace, "%s MULTIEXIT: %d\n", mesg,
	 attr->multiexit);
   if (BIT_TEST (attr->attribs, O_DPA))
     trace(TR_PACKET, _trace, "%s DPA: AS%d %d\n", mesg,
	   attr->dpa->AS, attr->dpa->value);
}

/* bgp_attr_toa
 */
char *bgp_attr_toa (bgp_attr_t *attr) 
{
   static char tmp[MAXLINE];
   static char *s_origins[] = {"IGP", "EGP", "INCOMPLETE"};

   sprintf (tmp, "%s|%s|%s|%d|%d", 
	    aspath_toa (attr->aspath),
	    s_origins[attr->origin],
	    prefix_toa (attr->nexthop),
	    (int) attr->local_pref,
	    (int) attr->multiexit);

   return (tmp);
}


/* bgp_delete_attr
 */
int bgp_delete_attr (bgp_attr_t *tmp) {
   if (tmp == NULL)
      return (-1);

   Delete_ASPATH (tmp->aspath);
   Delete (tmp);
   return (1);
}

/* bgp_compare_attr
 * return 1 if BGP attributes are the same 
 */
int bgp_compare_attr (bgp_attr_t *a1, bgp_attr_t *a2)
{
   if (a1->gateway->AS != a2->gateway->AS)
      return (-1);

   if (a1->attribs ^ a2->attribs)
      return (-1);

   if (a1->origin != a2->origin)
      return (-1);

   if (compare_aspaths (a1->aspath, a2->aspath) < 0)
      return (-1);

   return (1);
}


dpa_t* New_DPA (short as, u_long value) {
  dpa_t *tmp = New (dpa_t);

  tmp->AS = as;
  tmp->value = value;

  return (tmp);
}


