/*
 * $Id: aspath.c,v 1.6 1996/12/23 14:36:48 masaki Exp $
 */

#include <stdio.h>
#include <sys/types.h>
#include <stdlib.h>
#include <string.h>
#include <array.h>
#include <hash.h>
#include <bgp_proto.h>
#include <mrt.h>
#include <aspath.h>
#include <lbgp.h>

static void Delete_ASPATH_segment (aspath_segment_t *tmp);
static aspath_segment_t* New_ASPATH_segment (short type, int len, u_char **p_cp);


/* New_ASPATH 
 */
aspath_t* New_ASPATH () {
  aspath_t *aspath = New (aspath_t);

  aspath->ll_segments = 
    LL_Create (LL_DestroyFunction,  Delete_ASPATH_segment, NULL); 

  return (aspath);
}

/* Delete_ASPATH
 */
int Delete_ASPATH (aspath_t *aspath) {
  if (aspath == NULL) return (1);

  LL_Destroy (aspath->ll_segments);
  Delete (aspath);
  return (1);
}


/* munge_aspath
 * Given a bitlength and byte string from a BGP packet, 
 * return an aspath structure
 */
aspath_t* munge_aspath (int len, u_char **p_cp)
{
   u_char seg_type;
   short seg_len;
   aspath_segment_t *as_seg;
   aspath_t *aspath;
   int tmp_len = len;

   aspath = New_ASPATH ();

   while (tmp_len > 0) {
      BGP_GET_BYTE(seg_type, *p_cp);
      BGP_GET_BYTE(seg_len, *p_cp);

      /*if (seg_len > 100)
	 err_dump ("seg_len > 100");*/

      tmp_len -= 2;
      tmp_len -= 2*seg_len;

      as_seg = (aspath_segment_t *) 
	New_ASPATH_segment (seg_type, seg_len, p_cp);
      LL_Add (aspath->ll_segments, as_seg);
   }

   return (aspath);
}


/* unmunge_aspath
 * Given an aspath and a pointer to memory, fill in memory
 * with ASPATH and return pointer to end of ASPATH. If as > 0,
 * then add as to end of path
 *
 * NOTE: We assume enough memory has been allocated.
 */

u_char* unmunge_aspath (aspath_t *aspath, u_char *cp, short as) {
   aspath_segment_t *aspath_segment, *tail;

   if (aspath == NULL) {
     if (as <= 0) return (cp);
     
      BGP_PUT_BYTE (PA_PATH_SEQ, cp);
      BGP_PUT_BYTE (1, cp);
      BGP_PUT_SHORT (as, cp);
      return (cp);
   }

   tail = LL_GetTail (aspath->ll_segments);

   LL_Iterate (aspath->ll_segments, aspath_segment) {
     BGP_PUT_BYTE (aspath_segment->type, cp);

     /* add an AS to last segment if > zero */
     if ((as > 0) && (aspath_segment == tail)) {
       BGP_PUT_BYTE (aspath_segment->len + 1, cp);
       memcpy (cp, aspath_segment->as, aspath_segment->len*2);
       cp += aspath_segment->len*2;
       memcpy (cp, &as, 2);
       cp +=2;
     }
     /* just copy old ASpath */
     else {
       BGP_PUT_BYTE (aspath_segment->len, cp);
       memcpy (cp, aspath_segment->as, aspath_segment->len*2);
       cp += aspath_segment->len*2;
     }
   }

   return (cp);
}




/* New_ASPATH_segment
 * Create and fill in an aspath_segment_t structure. Len is number of
 * AS's -- NOT length of buffer
 */
static aspath_segment_t*
New_ASPATH_segment (short type, int len, u_char **p_cp)
{
   aspath_segment_t *as_seg = (aspath_segment_t *) New (aspath_segment_t);

   as_seg->type = type;
   as_seg->len = len;
   as_seg->as = (short *) malloc ( (len * 2));
   memcpy (as_seg->as, *p_cp, len*2);
   *p_cp += 2*len;
   return (as_seg);
}

/* Delete_ASPATH_segment 
 */
static void Delete_ASPATH_segment (aspath_segment_t *tmp) 
{ Delete (tmp->as); Delete (tmp);}




/* aspath_toa
 */
char *aspath_toa (aspath_t *aspath) {
   static u_char stmp[MAXLINE];
   register int i;
   u_char *tmp = stmp;
   aspath_segment_t *as_seg;

   memset (tmp, 0, MAXLINE);

   LL_Iterate (aspath->ll_segments, as_seg) {
      int seg_len = as_seg->len;
      u_char *cp = (u_char *) as_seg->as;
      
      /*if (tmp - stmp > 100)
	 err_dump ("> 100");*/

      if (as_seg->type == 0)
         sprintf(tmp++, "?");

      if (as_seg->type == PA_PATH_SET)
         sprintf(tmp++, "[");

      while (seg_len-- > 0 ) {
            i = (*cp++) << 8;
            i |= *cp++;
            sprintf (tmp, "%d ", i);
            
	    /*if (i > BGP_AS_HIGH)
	       err_dump ("> 65535");*/
	    
            if (i < 10) tmp += 2;
            else if (i < 100) tmp +=3;
            else if (i < 1000) tmp +=4;
            else tmp +=5;
         }

      if (as_seg->type == PA_PATH_SET)
         sprintf(tmp++, "]");
   }

   return (stmp);
}




/* bgp_get_home_AS
 * Return HOMEAS (last AS in AS segment list) when 
 * given A bgp_attr_t structure
 */
short bgp_get_home_AS(bgp_attr_t *bgp_attr) {
   short home_AS;
   aspath_segment_t *segment;
   
   if (bgp_attr == NULL) return (0);
   

   segment = (aspath_segment_t *) LL_GetTail (bgp_attr->aspath->ll_segments);

   if (segment == NULL)
     return (0);

   home_AS = segment->as[segment->len - 1];

   return (home_AS);
}




/* aspath_length
 * Given an aspath (really a linked list of AS_PATH_SEG 
 * structs), calculate total length of aspath attribute for
 * inclusion in BGP packet
 */
int aspath_length (aspath_t *aspath, short AS) {
   aspath_segment_t *segment;
   int l = 0;

   if (aspath == NULL) {
     if (AS <= 0) {return (0);}
     return (4);
   }
   
   LL_Iterate (aspath->ll_segments, segment) {
     l += 2; /* segmentq type field, segment length field */
     l += 2 * segment->len;
   }

   if (AS > 0) { l+=2;}

   return (l);
}


/* compare_aspaths
 * should have speaical code for sets! But  noone uses sets 
 */
int compare_aspaths (aspath_t* aspath1, aspath_t *aspath2)
{
   aspath_segment_t *seg1, *seg2;
   
   seg1 = LL_ContGetHead (aspath1->ll_segments);
   seg2 = LL_ContGetHead (aspath2->ll_segments);

   while ((seg1 != NULL) && (seg2 != NULL)) {
      if (seg1->type != seg2->type)
	 return (-1);
      if (seg1->len != seg2->len)
	 return (-1);
      if (memcmp (seg2->as, seg1->as, seg1->len *2))
	 return (-1);

      seg1 = (aspath_segment_t *) LL_ContGetNext (aspath1->ll_segments, seg1);
      seg2 = (aspath_segment_t *) LL_ContGetNext (aspath2->ll_segments, seg2);

      /* both NULL */
      if (seg1 == seg2)
	 return (1);
   }

   /* one is NULL and one is not */
   return (-1);
}

u_int aspath_hash_fn (aspath_t *aspath, u_int size) {
  u_int tmp, i;
  aspath_segment_t* as_seg;
  
  tmp = 0;
  if ((aspath == NULL) || (aspath->ll_segments == NULL)) {return (0);}

  LL_Iterate (aspath->ll_segments, as_seg) {
      int seg_len = as_seg->len;
      u_char *cp = (u_char *) as_seg->as;
      
      while (seg_len-- > 0 ) {
            i = (*cp++) << 8;
            i |= *cp++;
      
	    tmp += i;
      }
  }

  tmp = tmp % size;
  return (tmp);
}



aspath_t *aspth_from_string (char *path) {
  aspath_t *tmp;
  char *cp = path;
  short as, flag;
  u_char segment[PA_PATH_MAXSEGLEN * 2];
  u_char *pseg = segment;
  u_char *tmp_seg;
  u_int length;

  tmp = New_ASPATH();
  /* printf ("%s\n", path);*/

  as = 0;
  flag = 0;
  while (*cp != '\0') {
    if (isdigit (*cp)) {
      /*printf( "%c\n", *cp);*/
      if (flag == 0) as = (*cp - 48);
      else { 
	as = as * 10;
	as += (*cp - 48);
      }
	flag = 1;
    }
    else {
      if (flag == 1) {
	/* printf ("Adding as %d\n", as);*/
	BGP_PUT_SHORT (as, pseg);
      }
      as = 0;
      flag = 0;
    }
    cp++;
  }
  if (flag == 1) {
    /* printf ("Adding as %d\n", as);*/
    BGP_PUT_SHORT (as, pseg);
  }

  length = pseg - segment;
  pseg = segment;
  tmp_seg = (u_char *)New_ASPATH_segment (2, length/2, &pseg);
  LL_Add (tmp->ll_segments, tmp_seg);

  return (tmp);
}


/*
 * The following is for supporting community. -- masaki
 */

static community_t* New_community (int len, u_long *value);

/* munge_community
 * Given a bytelength and byte string from a BGP packet, 
 * return an community structure
 */
community_t* munge_community (int plen, u_char **p_cp)
{
   community_t *tmp = NULL;
   u_long value;
   u_long community[PA_COMMUNITY_MAXLEN];
   int len = 0;

   while (plen > 0) {
      BGP_GET_LONG(value, *p_cp);
      community[len++] = value;
      plen -= 4;
   }
   assert (plen == 0); /* check multiple of 4 bytes */
   if (len > 0) {
      tmp = New_community(len, community);
   }

   return (tmp);
}


/* unmunge_community
 * Given an community and a pointer to memory, fill in memory
 * with community values and return pointer to end of memory. 
 *
 * NOTE: We assume enough memory has been allocated.
 */

u_char* unmunge_community (community_t *community, u_char *cp) {
   int i;

   if (community && community->len) {
      for (i = 0; i<community->len; i++)
	  BGP_PUT_LONG (community->value[i], cp);
   }

   return (cp);
}


/* New_community
 * Create and fill in an community_t structure. Len is number of
 * community values -- NOT length of buffer
 */
static community_t*
New_community (int len, u_long *value)
{
   community_t *community = (community_t *) New (community_t);

   community->len = len;
   community->value = (u_long *) malloc ((len * sizeof (*community->value)));
   memcpy (community->value, value, len*sizeof (*community->value));
   return (community);
}

/* Delete_community 
 */
static void Delete_community (community_t *tmp) 
{ Delete (tmp->value); Delete (tmp);}


community_t *community_from_string (char *cp) {
  community_t *tmp = NULL;
  u_long community[PA_COMMUNITY_MAXLEN];
  int low = 0, high = 0; /* low should be int for case of no ':' */
  int len = 0, flag = 0;

  while (*cp != '\0') {
    if (isdigit (*cp)) {
	low *= 10;
	low += (*cp - '0');
	flag = 1;
    }
    else if (*cp == ':') {
        high = low;
	low = 0;
	flag = 1; /* single ':' means zero value */
    }
    else {
      if (flag) {
	community[len++] = (high << 16) + low; /* low&0xffff ? */
        low = high = flag = 0;
      }
    }
    cp++;
  }
  if (flag) {
    community[len++] = (high << 16) + low;
  }

  if (len > 0) {
    tmp = New_community(len, community);
  }

  return (tmp);
}


/* community_toa
 */
char *community_toa (community_t *community) {
   static u_char stmp[MAXLINE];
   u_char *tmp = stmp;
   register int i;

   *tmp = '\0';
   if (community && community->len) {
      for (i = 0; i<community->len; i++) {
	 if (tmp + 3 /* length of "..." */ >= stmp + sizeof (stmp)) {
	    sprintf(tmp, "...");
	    tmp += strlen(tmp);
	    break;
	 }
         sprintf(tmp, "%d:%d ", (community->value[i]>>16)&0xffff,
         			 community->value[i]&0xffff);
	 tmp += strlen(tmp);
      }
   }
   return (stmp);
}

