/* 
 * $Id: simple_bgp.c,v 1.8 1996/06/20 20:20:01 labovit Exp $ 
 */

/* sbgp: Simple BGP4 speaker and listner. Provides way to dump
 * BGP routing stream to other MRT tools, as well as to save stream
 * to file. Can also inject BGP information into peering with real
 * router */

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include <mrt.h>
#include <select.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <timer.h>
#include <bgp.h>
#include <io.h>
#include <lbgp.h>

/*
 * GLOBALS
 */
bgp_t *BGP;
io_t *IO;
trace_t *default_trace = NULL;
mtimer_t *timer;
long last = 0; /* last time we sent a packet */

int init_gateways ();
int init_schedules ();

static void sbgp_process_update (bgp_peer_t *peer, u_char *cp, int len);
static int sbgp_peer_down (bgp_peer_t *peer);
static int sbgp_peer_established (bgp_peer_t *peer);
static int sbgp_process_input ();
static int process_command_line_args (int argc, char *argv[]);
void io_timer (mtimer_t* timer);

/* main
 */
int main (int argc, char *argv[])
{
   bgp_peer_t *peer;

   default_trace = New_Trace ();
   set_trace (default_trace, TRACE_FLAGS, TR_ALL, 
	      TRACE_LOGFILE, "stdout",
	      NULL);

   init_mrt (default_trace);
   init_BGP (default_trace);

   IO = (io_t *) New_IO (default_trace);
   timer = (mtimer_t *) New_Timer (io_timer, 0, "IO timer", NULL);

   set_BGP (BGP_PEER_ESTABLISHED_FN, sbgp_peer_established,
	    BGP_RECV_UPDATE_FN, sbgp_process_update,
	    NULL);

   process_command_line_args (argc, argv);

   start_bgp ();

#ifndef HAVE_LIBPTHREAD
   while (1) {
     process_all_schedules();
     mrt_select ();
   }
#endif /* HAVE_LIBPTHREAD */
   
   /* loop */
   while (1); 

}



/* sbgp_process_update
 *  Just write update as MRT output mesage
 */
void sbgp_process_update (bgp_peer_t *peer, u_char *buf, int len) {
  char tmp[MAX_MSG_SIZE], *cp;
  
  cp = tmp;

  /* from */
  BGP_PUT_SHORT (peer->gateway->AS, cp);
  BGP_PUT_LONG (prefix_tolong (peer->gateway->prefix), cp);

   /* to */
   BGP_PUT_SHORT (0, cp);
   BGP_PUT_LONG (0, cp);

   memcpy (cp, buf, len);
   len += 12;

   io_write (IO, 0, MSG_PROTOCOL_BGP, MSG_BGP_UPDATE, len, tmp);
}



/* sbgp_peer_down
 */
int sbgp_peer_down (bgp_peer_t *peer) {
   Timer_Turn_ON (peer->timer_Start);
   select_delete_fd (IO->in.fd); /* this should really be in IO routines ... */
   return (1);
}


/* sbgp_peer_established
 */
int sbgp_peer_established (bgp_peer_t *peer) {
  io_set_notify (IO, 1, sbgp_process_input);
  return (1);
}


/* sbgp_process_input
 * called by IO. Read a packet and send it on its way via BGP
 */
int sbgp_process_input (mrt_msg_t *msg)
{
  bgp_peer_t *peer;
  char *buf;
  
  /*while ((msg = (mrt_msg_t *) io_read (IO)) != NULL) {*/
  if ((msg = (mrt_msg_t *) io_read (IO)) != NULL) {

    LL_Iterate (BGP->ll_bgp_peers, peer) {
      u_char *cp = msg->value;
      int length = msg->length;

      if (peer->state != BGPSTATE_ESTABLISHED) { continue;}
      
      length -= 12;
      cp += 12;

      buf = NewArray (u_char, length);
      memcpy (buf, cp, length);

      bgp_send_update (peer, length, buf);
    }
  }
  
  if (last > 0) {
    /* another packet at same second */
     if (msg->tstamp == last) {
       select_enable_fd (IO->in.fd);
     }
     else {
       Timer_Set_Time (timer, msg->tstamp - last);
       trace (NORM, BGP->trace, "IO waiting for %d seconds\n", 
	      msg->tstamp - last);
     }
  } else {
    select_enable_fd (IO->in.fd);
  }
  last = msg->tstamp;

  /* free memory here */
  Delete (msg);

  return (1);
}





/* process_command_line_args
 */
int process_command_line_args (int argc, char *argv[])
{
   char c;
   extern char *optarg;	/* getopt stuff */
   extern int optind;	/* getopt stuff */
   prefix_t *prefix;
   char *usage ="Usage: sbgp [-av] [-i binary_data_in_file] [-o binary_data_out_file]\n\t    [-l log_file] [-f config_file] [-c port] [-d port]\n\t    [my AS] [peer_IP peer_AS]...\n";
   bgp_peer_t *peer;
   int errors = 0;
   
   while ((c = getopt(argc, argv, "r:w:hvo:i:l:c:d:")) != -1)
      switch (c) {
      case 'd': /* port for BGP daemon to listen on */
	 BGP->lport = (int) atol(optarg);
	 break;
      case 'c': /* port for BGP to connect to  */
	 BGP->cport = (int) atol(optarg); 
	 break;
      case 'o':	/* out data file name */
	if (io_set(IO, IO_OUTFILE, (char *) (optarg), NULL) < 0) {
	  printf ("\nError setting outfile %s\n", optarg);
	  errors++;
	} 
	break;
      case 'w':	/* out message key name */
	if (io_set(IO, IO_OUTMSGQ, (char *) (optarg), NULL) < 0) {
	  printf ("\nError setting outfile %s\n", optarg);
	  errors++;
	} 
	break;
      case 'i':	/* in data file name */
	if (io_set(IO, IO_INFILE, (char *) (optarg), NULL) < 0) {
	  printf ("\nError setting infile %s\n", optarg);
	  exit (0);
	}
	 break;
      case 'r':	/* in message key */
	if (io_set(IO, IO_INMSGQ, (char *) (optarg), NULL) < 0) {
	  printf ("\nError setting inmsg %s\n", optarg);
	  errors++;
	}
	 break;
      case 'l':	/* log file name */
	/*Set_trace_t (BGP->trace, TRACE_LOGFILE, optarg, NULL);*/
	 break;
      case 'v': /* verbose */
	 BGP->trace = New_Trace ();
	 set_trace (BGP->trace, TRACE_FLAGS, TR_ALL, 
		    TRACE_LOGFILE, "stdout",
		    NULL);
	 break;
      case 'f': /* config file */
	 fprintf (stdout, "\nConfig file option not yet supported");
	 errors++;
	 break;
      case 'h':
      default:
	errors++;
	break;
      }

   if (errors) {
     fprintf(stderr, usage);
     printf ("\nMRT version (%s) compiled on %s\n\n",
	     MRT_VERSION, __DATE__);
     exit (0);
   }

   /* set my AS number */
   if ((optind < argc) && (!strncasecmp ("AS", argv[optind], 2))) 
      BGP->my_as = (short) atol (2 + argv[optind++]);


   /* add peers: peer_ip [peer_AS] */
   if ((optind+1 < argc) && (!strncasecmp ("AS", argv[optind+1], 2))) {
     prefix = ascii2prefix (AF_INET, (char *) argv[optind++]);
     if (prefix == NULL) {
       printf ("Unknown host %s\n",argv[optind -1]);
       exit (0);
     }

     Add_BGP_Peer (prefix,
		   (short) atol (2 + argv[optind++]),
		   BGP->trace);
   }

   return (1);
}





/* io_timer
 * renable timer after x seconds so we can recreate time-based events
 */
void io_timer (mtimer_t* timer) {
  select_enable_fd (IO->in.fd);
}
