/*
 * $Id: mrt.c,v 1.27 1997/02/18 23:17:33 masaki Exp $
 */

#include <stdio.h>
#include <string.h>
#include <mrt.h>
#include <trace.h>
#include <interface.h>
#include <rip.h>
#ifdef HAVE_IPV6
#include <ripng.h>
#endif /* HAVE_IPV6 */
#include <bgp.h>
#include <signal.h>

#include "rib.h"
#include "config.h"

#ifndef SETPGRP_VOID
#include <fcntl.h>
#include <sys/termios.h>
#endif

rip_t *RIP;
#ifdef HAVE_IPV6
ripng_t *RIPNG;
rib_t *RIBv6;
#endif /* HAVE_IPV6 */
rib_t *RIB;
bgp_t *BGP;
int kernel_install_flag = 1;
int IPV4 = 0;
int IPV6 = 0;

void rip_recv_update ();
void start_config ();
void show_rip ();
void check_passwd ();
int show_ip_routes ();
int show_rip_routing_table();
int show_bgp_routing_table();
int rip_debug ();
int show_interfaces ();
#ifdef HAVE_IPV6
void show_ripng ();
int show_ipv6_routes ();
int show_ripng_routing_table ();
#endif /* HAVE_IPV6 */

int show_threads ();
int show_view ();
int rib_add_rip_route (prefix_t *prefix, rip_attr_t *attr);
int rib_add_rib_route (prefix_t *prefix, bgp_attr_t *attr);
#ifdef HAVE_IPV6
int rib_update_ripng_route (int cmd, prefix_t *prefix, ripng_attr_t *attr);
#endif HAVE_IPV6
int process_bgp_update ();
int show_bgp ();
int kill_peer ();
static int add_interfaces_to_rib ();
static int add_static_to_rib ();
void add_kernel_route (int family, char *dest, char *nexthop, int masklen);
int init_rib (trace_t *trace);
static void daemonize();


main (int argc, char *argv[]) {
  char c, *p, *name = argv[0];
  extern char *optarg;	/* getopt stuff */
  extern int optind;	/* getopt stuff */
  int errors = 0;
  char tmp[MAXLINE];
  int daemon = 0;
  char *port = "mrt";
  time_t start, now;

  char *usage ="Usage: %s [-f config_file] [-p uii_port ] [-v] [-n]\n";
  char *config_file = NULL;
  trace_t *default_trace;

  if (p = strrchr (name, '/')) {
    name = p+1;
  }
  if (strcasecmp (name, "mrtd") == 0) {
    config_file = "/etc/mrtd.conf"; /* unix convension */
    daemon = 1;
  }
#ifdef MCHECK
  mcheck(0);
#endif
  default_trace = New_Trace ();

  while ((c = getopt (argc, argv, "hnvf:p:")) != -1) 
     switch (c) {
     case 'v': /* verbose */
       set_trace (default_trace, TRACE_FLAGS, TR_ALL, 
		  TRACE_LOGFILE, "stdout",
		  NULL);
	daemon = 0;
       break;
     case 'f': /* config file */
       config_file = optarg;
       break;
     case 'n': /* no kernel installation */
       kernel_install_flag = 0;
	daemon = 0;
       break;
     case 'p': /* uii port number */
       port = optarg;
       break;
     case 'h':
     default:
       errors++;
       break;
     }


   if (errors) {
     fprintf(stderr, usage, name);
     printf ("\nMRT %s compiled on %s\n\n",
	     MRT_VERSION, __DATE__);
     exit (1);
   }
  
   if (daemon) {
        if (getuid()) {
            fprintf(stderr, "mrt: must be root\n");
            exit(1);
        }
   }
   init_trace (name, daemon);

   init_mrt (default_trace);
   init_config (default_trace);

   init_rip (default_trace);
   /* install the route in the kernel */
   RIP->add_call_fn = rib_add_rip_route;

   init_BGP (default_trace);
   set_BGP (BGP_RECV_UPDATE_FN, process_bgp_update, NULL);

   set_uii (UII, UII_PROMPT, 0, "password> ");
   set_uii (UII, UII_PROMPT, 1, "MRT-1.3A> ");

#ifdef HAVE_IPV6 
   init_ripng (default_trace);
   set_ripng (RIPNG_RT_UPDATE_FN, rib_update_ripng_route, NULL);
#endif /* HAVE_IPV6 */

   /* read information on all interfaces from the kernel */
#ifdef notdef
   init_interfaces (default_trace);
#endif
   init_rib (default_trace);

   /* default trace flag should be set by here */

  /*
   * read configuration here
   */
   config_from_file (config_file);

   if (daemon) {
      /*
       * Now going into daemon mode right after reading configuration
       */
       daemonize();
   }

   if (BIT_TEST (MRT->protocols, (1 << PROTO_RIP))) IPV4++;
   if (BIT_TEST (MRT->protocols, (1 << PROTO_BGP))) IPV4++;
   if (BIT_TEST (MRT->protocols, (1 << PROTO_RIPNG))) IPV6++;

#ifdef __linux__
   kernel_init ();
#endif
   kernel_read_rt_table (add_kernel_route); /* add kernel route first */

/* CONFIGURATION DEPENDENT INITIALIZATION PART */

   add_interfaces_to_rib (); /* override interface route next */
   add_static_to_rib (); /* overide static route last */

   /* distribute ipv4 rib protocols into other protocols */
   if (IPV4) redistribute ();

   /* distribute ipv6 rib protocols into other protocols */
#ifdef HAVE_IPV6
   if (IPV6) redistribute_ipv6 ();
#endif /* HAVE_IPV6 */

   uii_add_command (0, "", (void *) check_passwd);
   uii_add_command (1, "config", (void *) start_config);
   uii_add_command (1, "show rip", (void *) show_rip);
   uii_add_command (1, "show bgp", (void *) show_bgp);
   uii_add_command (1, "show ip", (void *) show_ip_routes);
#ifdef HAVE_IPV6 
   uii_add_command (1, "show ripng", (void *) show_ripng);
   uii_add_command (1, "show ripng routes", (void *) show_ripng_routing_table);
   uii_add_command (1, "show ipv6", (void *) show_ipv6_routes);
#endif /* HAVE_IPV6 */
   uii_add_command (1, "show view", (void *) show_view);
   uii_add_command (1, "show interfaces", (void *) show_interfaces);
   uii_add_command (1, "show rip routes", (void *) show_rip_routing_table);
   uii_add_command (1, "show bgp routes", (void *) show_bgp_routing_table);
   uii_add_command (1, "clear bgp", (void *) kill_peer);

if (BIT_TEST (MRT->protocols, (1 << PROTO_RIP)))
   start_rip ();
#ifdef HAVE_IPV6 
if (BIT_TEST (MRT->protocols, (1 << PROTO_RIPNG)))
   start_ripng ();
#endif /* HAVE_IPV6 */
if (BIT_TEST (MRT->protocols, (1 << PROTO_BGP)))
   start_bgp ();

   set_uii (UII, UII_PROMPT, UII_UNPREV, "Password> ");
   set_uii (UII, UII_PROMPT, UII_NORMAL, "MRT> ");
   set_uii (UII, UII_PROMPT, UII_CONFIG, "Config> ");
   set_uii (UII, UII_PROMPT, UII_CONFIG_ROUTER, "Router> ");
   set_uii (UII, UII_PROMPT, UII_CONFIG_INTERFACE, "Interface> ");
#ifdef notdef
   sprintf (tmp, "MRT %s> ", MRT_VERSION);
   set_uii (UII, UII_PROMPT, 1, tmp);
#endif
   listen_uii2 (port);

   time (&start);
   while (1) {
#ifndef HAVE_LIBPTHREAD
     while (process_all_schedules()); /* process all first */
     mrt_select ();
#else
     sleep (1);
#endif /* THREADS */
     if (start) {
       time (&now);
#define KERNEL_ROUTE_TIMEOUT 120
       if (now - start > KERNEL_ROUTE_TIMEOUT) {
	    if (IPV4) {
   	   	rib_open (RIB);
   	   	rib_kernel_route_sweep (RIB);
   	   	rib_close (RIB);
	    }
#ifdef HAVE_IPV6
	    if (IPV6) {
   	   	rib_open (RIBv6);
   	   	rib_kernel_route_sweep (RIBv6);
   	    	rib_close (RIBv6);
	    }
#endif /* HAVE_IPV6 */
	   start = 0;
       }
     }
   }
}


int rib_add_rip_route (prefix_t *prefix, rip_attr_t *attr) {
  rib_open (RIB);
  rib_add_route (RIB, prefix, (generic_attr_t *) attr, RIP_PREF, 0);
  rib_close (RIB);
  return (1);
}

int rib_add_bgp_route (prefix_t *prefix, bgp_attr_t *attr) {
  rib_open (RIB);
  rib_add_route (RIB, prefix, (generic_attr_t *) attr, BGP_PREF, 0);
  rib_close (RIB);
  return (1);
}


#ifdef HAVE_IPV6 
/* rib_update_ripng_route
 * 0 = add, 1 = change, 2= delete 
 */
int rib_update_ripng_route (int cmd, prefix_t *prefix, ripng_attr_t *attr) {

  rib_open (RIBv6);
  switch (cmd) {
    case KERNEL_UPDATE_ADD:
      rib_add_route (RIBv6, prefix, (generic_attr_t *) attr, RIPNG_PREF, 0);
      break;
    case KERNEL_UPDATE_DEL:
      rib_del_route (RIBv6, prefix, (generic_attr_t *) attr);
      break;
    case KERNEL_UPDATE_HOLD:
      rib_hold_route (RIBv6, prefix, (generic_attr_t *) attr, HOLD_PREF);
      break;
    case KERNEL_UPDATE_UNHOLD:
      rib_hold_route (RIBv6, prefix, (generic_attr_t *) attr, RIPNG_PREF);
      break;
    default:
      assert (0);
      break;
  }
  rib_close (RIBv6);
  return (1);
}
#endif /* HAVE_IPV6 */

int show_ip_routes (uii_connection_t *uii) {
  show_rib_routes (uii, RIB);
}

#ifdef HAVE_IPV6 
int show_ipv6_routes (uii_connection_t *uii) {
  show_rib_routes (uii, RIBv6);
}
#endif /* HAVE_IPV6 */

int process_bgp_update (bgp_peer_t *peer, u_char *cp, int length) {
  bgp_attr_t *tmp_attr, *attr;
  LINKED_LIST *ll_with_prefixes;
  LINKED_LIST *ll_ann_prefixes;
  prefix_t *prefix;
  int i;

  bgp_process_update_packet (cp, length, peer->gateway, &tmp_attr,
			     &ll_with_prefixes, &ll_ann_prefixes);

  attr = (bgp_attr_t *) bgp_get_attr (tmp_attr);

  bgp_trace_attr ("BGP", peer->trace, attr);
  trace_prefix_list ("BGP recv", peer->trace, ll_ann_prefixes);

  /* iterate through views adding the announce/withdraws */
  for (i=0; i<32; i++) {
    if (BGP->views[i] == NULL) {continue;}

    view_open (BGP->views[i]);
  
    if (ll_ann_prefixes) {
      LL_Iterate (ll_ann_prefixes, prefix) {
	/* apply policy */
	bgp_add_route (BGP->views[i], prefix, attr, 0);
      }
    }

    /* process change list -- send out updates to peers */
    bgp_process_changes (BGP->views[i]);

    view_close (BGP->views[i]);
  }

  return (1);
}

int kill_peer (uii_connection_t *uii)
{
  prefix_t *prefix;
  int asl;
  short as;
  gateway_t *gateway;
  bgp_peer_t *peer;

  if (parse_line (uii->cp, "%p %d", &prefix, &asl) != 2) {
    uii_send_data (uii, "ERROR: close peer d.d.d.d as\r\n");
    return (-1);
  }

  as = (short) asl;
  if ((gateway = find_gateway (prefix, as)) == NULL) {return (-1);}

  peer = find_bgp_peer (gateway);
  /*bgp_peer_dead (peer);
  view_delete_peer (BGP->view, gateway);*/
  bgp_sm_process_event (peer, peer->state, BGPEVENT_ERROR);
  return (1);
}


/* redistribute
 */
int redistribute () {
  route_head_t *head = NULL;

  rib_open (RIB);
  view_open (BGP->view);

  HASH_Iterate (RIB->hash, head) {
    /* redistribute static */
    if (head->active == NULL) printf("NULL ACTIVE: %s\n", prefix_toa (head->prefix));
    if (head->active == NULL) continue;
    if (head->active->attr->type == PROTO_STATIC) {
      if (BIT_TEST (MRT->rdist_static, PROTO_BGP)) {
        bgp_attr_t *attr = New (bgp_attr_t);
        attr->type = head->active->attr->type;
        attr->gateway = head->active->attr->gateway;
        attr->nexthop = Ref_Prefix (head->active->attr->nexthop);
        attr->ref_count = 1;
        bgp_add_route (BGP->view, head->prefix, attr, 0);
      }
    }
  }
  view_close (BGP->view);

  HASH_Iterate (RIB->hash, head) {
    /* redistribute static */
    if (head->active == NULL) printf("NULL ACTIVE: %s\n", prefix_toa (head->prefix));
    if (head->active == NULL) continue;
    if ((head->active->attr->type == PROTO_STATIC &&
          BIT_TEST (MRT->rdist_static, PROTO_RIP)) ||
/*        head->active->attr->type == PROTO_KERNEL || */
       (head->active->attr->type == PROTO_CONNECTED &&
        /* only advertise route for interface enabled for this proto */
             BIT_TEST (RIP->interface_mask, 
                  head->active->attr->gateway->interface->bit))) {
      rip_attr_t *attr = New_Rip_Attr (1);
      attr->type = head->active->attr->type;
      attr->nexthop = Ref_Prefix (head->active->attr->nexthop);
      attr->gateway = head->active->attr->gateway;
      attr->metric = 1; 
      attr->tag = 0; 
      attr->ref_count = 1;  /* should be done in New_Rip_Attr */
      New_Rip_Route (head->prefix, attr);
      /* Deref_Ripn_Attr (attr) or Deref_Generic_Attr (attr) */
    }
  }

  rib_close (RIB);  
}

/* redistribute_ipv6
 */
#ifdef HAVE_IPV6
int redistribute_ipv6 () {
  route_head_t *head = NULL;
  interface_t *interface;

  rib_open (RIBv6);

  HASH_Iterate (RIBv6->hash, head) {
    /* redistribute static */
    if (head->active == NULL) continue;

    if ((head->active->attr->type == PROTO_STATIC &&
          BIT_TEST (MRT->rdist_static, PROTO_RIPNG)) ||
	/* currently PROTO_KERNEL is imported 
	   into RIPNG to delete it by the RIPNG timer.
	   it should be done by the RIB timer which is not yet implemented */
/*        head->active->attr->type == PROTO_KERNEL || */
       (head->active->attr->type == PROTO_CONNECTED &&
        /* only advertise route for interface enabled for this proto */
             BIT_TEST (RIPNG->interface_mask, 
                  head->active->attr->gateway->interface->bit))) {

      ripng_attr_t *attr;
      struct in6_addr *addr6 = 
	prefix_toaddr6 (head->prefix);

    if (ipv6_multicast_addr (addr6) ||
       (ipv6_compat_addr (addr6) && head->prefix->bitlen > 0) ||
        ipv6_link_local_addr (addr6)) continue;

      attr = New_Ripng_Attr (1);
      attr->type = head->active->attr->type;
      attr->nexthop = Ref_Prefix (head->active->attr->nexthop);
      attr->gateway = head->active->attr->gateway;
      attr->tag = 0; 
      New_Ripng_Route (head->prefix, attr);
      Deref_Ripng_Attr (attr);
    }
  }

  rib_close (RIBv6);  

}
#endif /* HAVE_IPV6 */


/* add_interfaces
 * add all of the interfaces to thge appropriate ribs
 */
static int add_interfaces_to_rib () {
  interface_t *interface;
  ll_addr_t *ll_addr;
  prefix_t *network, *gw;
  generic_attr_t *attr;

  /* ipv4 */
  rib_open (RIB);
  LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {

    if (interface->ll_addr == NULL) continue;
    if (!BIT_TEST (interface->flags, IFF_UP)) continue;
    /* i'm not sure but for now */
    if (BIT_TEST (interface->flags, IFF_POINTOPOINT)) continue;
    LL_Iterate (interface->ll_addr, ll_addr) {
      if (ll_addr->prefix == NULL) continue;
      if (ll_addr->prefix->family != AF_INET) continue;

      network = New_Prefix (AF_INET,
	prefix_tochar (ll_addr->prefix), ll_addr->prefix->bitlen);
      attr = New (generic_attr_t);
      attr->type = PROTO_CONNECTED;
      /* myself for split horizen */
      gw = copy_prefix (ll_addr->prefix);
      gw->bitlen = 32;
      attr->gateway = add_gateway (gw, 0, interface); 
      attr->nexthop = gw;
      /* masking may not be needed */
      netmasking (network->family, prefix_tochar (network), network->bitlen);

      trace (NORM, MRT->trace, "Adding a route for %s\n", interface->name);
      rib_add_route (RIB, network, attr, CONNECTED_PREF, 0);
    }
  }
  rib_close (RIB);

#ifdef HAVE_IPV6
  /* ipv6 */
  rib_open (RIBv6);
  LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
    if (interface->ll_addr == NULL) continue;
    if (!BIT_TEST (interface->flags, IFF_UP)) continue;

    LL_Iterate (interface->ll_addr, ll_addr) {

      if (ll_addr->prefix == NULL) continue;
      if (ll_addr->prefix->family != AF_INET6) continue;

      if (BIT_TEST (interface->flags, IFF_POINTOPOINT)) {
        if (!BIT_TEST (interface->flags, IFF_USERDEF) ||
            ll_addr->broadcast == NULL) continue;
        network = ll_addr->broadcast;
        gw = (interface->link_local)?interface->link_local->broadcast:
		interface->primary6->broadcast;
      }
      else {
        network = ll_addr->prefix;
        gw = (interface->link_local)?interface->link_local->prefix:
		interface->primary6->prefix;
      }

      if (ipv6_compat_addr (prefix_toaddr6 (network)) 
	    && ll_addr->prefix->bitlen > 0) continue;
      if (ipv6_link_local_addr (prefix_toaddr6 (network))) 
	    continue;
      if (ipv6_multicast_addr (prefix_toaddr6 (network))) 
	    continue;

      /* too strict for now ??? */
      /* if (!ipv6_link_local_addr (prefix_toaddr6 (gw))) 
	    continue; */

      network = New_Prefix (AF_INET6, 
	prefix_tochar (network), network->bitlen);
      attr = New (generic_attr_t);
      attr->type = PROTO_CONNECTED;
      /* myself for split horizen */
      gw = copy_prefix (gw);
      gw->bitlen = 128;
      attr->gateway = add_gateway (gw, 0, interface); 
      attr->nexthop = gw;
      /* masking may not be needed */
      netmasking (network->family, prefix_tochar (network), network->bitlen);

      trace (NORM, MRT->trace, "Adding a route for %s\n", interface->name);
      rib_add_route (RIBv6, network, attr, CONNECTED_PREF, 0);
    }
  }
  rib_close (RIBv6);
#endif /* HAVE_IPV6 */
  
  return (1);
}


/* 
 * add all configured static routes to the appropriate rib
 */
static int add_static_to_rib () {
  prefix_t *prefix;
  generic_attr_t *attr;
  extern config_master_t *CONFIG;

  if (CONFIG->ll_static_dest == NULL)
     return;
  assert (CONFIG->ll_static_attr);
  /* ipv4 */
  rib_open (RIB);
#ifdef HAVE_IPV6
  rib_open (RIBv6);
#endif /* HAVE_IPV6 */
  attr = LL_GetHead (CONFIG->ll_static_attr);
  LL_Iterate (CONFIG->ll_static_dest, prefix) {
    if (prefix->family == AF_INET)
      rib_add_route (RIB, prefix, attr, STATIC_PREF, 0);
#ifdef HAVE_IPV6
    else if (prefix->family == AF_INET6)
      rib_add_route (RIBv6, prefix, attr, STATIC_PREF, 0);
#endif /* HAVE_IPV6 */
    else {
    }

    attr = LL_GetNext (CONFIG->ll_static_attr, attr);
  }
  rib_close (RIB);
#ifdef HAVE_IPV6
  rib_close (RIBv6);
#endif /* HAVE_IPV6 */
#ifdef XXX
  LL_Destroy (CONFIG->ll_static_dest);
  LL_Destroy (CONFIG->ll_static_attr);
#endif
  CONFIG->ll_static_dest = NULL;
  CONFIG->ll_static_attr = NULL;
} 


void add_kernel_route (int family, char *dest, char *nexthop, int masklen) {

  prefix_t *prefix;
  generic_attr_t *attr;
  char tmp[MAXLINE];
  extern rib_t *RIB;
#ifdef HAVE_IPV6
  struct in6_addr zero6;
  extern rib_t *RIBv6;
#endif /* HAVE_IPV6 */
  int len;
  rib_t *rib;

#ifdef HAVE_IPV6
  assert (family == AF_INET || family == AF_INET6);
#else
  assert (family == AF_INET);
#endif /* HAVE_IPV6 */

  if (family == AF_INET) {
    if (*(u_long *)nexthop == INADDR_ANY) return;
    len = 32;
    rib = RIB;
    rib_open (rib);
  }
#ifdef HAVE_IPV6
  else {
    memset (&zero6, 0, sizeof (zero6));
    /* linux way of direct connected route */
    if (memcmp (nexthop, &zero6, 16) == 0) return;
    /*
     * I'm not sure but for now these prefix are ignored
     */
    if (((memcmp (dest, &zero6, 16) != 0 || masklen > 0) &&
         ipv6_compat_addr ((struct in6_addr *)dest)) ||
        ipv6_link_local_addr ((struct in6_addr *)dest) ||
        ipv6_multicast_addr ((struct in6_addr *)dest)) return;
    len = 128;
    rib = RIBv6;
    rib_open (rib);
  }
#endif /* HAVE_IPV6 */

  attr = New (generic_attr_t);
  attr->type = PROTO_KERNEL;
  prefix = New_Prefix (family, nexthop, len);
  attr->nexthop = prefix;
  attr->gateway = add_gateway (prefix, 0, 0); 

  prefix = New_Prefix (family, dest, masklen);
  trace (NORM, MRT->trace, "ROUTE from KERNEL %s/%d %s\n",
	prefix_toa (prefix), masklen, prefix_toa2 (attr->nexthop, tmp));
  rib_add_route (rib, prefix, attr, KERNEL_PREF, 0);
  Deref_Prefix (prefix);

  rib_close (rib);
}


int init_rib (trace_t *trace) {

   assert (RIB == NULL);
   RIB = (rib_t *) New_Rib (32);
   RIB->trace = trace;

#ifdef HAVE_IPV6
   assert (RIBv6 == NULL);
   RIBv6 = (rib_t *) New_Rib (128);
   RIBv6->trace = trace;
#endif /* HAVE_IPV6 */
}


static void
daemonize() {
    int pid, t;

    if ((pid = fork()) == -1) {
	perror("mrt: fork");
	exit(1);
    }
    else if (pid != 0) { /* parent */
	exit(0);
    }

#ifdef SETPGRP_VOID
    if ((t = setpgrp()) < 0) {
	perror("mrt: setpgrp");
	exit(1);
    }
    signal(SIGHUP, SIG_IGN);

    /* fork again so that not being a process group leader */
    if ((pid = fork()) == -1) {
	perror("mrt: fork");
	exit(1);
    }
    else if (pid != 0) { /* parent */
	exit(0);
    }
#else /* !SETPGRP_VOID */
    if ((t = setpgrp(0, getpid())) < 0) {
	perror("mrt: setpgrp");
	exit(1);
    }

    /* Remove our association with a controling tty */
    if ((t = open("/dev/tty", O_RDWR, 0)) >= 0) {
        if (ioctl(t, TIOCNOTTY, NULL) < 0) {
	    perror("mrt: TIOCNOTTY");
		exit(1);
	}
	(void) close(t);
    }
#endif /* SETPGRP_VOID */

#ifdef notdef
    /* Close all open files --- XXX need to check for logfiles */
    for (t = 0; t < 2; t++)
	(void) close(t);
#endif

    umask(022);

    trace (INFO|NORM, MRT->trace, "MRT %s compiled on %s started\n",
        MRT_VERSION, __DATE__);
}
