/* 
 * $Id: config.c,v 1.18 1997/02/01 01:19:54 masaki Exp $
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <mrt.h>
#include <trace.h>
#include <io.h>
#include <select.h>
#include <stdarg.h>
#include <interface.h>
#include <rip.h>
#ifdef HAVE_IPV6
#include <ripng.h>
#endif /* HAVE_IPV6 */
#include <bgp.h>
#include "config.h"
#include "rib.h"

/* globals */
config_master_t *CONFIG;
extern rib_t	*RIB;
#ifdef HAVE_IPV6 
extern rib_t	*RIBv6;
#endif /* HAVE_IPV6 */

int rip_debug (uii_connection_t *uii);
int bgp_debug (uii_connection_t *uii);
#ifdef HAVE_IPV6 
int ripng_debug (uii_connection_t *uii);
#endif /* HAVE_IPV6 */
#ifdef notdef
int config_ip_route (uii_connection_t *uii);
int config_router_network (uii_connection_t *uii);
int config_router_neighbor (uii_connection_t *uii);
int parse_line (u_char *line, char *format, ...);
int config_interface (uii_connection_t *uii);
int config_tunnel_destination (uii_connection_t *uii);
int config_tunnel_source (uii_connection_t *uii);
int config_access_list (uii_connection_t *uii);
int config_distribute_list (uii_connection_t *uii);
#endif

/* Comments...
 * Config can work from a file or from user configuration via a
 * socket.
 */

/* start_config
 * call by uii thread. 
 */
void start_config (uii_connection_t *uii) {
  
  uii->state = UII_CONFIG;
}


/* config_from_file
 */
int config_from_file (char *filename) {
  FILE *fd;
  uii_connection_t *uii = New (uii_connection_t);

#define MRT_CONFIG_DEFAULT_1 "/etc/mrt.config"
#define MRT_CONFIG_DEFAULT_2 "./mrt.config"
/* the third one is for backward compatibility */
#define MRT_CONFIG_DEFAULT_3 "./config"
  if (filename == NULL) {
    filename = MRT_CONFIG_DEFAULT_1;
    if ((fd = fopen (filename, "r")) == NULL) {
      filename = MRT_CONFIG_DEFAULT_2;
      if ((fd = fopen (filename, "r")) == NULL) {
        filename = MRT_CONFIG_DEFAULT_3;
        if ((fd = fopen (filename, "r")) == NULL) {
          trace (FATAL, CONFIG->trace, 
            "MRT failed to open all default config files %s, %s, and %s\n",
             MRT_CONFIG_DEFAULT_1, MRT_CONFIG_DEFAULT_2, MRT_CONFIG_DEFAULT_3);
          return (-1);
        }
      }
    }
  }
  else {
    if ((fd = fopen (filename, "r")) == NULL) {
      trace (FATAL, CONFIG->trace, 
         "MRT failed to open the config file %s\n", filename);
      return (-1);
    }
  }
  trace (NORM, CONFIG->trace, 
     "MRT opened the config file %s\n", filename);

  MRT->config_file_name = strdup (filename);
  start_config (uii);
  CONFIG->line = 0;

  while (fgets (uii->buffer, sizeof (uii->buffer)-1, fd) > 0) {
    CONFIG->line++;
    /* process_config (0, line);*/
    uii_proccess_command (uii);
  }
  
  fclose (fd);
  Delete (uii);
  return (0);
}


/* config_ip_route
 * config a static ip route
 */
int config_ip_route (uii_connection_t *uii) {
  prefix_t *prefix, *nexthop;
  generic_attr_t *attr;
  char tmp[MAXLINE];
  rib_t *rib;

  if (parse_line (uii->cp, "%p %p", &prefix, &nexthop) == 2
#ifdef HAVE_IPV6 
   || parse_line (uii->cp, "%P %P", &prefix, &nexthop) == 2
#endif /* HAVE_IPV6 */
   ) {

    trace (NORM, CONFIG->trace, "CONFIG static route %s via %s\n",
	 prefix_toa (prefix),  prefix_toa2 (nexthop, tmp));
  
    attr = New (generic_attr_t);
    attr->type = PROTO_STATIC;
    attr->gateway = add_gateway (nexthop, 0, 0);
    attr->nexthop = nexthop;
#if 0
    if (prefix->family == AF_INET)
      rib = RIB;
#ifdef HAVE_IPV6 
    else if (prefix->family == AF_INET6)
      rib = RIBv6;
#endif /* HAVE_IPV6 */
    else
      return (-1);
    rib_open (rib);
    rib_add_route (rib, prefix,  attr, STATIC_PREF, 0);
    rib_close (rib);
#else
    if (CONFIG->ll_static_dest == NULL) {
      CONFIG->ll_static_dest = LL_Create (0);
      CONFIG->ll_static_attr = LL_Create (0);
    }
    LL_Add (CONFIG->ll_static_dest, prefix);
    LL_Add (CONFIG->ll_static_attr, attr);
#endif
    return (1);
  }
  else {
    return (0);
  }
}


#define MAX_ALIST 100
static LINKED_LIST *access_list[MAX_ALIST];
typedef struct _condition_t {
   int permit;
   int refine;
   prefix_t *prefix;
} condition_t;

/* 
 * config_access_list
 */
int config_access_list (uii_connection_t *uii) {
/*
 * For now, only for ipv6
 */
  u_int num;
  prefix_t *prefix = NULL;
  int permit = 0, refine = 0;
  condition_t *condition;
  char tmp[MAXLINE];
  
  if (parse_line (uii->cp, "%d deny all", &num) >= 3) {
    /* nothing */
  } else if (parse_line (uii->cp, "%d permit all", &num) >= 3) {
    permit++;
#ifdef HAVE_IPV6 
  } else if (parse_line (uii->cp, "%d deny %P refine", &num, &prefix) >= 4) {
    refine++;
  } else if (parse_line (uii->cp, "%d deny %P exact", &num, &prefix) >= 3) {
    /* nothing */
  } else if (parse_line (uii->cp, "%d permit %P refine", &num, &prefix) >= 4) {
    permit++;
    refine++;
  } else if (parse_line (uii->cp, "%d permit %P exact", &num, &prefix) >= 3) {
    permit++;
#endif /* HAVE_IPV6 */
  } else if (parse_line (uii->cp, "%d deny %p refine", &num, &prefix) >= 4) {
    refine++;
  } else if (parse_line (uii->cp, "%d deny %p exact", &num, &prefix) >= 3) {
    /* nothing */
  } else if (parse_line (uii->cp, "%d permit %p refine", &num, &prefix) >= 4) {
    permit++;
    refine++;
  } else if (parse_line (uii->cp, "%d permit %p exact", &num, &prefix) >= 3) {
    permit++;
  } else {
    return (-1);
  }
  if (num <= 0 && num >= MAX_ALIST) {
    trace (NORM, CONFIG->trace, "CONFIG invalid access-list number %d\n", num);
    return (0);
  }

  if (prefix)
    sprintf (tmp, "%s/%d %s", prefix_toa (prefix), prefix->bitlen,
	(refine)?"refine":"exact");
  else
    sprintf (tmp, "all");
  trace (NORM, CONFIG->trace, "CONFIG access-list %d %s %s\n",
         num, (permit)?"permit":"deny", tmp);
  
  if (access_list[num] == NULL) {
     access_list[num] = LL_Create (0);
  }
  condition = New (condition_t);
  condition->permit = permit;
  condition->refine = refine;
  condition->prefix = prefix;
  LL_Add (access_list[num], condition);
  return (1);
}


/*
 * apply_access_list
 * return 1 if permit, 0 otherwise
 */
int apply_access_list (int num, prefix_t *prefix) {

   condition_t *condition;

   assert (0 <= num && num < MAX_ALIST);
   if (num == 0)
      return (1); /* cisco feature */
   if (access_list[num] == NULL) {
      /* I'm not sure how cisco works for undefined access lists */
     return (0); /* assuming deny for now */
   }

   LL_Iterate (access_list[num], condition) {

      if (condition->prefix == NULL) /* all */
	  return (condition->permit);

      if ((prefix->family == condition->prefix->family) && (
          (condition->refine && (prefix->bitlen > condition->prefix->bitlen)) ||
          (!condition->refine && 
              (prefix->bitlen == condition->prefix->bitlen))) &&
           comp_with_mask (prefix_tochar (prefix), 
                           prefix_tochar (condition->prefix),
                           condition->prefix->bitlen))
         return (condition->permit);
   }
   return (0);
}


/* 
 * config_distribute_list
 */
int config_distribute_list (uii_connection_t *uii) {
   u_int num;
   char buff[MAXLINE];
   int in = 0;
   interface_t *interface;
  
   buff[0] = '\0';
   if (parse_line (uii->cp, "%d in %s", &num, buff) >= 2) {
      in++;
   }
   else if (parse_line (uii->cp, "%d out %s", &num, buff) >= 2) {
   }
   else {
     printf("\nerror --");
     return (0);
   }
   if (num < 0 && num >= MAX_ALIST) {
     printf("\ninvalid access-list number (%d) --", num);
     return (0);
   }
 
   trace (NORM, CONFIG->trace, "CONFIG distribute-list %d %s %s\n",
          num, (in)?"in":"out", (buff[0])?buff:"all");
   
   if (buff[0]) {
      if ((interface = find_interface_byname (buff)) == NULL) {
         trace (NORM, CONFIG->trace, "CONFIG unkown interface %s\n", buff);
         return (0);
      }
      if (in) interface->dlist_in[uii->protocol] = num;
      else    interface->dlist_out[uii->protocol] = num;
   }
   else {
      LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
         if (in) interface->dlist_in[uii->protocol] = num;
         else    interface->dlist_out[uii->protocol] = num;
      }
   }

   return (1);
}


#ifdef HAVE_IPV6 

/* config_interface
 * start interface configuration 
 */
int config_interface (uii_connection_t *uii) {
  char name[MAXLINE];
  interface_t *interface;

  if (parse_line (uii->cp, "%s", name) >= 1) {
     if ((interface = find_interface_byname (name)) == NULL) {
        interface = 
     	   new_interface (name, IFF_POINTOPOINT|IFF_USERDEF|IFF_TUNNEL, 0, 0);
     }
     CONFIG->interface = interface;
     CONFIG->tunnel_source = NULL;
     trace (NORM, CONFIG->trace, "CONFIG INTERFACE %s\n", name);
     uii->state = UII_CONFIG_INTERFACE;

     return (1);
  }
  return (0);
}

int config_tunnel_destination (uii_connection_t *uii) {

  char tmp[MAXLINE];
  prefix_t *prefix = NULL, *prefix6 = NULL;

  if (BIT_TEST (CONFIG->interface->flags, IFF_USERDEF) == 0) {
     trace (NORM, CONFIG->trace, 
	"CONFIG tunnel destination for existent if %s!\n", 
	    CONFIG->interface->name);
     return (0);
  }

  if (parse_line (uii->cp, "%p %P", &prefix, &prefix6) >= 2) {
    trace (NORM, CONFIG->trace, 
	"CONFIG tunnel destination %s %s\n",
	   prefix_toa (prefix), prefix_toa2 (prefix6, tmp));
    assert (CONFIG->interface);
    if (CONFIG->tunnel_source == NULL) {
       trace (NORM, CONFIG->trace, 
	   "CONFIG tunnel source not defined!\n");
       return (0);
    }
#if 0
    memcpy (&CONFIG->interface->dest, prefix_tochar (prefix), 4);
    memcpy (&CONFIG->interface->dest6, prefix_tochar (prefix6), 16);
#endif
    CONFIG->interface->mtu = CONFIG->tunnel_source->mtu;
    add_addr_to_interface (CONFIG->interface, AF_INET,
       prefix_tochar (CONFIG->tunnel_source->primary->prefix),
       32, prefix_tochar (prefix));
    add_addr_to_interface (CONFIG->interface, AF_INET6,
       prefix_tochar (CONFIG->tunnel_source->primary6->prefix),
       128, prefix_tochar (prefix6));
    CONFIG->interface->flags |= IFF_UP;

    Delete (prefix); /* local use */
    Delete (prefix6); /* local use */
    return (1);
  }
  else if (parse_line (uii->cp, "%p alias", &prefix) >= 1 ||
           parse_line (uii->cp, "%P alias", &prefix) >= 1 ) {
    char *addr = NULL;

    /* permit without alias keyword */
    trace (NORM, CONFIG->trace, 
	"CONFIG tunnel destination %s\n",
	   prefix_toa (prefix));
    assert (CONFIG->interface);
    if (CONFIG->tunnel_source == NULL) {
       trace (NORM, CONFIG->trace, 
	   "CONFIG tunnel source not defined!\n");
       Delete (prefix); /* local use */
       return (0);
    }
    if (prefix->family == AF_INET6) {
       if (ipv6_link_local_addr (prefix_toaddr6 (prefix))) {
	  if (CONFIG->tunnel_source->link_local)
             addr = prefix_tochar (CONFIG->tunnel_source->link_local->prefix);
       }
       else {
	  if (CONFIG->tunnel_source->primary6)
             addr = prefix_tochar (CONFIG->tunnel_source->primary6->prefix);
       }
    }
    else {
       if (CONFIG->tunnel_source->primary)
          addr = prefix_tochar (CONFIG->tunnel_source->primary->prefix);
    }
    if (addr == NULL) {
       trace (NORM, CONFIG->trace, 
"CONFIG tunnel source %s doesn't have any addresses with such a family\n",
          CONFIG->tunnel_source->name);
       Delete (prefix); /* local use */
       return (0);
    }
    add_addr_to_interface (CONFIG->interface, prefix->family,
       addr, (prefix->family==AF_INET)?32:128, prefix_tochar (prefix));
    CONFIG->interface->flags |= IFF_UP;

    Delete (prefix); /* local use */
    return (1);
  }
  if (prefix) Delete (prefix); /* local use */
  if (prefix6) Delete (prefix6); /* local use */
  return (0);
}


int config_tunnel_source (uii_connection_t *uii) {

  char name[MAXLINE];
  interface_t *interface;
  prefix_t *prefix, *prefix6;

  if (BIT_TEST (CONFIG->interface->flags, IFF_USERDEF) == 0) {
     trace (NORM, CONFIG->trace, 
	"CONFIG tunnel source for existent if %s!\n", CONFIG->interface->name);
     return (0);
  }

  if (parse_line (uii->cp, "%s", name) >= 1) {
    if ((interface = find_interface_byname (name)) == NULL) {
       trace (NORM, CONFIG->trace, 
	   "CONFIG tunnel source %s not exist!\n", name);
       return (0);
    }
    trace (NORM, CONFIG->trace, "CONFIG tunnel source %s\n", name);

    CONFIG->tunnel_source = interface;
    CONFIG->interface->mtu = interface->mtu;
#if 0
    assert (CONFIG->interface);
    CONFIG->interface->prefix = copy_prefix (interface->prefix);
    CONFIG->interface->prefix->bitlen = 32;
    CONFIG->interface->prefix6 = copy_prefix (interface->prefix6);
    CONFIG->interface->prefix6->bitlen = 128;
#endif
    return (1);
  }
}

int config_gateway (uii_connection_t *uii) {

  interface_t *interface;
  prefix_t *prefix6;

  if (BIT_TEST (CONFIG->interface->flags, IFF_USERDEF) != 0) {
     trace (NORM, CONFIG->trace, 
	"CONFIG gateway for tunnel if %s!\n", CONFIG->interface->name);
     return (0);
  }

  if (parse_line (uii->cp, "%P", &prefix6) >= 1) {
    trace (NORM, CONFIG->trace, 
	"CONFIG gateway %s\n", prefix_toa (prefix6));
    add_gateway (prefix6, 0, CONFIG->interface);
    Deref_Prefix (prefix6);
    return (1);
  }
}


/*
 * config_router_ripng
 * start RIPNG protocol configuration 
 */
int config_router_ripng (uii_connection_t *uii) {
  char *token;

  trace (NORM, RIPNG->trace, "CONFIG RIPNG\n");
  uii->protocol = PROTO_RIPNG;
  uii->state = UII_CONFIG_ROUTER;
  MRT->protocols |= (1 << PROTO_RIPNG);
  return (1);
}
#endif /* HAVE_IPV6 */


/* config_router_rip
 * start RIP protocol configuration 
 */
int config_router_rip (uii_connection_t *uii) {
  char *token;

  trace (NORM, RIP->trace, "CONFIG RIP\n");
  uii->protocol = PROTO_RIP;
  uii->state = UII_CONFIG_ROUTER;
  MRT->protocols |= (1 << PROTO_RIP);
  return (1);
}


int config_router_bgp (uii_connection_t *uii) {
  char *token;
  u_int as;

  parse_line (uii->cp, "%d", &as);
  set_BGP (BGP_MY_AS, as, NULL);

  trace (NORM, BGP->trace, "CONFIG BGP AS%d\n", as);
  uii->protocol = PROTO_BGP;
  uii->state = UII_CONFIG_ROUTER;
  MRT->protocols |= (1 << PROTO_BGP);
  return (1);
}


/* config_router_neighbor 
 * just used by BGP to configure peer (at least for now)
 */
int config_router_neighbor (uii_connection_t *uii) {
  u_int as, weight, view = 0;
  prefix_t *prefix;
  char tmp[MAXLINE];
  bgp_peer_t *peer;
  char name[MAXLINE];
  gateway_t *gateway;
  
  if (parse_line (uii->cp, "%p remote-as %d view %d", 
		  &prefix, &as, &view) > 2) {
    peer = Add_BGP_Peer (prefix, as, BGP->trace);
    trace (NORM, BGP->trace, "CONFIG BGP neighbor %s AS%d\n",
	   rinet_ntoa (tmp, MAXLINE, prefix), as);

    /* by default, all peers use view 0 */
    if (view == 0) {return (1);}

    if ((view >32) || (view <0)) {
      trace (NORM, BGP->trace, 
	     "*ERROR* CONFIG invalid view %d for BGP neighbor %s\n",
	     view, rinet_ntoa (tmp, MAXLINE, prefix));
      uii_send_data (uii, "Invalid view %d [0-32]\r\n", view);
      return (-1);
    }

    /* a new view */
    BGP->views[view] = New_View (32);
    peer->view = view;
  }

  /* neighbor x.x.x.x weight xx */
  if (parse_line (uii->cp, "%p weight %d", &prefix, &weight) >= 3) {
    trace (NORM, BGP->trace, "CONFIG BGP neighbor %s weight %d\n",
	   rinet_ntoa (tmp, MAXLINE, prefix), weight);
    return (1);
  }

  if (parse_line (uii->cp, "%p transparent-as", &prefix) >= 2) {
    gateway_t * gateway;
    trace (NORM, BGP->trace, "CONFIG BGP neighbor %s transparent-as\n",
	   rinet_ntoa (tmp, MAXLINE, prefix), weight);
    gateway = find_gateway (prefix, 0);
    peer = find_bgp_peer (gateway);

    if (peer != NULL) {
      peer->options |= BGP_TRANSPARENT_AS;
      return (1);
    }
  }

#if 0
  /* neighbor x.x.x.x name xx */
  if (parse_line (uii->cp, "%P name %s", &prefix, name) >= 3) {
    if (*name == '"') {
      char *p;
      strcpy (name, name+1);
      if (p = strrchr (name, '"'))
	*p = '\0';
    }
    trace (NORM, BGP->trace, "CONFIG neighbor %s name %s\n",
	   prefix_toa (prefix), name);
    gateway = add_gateway (prefix, 0, 0);
    gateway->name = strdup (name);
    Deref_Prefix (prefix);
    return (1);
  }
#endif

  return (-1);
}


/* config_router_network
 */
int config_router_network (uii_connection_t *uii) {
  prefix_t *prefix;
  interface_t *interface;
  LINKED_LIST *ll;
  char name[MAXLINE];
  
  if ((uii->protocol == PROTO_RIP && parse_line (uii->cp, "%p", &prefix) >= 1)
#ifdef HAVE_IPV6 
    || (uii->protocol == PROTO_RIPNG && 
           parse_line (uii->cp, "%P", &prefix) >= 1)
#endif /* HAVE_IPV6 */
       ) {
    if ((ll = find_network (prefix)) != NULL) {
      LL_Iterate (ll, interface) {
        switch (uii->protocol) {
          case PROTO_RIP:
            if (BIT_TEST (RIP->interface_mask, interface->bit))
              break;
            RIP->interface_mask |= interface->bit;
            trace (NORM, RIP->trace, "CONFIG RIP on interface %s\n",
                   interface->name);
            break;
#ifdef HAVE_IPV6 
          case PROTO_RIPNG:
            if (BIT_TEST (RIPNG->interface_mask, interface->bit))
              break;
	    if (!BIT_TEST (interface->flags, IFF_MULTICAST) &&
	         BIT_TEST (interface->flags, IFF_POINTOPOINT) &&
                (interface->link_local == NULL ||
                 interface->link_local->broadcast == NULL)) {
              trace (NORM, RIPNG->trace, 
"CONFIG RIPNG on interface %s ignored due to no multicast or link-local dest addr\n",
                  interface->name);
              break;
            }
            RIPNG->interface_mask |= interface->bit;
            trace (NORM, RIPNG->trace, "CONFIG RIPNG on interface %s\n",
                   interface->name);
            break;
#endif /* HAVE_IPV6 */
        }
      }
#ifdef XXX
      LL_Destroy (ll);
#endif
    }
    Delete_Prefix (prefix);
  }
  /* This part must be the last otherwise some ipv6 address will be mistook */
  else if (parse_line (uii->cp, "%s", name) == 1) {
    if ((interface = find_interface_byname (name)) != NULL) {
      if (uii->protocol == PROTO_RIP) {
        RIP->interface_mask |= interface->bit;
        trace (NORM, RIP->trace, "CONFIG RIP on interface %s\n",
	     interface->name);
      }
#ifdef HAVE_IPV6 
      else if (uii->protocol == PROTO_RIPNG) {
        if (!BIT_TEST (RIPNG->interface_mask, interface->bit)) {
	  if (!BIT_TEST (interface->flags, IFF_MULTICAST) &&
	     BIT_TEST (interface->flags, IFF_POINTOPOINT) &&
            (interface->link_local == NULL ||
             interface->link_local->broadcast == NULL)) {
            trace (NORM, RIPNG->trace, 
"CONFIG RIPNG on interface %s ignored due to no multicast or link-local dest addr\n",
                interface->name);
          }
	  else {
            RIPNG->interface_mask |= interface->bit;
            trace (NORM, RIPNG->trace, "CONFIG RIPNG on interface %s\n",
	         interface->name);
	  }
        }
      }
#endif /* HAVE_IPV6 */
    }
    else
      return (-1);
  }
  else
    return (-1);

  return (1);
}


int show_config (uii_connection_t *uii) {
  char tmp[MAXLINE];
  FILE *fd;
  
  if ((fd = fopen (MRT->config_file_name, "r")) == NULL) {return (-1);}

  while (fgets (tmp, sizeof (tmp) - 1, fd)) {
    uii_send_data (uii, "%s\r", tmp);
  }
  fclose (fd);
}


int config_redistribute (uii_connection_t *uii) {

  MRT->rdist_static |= uii->protocol;
  if (uii->protocol == PROTO_BGP)
    trace (NORM, BGP->trace, "CONFIG BGP redistribute static\n");
  
  return (1);
}

int config_comment (uii_connection_t *uii) {
  if (strlen (uii->cp) < 2) {
    trace (TR_PARSE, CONFIG->trace, "COMMENT -- STATE EOF %s", uii->cp);
    uii->state = UII_CONFIG;
  }
  
  trace (TR_PARSE, CONFIG->trace, "COMMENT %s", uii->cp);
  return (1);
}


/* strip off the begining and trailing space characters */
char *strip_spaces (char *tmp) {
    char *cp = tmp, *pp;
    while (*cp && isspace (*cp)) cp++;
    if (*cp) {
      pp = cp+strlen(cp)-1;
      while (pp > cp && isspace (*pp)) pp--;
      pp[1] = '\0';
    }
    return (cp);
}


int password (uii_connection_t *uii) {
  char tmp[MAXLINE], *cp;

  tmp[0] = '\0';
  if (parse_line (uii->cp, "%S", tmp) >= 1) {
    cp = strip_spaces (tmp);
  }
  if (*cp) {
    UII->initial_state = 0; /* password required */
    CONFIG->password = strdup (cp);
  }
}


void check_passwd (uii_connection_t *uii) {
  char tmp[MAXLINE], *cp = tmp;

  tmp[0] = '\0';
  if (parse_line (uii->cp, "%S", tmp) >= 1) {
    cp = strip_spaces (tmp);
  }
  if (CONFIG->password == NULL || strcmp (CONFIG->password, cp) == 0)
    uii->state = 1;
  else {
    trace (INFO|NORM, MRT->trace, "INVALID PASSWORD: [%s]\n", cp);
  }
}

int parse_debug (uii_connection_t *uii, trace_t *trace) {
  char *token, *line;
  u_long flag = 0;
  line = uii->cp;

  /* get flag */
  if ((token = (char *) uii_parse_line (&line)) == NULL) return (-1);
  flag = trace_flag (token);
  set_trace (trace, TRACE_FLAGS, flag, NULL);

  /* get file name (OPTIONAL) */
  if ((token = (char *) uii_parse_line (&line)) == NULL) return (1);
  set_trace (trace, TRACE_LOGFILE, token, NULL);

  return (1);
}

config_debug (uii_connection_t *uii) {
  parse_debug (uii, CONFIG->trace);
}

rib_debug (uii_connection_t *uii) {
  parse_debug (uii, RIB->trace);
#ifdef HAVE_IPV6 
  parse_debug (uii, RIBv6->trace);
#endif /* HAVE_IPV6 */
}

rib4_debug (uii_connection_t *uii) {
  parse_debug (uii, RIB->trace);
}

#ifdef HAVE_IPV6 
rib6_debug (uii_connection_t *uii) {
  parse_debug (uii, RIBv6->trace);
}
#endif /* HAVE_IPV6 */

kernel_debug (uii_connection_t *uii) {
  parse_debug (uii, INTERFACE_MASTER->trace);
}

timer_debug (uii_connection_t *uii) {
  parse_debug (uii, TIMER_MASTER->trace);
}

io_debug (uii_connection_t *uii) {
/*  parse_debug (uii, MASTER_IO->trace); */
  parse_debug (uii, SELECT_MASTER->trace);
}

thread_debug (uii_connection_t *uii) {
  parse_debug (uii, MRT->trace);
  parse_debug (uii, SCHEDULE_MASTER->trace);
}


int init_config (trace_t *trace) {

  CONFIG = New (config_master_t);
  CONFIG->trace = trace_copy (trace);
  CONFIG->ll_static_dest = NULL;
  CONFIG->ll_static_attr = NULL;
  CONFIG->interface = NULL;
  CONFIG->tunnel_source = NULL;
  CONFIG->password = NULL;

  UII->initial_state = 1; /* no password required */
  uii_add_command (UII_NORMAL, "show config", (void *) show_config);

  /* config top level */
  uii_add_command (UII_CONFIG, "password", (void *) password);
#ifdef HAVE_IPV6 
  uii_add_command (UII_CONFIG, "debug ripng", (void *) ripng_debug);
  uii_add_command (UII_CONFIG, "debug rib6", (void *) rib6_debug);
  uii_add_command (UII_CONFIG, "router ripng", (void *) config_router_ripng);
  uii_add_command (UII_CONFIG, "ipv6 route", (void *) config_ip_route);
  uii_add_command (UII_CONFIG, "interface", (void *) config_interface);
  uii_add_command (UII_CONFIG_INTERFACE, "gateway", 
	(void *) config_gateway);
  uii_add_command (UII_CONFIG_INTERFACE, "tunnel destination", 
	(void *) config_tunnel_destination);
  uii_add_command (UII_CONFIG_INTERFACE, "tunnel source", 
	(void *) config_tunnel_source);
  uii_add_command (UII_CONFIG_INTERFACE, "!", (void *) config_comment);
#endif /* HAVE_IPV6 */
  uii_add_command (UII_CONFIG, "debug rip", (void *) rip_debug);
  uii_add_command (UII_CONFIG, "debug bgp", (void *) bgp_debug);
  uii_add_command (UII_CONFIG, "router bgp", (void *) config_router_bgp);
  uii_add_command (UII_CONFIG, "router rip", (void *) config_router_rip);
  uii_add_command (UII_CONFIG, "ip route", (void *) config_ip_route);
  uii_add_command (UII_CONFIG, "access-list", (void *) config_access_list);
  uii_add_command (UII_CONFIG, "!", (void *) config_comment);
  uii_add_command (UII_CONFIG, "debug config", (void *) config_debug);
  uii_add_command (UII_CONFIG, "debug rib", (void *) rib_debug);
  uii_add_command (UII_CONFIG, "debug rib4", (void *) rib4_debug);
  uii_add_command (UII_CONFIG, "debug kernel", (void *) kernel_debug);
  uii_add_command (UII_CONFIG, "debug timer", (void *) timer_debug);
  uii_add_command (UII_CONFIG, "debug io", (void *) io_debug);
  uii_add_command (UII_CONFIG, "debug thread", (void *) thread_debug);

  /* router [rip|bgp|ripng] */
  uii_add_command (UII_CONFIG_ROUTER, "network", 
	(void *) config_router_network);
  uii_add_command (UII_CONFIG_ROUTER, "neighbor", 
	(void *) config_router_neighbor);
  uii_add_command (UII_CONFIG_ROUTER, "redistribute static", 
	(void *) config_redistribute);
  uii_add_command (UII_CONFIG_ROUTER, "distribute-list", 
	(void *) config_distribute_list);
  uii_add_command (UII_CONFIG_ROUTER, "!", (void *) config_comment);

  return (1);
}
