#!/usr/local/bin/perl 
# rt_entry is "time(4)|aspath|origin|nexthop"
# binary file format is
# version (2)
# num aspaths (2)
#   AS path id (2) | aspath len (2) | aspath (aspath len * 2)
# num prefixes (4)
#   prefix (4) | bitlen (2) | num peers (2)
#     peer (4) | time (4) | short AS path id (2) | origin (2)
#

use MRT;
require "getopts.pl";
$VERSION = 1;

&Getopts("i:o:shd");
if(defined $opt_i) { 
    $in = $opt_i;
}
if (defined $opt_o) {
    $RTFILE = $opt_o;
}
if (defined $opt_s) {
    $SORT_FLAG = 1;
}
if (defined $opt_d) {
    $DEBUG = 1;
}

if (defined $opt_h) {&error;}
if ($RTFILE eq "") {&error;}

# read earlier routing table
if ($in ne "") { 
    print "Reading rib file $in\n";
    &read_rib ($in);
}

foreach $arg (@ARGV) {
    print "Reading update spool file $arg\n";
    open_mrt_data ($arg);
    &read_logfile;
}
&write_table ($RTFILE);
exit;



sub read_logfile {
    while (1) {
	($time, $type, $subtype, $length) = &get_mrt_header;

	if ($time == -1) {return;}

	# get packet
	($srcip, $srcas, $dstip, $dstas, $ann_prefix, $with_prefix, 
	 $aspath, $origin, $nexthop) = 
	     &get_mrt_bgp_msg ($length);

	# hash aspaths and assign each a number to save space
	if (($ann_prefix ne "") && ($ASPATH{$aspath} eq "")) {
	    $ASPATH{$aspath} = $NASPATHS++;
	}
	$aspath = $ASPATH{$aspath};

	$PEER{$srcip} += 1;
	@ann = split (/\s/, $ann_prefix);
	@with = split (/\s/, $with_prefix);
    
	foreach $ann (@ann) {
	    $RT{$ann}{$srcip} = "$time|$aspath|$origin|$nexthop";
	}
	foreach $with (@with) {
	    delete $RT{$with}{$srcip};
	    $count = keys %{$RT{$with}};
	    if ($count == 0) {delete $RT{$with};};
	}
    }
}


sub write_table {
    local ($file) = @_;

# num aspaths (2)
#   AS path id (2) | aspath len (2) | aspath (aspath len * 2)
# num prefixes (4)
#   prefix (4) | bitlen (2) | num peers (2)
#     peer (4) | time (4) | short AS path id (2) 
#

    open (OUT, "> $file");
    print "Writing rib to $file\n";


    # print version
    print OUT pack ("S", $VERSION);

    @aspaths = keys %ASPATH;
    $paths = $#aspaths + 1;
    $buf = pack ("S", $paths);

    # num aspaths
    print OUT $buf; 
    if ($DEBUG == 1) {print "$paths paths\n";}

    # aspaths
    foreach $aspath (@aspaths) {
	@as = split (/\s/, $aspath);
	$aslen = $#as + 1;
	if ($DEBUG == 1) {
	    print "$ASPATH{$aspath} $aslen @as\n";
	}
	$buf = pack ("SSS$aslen", $ASPATH{$aspath}, $aslen, @as);
	print OUT $buf;
    }

    # number of prefixes
    @prefix = keys %RT;
    $numprefixes = $#prefix + 1;
    print OUT pack ("L", $numprefixes);
    if ($DEBUG == 1) {print "$numprefixes prefixes\n"; }

    # this can be pretty expensive
    if ($SORT_FLAG == 1) {
	print "Sorting $numprefixes prefixes\n";
	@prefix = sort mysort @prefix;
    }


    foreach $prefix (@prefix) {
	# prefix
	($ip, $bitlen) = split (/\//, $prefix);
	@ip = split (/\./, $ip);
	$ip = pack ("C4", @ip);
	# prefix
	print OUT $ip;
	#bitlen
	print OUT pack ("S", $bitlen);
	# number of peers announcing this prefix
	@peers = keys %{$RT{$prefix}};
	$numpeers = $#peers + 1;
	print OUT pack ("S", $numpeers);

	if ($DEBUG == 1) { print "@ip/$bitlen  $numpeers peers\n";}
	
	foreach $peer (@peers) {
	    # peer ip
	    @ip = split (/\./, $peer);
	    print OUT pack ("C4", @ip);
	    if ($DEBUG == 1) {
		print "\t @ip\n";
	    }
	    # time | aspath ID | origin
	    # rt_entry is "time(4)|aspathid|origin|nexthop"
	    ($time, $aspathid, $origin, $nexthop) = 
		split (/\|/, $RT{$prefix}{$peer});
	    print OUT pack ("LSS", $time, $aspathid, $origin) ;
	}
    }
		     

    close (OUT);

}

sub read_rib {
    local ($in) = @_;
    
    open (INPUT, $in) || die "Could not open $in: $!\n";

    # version
    $buf = &get (2);
    $version = unpack ("S", $buf);
    if ($version != 1) {
	print "ERROR -- unknown version number $version\n";
	exit;
    }

    # num aspaths
    $buf = &get(2);
    $numaspaths = unpack ("S", $buf);
    if ($DEBUG == 1) {
	print "$numaspaths distinct ASPATHs\n";
    }

    # aspaths
    while ($numaspaths--) {
	$buf = &get (4);
	($asID, $aslen) = unpack ("SS", $buf);

	$buf = &get ($aslen*2);
	(@aspath) = unpack ("S$aslen", $buf);
	if ($DEBUG == 1) {
	    #print "id=$asID len=$aslen @aspath\n";
	}
	$ASPATH{"@aspath"} = $asID;
    }

    # number of prefixes
    $buf = &get(4);
    $numprefix = unpack ("L", $buf);
    if ($DEBUG == 1) {print "$numprefix prefixes\n";}

    while ($numprefix--) {
	$buf = &get(4);
	@ip = unpack ("C4", $buf);
	$ip = join (".", @ip);
	
	$buf = &get(4);
	($bitlen, $numpeer) = unpack ("SS", $buf);
	$ip = "$ip/$bitlen";
	$route = $ip;
	#if ($DEBUG == 1) {print "$ip $numpeer\n";}

	while ($numpeer--) {
	    $buf = &get(4);
	    @ip = unpack ("C4", $buf);
	    $ip = join (".", @ip);
	    
	    $buf = &get(8);
	    ($time, $asID, $origin) = unpack ("LSS", $buf);
	    if ($DEBUG == 1) {
		#print "\t$ip $time $asID $origin\n";
	    }
	    # rt_entry is "time(4)|aspathid|origin|nexthop"
	    $RT{$route}{$ip} = "$time|$asID|$origin|$nexthop";
	}
    }
}


sub get {
    local ($bytes) = @_;
    
    $buf = "";
    if (($n = read (INPUT, $buf, $bytes)) != $bytes) {
	if (($n == 0) && ($bytes == 12)) {return "";}
	print "read $n bytes, expected $bytes\n";
	exit;
    }
    
    $read += $bytes;
    return ($buf);
}


sub error {

    print "Usage: rtbuild.pl -o rib [-i input_rib] updatefile [[updatefile]...]\n";
    exit;
}

    
sub mysort {
  $a =~ /([\d\.]+)/;
  $aq = $1; 
  $b =~ /([\d\.]+)/;
  $bq = $1;

   #print "$aq $bq\n";

  @aa = split (/\./, $aq);
  @bb = split (/\./, $bq);

  if ($aa[0] > $bb[0]) {return 1;}
  elsif ($aa[0] <  $bb[0]) {return -1;} 

  if ($aa[1] > $bb[1]) {return 1;}
  elsif ($aa[1] < $bb[1]) {return -1;}

  if ($aa[2] > $bb[2]) {return 1;}
  elsif ($aa[2] < $bb[2]) {return -1;}

  if ($aa[3] > $bb[3]) {return 1;}
  elsif ($aa[3] < $bb[3]) {return -1;}
  
  return 0;
}



