/* 
 * $Id: prefix.c,v 1.17 1997/02/18 23:17:15 masaki Exp $
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>
#include <stdio.h>
#include <netdb.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <array.h>
#include <hash.h>
#include <mrt.h>

/* New_Prefix
 */
prefix_t *New_Prefix (int family, u_char *dest, int bitlen)
{

   prefix_t *prefix = New (prefix_t);
   prefix->bitlen = bitlen;
   prefix->family = family;

   if (family == AF_INET) {
     memcpy (&prefix->add.sin, dest, 4);
   }
#ifdef HAVE_IPV6
   else if (family == AF_INET6) {
     memcpy (&prefix->add.sin6, dest, 16);
   }
#endif /* HAVE_IPV6 */
   else {
     Delete (prefix);
     return (NULL);
   }

   prefix->ref_count = 1;
   return (prefix);
}


prefix_t *Ref_Prefix (prefix_t *prefix)
{
   assert (prefix->ref_count > 0);
   prefix->ref_count++;
   return (prefix);
}


void Deref_Prefix (prefix_t *prefix)
{
   prefix->ref_count--;
   assert (prefix->ref_count >= 0);
   if (prefix->ref_count <= 0) {
     Delete (prefix);
   }
}


/* copy_prefix
 */
prefix_t *copy_prefix (prefix_t *prefix)
{
   assert (prefix->ref_count > 0);
   return (New_Prefix  (prefix->family, (char *) &prefix->add, prefix->bitlen));
}


/* ascii2prefix
 */
prefix_t* ascii2prefix (int family, char *string) {
  u_long l, bitlen, maxbitlen = 0;
  u_char *cp;
#ifdef HAVE_IPV6
  struct in6_addr sin6;
#endif /* HAVE_IPV6 */
  int result;

  if (family == AF_INET) {
    maxbitlen = 32;
  }
#ifdef HAVE_IPV6
  else if (family == AF_INET6) {
    maxbitlen = 128;
  }
#endif /* HAVE_IPV6 */

  if ((cp = strchr (string, '/')) != NULL) {
    bitlen = atol (cp+1);
    *cp = '\0';
    if (bitlen < 0 || bitlen > maxbitlen)
      bitlen = maxbitlen;
  }
  else {
    bitlen = maxbitlen;
  }

  if (family == AF_INET) {
    /* MRT uses short-hand expressions like 192.168.10/24,
       but inet_addr() doesn't handle them properly */
#if 1
    int i;
    char *xp = (unsigned char *)&l;
    memset (xp, 0, sizeof (l));
    for (i = 0; i < 4; i++) {
      xp[i] = atoi (string);
      if ((cp = strchr (string, '.')) == NULL)
        break;
      string = cp+1;
    }
#else
    l = inet_addr (string);
#endif
    return (New_Prefix  (AF_INET, (char *) &l, bitlen));
  }
#ifdef HAVE_IPV6
  else if (family == AF_INET6) {
    if ((result = inet_pton (AF_INET6, string, &sin6)) < 0) 
      return (NULL);
    return (New_Prefix  (AF_INET6, (char *) &sin6, bitlen));
  }
#endif /* HAVE_IPV6 */
  else 
    return (NULL);
}


void Delete_Prefix (prefix_t *prefix) {
   assert (prefix->ref_count >= 0);
   Delete (prefix);
}


/* 
 * print_prefix_list
 */
void print_prefix_list (LINKED_LIST *ll_prefixes) {
   prefix_t *prefix;
   
   if (ll_prefixes == NULL)
      return;

   LL_Iterate (ll_prefixes, prefix) {
      printf ("\n  %-15s", 
              prefix_toa (prefix));
   }
}


/*
 * print_pref_prefix_list
 */
void print_pref_prefix_list (LINKED_LIST *ll_prefixes, char *pref) {
   prefix_t *prefix;
   char tmp[MAXLINE];
   int i = 0;

   if ((ll_prefixes == NULL) || (pref == NULL))
      return;
   
   LL_Iterate (ll_prefixes, prefix) {
      printf ("\n  %-15s %d", 
	      prefix_toa (prefix), pref[i++]);
   }
}


int prefix_compare (prefix_t *p1, prefix_t *p2) {
  
  assert (p1->ref_count > 0);
  assert (p2->ref_count > 0);
  return ((p1->bitlen == p2->bitlen) &&
      (p1->bitlen == 0 || comp_with_mask (
	prefix_tochar (p1), prefix_tochar (p2), p1->bitlen)));
#ifdef notdef
  if (p1->family == AF_INET) {
    if ((p1->bitlen == p2->bitlen) &&
	memcmp (&p1->add, &p2->add, 4) == 0)
      return (1);
  }
#ifdef HAVE_IPV6
  else if (p1->family == AF_INET6) {
    if ((p1->bitlen == p2->bitlen) &&
	memcmp (&p1->add, &p2->add, 16) == 0) 
      return (1);
  }
#endif /* HAVE_IPV6 */
#endif

  return (0);
}

/* prefix_toa2
 * convert prefix information to ascii string but use static buffer area
 */
char *prefix_toa (prefix_t *prefix) {
  return (prefix_toa2 (prefix, (char *)NULL));
}

/* prefix_toa2
 * convert prefix information to ascii string
 */
char *prefix_toa2 (prefix_t *prefix, char *tmp) {
  static char buff[40];

  assert (prefix->ref_count > 0);
  if (tmp == NULL) tmp = buff;
  if (prefix->family == AF_INET) {
    unsigned char *a = prefix_tochar (prefix);
    sprintf(tmp, "%d.%d.%d.%d", a[0], a[1], a[2], a[3]);
    return (tmp);
  }
#ifdef HAVE_IPV6
  else if (prefix->family == AF_INET6) {
    (void) inet_ntop (AF_INET6, &prefix->add.sin6, tmp, 40);
                                         /* 40 is a guess */
    return (tmp);
  }
#endif /* HAVE_IPV6 */
  else 
    return (NULL);
}


#ifdef notdef
/* prefix_tolong
 * convert prefix information to long
 */
long prefix_tolong (prefix_t *prefix) {
  int l;
  return (prefix->add.sin.s_addr);
}


/* prefix_tochar
 * convert prefix information to bytes
 */
u_char *prefix_tochar (prefix_t *prefix) {

  return ((u_char *) &prefix->add.sin);
}
#endif


void trace_prefix_list (char *msg, trace_t *ltrace, LINKED_LIST *ll_prefix) {
  char tmp[4][MAXLINE];
  prefix_t *prefix = NULL;
  int n = 0;

  if (ll_prefix == NULL) return;
  
  LL_Iterate (ll_prefix, prefix) {
    n++;
    if (n == 4) {
      sprintf (tmp[n-1], "%s", prefix_toa (prefix));
      trace (TR_PACKET, ltrace, "%s %s %s %s %s\n",
	     msg, tmp[0], tmp[1], tmp[2], tmp[3]);
      n = 0;
      memset (tmp[0], 0, MAXLINE);
      memset (tmp[1], 0, MAXLINE);
      memset (tmp[2], 0, MAXLINE);
      memset (tmp[3], 0, MAXLINE);
    }
    else {
      sprintf (tmp[n-1], "%s", prefix_toa (prefix));
    }
  }
   
  if (n == 3) 
    trace (TR_PACKET, ltrace, "%s %s %s %s\n",
	   msg, tmp[0], tmp[1], tmp[2]);
  else if (n == 2) 
    trace (TR_PACKET, ltrace, "%s %s %s\n",
	   msg, tmp[0], tmp[1]);
  else if (n == 1)
    trace (TR_PACKET, ltrace,"%s %s\n",
	   msg, tmp[0]);
}


#ifdef HAVE_IPV6

int ipv6_global_unicast_addr (struct in6_addr *sin6) {

    return ((sin6->s6_addr[0] & 0xe0) == 0x04 ||
         (sin6->s6_addr[0] & 0xe0) == 0x08);
}

int ipv6_multicast_addr (struct in6_addr *sin6) {

    struct in6_addr zero;

    zero.s6_addr[0] = 0xff;
    return (comp_with_mask ((char *)&zero, (char *)sin6, 8));
}

int ipv6_link_local_addr (struct in6_addr *sin6) {

    struct in6_addr zero;

    zero.s6_addr[0] = 0xfe;
    zero.s6_addr[1] = 0x80;

    return (comp_with_mask ((char *)&zero, (char *)sin6, 10));
}

int ipv6_ipv4_addr (struct in6_addr *sin6) {

    struct in6_addr zero;

    memset (&zero, 0, 12);
    zero.s6_addr[10] = 0xff;
    zero.s6_addr[11] = 0xff;
    return (comp_with_mask ((char *)&zero, (char *)sin6, 96));
}

int ipv6_compat_addr (struct in6_addr *sin6) {

    struct in6_addr zero;

    memset (&zero, 0, 12);
    return (comp_with_mask ((char *)&zero, (char *)sin6, 96));
}

int ipv6_any_addr (struct in6_addr *sin6) {

    struct in6_addr zero;

    memset (&zero, 0, 16);
    return (comp_with_mask ((char *)&zero, (char *)sin6, 128));
}

#endif /* HAVE_IPV6 */

u_char *
netmasking (int family, u_char *addr, int bitlen) {

    if (family == AF_INET) {
        assert (0 <= bitlen && bitlen <= 32);
        if (bitlen == 32) return;
        memset (addr + (bitlen+7)/8, 0, 4 - (bitlen+7)/8);
        addr[bitlen/8] &=  (0xff00 >> (bitlen%8));
    }
#ifdef HAVE_IPV6
   else if (family == AF_INET6) {
        assert (0 <= bitlen && bitlen <= 128);
        if (bitlen == 128) return;
        memset (addr + (bitlen+7)/8, 0, 16 - (bitlen+7)/8);
        addr[bitlen/8] &=  (0xff00 >> (bitlen%8));
    }
#endif /* HAVE_IPV6 */
    else
	assert (0);
    return (addr);
}


int comp_with_mask (char *addr, char *dest, int mask) {
   
    if (/* mask/8 == 0 ||*/ memcmp (addr, dest, mask/8) == 0) {
        int n = mask/8;
        int m = ((-1) << (8-(mask%8)));
   
        if (mask%8 == 0 || (addr[n] & m) == (dest[n] & m))
                return (1);
    }
    return (0); 
}

#if !defined(HAVE_INET_NTOP) && defined(HAVE_ADDR2ASCII)
/*
 * mrt uses inet_ntop() defined in draft-ietf-ipngwg-bsd-api-06.txt,
 * which is different from draft-ietf-ipngwg-bsd-api-05.txt, and
 * also different from the old addr2ascii(). At least, BIND 4.9.5 has
 * the expected version of inet_ntop() in its library (libresolv.a).
 */

const char *inet_ntop (int af, const void *src, char *dst, size_t size) {
    return (addr2ascii (af, src, (af==AF_INET)?4:16, dst));
}

int inet_pton(int af, const char *src, void *dst) {
    return (ascii2addr (af, src, dst));
}
#endif
