/*
 * $Id: mrt.c,v 1.9 1997/02/18 23:17:13 masaki Exp $
 */

#include <mrt.h>
#include <trace.h>
#include <string.h>
#include <signal.h>

mrt_t *MRT;

void	mrt_process_signal ();
int	show_threads (uii_connection_t *uii);


/* init_mrt
 * initialize MRT bookeeping structures. *ALL* MRT programs must call this 
 * routine.
 */
int init_mrt(trace_t *trace) {
   mrt_thread_t *mrt_thread;

   assert (MRT == NULL);
#ifdef HAVE_THR_SETCONCURRENCY
   thr_setconcurrency (10);
#else
   pthread_init ();
#endif

   signal (SIGPIPE, mrt_process_signal);
   signal (SIGINT, mrt_process_signal);
   
   MRT = New (mrt_t);
   MRT->ll_threads = LL_Create (0);
   MRT->ll_signal_call_fn = LL_Create (0);
   MRT->ll_gateways = LL_Create (0);
   MRT->trace = trace_copy (trace);
   MRT->config_file_name = NULL;

   pthread_mutex_init (&MRT->mutex_lock, NULL);
   init_schedules (trace);

   mrt_thread = New (mrt_thread_t);
   memcpy (mrt_thread->name, "MAIN thread", 11);
   mrt_thread->thread = pthread_self ();
   LL_Add (MRT->ll_threads, mrt_thread);

   MRT->threadn = 10; /* only used on non-thread capable machines */

   init_uii (trace);
   init_timer (trace);
   init_select (trace);
   init_interfaces (trace);

   /* add some user interface hooks */
   uii_add_command (1, "show threads", show_threads);
}


int
mrt_exit(status)
    int status;
{
    trace (NORM|INFO, MRT->trace, "exit (%d)\n", status);
    exit(status);
}
  

/* mrt_thread_create
 * a wrapper around pthread create, but also adds
 * thread to bookkeeping structures
 * Note: not locking down structure because (for moment) only main
 *       thread calls this routine. Maybe this will change.
 */
mrt_thread_t *mrt_thread_create (char *name, schedule_t *schedule,
				 void (*call_fn)(), void *arg) 
{
  mrt_thread_t *mrt_thread;
  pthread_t thread = 0;

  mrt_thread = New (mrt_thread_t);
  memcpy (mrt_thread->name, name, 100);

#ifdef HAVE_LIBPTHREAD
  if (pthread_create (&thread, NULL, (void *) call_fn, arg) < 0) {
    trace (NORM, MRT->trace, "ERROR -- Could not start %s thread: %s\n",
	   name, strerror (errno));
    return (NULL);
  }
#else
  thread = MRT->threadn++;
  set_thread_id (thread);
  if (schedule)
     schedule->self = thread;
  (*call_fn) (arg);
#endif /* HAVE_LIBPTHREAD */

  mrt_thread->thread = thread;
  mrt_thread->schedule = schedule;
  LL_Add (MRT->ll_threads, mrt_thread);
}

/*
 * mrt_thread_exit
 */
int mrt_thread_exit () {
  mrt_thread_t *mrt_thread;
  pthread_t thread = pthread_self ();

  assert (thread > 0);
  pthread_mutex_lock (&MRT->mutex_lock);
  LL_Iterate (MRT->ll_threads, mrt_thread) {
     if (mrt_thread->thread == thread) {
        LL_Remove (MRT->ll_threads, mrt_thread);
        Delete (mrt_thread);
	break;
     }
  }
  pthread_mutex_unlock (&MRT->mutex_lock);

#ifdef HAVE_LIBPTHREAD
  pthread_exit (0);
#endif /* HAVE_LIBPTHREAD */
}

/* show_threads
 * Display thread and schedule info. Usually called by UII
 */
int show_threads (uii_connection_t *uii) {
  mrt_thread_t *mrt_thread;

#ifndef HAVE_LIBPTHREAD  
  uii_send_data (uii, "*** NO THREAD SUPPORT ***\r\n");
  uii_send_data (uii, "(Running in degraded mode)\r\n");
#endif /* HAVE_LIBPTHREAD */
  uii_send_data (uii, "Unix Process ID is %d\r\n", getpid ());

  uii_send_data (uii, "[id]\t%-25s\t%-15s%s\t%-15s\r\n", "Name", 
		 "Status", "Queue", "Blocked");

  LL_Iterate (MRT->ll_threads, mrt_thread) {
    uii_send_data (uii, "[%-2d]\t%-25s\t%-15s%d\t%d\r\n",
		   mrt_thread->thread, mrt_thread->name, "Running", 
		   schedule_count (mrt_thread->schedule), 0);
    
  }
}


/* mrt_process_signal
 * invoke registered handler routines, and exit on sigint
 */
void mrt_process_signal (int sig) {
  trace (NORM, MRT->trace, "signal (%d) received\n", sig);
  if (sig == SIGQUIT) mrt_exit (2);
  if (sig == SIGINT) mrt_exit (2);
}

#ifndef HAVE_LIBPTHREAD
#undef malloc
#undef realloc
#undef free

/*
 * Limitations: 1) those three functions only
 *  2) block SIGALRM only
 * Use libpthread instead
 */

void *safe_malloc (unsigned int size) {
    void *ptr;
    sigset_t omask;

    omask = block_signal (SIGALRM);
    ptr = (void *) malloc (size);
    recover_signal (omask);
    return (ptr);
}

void *safe_realloc (void *ptr, unsigned int size) {
    void *ret;
    sigset_t omask;

    omask = block_signal (SIGALRM);
    ret = (void *) realloc (ptr, size);
    recover_signal (omask);
    return (ret);
}


void safe_free (void *ptr) {
    sigset_t omask;

    omask = block_signal (SIGALRM);
    free (ptr);
    recover_signal (omask);
}

#endif /* HAVE_LIBPTHREAD */


