/*
 * $Id: freebsd.c,v 1.3 1997/02/18 23:17:09 masaki Exp $
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <version.h>
#include <sys/ioctl.h>
#include <sys/sockio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/param.h>
#include <net/route.h>
#include <sys/sysctl.h>
#include <syslog.h>
#include <New.h>
#include <linked_list.h>
#include <interface.h>

#ifdef HAVE_IPV6
#include <netinet/in6_var.h>
#endif /* HAVE_IPV6 */

extern interface_master_t *INTERFACE_MASTER;
static interface_t *ifstatus (char *name);
static int ifstatus_alias ();

int read_interfaces (trace_t *ltrace) {
  struct ifconf ifc;
  struct ifreq *ifptr, *end;
  int s, n, bufsize;
  char buffer[MAX_INTERFACES * sizeof(struct ifreq)];
  interface_t *interface;

  if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
    close(s);
    return (-1);
  }

  ifc.ifc_len = sizeof (buffer);
  ifc.ifc_buf = buffer;
  if (ioctl(s, SIOCGIFCONF, (char *)&ifc) < 0) {
    close(s);
    return (-1);
  }
  close (s);

  end = (struct ifreq *) (ifc.ifc_buf + ifc.ifc_len);
  ifptr = ifc.ifc_req;

  while (ifptr < end) {
    
    if (find_interface_byname (ifptr->ifr_name) == NULL) {
      interface = ifstatus (ifptr->ifr_name);
    }

    if (ifptr->ifr_addr.sa_len)	/* Dohw! */ 
      ifptr = (struct ifreq *) ((caddr_t) ifptr +
				ifptr->ifr_addr.sa_len -
				sizeof(struct sockaddr));
    ifptr++;
  }
  ifstatus_alias ();
}


/* get interface configuration */
static interface_t *ifstatus (char *name) {
  struct ifreq ifr;
  int s;
  u_long flags, mtu = 0;
  char tmp[MAXLINE];
  interface_t *interface;

  if ((s = socket (AF_INET, SOCK_DGRAM, 0)) < 0) {
    trace (FATAL, INTERFACE_MASTER->trace, 
	     "socket() failed for %s\n", name);
    return (NULL);
  }

  strncpy (ifr.ifr_name, name, sizeof (ifr.ifr_name));

  if (ioctl(s, SIOCGIFFLAGS, (caddr_t)&ifr) < 0) {
    trace (FATAL, INTERFACE_MASTER->trace, 
	     "SIOCGIFFLAGS failed for %s\n", name);
    close(s);
    return (NULL);
  }
  assert (sizeof (ifr.ifr_flags) == 2);
  flags = ifr.ifr_flags&0x0000ffff; /* short */

#ifdef notdef
/* sysctl () does this */
#ifdef SIOCGIFMTU
  if (ioctl(s, SIOCGIFMTU, (caddr_t)&ifr) < 0) {
    trace (FATAL, INTERFACE_MASTER->trace, 
	     "SIOCGIFMTU failed for %s\n", name);
    close(s);
    return (NULL);
  }
  mtu = ifr.ifr_mtu;
#endif
#endif
  interface = new_interface (name, flags, mtu, 0);
  close(s);
  return (interface);
}


union sockunion {
   struct sockaddr sa;
   struct sockaddr_in sin;
#ifdef HAVE_IPV6
   struct sockaddr_in6 sin6;
#endif /* HAVE_IPV6 */
} so_dst, so_gate, so_mask, so_genmask, so_ifa, so_ifp, so_brd;

#define ROUNDUP(a) \
   ((a) > 0 ? (1 + (((a) - 1) | (sizeof(long) - 1))) : sizeof(long))
      
#define NEXTADDR(w, u) \
   if (ifm->ifm_addrs & (w)) {  \
           l = ROUNDUP(((struct sockaddr *)cp)->sa_len); \
           bcopy(cp, (char *)&(u), l); cp += l; \
   }
        
static char *sockunion2char (union sockunion *u) {
    if (u->sa.sa_family == AF_INET) {
       return ((char *)&u->sin.sin_addr);
    }
#ifdef HAVE_IPV6
    else if (u->sa.sa_family == AF_INET6) {
       return ((char *)&u->sin6.sin6_addr);
    }
#endif /* HAVE_IPV6 */
    return (NULL);
}


static int ifstatus_alias () {

   size_t needed;
   char *buf, *next, *lim;
   register struct if_msghdr *ifm;
   register struct ifa_msghdr *ifam;
   interface_t *interface;

   register int l; 
   register char *cp;

   static int mib[] = {CTL_NET, PF_ROUTE, 0, 0, NET_RT_IFLIST, 0};

   if (sysctl (mib, 6, NULL, &needed, NULL, 0) < 0) {
      trace (FATAL, INTERFACE_MASTER->trace, "sysctl: %s\n", strerror (errno));
      return;
   }
   needed = ROUNDUP (needed);
   if ((buf = malloc (needed)) == 0) {
      trace (FATAL, INTERFACE_MASTER->trace, "malloc: %s\n", strerror (errno));
      return;
   }
   if (sysctl (mib, 6, buf, &needed, NULL, 0) < 0) {
      trace (FATAL, INTERFACE_MASTER->trace, "sysctl: %s\n", strerror (errno));
      return;
   }

   lim  = buf + needed;
   for (next = buf; next < lim; next += ifm->ifm_msglen) {

      int family = 0;
      char *addr = NULL;
      char *broadcast = NULL;
      int masklen = 0;
   
      ifm = (struct if_msghdr *)next;

      if (ifm->ifm_version != RTM_VERSION) {
         trace (NORM, INTERFACE_MASTER->trace, 
            "ifm_version mismatch: %d should be %d\n",
		ifm->ifm_version, RTM_VERSION);
      }

      if (ifm->ifm_type == RTM_IFINFO) {
         struct if_msghdr *ifm;
         ifm = (struct if_msghdr *)next;

	 interface = find_interface_byindex (ifm->ifm_index);
	 assert (interface);
	 interface->mtu = ifm->ifm_data.ifi_mtu;
      }
      if (ifm->ifm_type == RTM_NEWADDR) {
         ifam = (struct ifa_msghdr *)next;
         cp = (char *)ifam + sizeof (*ifam);

         NEXTADDR (RTA_DST, so_dst);
         NEXTADDR (RTA_GATEWAY, so_gate);
         NEXTADDR (RTA_NETMASK, so_mask);
         NEXTADDR (RTA_GENMASK, so_genmask);
         NEXTADDR (RTA_IFP, so_ifp);
         NEXTADDR (RTA_IFA, so_ifa);
         NEXTADDR (RTA_BRD, so_brd);

	 interface = find_interface_byindex (ifam->ifam_index);
	 assert (interface);

         if (ifam->ifam_addrs & RTA_BRD) {
            broadcast = sockunion2char (&so_brd);
         }
         if (ifam->ifam_addrs & RTA_IFA) {
	    family = so_ifa.sa.sa_family;
	    addr = sockunion2char (&so_ifa);
            if (ifam->ifam_addrs & RTA_NETMASK) {
	       so_mask.sa.sa_family = so_ifa.sa.sa_family;
	       masklen = mask2len (sockunion2char (&so_mask),
                                   (family == AF_INET)? 4: 
#ifdef HAVE_IPV6
				   (family == AF_INET6)? 16: 
#endif /* HAVE_IPV6 */
                                   0);
            }
            if (addr)
               add_addr_to_interface (interface, family, addr, masklen, 
		  broadcast);
         }
      }
   }
   free (buf);
}


/*
 * $Id: freebsd.c,v 1.3 1997/02/18 23:17:09 masaki Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <mrt.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <trace.h>
#include <signal.h>
#include <sys/param.h>
#include <sys/sysctl.h>
#include <sys/ioctl.h>
#include <sys/sockio.h>
#include <netinet/in.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <kvm.h>
#include <nlist.h>
#include <limits.h>
#include <mrt.h>
#include <interface.h>
#include <rib.h>

struct {
	struct	rt_msghdr m_rtm;
	char	m_space[512];
} m_rtmsg;

#ifdef notdef
union	sockunion {
	struct	sockaddr sa;
	struct	sockaddr_in sin;
#ifdef HAVE_IPV6
	struct	sockaddr_in6 sin6;
#endif /* HAVE_IPV6 */ 
} so_dst, so_gate, so_mask, so_genmask, so_ifa, so_ifp, so_brd;
#endif

#ifndef ROUNDUP
#define ROUNDUP(a) \
	((a) > 0 ? (1 + (((a) - 1) | (sizeof(long) - 1))) : sizeof(long))
#endif


/* 
 * kernel_update_route
 *   cmd should be one of:
 *     KERNEL_UPDATE_ADD (0)
 *     KERNEL_UPDATE_CHG (1) [not yet]
 *     KERNEL_UPDATE_DEL (2)
 * some flags for the route should be passed
 */
int kernel_update_route (int cmd, prefix_t *dest, prefix_t *nexthop)
{
  int s, rlen;
  char *cp = m_rtmsg.m_space;

  int flags, rtm_addrs;
  static int seq;
  struct rt_metrics rt_metrics;
  u_long  rtm_inits;
  register int l;
  interface_t *interface;
  int len = dest->bitlen;

  bzero((char *)&m_rtmsg, sizeof(m_rtmsg));
  bzero((char *)&rt_metrics, sizeof(rt_metrics));

  if (dest->family == AF_INET) {
    so_dst.sin.sin_addr.s_addr = prefix_tolong (dest);
    so_dst.sa.sa_len = sizeof (struct sockaddr_in);
    so_dst.sin.sin_family = AF_INET;

    so_gate.sin.sin_addr.s_addr = prefix_tolong (nexthop);
    so_gate.sa.sa_len = sizeof (struct sockaddr_in);
    so_gate.sin.sin_family = AF_INET;

    so_mask.sa.sa_len = sizeof(struct sockaddr_in);
    len2mask (len, (char *)&so_mask.sin.sin_addr, 4);
  }
#ifdef HAVE_IPV6
  else if (dest->family == AF_INET6) {

    memcpy (&so_dst.sin6.sin6_addr, prefix_tochar (dest), 16);
    so_dst.sa.sa_len = sizeof (struct sockaddr_in6);
    so_dst.sin.sin_family = AF_INET6;

/*
 * CAUTION: INRIA IPV6 DEPENDING PART
 * THE FOLLOWING CODE SHOULD BE REMOVED AFTER REVISED.
 */

    if ((interface = find_interface (nexthop))
                  && BIT_TEST (interface->flags, IFF_TUNNEL)
		  && BIT_TEST (interface->flags, IFF_POINTOPOINT)) {
        /* replace the nexthop with a compatible address of 
           configured tunnel destination */
        memset (&so_gate.sin6.sin6_addr, 0, 12);
        memcpy (&so_gate.sin6.sin6_addr+12, &interface->dest, 4);
    }
    else
/*  END OF PART */

    memcpy (&so_gate.sin6.sin6_addr, prefix_tochar (nexthop), 16);
    so_gate.sa.sa_len = sizeof (struct sockaddr_in6);
    so_gate.sin.sin_family = AF_INET6;

    so_mask.sa.sa_len = sizeof(struct sockaddr_in6);
    len2mask (len, (char *)&so_mask.sin6.sin6_addr, 16);
  }
#endif /* HAVE_IPV6 */
  else /* unknown family */
    return (-1);

  so_mask.sin.sin_family = dest->family;
  flags = RTF_UP;
  if (dest->family == AF_INET) {
    if (len == 32)
      flags |= RTF_HOST;
    if (so_dst.sin.sin_addr.s_addr != INADDR_ANY)
      flags |= RTF_GATEWAY;
  }
#ifdef HAVE_IPV6
  else if (dest->family == AF_INET6) {
    if (len == 128)
      flags |= RTF_HOST;
    if (!ipv6_any_addr (&so_dst.sin6.sin6_addr))
      flags |= RTF_GATEWAY;
  }
#endif /* HAVE_IPV6 */

  if ((s = socket(PF_ROUTE, SOCK_RAW, 0)) < 0) {
    trace (ERROR, MRT->trace, "kernel socket (%s)\n", strerror (errno));
    return (-1);
  }

  rtm_addrs =  RTA_DST | RTA_GATEWAY | RTA_NETMASK;

#undef NEXTADDR(w, u)
#define NEXTADDR(w, u) \
	if (rtm_addrs & (w)) {\
	    l = ROUNDUP(u.sa.sa_len); bcopy((char *)&(u), cp, l); cp += l;\
	}

#define rtm m_rtmsg.m_rtm
  assert (cmd == KERNEL_UPDATE_ADD || cmd == KERNEL_UPDATE_DEL);
  if (cmd == KERNEL_UPDATE_ADD) 	
    rtm.rtm_type = RTM_ADD;
  else if (cmd == KERNEL_UPDATE_DEL)
    rtm.rtm_type = RTM_DELETE;
  rtm.rtm_flags = flags;
  rtm.rtm_version = RTM_VERSION;
  rtm.rtm_seq = ++seq;
  rtm.rtm_addrs = rtm_addrs;
  rtm.rtm_rmx = rt_metrics;
  /*rtm.rtm_inits = 0;*/

  NEXTADDR(RTA_DST, so_dst);
  NEXTADDR(RTA_GATEWAY, so_gate);
  NEXTADDR(RTA_NETMASK, so_mask);
#ifdef notdef
  NEXTADDR(RTA_GENMASK, so_genmask);
  NEXTADDR(RTA_IFP, so_ifp);
  NEXTADDR(RTA_IFA, so_ifa);
#endif
  rtm.rtm_msglen = l = cp - (char *)&m_rtmsg;
#undef NEXTADDR
#undef rtm

  if ((rlen = write(s, (char *)&m_rtmsg, l)) < 0) {
    close (s);
    if (errno == EEXIST) {return (1);} /* route already exists */
    trace (ERROR, MRT->trace, "kernel write (%s)\n", strerror (errno));
    return (-1);
  }
  close (s);
  return (1);
}


#ifdef notdef
static char *sockunion2char (union sockunion *u) {
    if (u->sa.sa_family == AF_INET) {
       return ((char *)&u->sin.sin_addr);
    }
#ifdef HAVE_IPV6
    else if (u->sa.sa_family == AF_INET6) {
       return ((char *)&u->sin6.sin6_addr);
    }
#endif /* HAVE_IPV6 */
    return (u->sa.sa_data);
}   
#endif

#define NEXTADDR(w, u) \
   if (rtm->rtm_addrs & (w)) {  \
           l = ROUNDUP(((struct sockaddr *)cp)->sa_len); \
           bcopy(cp, (char *)&(u), l); cp += l; \
   }

kernel_read_rt_table (void (*add_kernel_route)()) {
   size_t needed;
   static int mib[] = {CTL_NET, PF_ROUTE, 0, 0, NET_RT_DUMP, 0};
   char *buf, *next, *lim, *cp;
   struct rt_msghdr *rtm;

   int family, masklen, l;
   u_char *dest, *nexthop, *mask;

   if (sysctl (mib, 6, NULL, &needed, NULL, 0) < 0) {
      trace (NORM, MRT->trace, "sysctl: %s\n", strerror (errno));
      return;
   }
   if ((buf = malloc (needed)) == 0) {
      trace (NORM, MRT->trace, "malloc: %s\n", strerror (errno));
      return;
   }
   if (sysctl (mib, 6, buf, &needed, NULL, 0) < 0) {
      trace (NORM, MRT->trace, "sysctl: %s\n", strerror (errno));
      return;
   }

   lim  = buf + needed;
   for (next = buf; next < lim; next += rtm->rtm_msglen) {
      rtm = (struct rt_msghdr *)next;

      if (rtm->rtm_version != RTM_VERSION) { 
         trace (NORM, MRT->trace,
            "rtm_version mismatch: %d should be %d\n",
                rtm->rtm_version, RTM_VERSION);
      }
   
      if (rtm->rtm_type == RTM_GET) {

#if 0
printf("rtm->rtm_flags=%x rtm->rtm_addrs=%x\n", rtm->rtm_flags, rtm->rtm_addrs);
#endif
         if (BIT_TEST (rtm->rtm_flags, ~(RTF_UP|RTF_GATEWAY|RTF_HOST))) 
	    continue;
         if (!BIT_TEST (rtm->rtm_flags, RTF_UP)) continue;
         if (!BIT_TEST (rtm->rtm_flags, RTF_GATEWAY)) continue;

         cp = (char *)rtm + sizeof (*rtm);

         NEXTADDR (RTA_DST, so_dst);
         NEXTADDR (RTA_GATEWAY, so_gate);
         NEXTADDR (RTA_NETMASK, so_mask);

         family = so_dst.sa.sa_family;
         dest = sockunion2char (&so_dst);
         nexthop = sockunion2char (&so_gate);
         so_mask.sa.sa_family = family;
         mask = sockunion2char (&so_mask);

         if (family != AF_INET
#ifdef HAVE_IPV6
             && family != AF_INET6
#endif /* HAVE_IPV6 */
             ) continue;

         if (BIT_TEST (rtm->rtm_flags, RTF_HOST)) {
            masklen = (family == AF_INET)?32:128;
         }
         else {
            /* I don't know the reason, 
               but it's needed for getting rid of strange subnetmask */
            if (family == AF_INET && *(u_long *)dest == *(u_long *)mask)
               so_mask.sa.sa_len = 0;

            if (!BIT_TEST (rtm->rtm_addrs, RTA_NETMASK) || so_mask.sa.sa_len <= 2) {
               /* not specific netmask */
               if (family == AF_INET) {
                  /* natural mask */
                  if (*(u_long *)dest == INADDR_ANY)
                     masklen = 0;
                  else
                     masklen = (dest[0]<128)?8:
                               (dest[0]<192)?16:
                               (dest[0]<224)?24:32;
               }
               else {
                  masklen = 0;
               }
            }
            else {
               char tmp[16];
               memset (tmp, 0, sizeof (tmp));
               memcpy (tmp, sockunion2char (&so_mask), so_mask.sa.sa_len - 
                           (sockunion2char (&so_mask) - (char *)&so_mask));
               masklen = mask2len (tmp, (family==AF_INET)?4:16);
            }
         }

         if (so_gate.sa.sa_family != family) continue;
         add_kernel_route (family, dest, nexthop, masklen);

      }
   }
   free (buf);
}

