/*
 * $Id: common.c,v 1.2 1997/02/01 01:19:39 masaki Exp $
 */

#include <config.h>
#include <stdio.h>
#include <version.h>
#include <sys/ioctl.h>
/*#include <sys/sockio.h>*/
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <New.h>
#include <linked_list.h>
#include <interface.h>


interface_master_t *INTERFACE_MASTER;
interface_t *index2if[MAX_INTERFACES];
char *print_iflags (char *tmp, int len, u_long flags);


interface_t *new_interface (char *name, u_long flags, u_long mtu, int index) {
   interface_t *interface;
   char tmp[MAXLINE];

   if (interface = find_interface_byname (name)) {
     return (interface);
   }
   if (index <= 0)
     index = INTERFACE_MASTER->number+1;

   interface = New (interface_t);
   memset (interface, 0, sizeof (*interface));
   LL_Add (INTERFACE_MASTER->ll_interfaces, interface);

   interface->ll_addr = LL_Create (0);
   interface->flags = flags;
   interface->mtu = mtu;
   memset (interface->dlist_in, 0, sizeof (interface->dlist_in));
   memset (interface->dlist_out, 0, sizeof (interface->dlist_out));
   interface->metric_in = 1;
   interface->metric_out = 0;
   strcpy (interface->name, name);

   /* set bit for easy identification */
   assert (INTERFACE_MASTER->number < sizeof (interface->bit) * 8);
   interface->bit = 1 << INTERFACE_MASTER->number;
   INTERFACE_MASTER->number++;   
   interface->index = index;
   index2if[index] = interface;

   if (INTERFACE_MASTER->max_mtu < mtu)
     INTERFACE_MASTER->max_mtu = mtu;

   trace (NORM, INTERFACE_MASTER->trace, 
     "Interface %s flags %s mtu %d index %d\n", 
     name, print_iflags (tmp, sizeof (tmp), flags), mtu, index);

   return (interface);
}


ll_addr_t * New_Addr (int family, u_char *addr, int bitlen, 
	u_char *broadcast) {

   ll_addr_t *if_addr;

   if_addr = New (ll_addr_t);
   assert (addr);
   if_addr->prefix = New_Prefix (family, addr, bitlen);
   if (broadcast)
      if_addr->broadcast = New_Prefix (family, broadcast, bitlen);
   else
      if_addr->broadcast = NULL;

   return (if_addr);
}


interface_t *add_addr_to_interface (interface_t *interface, int family,
	char *addr, int len, char *broadcast) {

   char tmp[MAXLINE];
   ll_addr_t *if_addr;

   assert (interface);

#ifdef HAVE_IPV6 
   assert (family == AF_INET || family == AF_INET6);
#else
   assert (family == AF_INET);
#endif /* HAVE_IPV6 */

   if_addr = New_Addr (family, addr, len, broadcast);
   LL_Add (interface->ll_addr, if_addr);
   
   if (family == AF_INET && interface->primary == NULL) {
      interface->primary = if_addr;
   }
#ifdef HAVE_IPV6 
   else if (family == AF_INET6) {
      if (interface->primary6 == NULL || (!ipv6_global_unicast_addr (
	     prefix_toaddr6 (interface->primary6->prefix)) &&
	     ipv6_global_unicast_addr ((struct in6_addr *) addr))) {
         interface->primary6 = if_addr;
      }
      if (interface->link_local == NULL &&
	 ipv6_link_local_addr ((struct in6_addr *) addr)) {
         interface->link_local = if_addr;
      }
   }
#endif /* HAVE_IPV6 */

#ifdef mrt_compat
   if (family == AF_INET && interface->prefix == NULL) {
      interface->prefix = New_Prefix (family, addr, len);
      if (broadcast)
         memcpy (&interface->broadcast, broadcast, 4);
   }
#ifdef HAVE_IPV6 
   else if (family == AF_INET6 && interface->prefix6 == NULL) {
      interface->prefix6 = New_Prefix (family, addr, len);
      if (broadcast)
         memcpy (&interface->dest6, broadcast, 16);
   }
#endif /* HAVE_IPV6 */
#endif

#ifdef HAVE_IPV6
   /* the first non-loopback ipv6 capable interface becomes our default */
   if (family == AF_INET6 &&
	 strncmp (interface->name, "sit", 3) != 0 && /* XXX */
         INTERFACE_MASTER->default_interface == NULL &&
         !BIT_TEST (interface->flags, IFF_LOOPBACK) &&
	 ipv6_link_local_addr ((struct in6_addr *) addr)) {
     INTERFACE_MASTER->default_interface = interface;
     trace (NORM, INTERFACE_MASTER->trace,
          "DEFAULT Interface %s\n", interface->name);
   }
#else
   /* the first non-loopback interface becomes our default */
   if (family == AF_INET &&
         INTERFACE_MASTER->default_interface == NULL &&
         !BIT_TEST (interface->flags, IFF_LOOPBACK)) {
     INTERFACE_MASTER->default_interface = interface;
     trace (NORM, INTERFACE_MASTER->trace,
          "DEFAULT Interface %s\n", interface->name);
   }
#endif

   tmp[0] = '\0';
   if (family == AF_INET) {
      if (broadcast)
         sprintf(tmp, " %s %s", BIT_TEST (interface->flags, IFF_POINTOPOINT)? 
                             "dest": "broadcast", 
            inet_ntoa (*(struct in_addr *)broadcast));
      trace (NORM, INTERFACE_MASTER->trace,
          "Interface %s inet %s/%d%s\n", interface->name, 
          inet_ntoa (*(struct in_addr *)addr), len, tmp);
   }
#ifdef HAVE_IPV6 
   else if (family == AF_INET6) {
      if (broadcast)
         sprintf(tmp, " %s %s", BIT_TEST (interface->flags, IFF_POINTOPOINT)? 
                             "dest": "broadcast", 
         inet_ntop (family, broadcast, NULL, 0));
      trace (NORM, INTERFACE_MASTER->trace,
          "Interface %s inet6 %s/%d%s\n", interface->name, 
		inet_ntop (family, addr, NULL, 0), 
                len, tmp);
   }
#endif /* HAVE_IPV6 */

   return (interface);
}


int init_interfaces (trace_t *ltrace) {

    assert (INTERFACE_MASTER == NULL);
    INTERFACE_MASTER = New (interface_master_t);
    INTERFACE_MASTER->ll_interfaces = LL_Create (0);
    INTERFACE_MASTER->trace = trace_copy (ltrace);
    INTERFACE_MASTER->number = 0;
    INTERFACE_MASTER->max_mtu = 576;

    read_interfaces ();
}


/* 
 * Given a prefix, find an interface that prefix is on
 */
interface_t *find_interface (prefix_t *prefix) {

  return (find_interface_flags (prefix, 0));
} 


/* 
 * Given a prefix, find an interface with the specified flags 
 * that prefix is on if flags == 0, then see everything
 */
interface_t *find_interface_flags (prefix_t *prefix, int flags) {
  interface_t *interface;
  ll_addr_t *ll_addr;
  u_char *addr, *dest;
  int llocal = 0;
   
  addr = prefix_tochar (prefix);
#ifdef HAVE_IPV6 
  if (ipv6_link_local_addr ((struct in6_addr *)addr)) {
     llocal++;
  }
#endif /* HAVE_IPV6 */
  LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
    if (flags && !BIT_TEST (interface->flags, flags)) continue;

    if (interface->ll_addr) {
      LL_Iterate (interface->ll_addr, ll_addr) {
        if (ll_addr->prefix == NULL) continue;
        if (ll_addr->prefix->family != prefix->family) continue;
        if (!llocal && ll_addr->prefix->bitlen > prefix->bitlen) continue;
        
        dest = prefix_tochar (ll_addr->prefix);
        if (BIT_TEST (interface->flags, IFF_POINTOPOINT)) {
           if (ll_addr->broadcast == NULL) continue;
           dest = prefix_tochar (ll_addr->broadcast);
        }
  
#if 0
printf("name = %s ", interface->name);
if (ll_addr->prefix->family == AF_INET6) {
printf("addr = %s ", inet_ntop (AF_INET6, addr, NULL, 0));  

printf("dest = %s ", inet_ntop (AF_INET6, dest, NULL, 0));
}   
else {
printf("addr = %s ", inet_ntop (AF_INET, addr, NULL, 0));  
printf("dest = %s ", inet_ntop (AF_INET, dest, NULL, 0));
}   
printf("bitlen = %d\n", ll_addr->prefix->bitlen);
#endif  
        if (addr && dest && comp_with_mask (addr, dest, 
              (llocal)?128:ll_addr->prefix->bitlen))
           return (interface);
        
      }
    }   
  }
           
  return (NULL);
} 


/*
 * Given a prefix, find interfaces on that prefix's network
 *   matches exactly or includes
 */
LINKED_LIST *find_network (prefix_t *prefix) {
  interface_t *interface;
  ll_addr_t *ll_addr;
  u_char *addr, *dest;
  LINKED_LIST *ll = NULL;

  addr = prefix_tochar (prefix);

  LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
#ifdef notdef
    if (BIT_TEST (interface->flags, IFF_LOOPBACK)) continue;
#endif

    if (interface->ll_addr) {
      LL_Iterate (interface->ll_addr, ll_addr) {
        if (ll_addr->prefix == NULL) continue;
        if (ll_addr->prefix->bitlen < prefix->bitlen) continue;

        dest = prefix_tochar (ll_addr->prefix);
        if (BIT_TEST (interface->flags, IFF_POINTOPOINT)) {
	   if (ll_addr->broadcast == NULL) continue;
           dest = prefix_tochar (ll_addr->broadcast);
        }

#if 0
printf("name = %s ", interface->name);
if (ll_addr->prefix->family == AF_INET6) {
printf("addr = %s ", ineto_ntop (AF_INET6, addr, NULL, 0));
printf("dest = %s ", ineto_ntop (AF_INET6, dest, NULL, 0));
}
else {
printf("addr = %s ", ineto_ntop (AF_INET, addr, NULL, 0));
printf("dest = %s ", ineto_ntop (AF_INET, dest, NULL, 0));
}
printf("bitlen = %d\n", prefix->bitlen);
#endif
        if (addr && dest && 
            comp_with_mask (addr, dest, prefix->bitlen)) {
          if (ll == NULL)
            ll = LL_Create (0);
	  LL_Add (ll, interface);
          break;
        }
      }
    }
  }

  return (ll);
}


/* 
 * Given a name, find an interface that has the name
 */
interface_t *find_interface_byname (char *name) {
  interface_t *interface;

  LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
    if (strcasecmp (name, interface->name) == 0)
      return (interface);
  }
  return (NULL);
}


/* 
 * if it belongs to a local interface, its pointer returned
 * this is used to detect when we have received a broadcast packet from 
 * ourselves in rip, and ripng
 */
interface_t *find_interface_local (prefix_t *prefix) {
  interface_t *interface;
  ll_addr_t *ll_addr;
  int family = prefix->family;
  char *addr = prefix_tochar (prefix);

#ifndef HAVE_IPV6
  assert (prefix->family == AF_INET);
#else
  assert (prefix->family == AF_INET || prefix->family == AF_INET6);
#endif /* HAVE_IPV6 */

  LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
    if (interface->ll_addr) {
      LL_Iterate (interface->ll_addr, ll_addr) {
        if (ll_addr->prefix == NULL) continue;
        if (ll_addr->prefix->family != family) continue;

       /*
        * The local side addresses are checked even in case of p-to-p
        */
        addr = prefix_tochar (ll_addr->prefix);
        if (memcmp (prefix_tochar (prefix), addr, (family==AF_INET)?4:16) == 0)
	    return (interface);
      }
    }
  }
  
  return (NULL);
}


/* 
 * compatible purpose
 */
interface_t *local_interface (int family, u_char *cp) {
  prefix_t ptmp, *prefix = &ptmp;

  if ((prefix->family = family) == AF_INET) {
    prefix->bitlen = 32;
    memcpy (&prefix->add.sin, cp, 4);
  }
#ifdef HAVE_IPV6
  else if (family == AF_INET6) {
    prefix->bitlen = 128;
    memcpy (&prefix->add.sin6, cp, 16);
  }
#endif /* HAVE_IPV6 */

  return (find_interface_local (prefix));
}


/* show_interfaces
 * Dump various interface stats/info to a socket
 */
int show_interfaces (uii_connection_t *uii) {
  interface_t *interface;
  ll_addr_t *ll_addr;
  char tmp[MAXLINE];

  LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
    uii_send_data (uii, "Interface %s %s mtu %d\r\n", 
		   interface->name, 
		   print_iflags (tmp, MAXLINE, interface->flags),
		   interface->mtu);
    
#ifdef notdef
    if (interface->prefix != NULL) {
      uii_send_data (uii, "  inet4 %s\r\n", prefix_toa (interface->prefix));
    }
#ifdef HAVE_IPV6 
    if (interface->prefix6 != NULL) {
      uii_send_data (uii, "  inet6 %s\r\n", prefix_toa (interface->prefix6));
    }
#endif /* HAVE_IPV6 */
#endif
    LL_Iterate (interface->ll_addr, ll_addr) {
      assert (ll_addr->prefix);

      tmp[0] = '\0';
      if (ll_addr->broadcast)
         sprintf(tmp, " %s %s", BIT_TEST (interface->flags, IFF_POINTOPOINT)? 
           "-->": "broadcast", prefix_toa (ll_addr->broadcast));

      uii_send_data (uii, "  %s %s/%d%s\r\n", 
	(ll_addr->prefix->family == AF_INET)?"inet":
#ifdef HAVE_IPV6 
	(ll_addr->prefix->family == AF_INET6)?"inet6":
#endif /* HAVE_IPV6 */
        "???",
        prefix_toa (ll_addr->prefix), ll_addr->prefix->bitlen, tmp);
    }
    uii_send_data (uii, "\r\n");
  }
}


typedef struct {
    u_int bit;
    char *name;
} bits;
static bits iflags[] = {
    {IFF_UP,            "UP" },
    {IFF_BROADCAST,     "BROADCAST" },
    {IFF_DEBUG,         "DEBUG" },
    {IFF_LOOPBACK,      "LOOPBACK" },
    {IFF_POINTOPOINT,   "POINTOPOINT" }, 
#ifdef  IFF_NOTRAILERS
    {IFF_NOTRAILERS,    "NOTRAILERS" },
#endif  /* IFF_NOTRAILERS */
    {IFF_RUNNING,       "RUNNING" },
#ifdef  IFF_NOARP
    {IFF_NOARP,         "NOARP" },
#endif  /* IFF_NOARP */
    {IFF_PROMISC,       "PROMISC" },
    {IFF_ALLMULTI,      "ALLMULTI" },
#ifdef  IFF_OACTIVE
    {IFF_OACTIVE,       "OACTIVE" },
#endif  /* IFF_OACTIVE */
#ifdef  IFF_SIMPLEX
    {IFF_SIMPLEX,       "SIMPLEX" },
#endif  /* IFF_SIMPLEX */
#ifdef  IFF_LINK0
    {IFF_LINK0,         "LINK0" },
    {IFF_LINK1,         "LINK1" },
    {IFF_LINK2,         "LINK2" },
#endif  /* IFF_LINK0 */
#ifdef  IFF_MULTICAST
    {IFF_MULTICAST,    "MULTICAST" },
#endif  /* IFF_MULTICAST */
#ifdef  IFF_USERDEF
    {IFF_USERDEF,       "USERDEF" },
#endif  /* IFF_USERDEF */
#ifdef  IFF_TUNNEL
    {IFF_TUNNEL,        "TUNNEL" },
#endif  /* IFF_TUNNEL */
    {0, NULL}
};  


char *print_iflags (char *tmp, int len, u_long flags) {
  char *cp = tmp;
  int i, l, n = 0;

  assert (len >= 3);
  tmp[n++] = '<';
  for (i = 0; iflags[i].bit; i++) {
    if (!BIT_TEST (flags, iflags[i].bit))
      continue;
    if (n > 1)
      tmp[n++] = ',';
    l = strlen (iflags[i].name);
    if (n+l+1 > len-1)
      break;
    strcpy (tmp+n, iflags[i].name);
    n += l;
  }
  tmp[n++] = '>';
  tmp[n] = '\0';
  return (tmp);
}


int mask2len (u_char *mask, int bytes) {
  int i, j, bitlen = 0;

  if (bytes == 0 || mask == NULL)
      return (bitlen);
  for (i = 0; i < bytes; i++, bitlen += 8) {
    if (mask[i] != 0xff) break;
  } 
  if (i != bytes && mask[i]) {
    for (j = 7; j > 0; j--, bitlen++) {
      if ((mask[i] & (1 << j)) == 0) break;
    } 
  } 
  /* this doesn't check if the rest of bits are all 0 or not */
  /* test in another place if contiguous or not */
  
#if 0
  if (bitlen == 0)
      bitlen = bytes*8;
#endif
  return (bitlen);
}


char *len2mask (int bitlen, u_char *mask, int bytes) {
  int i;

  assert (bytes*8 >= bitlen);
  if (bitlen == 0) {
    memset (mask, 0, bytes);
    return (mask);
  }
  if (mask == NULL || bytes == 0)
    return (NULL);
  for (i = 0; i < (bitlen / 8); i++)
    mask[i] = 0xff;
  if (bitlen < (bytes * 8)) 
    mask[i++] = 0xff << (8 - (bitlen & 7)); 
  for (; i < bytes; i++) 
    mask[i] = 0;
  return (mask);
}


interface_t *find_interface_byindex (int index) {
  interface_t *interface;

#ifdef notdef
  LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
    if (interface->index == index)
      return (interface);
  }
  return (NULL);
#else
  assert (index < MAX_INTERFACES);
  return (index2if[index]);
#endif
}

