/*
 * $Id: bgp.h,v 1.5 1997/02/01 01:19:30 masaki Exp $
 */

#ifndef _BGP_H
#define _BGP_H

#include <mrt.h>
#include <select.h>
#include <timer.h>
#include <bgp_proto.h>
#include <trace.h>
#include <io.h>
#include <schedule.h>
#include <hash.h>
#include "view.h"

/* various settable attributes */
#define DEFAULT_PEER_NAME	"glasshouse.merit.edu"
#define DEFAULT_PEER_AS		185
#define DEFAULT_MY_AS		6503

#define DEFAULT_DATAFILE	"/tmp/bgp.data"


#define DEFAULT_BGP_VERSION	4

#define DEFAULT_CONNET_RETRY_INTERVAL	60
#define DEFAULT_START_INTERVAL		10
#define DEFAULT_KEEPALIVE_INTERVAL	30
#define DEFAULT_HOLDTIME_INTERVAL	180

/* pdu types */
#define BGP_OPEN		1
#define BGP_UPDATE		2
#define BGP_NOTIFICATION	3
#define BGP_KEEPALIVE		4

extern char *sbgp_states[];
extern char *sbgp_pdus[];
extern char *sbgp_events[];
extern char *s_bgp_error[];
extern char *s_bgp_error_header[];
extern char *s_bgp_error_open[];
extern char *s_bgp_error_update[];


#define BGP_TRANSPARENT_AS	1
#define BGP_TRANSPARENT_NEXTHOP	2


typedef struct _bgp_peer_t {
   pthread_mutex_t	mutex_lock;
   pthread_t		self;		/* my thread ID */
   gateway_t		*gateway;
   struct sockaddr_in	peer_addr;	/* socket open to peer */
   u_int		sockfd;		/* sockect connected to peer */
   u_int 		view;		/* RIB to use when sending routes */
   u_int		bit;		/* id bit (used by view in mask) */
   u_long		options;	/* see options above */


   u_int		state;
   u_char		code;
   u_char 		subcode;

   schedule_t		*schedule;	/* event processing */

   /* packet parsing storage */
   u_char               buffer[BGPMAXPACKETSIZE];
   u_char               packet[BGPMAXPACKETSIZE];
   u_char 		*read_ptr;      /* current ptr of read in buffer */
   u_char 		*start_ptr;     /* current ptr of start of packet */
   u_int		room_in_buffer; /* bytes left in buffer */
   
   /* timers */
  mtimer_t		*timer_Start;
  mtimer_t		*timer_ConnectRetry;
  mtimer_t		*timer_KeepAlive;
  mtimer_t		*timer_HoldTime;

   /* time value of intervals in seconds */
   u_int		Start_Interval;
   u_int		ConnetRetry_Interval;
   u_int		KeepAlive_Interval;
   u_int		HoldTime_Interval;

   /* tracing stuff */
   trace_t		*trace;
} bgp_peer_t;


typedef struct _bgp_t {
  short			my_as;
  u_long		my_id;
  int			cport;		/* port to send out open on */
  int			lport;		/* port to listen for open on */
  int			sockfd;		/* socket we are listening on */
  view_t		*view;		/* view from which it gets routes */

  LINKED_LIST 		*ll_bgp_peers;
  int			num_peers;	/* number of peers */
  int			accept_all_peers;
  trace_t		*trace;

  view_t*		views[32];
  HASH_TABLE*		attr_hash;	/* hash of bgp attribute blocks */
   
  void (*peer_down_call_fn)();
  void (*update_call_fn)();
  void (*peer_established_call_fn)();

} bgp_t;


extern bgp_t *BGP;

enum BGP_ATTR {
   BGP_NULL = 0,
   BGP_MY_AS,
   BGP_MY_ID,
   BGP_TRACE_STRUCT,
   BGP_PEER_DOWN_FN,
   BGP_PEER_ESTABLISHED_FN,
   BGP_RECV_UPDATE_FN,
   BGP_ACCEPT_ALL_PEERS,	/* start peering sesssion with anyone */
   BGP_LPORT,			/* port to listen for connections on */
   BGP_CPORT,			/* port to attempt connections to */
   BGP_USE_PORTSERVER,		/* use portserver library for listening */
};


/* a convenience structure to store all prefixes with same BGP
 * attributes while we are building BGP update packets
 */
typedef struct _bgp_bucket_t {
   LINKED_LIST *ll_prefix;
   void *attr;		
} bgp_bucket_t;



/* from gated aspath.h */

/*
 * Bit definitions for the attribute flags byte
 */
#define	PA_FLAG_OPT	0x80	/* attribute is optional */
#define	PA_FLAG_TRANS	0x40	/* attribute is transitive */
#define	PA_FLAG_PARTIAL	0x20	/* incomplete optional, transitive attribute */
#define	PA_FLAG_EXTLEN	0x10	/* extended length flag */

#define	PA_FLAG_ALL  (PA_FLAG_OPT|PA_FLAG_TRANS|PA_FLAG_PARTIAL|PA_FLAG_EXTLEN)
#define	PA_FLAG_OPTTRANS	(PA_FLAG_OPT|PA_FLAG_TRANS)


/*
 * BGP version 4 attribute type codes (the dorks moved metric!).
 
#define	PA4_TYPE_INVALID	0
#define	PA4_TYPE_ORIGIN		1
#define	PA4_TYPE_ASPATH		2
#define	PA4_TYPE_NEXTHOP	3
#define	PA4_TYPE_METRIC		4
#define	PA4_TYPE_LOCALPREF	5
#define	PA4_TYPE_ATOMICAGG	6
#define	PA4_TYPE_AGGREGATOR	7
#define	PA4_MAXTYPE		7*/

/*
 * BGP4 subcodes for the AS_PATH attribute
 *
#define	PA_PATH_NOTSETORSEQ	0	
#define	PA_PATH_SET		1
#define	PA_PATH_SEQ		2
#define	PA_PATH_MAXSEGLEN	255	*/





/* in bgp_pdu.c */
int bgp_get_pdu (bgp_peer_t *peer);
int process_pdu (bgp_peer_t *peer, int state, u_char *cp);
u_char *get_packet (bgp_peer_t *peer, int *more_flag);
int bgp_process_update (bgp_peer_t *peer, u_char *cp);
int bgp_process_open (bgp_peer_t *peer, u_char *cp);
int bgp_process_notify (bgp_peer_t *peer, u_char *cp);
void bgp_send_open (bgp_peer_t *peer);
void bgp_send_keepalive (bgp_peer_t *peer);
void bgp_send_notification (bgp_peer_t *peer, 
			    int sockfd, struct sockaddr_in *addr, 
			    u_char code, u_char subcode);
void bgp_send_update (bgp_peer_t *peer, int len, u_char *data);

/* in bgp_util.c */
char *bgp_notify_to_string (u_char code, u_char subcode);
int bgp_start_transport_connection (bgp_peer_t *peer);
void bgp_release_resources (bgp_peer_t *peer);
bgp_peer_t *Add_BGP_Peer (prefix_t *prefix, short AS, trace_t *trace);
void bgp_flush_socket (bgp_peer_t *peer);
void bgp_broken_pipe ();
void bgp_die ();
int default_bgp_peer_down_fd (bgp_peer_t *peer);
void bgp_start_peer_thread (bgp_peer_t *peer);
bgp_peer_t *find_bgp_peer (gateway_t *gateway);


/* in bgp_sm.c */
void bgp_connect_succeded (Descriptor_Struct *descrip, bgp_peer_t *peer);
void bgp_change_state (bgp_peer_t *peer, int state, int event);
void bgp_sm_state_idle (bgp_peer_t *peer, int event);
void bgp_sm_state_active (bgp_peer_t *peer, int event);
void bgp_sm_state_connect (bgp_peer_t *peer, int event);
void bgp_sm_state_opensent (bgp_peer_t *peer, int event);
void bgp_sm_state_openconfirm (bgp_peer_t *peer, int event);
void bgp_sm_state_established (bgp_peer_t *peer, int event);
void bgp_sm_process_event (bgp_peer_t *peer, int state, int event);


/* in bgp_timer.c */
void bgp_timer_ConnectRetry_fire (mtimer_t *timer, bgp_peer_t *peer);
void bgp_timer_KeepAlive_fire (mtimer_t *timer, bgp_peer_t *peer);
void bgp_timer_HoldTime_fire (mtimer_t *timer, bgp_peer_t *peer);
void bgp_timer_StartTime_fire (mtimer_t *timer, bgp_peer_t *peer);


/* in bgp_thread.c */
void bgp_schedule_timer (mtimer_t *timer, bgp_peer_t *peer);
void bgp_schedule_socket ();

/* in bgp_attr.c */
u_char *bgp_create_update_msg (short *size, 
                               bgp_attr_t *p_bgp_route_attr,
                               LINKED_LIST *ll_ann_prefixes,
                               LINKED_LIST *ll_with_prefixes);

/* public functions */
int init_BGP (trace_t *ltrace);

#endif /* _BGP_H */
