/*
%%% copyright-cmetz
This software is Copyright 1996 by Craig Metz, All Rights Reserved.
The Inner Net License Version 2 applies to this software.
You should have received a copy of the license with this software. If
you didn't get a copy, you may request one from <license@inner.net>.

v0.04 - NOT FOR REDISTRIBUTION
*/
#define NAME "netd"
#define VERSION "v0.04 - NOT FOR REDISTRIBUTION"
#define USAGE "[-Vhd] [-C config file]"

#include <sys/types.h>
#include <pwd.h>
#include <grp.h>
#include <netdb.h>
#include <unistd.h>
#include <syslog.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <signal.h>
#include <unistd.h>
#include <ctype.h>
#include <setjmp.h>

#include <netinet6/in6.h>

#include "support.h"

#include "inner-apps.h"

int minfd = INT_MAX, maxfd = -1;
fd_set allfds;
fd_set servicefds;
fd_set larvalfds;

#define INSTANCES_MAX 128
#define USERNAME_MAX 32

struct rule {
  struct rule *next;
  int line;
  int instances;
  su prefix;
  su mask;
  int ident;
  int checkname;
  int downgrade;
  char *argv[10];
  char clbuf[256];
  int ninstances;
  uid_t uid;
  gid_t gid;
  char user[USERNAME_MAX];
  char directory[PATH_MAX];
};

struct service {
  struct service *next;
  int line;
  struct rule *rules;
  int socktype;
  fd_set fds;
} *services = NULL;

#define FDTOSERVICE_PRIME 17
struct fdtoservice {
  struct fdtoservice *next;
  int fd;
  struct service *service;
} *fdtoservices[FDTOSERVICE_PRIME];

#define PIDTORULE_PRIME 17
struct pidtorule {
  struct pidtorule *next;
  pid_t pid;
  int fd;
  struct rule *rule;
} *pidtorules[PIDTORULE_PRIME];

struct larvalmsg {
  pid_t pid;
  struct rule *rule;
};

int debug = 0;
#define CF_DIR "/etc/"

int handshaketime = 5;

#define CF_TOPLEVEL 0
#define CF_SERVICE  1
#define CF_RULE     2

#define CF_TOKEN_END 0
#define CF_TOKEN_SERVICE 1
#define CF_TOKEN_RULE 2
#define CF_TOKEN_BIND 3
#define CF_TOKEN_SOCKTYPE 4
#define CF_TOKEN_FAMILY 5
#define CF_TOKEN_QUEUELEN 6
#define CF_TOKEN_PREFIX 7
#define CF_TOKEN_RUN 8
#define CF_TOKEN_INSTANCES 9
#define CF_TOKEN_USER 10
#define CF_TOKEN_DIRECTORY 11
#define CF_TOKEN_CHECKNAME 12
#define CF_TOKEN_IDENT 13
#define CF_TOKEN_DOWNGRADE 14
struct nrl_nametonum cf_tokens[] = {
  { CF_TOKEN_END, "}", 1 },
  { CF_TOKEN_SERVICE, "service", 2 },
  { CF_TOKEN_RULE, "rule", 2 },
  { CF_TOKEN_BIND, "bind", 2 },
  { CF_TOKEN_SOCKTYPE, "socktype", 2 },
  { CF_TOKEN_FAMILY, "family", 2 },
  { CF_TOKEN_QUEUELEN, "queuelen", 2 },
  { CF_TOKEN_PREFIX, "prefix", 2 },
  { CF_TOKEN_RUN, "run", 0 },
  { CF_TOKEN_INSTANCES, "instances", 2 },
  { CF_TOKEN_USER, "user", 2 },
  { CF_TOKEN_DIRECTORY, "directory", 2 },
  { CF_TOKEN_CHECKNAME, "checkname", 2 },
  { CF_TOKEN_IDENT, "ident", 2 },
  { CF_TOKEN_DOWNGRADE, "downgrade", 2 },
  { -1, NULL, 0 }
};

#define CF_STRENGTH_SKIP 0
#define CF_STRENGTH_TRY 1
#define CF_STRENGTH_REQUIRE 2
struct nrl_nametonum cf_strengths[] = {
  { CF_STRENGTH_SKIP, "skip", 0 },
  { CF_STRENGTH_TRY, "try", 0 },
  { CF_STRENGTH_REQUIRE, "require", 0 },
  { -1, NULL, 0 }
};

char buffer[256];

void turn()
{
  struct pidtorule **pp, *pidtorule;
  pid_t pid;
  struct rule *rule;

  while(1) {
    if ((pid = waitpid(-1, NULL, WNOHANG)) < 0) {
      if (errno == ECHILD)
	break;
      syslog(LOG_ERR, "turn: waitpid: %s(%d)", strerror(errno), errno);
      if (debug)
	abort();
      continue;
    };
    if (!pid)
      break;
    for (pp = &(pidtorules[pid % PIDTORULE_PRIME]); *pp && ((*pp)->pid != pid); pp = &((*pp)->next));
    if (!*pp) {
      if (debug) {
	syslog(LOG_DEBUG, "turn: internal error: unknown child %d died", pid);
	abort();
      };
      continue;
    };
    pidtorule = *pp;
    *pp = (*pp)->next;
    if (pidtorule->fd >= 0) {
      close(pidtorule->fd);
      FD_CLR(pidtorule->fd, &larvalfds);
      FD_CLR(pidtorule->fd, &allfds);
    };
    if (!(rule = pidtorule->rule)) {
      free(pidtorule);
      if (debug)
	syslog(LOG_DEBUG, "turn: turned larval child %d", pid);
      continue;
    };
    free(pidtorule);
    if (--rule->ninstances < 0)
      rule->ninstances = 0;
    if (debug)
      syslog(LOG_DEBUG, "turn: turned child %d of rule at line %d", pid, rule->line);
  };
};

void sigchldhandler(int unused)
{
  if (debug)
    syslog(LOG_DEBUG, "sigchld: entering handler");

  turn();

  if (debug)
    syslog(LOG_DEBUG, "sigchld: leaving handler");
};

void sigalrmhandler_exit(int unused)
{
  syslog(LOG_ERR, "child: parent failed to return handshake in time");
  if (debug)
    abort();
  exit(1);
};

jmp_buf timeout_env;
void sigalrmhandler_timeout(int i)
{
  syslog(LOG_ERR, "child: operation timed out");
  longjmp(timeout_env, i);
};

void sigusr1handler(int unused)
{
  struct stat stat;
  int i;
  char buf[128];

  if (!debug) {
    return;
  };

  syslog(LOG_DEBUG, "sigusr1: entering handler");
  
  for (i = 0; i < sysconf(_SC_OPEN_MAX); i++) {
    if (!fstat(i, &stat)) {
      strcpy(buf, "fd %d is a ");
      if (S_ISLNK(stat.st_mode))
	strcat(buf, "symlink");
      if (S_ISREG(stat.st_mode))
	strcat(buf, "file ");
      if (S_ISDIR(stat.st_mode))
	strcat(buf, "directory ");
      if (S_ISCHR(stat.st_mode))
	strcat(buf, "character device ");
      if (S_ISBLK(stat.st_mode))
	strcat(buf, "block device ");
      if (S_ISFIFO(stat.st_mode))
	strcat(buf, "FIFO/pipe ");
      if (S_ISSOCK(stat.st_mode))
	strcat(buf, "socket ");
      syslog(LOG_DEBUG, buf, i);
      syslog(LOG_DEBUG, "dev=%d ino=%d mode=%o nlink=%d uid=%d gid=%d rdev=%d size=%d blksize=%d blocks=%d atime=%d mtime=%d ctime=%d\n", i, stat.st_dev, stat.st_ino, stat.st_mode, stat.st_nlink, stat.st_uid, stat.st_gid, stat.st_rdev, stat.st_size, stat.st_blksize, stat.st_blocks, stat.st_atime, stat.st_mtime, stat.st_ctime);
    };
  };
  syslog(LOG_DEBUG, "sigusr1: leaving handler");
};

int main(int argc, char **argv)
{
  char *c;
  int i, j;

  INIT(argc, argv);

  for (i = sysconf(_SC_OPEN_MAX); i > 2; i--)
    close(i);

  openlog(myname, LOG_PID, LOG_DAEMON);

  sprintf(buffer, CF_DIR "%s.conf", myname);

  while ((i = getopt(argc, argv, STDOPTS_FLAGS "dC:")) != EOF) {
    switch(i) {
      case 'd':
        debug++;
        break;
      case 'C':
	if (strlen(optarg) >= (sizeof(buffer) - 1)) {
	  syslog(LOG_ERR, "%s: config file pathname too long", myname);
	  exit(1);
	};
	strcpy(buffer, optarg);
	break;
      STDOPTS_CASES
    };
  };

  {
    FILE *f;
    int line = 0;
    char *c, *c2;
    struct service **ps = &services;
    int argc;
    char *argv[10];
    struct passwd *passwd;

#define STATESTACK_MAX 4
#define PUSHSTATE(x) \
    if (++statesp >= STATESTACK_MAX) { \
      syslog(LOG_ERR, "%d: internal error (push %d overruns state stack)", x); \
      exit(1); \
    }; \
    if (debug > 2) \
      syslog(LOG_DEBUG, "%d: push %d/%d -> %d/%d", line, statestack[statesp-1].state, statesp-1, x, statesp); \
    statestack[statesp] = statestack[statesp-1]; \
    statestack[statesp].state = x; \
    st = &(statestack[statesp]);
#define POPSTATE \
    if (--statesp < 0) { \
      syslog(LOG_ERR, "%d: internal error (pop %d underruns state stack)", statestack[0].state); \
      exit(1); \
    }; \
    if (debug > 2) \
      syslog(LOG_DEBUG, "%d: pop %d/%d -> %d/%d", line, statestack[statesp+1].state, statesp+1, statestack[statesp].state, statesp); \
    st = &(statestack[statesp]);

    struct statevars {
      int state;

      int sline;
      int bindline;
      int family;
      int socktype;
      int queuelen;
      char shbuf[HNAME_MAX];
      char sbuf[SNAME_MAX];
      int rline;
      int prefixline;
      int prefixlen;
      int instances;
      char rhbuf[HNAME_MAX];
      char user[USERNAME_MAX];
      char directory[PATH_MAX];
      char *argv[10], clbuf[256];
      uid_t uid;
      gid_t gid;
      int ident;
      int checkname;
      int downgrade;
    } statestack[STATESTACK_MAX], *st = &(statestack[0]);
    int statesp = 0, uid;
    struct rule *srules, **pr;

    memset(statestack, 0, sizeof(statestack));
    setpwent();

    if ((uid = getuid())) {
      if (!(passwd = getpwuid(uid))) {
	syslog(LOG_ERR, "config: getpwuid: %s(%d)", strerror(errno), errno);
	exit(1);
      };
    } else {
      if (!(passwd = getpwnam("nobody"))) {
	syslog(LOG_ERR, "config: getpwnam(\"nobody\"): %s(%d)", strerror(errno), errno);
	exit(1);
      };
    };

    strcpy(st->user, passwd->pw_name);
    st->uid = passwd->pw_uid;
    st->gid = passwd->pw_gid;

    strcpy(st->directory, "/tmp");

    st->ident = CF_STRENGTH_TRY;
    st->checkname = CF_STRENGTH_TRY;

    if (!(f = fopen(buffer, "r"))) {
      syslog(LOG_ERR, "%s: can't open %s\n", myname, buffer);
      exit(1);
    };

    memset(fdtoservices, 0, sizeof(fdtoservices));

    FD_ZERO(&allfds);
    FD_ZERO(&servicefds);

    while(fgets(buffer, sizeof(buffer), f)) {
      if ((c = strchr(buffer, '#')))
	*c = 0;
      if ((c = strchr(buffer, ';')))
	*c = 0;
      line++;

      if (debug > 3)
	syslog(LOG_DEBUG, "%d: %s", line, buffer);

      if (!(argv[argc = 0] = strtok(buffer, " \t\n")))
	continue;

      if ((i = nrl_nametonum(cf_tokens, argv[0])) < 0) {
	syslog(LOG_ERR, "%d: syntax error: %s unknown or not valid here", line, argv[0]);
	exit(1);
      };

      while((argv[++argc] = strtok(NULL, " \t\n")));

      if ((j = nrl_numtoflags(cf_tokens, i)) && (argc != j)) {
	syslog(LOG_ERR, "%d: syntax error: invalid number of arguments (%d should be %d)", line, argc-1, j-1);
	exit(1);
      };

      switch(i) {
        case CF_TOKEN_END:
	  switch(st->state) {
	    case CF_TOPLEVEL:
	      syslog(LOG_ERR, "%d: syntax error: close-brace not allowed here", line);
	      exit(1);
	    case CF_SERVICE:
	      if (!st->bindline) {
		syslog(LOG_ERR, "%d: syntax error: no bind statement", st->sline);
		exit(1);
	      };

	      if (!(*ps = malloc(sizeof(struct service)))) {
		syslog(LOG_ERR, "%d: malloc: %s(%d)", line, strerror(errno), errno);
		exit(1);
	      };
	      memset(*ps, 0, sizeof(struct service));
	      (*ps)->line = st->sline;
	      (*ps)->rules = srules;
	      (*ps)->socktype = st->socktype;

	      {
		struct addrinfo req, *ai, *ai2;
		struct fdtoservice **pf;
		int k;

		memset(&req, 0, sizeof(struct addrinfo));
		req.ai_flags = AI_PASSIVE;
		req.ai_socktype = st->socktype;
		req.ai_family = st->family;

		if ((i = getaddrinfo(st->shbuf, st->sbuf, &req, &ai))) {
		  syslog(LOG_ERR, "%d: getaddrinfo %s.%s: %s(%d)", st->bindline, st->shbuf, st->sbuf, gai_strerror(i), i);
		  exit(1);
		};

		FD_ZERO(&((*ps)->fds));
		k = 0;
		for (ai2 = ai; ai2; ai2 = ai2->ai_next) {
		  if ((i = socket(ai2->ai_family, ai2->ai_socktype, ai2->ai_protocol)) < 0) {
		    if (debug)
		      syslog(LOG_DEBUG, "%d: socket: %s(%d)", line, strerror(errno), errno);
		    continue;
		  };
                  j = 1;
		  if (setsockopt(i, SOL_SOCKET, SO_REUSEADDR, &j, sizeof(int)) < 0) {
		    if (debug) {
		      syslog(LOG_DEBUG, "%d: setsockopt(..., SO_REUSEADDR, ...): %s(%d)", line, strerror(errno), errno);
		      abort();
		    };
		    close(i);
		    continue;
		  };
		  if (bind(i, ai2->ai_addr, ai2->ai_addrlen) < 0) {
		    if (debug)
		      syslog(LOG_DEBUG, "%d: bind: %s(%d)", line, strerror(errno), errno);
		    close(i);
		    continue;
		  };
		  if ((listen(i, st->queuelen) < 0) && (errno != EOPNOTSUPP)) {
		    if (debug) {
		      syslog(LOG_DEBUG, "%d: listen: %s(%d)", line, strerror(errno), errno);
		      abort();
		    };
		    close(i);
		    continue;
		  };
		  if (fcntl(i, F_SETFD, 1) < 0) {
		    if (debug) {
		      syslog(LOG_DEBUG, "%d: fcntl(..., F_SETFD, 1): %s(%d)", line, strerror(errno), errno);
		      abort();
		    };
		    close(i);
		    continue;
		  };
		  if ((j = fcntl(i, F_GETFL)) < 0) {
		    if (debug) {
		      syslog(LOG_DEBUG, "%d: fcntl(..., F_GETFL): %s(%d)", line, strerror(errno), errno);
		      abort();
		    };
		    close(i);
		    continue;
		  };
		  if (fcntl(i, F_SETFL, j | O_NONBLOCK) < 0) {
		    if (debug) {
		      syslog(LOG_DEBUG, "%d: fcntl(..., F_SETFL, ...): %s(%d)", line, strerror(errno), errno);
		      abort();
		    };
		    close(i);
		    continue;
		  };

		  for (pf = &(fdtoservices[i % FDTOSERVICE_PRIME]); *pf; pf = &((*pf)->next));
		  if (!(*pf = malloc(sizeof(struct fdtoservice)))) {
		    if (debug) {
		      syslog(LOG_DEBUG, "%d: malloc: %s(%d)", line, strerror(errno), errno);
		      abort();
		    };
		    close(i);
		    continue;
		  };
		  memset(*pf, 0, sizeof(struct fdtoservice));
		  (*pf)->fd = i;
		  (*pf)->service = *ps;

		  FD_SET(i, &((*ps)->fds));
		  FD_SET(i, &allfds);
		  FD_SET(i, &servicefds);
		  if (i < minfd)
		    minfd = i;
		  if (i > maxfd)
		    maxfd = i;
		  k++;
		};

		freeaddrinfo(ai);

		if (!k) {
		  syslog(LOG_ERR, "%d: syntax error: no sockets result from definition", st->sline);
		  exit(1);
		};
	      };

	      ps = &((*ps)->next);
	      POPSTATE;
	      break;
	    case CF_RULE:
	      if (!st->argv[0]) {
		syslog(LOG_ERR, "%d: syntax error: no run statement", st->rline);
		exit(1);
	      };

	      {
		struct addrinfo req, *ai, *ai2;
		u_int8_t *a, *m, *d;
		su mask;

		if (!strcmp(st->rhbuf, "*"))
		  st->rhbuf[0] = 0;

		memset(&req, 0, sizeof(struct addrinfo));
#if 0
		req.ai_flags = AI_PASSIVE;
#endif /* 0 */
		req.ai_socktype = st->socktype;
		req.ai_family = st->family;

		if (!st->rhbuf[0])
		  strcpy(st->rhbuf, "*");

		if (!strcmp(st->rhbuf, "*"))
		  st->prefixlen = 0;

		if ((i = getaddrinfo(st->rhbuf, NULL, &req, &ai))) {
		  syslog(LOG_ERR, "%d: getaddrinfo %s.*: %s(%d)", st->prefixline, st->rhbuf, gai_strerror(i), i);
		  exit(1);
		};

		for (ai2 = ai; ai2; ai2 = ai2->ai_next) {
		  if (nrl_build_samask((struct sockaddr *)&mask, ai2->ai_family, st->prefixlen)) {
		    if (debug)
		      syslog(LOG_DEBUG, "%d: can't build %d-bit mask for %s sockaddr", st->prefixline, st->prefixlen, nrl_afnumtoname(ai2->ai_family));
		    continue;
		  };

		  if (!(*pr = malloc(sizeof(struct rule)))) {
		    syslog(LOG_ERR, "%d: malloc: %s(%d)", st->prefixline, strerror(errno), errno);
		    exit(1);
		  };
		  memset(*pr, 0, sizeof(struct rule));
		  (*pr)->line = st->rline;

		  a = (u_int8_t *)ai2->ai_addr;
		  m = (u_int8_t *)&mask;
		  d = (u_int8_t *)&((*pr)->prefix);
		  for (i = 0; i < ai2->ai_addrlen; i++)
		    *(d++) = *(a++) & *(m++);

		  memcpy(&((*pr)->mask), &mask, ai2->ai_addrlen);

		  (*pr)->checkname = st->checkname;
		  (*pr)->ident = st->ident;
		  (*pr)->downgrade = st->downgrade;
		  (*pr)->instances = st->instances;
		  (*pr)->uid = st->uid;
		  (*pr)->gid = st->gid;
		  strcpy((*pr)->user, st->user);
		  strcpy((*pr)->directory, st->directory);

		  for (i = 0, c = (*pr)->clbuf; st->argv[i]; i++) {
		    (*pr)->argv[i] = c;
		    strcpy(c, st->argv[i]);
		    c += strlen(c);
		    c++;
		  };

		  pr = &((*pr)->next);
		};

		if (st->rhbuf[0])
		  freeaddrinfo(ai);
	      };

	      POPSTATE;
	      break;
	    default:
	      syslog(LOG_ERR, "%d: internal error: invalid state %d", line, st->state);
	      exit(1);
	  };
	  break;
        case CF_TOKEN_SERVICE:
	  if (st->state != CF_TOPLEVEL) {
	    syslog(LOG_ERR, "%d: syntax error: can't start a service here", line);
	    exit(1);
	  };
	  srules = NULL;
	  pr = &srules;
	  PUSHSTATE(CF_SERVICE);
	  st->sline = line;
	  break;
        case CF_TOKEN_RULE:
	  if (st->state != CF_SERVICE) {
	    syslog(LOG_ERR, "%d: syntax error: can't start a rule here", line);
	    exit(1);
	  };
	  PUSHSTATE(CF_RULE);
	  st->rline = line;
	  break;
        case CF_TOKEN_BIND:
	  st->bindline = line;
	  if ((c2 = strrchr(argv[1], '.'))) {
	    c = argv[1];
	    *(c2++) = 0;
	  } else {
	    c2 = c;
	    c = "*";
	  };
	  if (strlen(c) >= HNAME_MAX) {
	    syslog(LOG_ERR, "%d: syntax error: %s is too long", line, c);
	    exit(1);
	  };
	  if (strlen(c2) >= SNAME_MAX) {
	    syslog(LOG_ERR, "%d: syntax error: %s is too long", line, c2);
	    exit(1);
	  };
	  strcpy(st->shbuf, c);
	  strcpy(st->sbuf, c2);
	  break;
        case CF_TOKEN_SOCKTYPE:
	  if ((st->socktype = nrl_socktypenametonum(argv[1])) < 0) {
	    syslog(LOG_ERR, "%d: syntax error: %s is an unknown/invalid socktype", line, argv[1]);
	    exit(1);
	  };
	  break;
        case CF_TOKEN_FAMILY:
          if (!strcmp(argv[1], "*"))
            st->family = 0;
          else
	    if ((st->family = nrl_afnametonum(argv[1])) < 0) {
	      syslog(LOG_ERR, "%d: syntax error: %s is an unknown/invalid address family", line, argv[1]);
	      exit(1);
	    };
	  break;
        case CF_TOKEN_QUEUELEN:
	  st->queuelen = strtoul(argv[1], &c, 10);
	  if ((*c < 0) || (st->queuelen < 1) || (st->queuelen > 1000)) {
	    syslog(LOG_ERR, "%d: syntax error: %s is an invalid queue length", line, argv[1]);
	    exit(1);
	  };
	  break;
        case CF_TOKEN_PREFIX:
	  st->prefixline = line;
	  if ((c = strrchr(argv[1], '/'))) {
	    *(c++) = 0;
	    st->prefixlen = strtoul(c, &c2, 10);
	    if (*c2) {
	      syslog(LOG_ERR, "%d: syntax error: %s is an invalid prefix length", line, c);
	      exit(1);
	    };
	  } else
	    st->prefixlen = -1;
	  if (strlen(argv[1]) >= HNAME_MAX) {
	    syslog(LOG_ERR, "%d: syntax error: %s is too long", line, c);
	    exit(1);
	  };
	  strcpy(st->rhbuf, argv[1]);
	  break;
        case CF_TOKEN_RUN:
	  {
	    char *c = st->clbuf;

	    if (argc < 2) {
	      syslog(LOG_ERR, "%d: syntax error: no command given to run");
	      exit(1);
	    };
	    
	    if (argc > 10) {
	      syslog(LOG_ERR, "%d: syntax error: too many tokens in command line (%d)", argc - 1);
	      exit(1);
	    };

	    for (i = 0; i < argc - 1; i++) {
	      st->argv[i] = c;
	      strcpy(c, argv[i+1]);
	      c += strlen(c);
	      c++;
	    };
	  };
	  break;
        case CF_TOKEN_INSTANCES:
	  st->instances = strtoul(argv[1], &c, 10);
	  if ((*c < 0) || (st->instances > INSTANCES_MAX)) {
	    syslog(LOG_ERR, "%d: syntax error: %s is an invalid number of instances", line, argv[1]);
	    exit(1);
	  };
	  break;
        case CF_TOKEN_USER:
	  if ((strlen(argv[1]) >= USERNAME_MAX) || !(passwd = getpwnam(argv[1]))) {
	    syslog(LOG_ERR, "%d: syntax error: %s is an invalid user name", line, argv[1]);
	    exit(1);
	  };
	  if ((uid != passwd->pw_uid) && uid) {
	    syslog(LOG_ERR, "%d: syntax error: only the superuser can change user ids", line);
	    exit(1);
	  };
	  strcpy(st->user, argv[1]);
	  st->uid = passwd->pw_uid;
	  st->gid = passwd->pw_gid;
	  break;
        case CF_TOKEN_DIRECTORY:
	  if (strlen(argv[1]) >= PATH_MAX) {
	    syslog(LOG_ERR, "%d: syntax error: %s is too long", line, argv[1]);
	    exit(1);
	  };
	  strcpy(st->directory, argv[1]);
	  break;
        case CF_TOKEN_CHECKNAME:
	  if ((st->checkname = nrl_nametonum(cf_strengths, argv[1])) < 0) {
	    syslog(LOG_ERR, "%d: syntax error: %s is an unknown strength", line, argv[1]);
	    exit(1);
	  };
	  break;
        case CF_TOKEN_IDENT:
	  if ((st->ident = nrl_nametonum(cf_strengths, argv[1])) < 0) {
	    syslog(LOG_ERR, "%d: syntax error: %s is an unknown strength", line, argv[1]);
	    exit(1);
	  };
	  break;
        case CF_TOKEN_DOWNGRADE:
	  if ((st->downgrade = nrl_nametonum(cf_strengths, argv[1])) < 0) {
	    syslog(LOG_ERR, "%d: syntax error: %s is an unknown strength", line, argv[1]);
	    exit(1);
	  };
	  break;
        default:
	  syslog(LOG_ERR, "%d: internal error: invalid token %d", line, i);
	  exit(1);
      };
    };
    fclose(f);
  };

  if (maxfd < 0) {
    syslog(LOG_ERR, "config: no services");
    exit(1);
  };

  endpwent();
  nrl_domainname();

  if (debug)
    syslog(LOG_DEBUG, "entering main loop");

  {
    struct sigaction sa;

    memset(&sa, 0, sizeof(struct sigaction));

    sa.sa_handler = sigchldhandler;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART;

    if (sigaction(SIGCHLD, &sa, NULL) < 0) {
      syslog(LOG_ERR, "main: sigaction: %s(%d)", strerror(errno), errno);
      exit(1);
    };

    sa.sa_handler = sigusr1handler;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART;
    
    if (sigaction(SIGUSR1, &sa, NULL) < 0) {
      syslog(LOG_ERR, "main: sigaction: %s(%d)", strerror(errno), errno);
      exit(1);
    };
  };

  {
    fd_set rfds, efds;
    struct rule *rule;
    struct fdtoservice *fdtoservice;
    su addr;
    int addrlen;
    int fds[2];
    struct larvalmsg larvalmsg;
    pid_t pid;
    struct pidtorule *pidtorule, **pp;
    sigset_t sigset;
    struct timeval timeval, timeval2;

    FD_ZERO(&larvalfds);

    sigemptyset(&sigset);
    sigaddset(&sigset, SIGCHLD);
    sigaddset(&sigset, SIGUSR1);

    if (sigprocmask(SIG_UNBLOCK, &sigset, NULL) < 0) {
      syslog(LOG_ERR, "sigprocmask(SIG_UNBLOCK, ...): %s(%d)", strerror(errno), errno);
      exit(1);
    };

    memset(&timeval, 0, sizeof(timeval));
    timeval.tv_usec = 100;

    while (1) {
      memcpy(&rfds, &allfds, sizeof(fd_set));
      memcpy(&efds, &allfds, sizeof(fd_set));
      memcpy(&timeval2, &timeval, sizeof(struct timeval));

      if (debug)
	syslog(LOG_DEBUG, "main: entering select");

      if (sigprocmask(SIG_UNBLOCK, &sigset, NULL) < 0) {
	syslog(LOG_ERR, "sigprocmask(SIG_UNBLOCK, ...): %s(%d)", strerror(errno), errno);
	exit(1);
      };

      if (select(maxfd + 1, &rfds, NULL, &efds, NULL) < 0) {
	if ((errno == EINTR) || (errno == EAGAIN))
	  continue;
	syslog(LOG_ERR, "select: %s(%d)", strerror(errno), errno);
	if (debug)
	  abort();
	exit(1);
      };

      if (sigprocmask(SIG_BLOCK, &sigset, NULL) < 0) {
	syslog(LOG_ERR, "sigprocmask(SIG_BLOCK, ...): %s(%d)", strerror(errno), errno);
	exit(1);
      };
      if (debug)
	syslog(LOG_DEBUG, "main: back from select");

      for (i = 0; i <= maxfd; i++) {
	if (!FD_ISSET(i, &rfds))
	  continue;
	if (FD_ISSET(i, &servicefds)) {
	  for (fdtoservice = fdtoservices[i % FDTOSERVICE_PRIME]; fdtoservice && (fdtoservice->fd != i); fdtoservice = fdtoservice->next);
	  if (!fdtoservice) {
	    syslog(LOG_ERR, "internal error: fd %d not in fdtoservice map?", i);
	    if (debug)
	      abort();
	    continue;
	  };
	  if (debug)
	    syslog(LOG_DEBUG, "main: matched service at line %d", fdtoservice->service->line);
	  if (socketpair(PF_UNIX, SOCK_DGRAM, 0, fds) < 0) {
	    if (debug) {
	      syslog(LOG_DEBUG, "main: socketpair: %s(%d)", strerror(errno), errno);
	      abort();
	    };
	    continue;
	  };
	  if (fcntl(fds[0], F_SETFD, 1) < 0) {
	    if (debug) {
	      syslog(LOG_DEBUG, "main: fcntl(..., F_SETFD, 1): %s(%d)", strerror(errno), errno);
	      abort();
	    };
	    close(fds[0]);
	    close(fds[1]);
	    continue;
	  };
	  if (fcntl(fds[1], F_SETFD, 1) < 0) {
	    if (debug) {
	      syslog(LOG_DEBUG, "main: fcntl(..., F_SETFD, 1): %s(%d)", strerror(errno), errno);
	      abort();
	    };
	    close(fds[0]);
	    close(fds[1]);
	    continue;
	  };
	  if ((j = accept(i, (struct sockaddr *)&addr, &addrlen)) < 0) {
	    if (debug) {
	      syslog(LOG_DEBUG, "main: accept: %s(%d)", strerror(errno), errno);
	      abort();
	    };
	    close(fds[0]);
	    close(fds[1]);
	    continue;
	  };
	  if (debug < 4) {
	    if ((pid = fork()) < 0) {
	      if (debug) {
		syslog(LOG_DEBUG, "main: fork: %s(%d)", strerror(errno), errno);
		abort();
	      };
	      close(j);
	      close(fds[0]);
	      close(fds[1]);
	      continue;
	    };
	    if (pid) {
	      close(j);
	      close(fds[1]);
	      if (debug)
		syslog(LOG_DEBUG, "main: forked pid %d, fd %d", pid, fds[0]);
	      
	      for (pp = &(pidtorules[pid % PIDTORULE_PRIME]); *pp; pp = &((*pp)->next));
	      if (!(*pp = malloc(sizeof(struct pidtorule)))) {
		syslog(LOG_ERR, "main: malloc: %s(%d)", strerror(errno), errno);
		if (debug)
		  abort();
		continue;
	      };
	      memset(*pp, 0, sizeof(struct pidtorule));
	      (*pp)->pid = pid;
	      (*pp)->fd = fds[0];
	      
	      FD_SET(fds[0], &larvalfds);
	      FD_SET(fds[0], &allfds);
	      if (i < minfd)
		minfd = i;
	      if (i > maxfd)
		maxfd = i;
	      
	      continue;
	    };
	  };

          signal(SIGALRM, sigalrmhandler_exit);
          alarm(15);

	  close(fds[0]);
	  for (i = 0; i <= maxfd; i++)
	    if (FD_ISSET(i, &allfds))
	      close(i);

	  {
	    su prefix;
	    u_int8_t *a, *m, *d;
	    char hbuf[3][HNAME_MAX], sbuf[2][SNAME_MAX];
	    int checkname = 0, ident = 0, downgrade = 0;
	    int timer;

	    /* XXX - kernel bug? */
	    if (getpeername(j, (struct sockaddr *)&addr, &addrlen) < 0) {
	      syslog(LOG_ERR, "child: getpeername: %s(%d)", strerror(errno), errno);
	      exit(1);
	    };

	    if (getsockname(j, (struct sockaddr *)&prefix, &i) < 0) {
	      syslog(LOG_ERR, "child: getsockname: %s(%d)", strerror(errno), errno);
	      exit(1);
	    };

	    if (getnameinfo((struct sockaddr *)&addr, addrlen, hbuf[0], sizeof(hbuf[0]), sbuf[0], sizeof(sbuf[0]), NI_NUMERICHOST | NI_NUMERICSERV)) {
	      syslog(LOG_ERR, "child: getnameinfo failed");
	      if (debug)
		abort();
	      exit(1);
	    };

	    if (getnameinfo((struct sockaddr *)&prefix, i, hbuf[1], sizeof(hbuf[1]), sbuf[1], sizeof(sbuf[1]), NI_NUMERICHOST | NI_NUMERICSERV)) {
	      syslog(LOG_ERR, "child: getnameinfo failed");
	      if (debug)
		abort();
	      exit(1);
	    };

	    hbuf[2][0] = 0;

	    syslog(LOG_INFO, "connection from %s.%s to %s.%s", hbuf[0], sbuf[0], hbuf[1], sbuf[1]);

	    if (debug > 2) {
	      syslog(LOG_DEBUG, "child: addrlen=%d, addr:", addrlen);
	      dump_sockaddr((struct sockaddr *)&addr, addrlen);
	    };

	    for (rule = fdtoservice->service->rules; rule; rule = rule->next) {
	      if (debug > 2) {
		syslog(LOG_DEBUG, "child: rule at %d: rule prefix:", rule->line);
		dump_sockaddr((struct sockaddr *)&rule->prefix, 0);
		syslog(LOG_DEBUG, "child: rule at %d: rule mask:", rule->line);
		dump_sockaddr((struct sockaddr *)&rule->mask, NRL_SA_LEN((struct sockaddr *)&rule->prefix));
	      };

	      d = (u_int8_t *)&prefix;
	      a = (u_int8_t *)&addr;
	      m = (u_int8_t *)&rule->mask;

	      for (i = 0; i < NRL_SA_LEN((struct sockaddr *)&rule->prefix); i++)
		*(d++) = *(a++) & *(m++);

	      if (debug > 2) {
		syslog(LOG_DEBUG, "child: rule at %d: incoming prefix:", rule->line);
		dump_sockaddr((struct sockaddr *)&prefix, 0);
	      };

	      if (memcmp(&prefix, &rule->prefix, NRL_SA_LEN((struct sockaddr *)&rule->prefix))) {
		if (debug)
		  syslog(LOG_DEBUG, "child: rule at %d: prefix test failed", rule->line);
		continue;
	      };

	      if (rule->checkname) {
		if (!checkname) {
		  checkname = 1;
		  if (!setjmp(timeout_env)) {
		    timer = alarm(0);
		    signal(SIGALRM, sigalrmhandler_timeout);
		    alarm(10);
		    if (getnameinfo((struct sockaddr *)&addr, addrlen, hbuf[2], sizeof(hbuf[2]), NULL, 0, NI_NAMEREQD)) {
		      if (debug)
			syslog(LOG_DEBUG, "child: rule at %d: checkname: getnameinfo failed -- no reverse DNS entry present?", rule->line);
		    } else {
		      struct addrinfo req, *ai;
		      memset(&req, 0, sizeof(struct addrinfo));

		      if (debug)
			syslog(LOG_DEBUG, "child: rule at %d: checkname: returned name is %s", rule->line, hbuf[2]);
		      
		      if ((addr.sa.sa_family != AF_INET6) || !IN6_IS_ADDR_V4MAPPED(&addr.sin6.sin6_addr))
			req.ai_family = addr.sa.sa_family;

		      if ((i = getaddrinfo(hbuf[2], sbuf[0], &req, &ai))) {
			if (debug)
			  syslog(LOG_DEBUG, "child: rule at %d: checkname: getaddrinfo %s.%s: %s(%d)", rule->line, hbuf[2], sbuf[0], gai_strerror(i), i);
		      } else {
			struct addrinfo *ai2;
			for (ai2 = ai; ai2; ai2 = ai2->ai_next) {
			  if (debug > 2) {
			    syslog(LOG_DEBUG, "child: rule at %d: checkname: ai_addrlen = %d, ai_addr:", rule->line, ai2->ai_addrlen);
			    dump_sockaddr(ai->ai_addr, ai->ai_addrlen);
			  };
			  if (!req.ai_family && (ai2->ai_family == AF_INET) && (((u_int32_t *)&addr.sin6.sin6_addr)[3] == *(u_int32_t *)&((struct sockaddr_in *)ai->ai_addr)->sin_addr)) {
			    if (debug)
			      syslog(LOG_DEBUG, "child: rule at %d: checkname: matched using IPv4-mapped rule", rule->line);
			    break;
			  };
			  if (addrlen != ai2->ai_addrlen)
			    continue;
			  if (!memcmp(ai->ai_addr, &addr, addrlen)) {
			    if (debug)
			      syslog(LOG_DEBUG, "child: rule at %d: checkname: matched using normal rule", rule->line);
			    break;
			  };
			};
			freeaddrinfo(ai);
			if (ai2)
			  checkname = 2;
		      };
		    };
		  };
		  alarm(0);
		  signal(SIGALRM, sigalrmhandler_exit);
		  alarm(timer);
		};

		if (checkname < rule->checkname) {
		  if (debug)
		    syslog(LOG_DEBUG, "child: rule at %d: checkname: test failed (%d < %d)", rule->line, checkname, rule->checkname);
		  continue;
		};
	      };

	      if (rule->ident) {
		if (!ident) {
		  if (!setjmp(timeout_env)) {
		    struct addrinfo req, *ai;

		    timer = alarm(0);
		    signal(SIGALRM, sigalrmhandler_timeout);
		    alarm(10);

		    memset(&req, 0, sizeof(struct addrinfo));
		    
		    ident = 1;
		    req.ai_family = addr.sa.sa_family;
		    req.ai_socktype = SOCK_STREAM;
		    
		    if ((i = getaddrinfo(hbuf[0], "ident", &req, &ai))) {
		      if (debug)
			syslog(LOG_DEBUG, "child: rule at %d: ident: getaddrinfo %s.ident: %s(%d)", rule->line, hbuf[2], gai_strerror(i), i);
		    } else {
		      struct addrinfo *ai2;
		      int k;
		      
		      sprintf(buffer, "%s,%s\r\n", sbuf[0], sbuf[1]);
		      k = strlen(buffer);
		      
		      for (ai2 = ai; ai2; ai2 = ai2->ai_next) {
			if ((i = socket(ai2->ai_family, ai2->ai_socktype, ai2->ai_protocol)) < 0) {
			  if (debug)
			    syslog(LOG_DEBUG, "child: rule at %d: ident: socket: %s(%d)", rule->line, strerror(errno), errno);
			  continue;
			};
			if (connect(i, ai2->ai_addr, ai2->ai_addrlen) < 0) {
			  if (debug)
			    syslog(LOG_DEBUG, "child: rule at %d: ident: connect: %s(%d)", rule->line, strerror(errno), errno);
			  close(i);
			  continue;
			};
			if (write(i, buffer, k) != k) {
			  if (debug)
			    syslog(LOG_DEBUG, "child: rule at %d: ident: write: %s(%d)", rule->line, strerror(errno), errno);
			  close(i);
			  continue;
			};
			for (c = buffer; c < (buffer + sizeof(buffer) - 1); c++) {
			  if (read(i, c, 1) != 1) {
			    if (debug)
			      syslog(LOG_DEBUG, "child: rule at %d: ident: read: %s(%d)", rule->line, strerror(errno), errno);
			    break;
			  };
			  if ((*c == '\r') || (*c == '\n')) {
			    *c = 0;
			  break;
			  };
			};
			close(i);
			if (*c) {
			  if (debug)
			    syslog(LOG_DEBUG, "child: rule at %d: ident: hit error or end of buffer", rule->line);
			  continue;
			};
			if (debug)
			  syslog(LOG_DEBUG, "child: rule at %d: ident: got %s", rule->line, buffer);
			if (!(c = strchr(buffer, ':'))) {
			  if (debug)
			    syslog(LOG_DEBUG, "child: rule at %d: ident: syntax error in response", rule->line, strerror(errno), errno);
			  continue;
			};
			c++;
			while(*c && isspace(*c)) c++;
			if (!*c) {
			  if (debug)
			    syslog(LOG_DEBUG, "child: rule at %d: ident: syntax error in response", rule->line, strerror(errno), errno);
			  continue;
			};
			if (strncasecmp(c, "userid", 6)) {
			  if (debug)
			    syslog(LOG_DEBUG, "child: rule at %d: ident: response was probably an error", rule->line, strerror(errno), errno);
			  continue;
			};
			c += 6;
			while(*c && isspace(*c)) c++;
			if (*(c++) != ':') {
			  if (debug)
			    syslog(LOG_DEBUG, "child: rule at %d: ident: syntax error in response", rule->line, strerror(errno), errno);
			  continue;
			};
			if (!(c = strchr(c, ':'))) {
			  if (debug)
			    syslog(LOG_DEBUG, "child: rule at %d: ident: syntax error in response", rule->line, strerror(errno), errno);
			  continue;
			};
			c++;
			while(*c && isspace(*c)) c++;
			if (!*c) {
			  if (debug)
			    syslog(LOG_DEBUG, "child: rule at %d: ident: syntax error in response", rule->line, strerror(errno), errno);
			  continue;
			};
			syslog(LOG_DEBUG, "child: ident: %s", c);
			ident = 2;
			break;
		      };
		      freeaddrinfo(ai);
		    };
		  };
		  alarm(0);
		  signal(SIGALRM, sigalrmhandler_exit);
		  alarm(timer);
		};

		if (ident < rule->ident) {
		  if (debug)
		    syslog(LOG_DEBUG, "child: rule at %d: ident: test failed (%d < %d)", rule->line, ident, rule->ident);
		  continue;
		};
	      };

	      if (rule->downgrade) {
		if (!downgrade) {
		  /* XXX */
		  if ((addr.sa.sa_family == AF_INET6) && IN6_IS_ADDR_V4MAPPED(&addr.sin6.sin6_addr))
		    downgrade = 2;
		  else
		    downgrade = 1;
		};
		if (downgrade < rule->downgrade) {
		  if (debug)
		    syslog(LOG_DEBUG, "child: rule at %d: downgrade: test failed (%d < %d)", rule->line, downgrade, rule->downgrade);
		  continue;
		};
	      };

	      if ((rule->instances > 0) && (rule->ninstances >= rule->instances)) {
		if (debug)
		  syslog(LOG_DEBUG, "child: rule at %d: instances test failed (%d >= %d)", rule->line, rule->ninstances, rule->instances);
		continue;
	      };
	      break;
	    };
	    if (!rule) {
	      if (debug) {
		syslog(LOG_DEBUG, "child: no rules matched");
		abort();
	      };
	      exit(1);
	    };

	    if (rule->uid != getuid()) {
	      if (setgid(rule->gid) < 0) {
		syslog(LOG_ERR, "child: setgid: %s(%d)", strerror(errno), errno);
		exit(1);
	      };
	      if (initgroups(rule->user, rule->gid) < 0) {
		syslog(LOG_ERR, "child: initgroups: %s(%d)", strerror(errno), errno);
		exit(1);
	      };
	      if (setuid(rule->uid) < 0) {
		syslog(LOG_ERR, "child: setuid: %s(%d)", strerror(errno), errno);
		exit(1);
	      };
	    };

	    memset(&larvalmsg, 0, sizeof(struct larvalmsg));
	    larvalmsg.pid = getpid();
	    larvalmsg.rule = rule;
	    if (write(fds[1], &larvalmsg, sizeof(larvalmsg)) != sizeof(larvalmsg)) {
	      syslog(LOG_ERR, "child: write: %s(%d)", strerror(errno), errno);
	      exit(1);
	    };
	    if (read(fds[1], &larvalmsg, sizeof(larvalmsg)) != sizeof(larvalmsg)) {
	      syslog(LOG_ERR, "child: read: %s(%d)", strerror(errno), errno);
	      exit(1);
	    };
	    close(fds[1]);

	    if ((larvalmsg.pid != getpid()) || (larvalmsg.rule != rule)) {
	      syslog(LOG_ERR, "child: internal error: echoed handshake doesn't match");
	      exit(1);
	    };

	    /* XXX */
	    if (downgrade == CF_STRENGTH_REQUIRE) {
	      int i = AF_INET;
	      if (setsockopt(j, SOL_IPV6, IPV6_ADDRFORM, &i, sizeof(int))) {
		syslog(LOG_ERR, "child: setsockopt(..., SOL_IPV6, IPV6_ADDRFORM, ...): %s(%d)", strerror(errno), errno);
		exit(1);
	      };
	    };

	    if (dup2(j, 0) < 0) {
	      syslog(LOG_ERR, "child: dup2(..., 0): %s(%d)", strerror(errno), errno);
	      exit(1);
	    };
	    if (dup2(j, 1) < 0) {
	      syslog(LOG_ERR, "child: dup2(..., 1): %s(%d)", strerror(errno), errno);
	      exit(1);
	    };
	    if (dup2(j, 2) < 0) {
	      syslog(LOG_ERR, "child: dup2(..., 2): %s(%d)", strerror(errno), errno);
	      exit(1);
	    };
	    close(j);

	    if (rule->directory[0])
	      if (chdir(rule->directory) < 0) {
		syslog(LOG_ERR, "child: chdir: %s(%d)", strerror(errno), errno);
		exit(1);
	      };

	    if (setsid() < 0) {
	      syslog(LOG_ERR, "child: setsid: %s(%d)", strerror(errno), errno);
	      exit(1);
	    };

	    if (sigprocmask(SIG_UNBLOCK, &sigset, NULL) < 0) {
	      syslog(LOG_ERR, "child: sigprocmask(SIG_UNBLOCK, ...): %s(%d)", strerror(errno), errno);
	      exit(1);
	    };

	    if (debug)
	      syslog(LOG_DEBUG, "child: running %s", rule->argv[0]);
	    closelog();
	    if (!debug)
	      for (i = 3; i < sysconf(_SC_OPEN_MAX); i++)
		close(i);
	    alarm(0);
	    execv(rule->argv[0], rule->argv);
	    syslog(LOG_ERR, "child: exec %s: %s(%d)", rule->argv[0], strerror(errno), errno);
	    if (debug)
	      abort();
	    exit(1);
	  };
	};
	if (FD_ISSET(i, &larvalfds)) {
	  if ((j = read(i, &larvalmsg, sizeof(struct larvalmsg))) != sizeof(struct larvalmsg)) {
	    if (j < 0)
	      syslog(LOG_ERR, "main: read: %s(%d)", strerror(errno), errno);
	    else
	      syslog(LOG_ERR, "main: read only %d/%d bytes of child's message", j, sizeof(struct larvalmsg));
	    if (debug)
	      abort();
	    close(i);
	    FD_CLR(i, &larvalfds);
	    FD_CLR(i, &allfds);
	    continue;
	  };
	  if (debug)
	    syslog(LOG_DEBUG, "main: got handshake from child %d", larvalmsg.pid);

	  for (pp = &(pidtorules[larvalmsg.pid % PIDTORULE_PRIME]); *pp && ((*pp)->pid != larvalmsg.pid); pp = &((*pp)->next));

	  if (((pid = waitpid(larvalmsg.pid, NULL, WNOHANG) < 0) && (errno == ECHILD)) || (pid == larvalmsg.pid)) {
	    if (debug)
	      syslog(LOG_DEBUG, "main: ... but it appears to have already died");

	    if (!*pp) {
	      if (debug)
		syslog(LOG_DEBUG, "main: ... and has already been cleaned up");
	      continue;
	    };
	    pidtorule = *pp;
	    *pp = (*pp)->next;
	    close(pidtorule->fd);
	    free(pidtorule);
	    if (debug)
	      syslog(LOG_DEBUG, "main: ... and wasn't already cleaned up");
	    continue;
	  };

	  if (pid < 0) {
	    syslog(LOG_ERR, "main: waitpid(%d, ...): %s(%d)", larvalmsg.pid, strerror(errno), errno);
	    if (debug)
	      abort();
	    close(i);
	    FD_CLR(i, &larvalfds);
	    FD_CLR(i, &allfds);
	    continue;
	  };

	  if (!*pp) {
	    if (debug)
	      syslog(LOG_DEBUG, "main: internal error: fd %d points to unknown child %d? killing...", i, pid);
	    if (kill(pid, SIGKILL) < 0) {
	      syslog(LOG_ERR, "main: couldn't kill child %d", pid);
	    };
	    if (debug)
	      abort();
	    close(i);
	    FD_CLR(i, &larvalfds);
	    FD_CLR(i, &allfds);
	    continue;
	  };

	  j = write(i, &larvalmsg, sizeof(struct larvalmsg));
	  close(i);
	  FD_CLR(i, &larvalfds);
	  FD_CLR(i, &allfds);
	  (*pp)->fd = -1;

	  if (j != sizeof(struct larvalmsg)) {
	    if (j < 0)
	      syslog(LOG_ERR, "main: write: %s(%d)", strerror(errno), errno);
	    else
	      syslog(LOG_ERR, "main: wrote only %d/%d bytes back to child", j, sizeof(struct larvalmsg));
	    if (kill(pid, SIGKILL) < 0) {
	      syslog(LOG_ERR, "main: couldn't kill child %d", pid);
	    };
	    if (debug)
	      abort();
	    continue;
	  };

	  (*pp)->rule = larvalmsg.rule;
	  larvalmsg.rule->ninstances++;
	};
      };
    };
  };
};
