#include <stdio.h>         
#include <stdlib.h>        
#include <sys/ioctl.h>

#include <sys/types.h>
#include <sys/socket.h>

#include <netinet/in.h>
#include <net/if.h>
#include <net/if_dl.h>
#include <sys/sockio.h>
#include <sys/syslog.h>

#include "support.h"


/*****************************************************************************
   if_nametoindex():  Returns the index number when given the interface
   name.
*****************************************************************************/
unsigned int if_nametoindex(const char *ifname)
{
  unsigned int sdl_index;
  int i, nifreqs;
  struct ifreq *ifreqs = NULL;
  int localhostfound = 0;

  struct ifconf ifconf;

  int buflen, buflen_i;

  char *dataline;
  char *dataendpiece;
  u_char datafamily;

  /*  Get a socket, set-up memory buffer and then call IOCTL with SIOGIFCONF. */
  {
    ifconf.ifc_len=0;
    ifconf.ifc_buf=NULL;

      if ((i = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0) {
         perror("socket");
         return NULL;
      }

      buflen = sizeof(struct ifreq) * 1000;

      while(1) {
         if (!(ifconf.ifc_buf = (void *)malloc(ifconf.ifc_len=buflen) )) {
            syslog(LOG_ERR, "malloc() failed!");
            return NULL;
          }

          if (ioctl(i, SIOCGIFCONF, (char *)&ifconf) < 0) {
             perror("ioctl(SIOCGIFCONF)");
             exit(1);
          }



          /*----------------------------------------------------------------
               Test to see returned ioctl data fits into our allocated
               buffer:   If so, then break out of loop; otherwise,
               free the newly created buffer and try a new size and
               while-loop block over again until correct.
          -----------------------------------------------------------------*/

          if (ifconf.ifc_len < buflen)
          break;

          free(ifconf.ifc_buf);
          buflen += sizeof(struct ifreq);

      }  /* End of "while (1)" */

      close(i);
  }  /* End of get, set-up, and call. */



  /*-------------------------------------------------------------------
     Start walking through the ifconf data (returned from ioctl call),
     looking for AF_LINK (sockaddr_dl) entries with the correct index.
     See figure 4.28 of Steven's TCP/IP Illustrated Vol. II, page 121.)
   *-------------------------------------------------------------------*/

  buflen_i   = 0;
  dataline   = ifconf.ifc_buf;
    
  while (buflen_i < (ifconf.ifc_len - 1))  {
       dataendpiece   = dataline + IFNAMSIZ;
       datafamily     = ((struct sockaddr *)dataendpiece)->sa_family;


       if ((int)datafamily == AF_LINK) {


               /*  Memory compare the interface name to find index number. */
               if (!memcmp(ifname,
                          ((struct sockaddr_dl *)dataendpiece)->sdl_data,
                          ((struct sockaddr_dl *)dataendpiece)->sdl_nlen  )) {
                   sdl_index = ((struct sockaddr_dl *)dataendpiece)->sdl_index;
                   free(ifconf.ifc_buf);
                   return( sdl_index );
               }

       }
               buflen_i += (IFNAMSIZ + ((struct sockaddr *)dataendpiece)->sa_len);
               dataline += (IFNAMSIZ + ((struct sockaddr *)dataendpiece)->sa_len);
   }  /*  End of while  */

   /*  Didn't find any?  Free up memory and return with NULL value  */
   free(ifconf.ifc_buf);
   return(NULL);
}





/*****************************************************************************
  if_indextoname():  Returns the interface strings when given a valid index
  number.
*****************************************************************************/

char *if_indextoname(unsigned int ifindex, char *ifname)
{
  int i, nifreqs;
  struct ifreq *ifreqs = NULL;
  int localhostfound = 0;

  struct ifconf ifconf;

  int buflen, buflen_i;

  char *dataline;
  char *dataendpiece;
  u_char datafamily;

  /*  Get a socket, set-up memory buffer and then call IOCTL with SIOGIFCONF. */
  {
    ifconf.ifc_len=0;
    ifconf.ifc_buf=NULL;

      if ((i = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0) {
         perror("socket");
         return NULL;
      }

      buflen = sizeof(struct ifreq) * 1000;

      while(1) {
         if (!(ifconf.ifc_buf = (void *)malloc(ifconf.ifc_len=buflen) )) {
            syslog(LOG_ERR, "malloc() failed!");
            return NULL;
          }

          if (ioctl(i, SIOCGIFCONF, (char *)&ifconf) < 0) {
             perror("ioctl(SIOCGIFCONF)");
             exit(1);
          }



          /*----------------------------------------------------------------
               Test to see returned ioctl data fits into our allocated
               buffer:   If so, then break out of loop; otherwise,
               free the newly created buffer and try a new size and
               while-loop block over again until correct.
          -----------------------------------------------------------------*/

          if (ifconf.ifc_len < buflen)
          break;

          free(ifconf.ifc_buf);
          buflen += sizeof(struct ifreq);

      }  /* End of "while (1)" */

      close(i);
  }  /* End of get, set-up, and call. */



  /*-------------------------------------------------------------------
     Start walking through the ifconf data (returned from ioctl call),
     looking for AF_LINK (sockaddr_dl) entries with the correct index.
     See figure 4.28 of Steven's TCP/IP Illustrated Vol. II, page 121.)
   *-------------------------------------------------------------------*/

  buflen_i   = 0;
  dataline   = ifconf.ifc_buf;
    
  while (buflen_i < (ifconf.ifc_len - 1))  {
       dataendpiece   = dataline + IFNAMSIZ;
       datafamily     = ((struct sockaddr *)dataendpiece)->sa_family;


       if ((int)datafamily == AF_LINK) {


               /*  Is the desire index number found?  If yes, copy data over to
                   the given function argument "ifname", free buffer and
                   and return with the address of the given "ifname" .       */
               if ( ifindex == ((struct sockaddr_dl *)dataendpiece)->sdl_index ) {
                   memcpy( ifname,
                           ((struct sockaddr_dl *)dataendpiece)->sdl_data,
                           ((struct sockaddr_dl *)dataendpiece)->sdl_nlen  );
                   ifname[((struct sockaddr_dl *)dataendpiece)->sdl_nlen]='\0';
                   free(ifconf.ifc_buf);
                   return( ifname );
               }

       }
               buflen_i += (IFNAMSIZ + ((struct sockaddr *)dataendpiece)->sa_len);
               dataline += (IFNAMSIZ + ((struct sockaddr *)dataendpiece)->sa_len);
   }  /*  End of while  */

   /*  Didn't find any?  Free up memory and return with NULL value  */
   free(ifconf.ifc_buf);
   return(NULL);
}




/*****************************************************************************
   if_nameindex():   This functions returns the list of index/interface_name
   pairs.  The function receives the information with an ioctl call.  The
   returned data is stepped through to grab both the index number and the
   appropriate string (name).

   We then count the number of bytes needed and create a huge block of memory.
   This block of memory will host (1) an array of structs (packed) which is
   (2) followed by a list of character strings packed.  A pointer to this
   block of data is returned to the calling function.
******************************************************************************/

struct if_nameindex *if_nameindex(void)
{
  int i;

  struct ifconf ifconf;
  int ifc_lenindex;    /*  Length index counter.  */


  int buflen;            /* Total buffer size       */

  u_char *dataline;
  u_char *dataendpiece;
  u_char datafamily;

  int count_sockaddr_entries;
  int count_char_bytes;

  struct if_nameindex *if_nameindex;
  struct if_nameindex *if_nameindex_struct_index;
  char *if_nameindex_charblock_start;
  char *if_nameindex_charblock_index;

  /*  Get a socket, set-up memory buffer and then call IOCTL with SIOGIFCONF. */
  {
    ifconf.ifc_len=0;
    ifconf.ifc_buf=NULL;

      if ((i = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0) {
         return NULL;
      }

      buflen = sizeof(struct ifreq) * 1000;

      while(1) {
         if (!(ifconf.ifc_buf = (void *)malloc(ifconf.ifc_len = buflen))) {
            return NULL;
          }

          if (ioctl(i, SIOCGIFCONF, (char *)&ifconf) < 0) {
             exit(1);
          }



          /*----------------------------------------------------------------
               Test to see returned ioctl data fits into our allocated
               buffer:   If so, then break out of loop; otherwise,
               free the newly created buffer and try a new size and
               while-loop block over again until correct.
          -----------------------------------------------------------------*/

          if (ifconf.ifc_len < buflen)
          break;

          free(ifconf.ifc_buf);
          buflen += sizeof(struct ifreq);

      }  /* End of "while (1)" */

      close(i);
  }  /* End of get, set-up, and call. */










  /*-------------------------------------------------------------------
     FIRST PASS -- Count the total number of sockaddr_dl entries &
     the total number of bytes of name characters:

     Start walking through the ifconf data (returned from ioctl call),
     looking for AF_LINK (sockaddr_dl) entries with the correct index.
     See figure 4.28 of Steven's TCP/IP Illustrated Vol. II, page 121.)
   *-------------------------------------------------------------------*/

  ifc_lenindex   = 0;
  dataline       = (u_char *)ifconf.ifc_buf;       /*  Start with the first chunk of data. */

  count_sockaddr_entries = 0;
  count_char_bytes = 0;
    
  while (ifc_lenindex < (ifconf.ifc_len - 1))  {
       dataendpiece   = dataline + IFNAMSIZ;
       datafamily     = ((struct sockaddr *)dataendpiece)->sa_family;


       if ((int)datafamily == AF_LINK) {
           count_sockaddr_entries++;
           count_char_bytes += (((struct sockaddr_dl *)dataendpiece)->sdl_nlen);
        }                


          /*  Point to the next chunk of data.  */
       ifc_lenindex += (IFNAMSIZ + (((struct sockaddr *)dataendpiece)->sa_len));
       dataline     += (IFNAMSIZ + (((struct sockaddr *)dataendpiece)->sa_len));
   }  /*  End of while  */



  /*---------------------------------------------------------------------
    Now, find the correct size of memory for returning the
    "struct if_nameindex arrays".


    I need memory allocation for (a) the `if_nameindex' structures,
    (b) each interface name (and (c) plus adding in their null termination) and
    (d) a final `if_nameindex' structure with a zero and a NULL value stored.
   ---------------------------------------------------------------------*/


  if (!(if_nameindex=(void *)malloc( ((count_sockaddr_entries + 1) * sizeof(struct if_nameindex))+
                                     (count_char_bytes * sizeof(char))+
                                     (count_sockaddr_entries * sizeof(char)) ) )) {
        return NULL;
   }


   
   if_nameindex_charblock_start =  (void *)if_nameindex +
                                   ((count_sockaddr_entries + 1) * sizeof(struct if_nameindex));

                                  

  /*-------------------------------------------------------------------
     SECOND PASS throught the returned ifconf data:
     Start walking through the ifconf data (returned from ioctl call),
     looking for AF_LINK (sockaddr_dl) entries with the correct index
     When found, add the index and name to array.
   *-------------------------------------------------------------------*/

  ifc_lenindex    = 0;
  dataline       = (u_char *)ifconf.ifc_buf;

  /*   Initialize struct `if_nameindex_struct_index' to start of allocated
       memory location.  */
  if_nameindex_struct_index = if_nameindex;
  if_nameindex_charblock_index = if_nameindex_charblock_start;
   

  /*  Start looking at the ifconf values   */
  while (ifc_lenindex < (ifconf.ifc_len - 1))  {
       dataendpiece   = dataline + IFNAMSIZ;

       if (((struct sockaddr *)dataendpiece)->sa_family == AF_LINK ) { 

           if_nameindex_struct_index->if_index = ((struct sockaddr_dl *)dataendpiece)->sdl_index;           
           if_nameindex_struct_index->if_name = if_nameindex_charblock_index;

           memcpy(if_nameindex_struct_index->if_name,
                  ((struct sockaddr_dl *)dataendpiece)->sdl_data,
                  ((struct sockaddr_dl *)dataendpiece)->sdl_nlen  );

           if_nameindex_charblock_index += ((struct sockaddr_dl *)dataendpiece)->sdl_nlen;
           *if_nameindex_charblock_index = '\0';
           if_nameindex_charblock_index++;

           if_nameindex_struct_index++;

       }  /* End of if. */

       /*  Point to the next chunk of data from ifconf.  */
       ifc_lenindex += (IFNAMSIZ + (((struct sockaddr *)dataendpiece)->sa_len));
       dataline     += (IFNAMSIZ + (((struct sockaddr *)dataendpiece)->sa_len));
   }  /*  End of while.  */

  if_nameindex_struct_index->if_index = 0;
  if_nameindex_struct_index->if_name  = NULL;
  free(ifconf.ifc_buf);
  return( if_nameindex );
}
