/*
 * in6.h  --  Basic definitions of structures and constants of IPv6
 *            internets.   Roughly corresponds to netinet/in.h.
 *
 * Copyright 1995 by Dan McDonald, Bao Phan, and Randall Atkinson,
 *	All Rights Reserved.  
 *      All Rights under this copyright have been assigned to NRL.
 */

/*----------------------------------------------------------------------
#       @(#)COPYRIGHT   1.1a (NRL) 17 August 1995

COPYRIGHT NOTICE

All of the documentation and software included in this software
distribution from the US Naval Research Laboratory (NRL) are
copyrighted by their respective developers.

This software and documentation were developed at NRL by various
people.  Those developers have each copyrighted the portions that they
developed at NRL and have assigned All Rights for those portions to
NRL.  Outside the USA, NRL also has copyright on the software
developed at NRL. The affected files all contain specific copyright
notices and those notices must be retained in any derived work.

NRL LICENSE

NRL grants permission for redistribution and use in source and binary
forms, with or without modification, of the software and documentation
created at NRL provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. All advertising materials mentioning features or use of this software
   must display the following acknowledgement:

        This product includes software developed at the Information
        Technology Division, US Naval Research Laboratory.

4. Neither the name of the NRL nor the names of its contributors
   may be used to endorse or promote products derived from this software
   without specific prior written permission.

THE SOFTWARE PROVIDED BY NRL IS PROVIDED BY NRL AND CONTRIBUTORS ``AS
IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL NRL OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation
are those of the authors and should not be interpreted as representing
official policies, either expressed or implied, of the US Naval
Research Laboratory (NRL).

----------------------------------------------------------------------*/

#ifndef _NETINET6_IN6_H
#define _NETINET6_IN6_H 1

/*
 * Next header types (called Protocols in netinet/in.h).
 */

#define IPPROTO_HOP		0	/* Hop-by-hop option header. */
#define IPPROTO_ICMPV4		1	/* ICMP message for IPv4. */
#define IPPROTO_IGMP		2	/* IGMP message. */
#define IPPROTO_IPV4		4	/* IPv4 in IPv6 (?!?) */
#define IPPROTO_IPV6		41	/* IPv6 in IPv6 */
#define IPPROTO_ROUTING		43	/* Routing header. */
#define IPPROTO_FRAGMENT	44	/* Fragmentation/reassembly header. */
#define IPPROTO_ESP		50	/* Encapsulating security payload. */
#define IPPROTO_AH		51	/* Authentication header. */
#define IPPROTO_ICMPV6		58      /* ICMP for IPv6 */
#define IPPROTO_NONE		59	/* No next header */
#define IPPROTO_DST		60	/* Destination options header. */

/*
 * Following are TBD, and subject to change rapidly
 */
#define IPPROTO_RAW		255	/* Payload of unknown type? */
#define IPPROTO_MAX		256	/* Upper bound for next header type. */

/* IPPROTO type macros. */

#define IS_PREFRAG(x)   ( (x)==IPPROTO_HOP || (x)==IPPROTO_ROUTING || \
			 (x) == IPPROTO_DST)
#define IS_IPV6OPT(x)   ( (x)==IPPROTO_FRAGMENT || (x) == IPPROTO_AH || \
			 IS_PRFRAG(x) )

/* Use netinet/in.h for port information.  Cheesy hack to include in.h
   iff a macro in in.h (IP_OPTIONS, in this case) isn't defined. */

#ifndef IP_OPTIONS
#include <netinet/in.h>
#endif

/*
 * Include vital type information (file is protected against multiple
 * inclusions).
 */

#include <netinet/in6_types.h>

/*
 * Macros to manipulate in6_addr.
 *
 * Note all macros for dealing with in6_addr assume you can use '.' rather
 * than '->'.  This is because we might support 16-byte ints, and
 * plug in simple assignment statements for these macros. 
 * (This rationale made much more sense when it was 8-byte SIP{,P}.
 *
 * These macros currently deal with in6_addr's 32-bits at a time.
 * Eventually, 64-bits at a time would be nice, and as mentioned, 128-would
 * be wicked awesome.
 *
 * It is recommended that statement macros are placed in {} in _if_
 * clauses, if they aren't inside them already.  This is so when they are
 * expanded, you don't have:
 *
 * if (foo == bar)
 *   MACRO(x);
 * else printf("error");
 *
 * turn into...
 *
 * if (foo == bar)
 *   {
 *     mystery();
 *   };
 * else printf("error");
 *
 * Because that semicolon will cause an error!
 *
 */

#define IN6_ADDR_ASSIGN(a,w1,w2,w3,w4) {(a).in6a_words[0]=(w1);\
					(a).in6a_words[1]=(w2);\
					(a).in6a_words[2]=(w3);\
					(a).in6a_words[3]=(w4);}
#define IN6_ADDR_COPY(a1,a2) IN6_ADDR_ASSIGN(a1,\
					     (a2).in6a_words[0],\
					     (a2).in6a_words[1],\
					     (a2).in6a_words[2],\
					     (a2).in6a_words[3])
#define IN6_ADDR_EQUAL(a1,a2) ( (a1).in6a_words[3] == (a2).in6a_words[3] &&\
                                 (a1).in6a_words[2] == (a2).in6a_words[2] &&\
                                 (a1).in6a_words[1] == (a2).in6a_words[1] &&\
                                 (a1).in6a_words[0] == (a2).in6a_words[0] )

/* These assume bytes in address 'a' are in network order... */
#define IS_IN6_UNSPEC(a) ( (a).in6a_words[3] == 0 &&\
                                 (a).in6a_words[2] == 0 &&\
                                 (a).in6a_words[1] == 0 &&\
                                 (a).in6a_words[0] == 0 )

/* NOTE:  IS_IN6_ALL* macros only check the 0x1 and 0x2 scoping levels.
          The IN6 ROAD document doesn't say those are good for higher
          scoping levels.
*/

#define IN6_INTRA_NODE 1  /*  intra-node scope */
#define IN6_INTRA_LINK 2  /*  intra-link scope */
/*            3  (unassigned)
            4  (unassigned)
*/
#define IN6_INTRA_SITE 5  /* intra-site scope */
/*            6  (unassigned)
            7  (unassigned)
*/
#define IN6_INTRA_ORG  8  /* intra-organization scope */
/*            9  (unassigned)
            A  (unassigned)
*/
#define IN6_INTRA_COMM 0xB/*  intra-community scope */
/*            C  (unassigned)
            D  (unassigned)
*/

#define IN6_GLOBAL   0xE  /* global scope*/
/*            F  reserved
*/

#if BYTE_ORDER ==  BIG_ENDIAN

#define SET_IN6_ALLNODES(a)  {(a).in6a_words[0]=0xff000000;(a).in6a_words[3]=1;\
                              (a).in6a_words[1]=0;(a).in6a_words[2]=0;}
#define SET_IN6_ALLROUTERS(a)  {(a).in6a_words[0]=0xff000000;(a).in6a_words[3]=2;\
                              (a).in6a_words[1]=0;(a).in6a_words[2]=0;}

#define SET_IN6_MCASTSCOPE(a,bits) {(a).in6a_words[0]&=0xfff0ffff;\
                                    (a).in6a_words[0]|=(bits<<16);}
#define GET_IN6_MCASTSCOPE(a) ( ((a).in6a_words[0] & 0x000f0000) >> 16  )

#define CREATE_IPV6_MAPPED(a, v4) {(a).in6a_words[0] = 0; \
				   (a).in6a_words[1] = 0; \
				   (a).in6a_words[2] = 0xffff; \
				   (a).in6a_words[3] = (v4);}

#define IS_IN6_LOOPBACK(a)  ( (a).in6a_words[0] == 0 && \
                               (a).in6a_words[1] == 0 && (a).in6a_words[2] == 0 && \
                               (a).in6a_words[3] == 1 )

#define IS_IN6_SITELOC(a)   (((a).in6a_words[0] & 0xFFC00000) == 0xFEC00000 )
#define IS_IN6_LINKLOC(a)   (((a).in6a_words[0] & 0xFFC00000) == 0xFE800000 )
#define IS_IN6_NSAP(a)      (((a).in6a_words[0] & 0xfc000000) == 0x02000000)
#define IS_IN6_IPX(a)       (((a).in6a_words[0] & 0xfc000000) == 0x04000000)
#define IS_IN6_MULTICAST(a) (((a).in6a_words[0] & 0xff000000) == 0xff000000 )
#define IS_IN6_MULTICAST_RESERVED(a)  \
  ((a).in6a_words[1] == 0 && (a).in6a_words[2] == 0 && (a).in6a_words[3] == 0 \
   && ((a).in6a_words[0] >> 16) >= 0xff00 \
   && ((a).in6a_words[0] >> 16) <= 0xff0f)

#define IS_IN6_IPV4_COMPAT(a) ((a).in6a_words[0] == 0 && (a).in6a_words[1]==0 \
			       && (a).in6a_words[2] == 0 && \
			       (a).in6a_words[3] != 0 && (a).in6a_words[3]!=1)
#define IS_IN6_IPV4_MAPPED(a) ((a).in6a_words[0] == 0 && (a).in6a_words[1]==0 \
			       && (a).in6a_words[2] == 0xffff)

/* Scoping values? */

#define IS_IN6_ALLNODES(a)  (((a).in6a_words[0] == 0xff010000 || \
                              (a).in6a_words[0] == 0xff020000 ) && \
                              (a).in6a_words[1] == 0 && (a).in6a_words[2] == 0 && \
                              (a).in6a_words[3] == 1   )
#define IS_IN6_ALLROUTERS(a)  (((a).in6a_words[0] == 0xff010000 || \
                              (a).in6a_words[0] == 0xff020000 ) && \
                              (a).in6a_words[1] == 0 && (a).in6a_words[2] == 0 && \
                              (a).in6a_words[3] == 2   )
#else   /* BYTE_ORDER == LITTLE_ENDIAN */

#define SET_IN6_ALLNODES(a)  {(a).in6a_words[0]=0xff;(a).in6a_words[3]=0x01000000;\
                              (a).in6a_words[1] = 0; (a).in6a_words[2] = 0;}
#define SET_IN6_ALLROUTERS(a)  {(a).in6a_words[0]=0xff;(a).in6a_words[3]=0x02000000;\
                              (a).in6a_words[1] = 0; (a).in6a_words[2] = 0;}

#define SET_IN6_MCASTSCOPE(a,bits) {(a).in6a_words[0]&=0xfffff0ff;\
                                    (a).in6a_words[0]|=(bits<<8);}
#define GET_IN6_MCASTSCOPE(a)  ( ((a).in6a_words[0] & 0x00000f00) >>8)

#define CREATE_IPV6_MAPPED(a, v4) {(a).in6a_words[0] = 0; \
				   (a).in6a_words[1] = 0; \
				   (a).in6a_words[2] = 0xffff0000; \
				   (a).in6a_words[3] = (v4);}



#define IS_IN6_LOOPBACK(a)  ((a).in6a_words[0] == 0 && \
			     (a).in6a_words[1] == 0 && (a).in6a_words[2] == 0 && \
			     (a).in6a_words[3]==0x01000000 )

#define IS_IN6_SITELOC(a)   (((a).in6a_words[0] & 0xc0ff) == 0xc0fe )
#define IS_IN6_LINKLOC(a)   (((a).in6a_words[0] & 0xc0ff) == 0x80fe )
#define IS_IN6_NSAP(a)     (((a).in6a_words[0] & 0xfc) == 0x02 )
#define IS_IN6_IPX(a)     (((a).in6a_words[0] & 0xfc) == 0x04 )
#define IS_IN6_MULTICAST(a) (((a).in6a_words[0] & 0xff) == 0xff )
#define IS_IN6_MULTICAST_RESERVED(a)  \
  ((a).in6a_words[1] == 0 && (a).in6a_words[2] == 0 && (a).in6a_words[3] == 0 \
   ((a).in6a_words[0] & 0xff) == 0xff && ((a).in6a_words[0] >> 8) <= 0xf )
#define IS_IN6_IPV4_COMPAT(a) ((a).in6a_words[0] == 0 && (a).in6a_words[1]==0 \
			       && (a).in6a_words[2] == 0 && \
			       (a).in6a_words[3] != 0 &&\
			       (a).in6a_words[3] != 0x1000000)
#define IS_IN6_IPV4_MAPPED(a) ((a).in6a_words[0] == 0 && (a).in6a_words[1]==0 &&\
                              (a).in6a_words[2] == 0xffff0000)


/* Scoping values? */

#define IS_IN6_ALLNODES(a)  (((a).in6a_words[0] == 0x000001ff || \
                              (a).in6a_words[0] == 0x000002ff ) && \
                              (a).in6a_words[1] == 0 && (a).in6a_words[2] == 0 && \
                              (a).in6a_words[3] == 0x01000000   )
#define IS_IN6_ALLROUTERS(a)  (((a).in6a_words[0] == 0x000001ff || \
                              (a).in6a_words[0] == 0x000002ff ) && \
                              (a).in6a_words[1] == 0 && (a).in6a_words[2] == 0 && \
                              (a).in6a_words[2] == 0x02000000   )

#endif  /* BYTE_ORDER == {BIG,LITTLE}_ENDIAN */

#define IN6_IS_ADDR_UNSPECIFIED(a) \
        ((((uint32_t *)(a))[0] == 0) && (((uint32_t *)(a))[1] == 0) && \
         (((uint32_t *)(a))[2] == 0) && (((uint32_t *)(a))[3] == 0))

#define IN6_IS_ADDR_LOOPBACK(a) \
        ((((uint32_t *)(a))[0] == 0) && (((uint32_t *)(a))[1] == 0) && \
         (((uint32_t *)(a))[2] == 0) && (((uint32_t *)(a))[3] == htonl(1)))

#define IN6_IS_ADDR_MULTICAST(a) (((uint8_t *)(a))[0] == 0xff)

#define IN6_IS_ADDR_LINKLOCAL(a) \
        ((((uint32_t *)(a))[0] & htonl(0xffc00000)) == htonl(0xfe800000))

#define IN6_IS_ADDR_SITELOCAL(a) \
        ((((uint32_t *)(a))[0] & htonl(0xffc00000)) == htonl(0xfec00000))

#define IN6_IS_ADDR_V4MAPPED(a) \
        ((((uint32_t *)(a))[0] == 0) && (((uint32_t *)(a))[1] == 0) && \
         (((uint32_t *)(a))[2] == htonl(0xffff)))

#define IN6_IS_ADDR_V4COMPAT(a) \
        ((((uint32_t *)(a))[0] == 0) && (((uint32_t *)(a))[1] == 0) && \
         (((uint32_t *)(a))[2] == 0) && (ntohl(((uint32_t *)(a))[3]) > 1))

#define IN6_IS_ADDR_MC_NODELOCAL(a) \
        (GET_IN6_MCASTSCOPE(*a) == IN6_INTRA_NODE)

#define IN6_IS_ADDR_MC_LINKLOCAL(a) \
        (GET_IN6_MCASTSCOPE(*a) == IN6_INTRA_LINK)

#define IN6_IS_ADDR_MC_SITELOCAL(a) \
        (GET_IN6_MCASTSCOPE(*a) == IN6_INTRA_SITE)

#define IN6_IS_ADDR_MC_ORGLOCAL(a) \
        (GET_IN6_MCASTSCOPE(*a) == IN6_INTRA_ORG)

#define IN6_IS_ADDR_MC_COMMLOCAL(a) \
        (GET_IN6_MCASTSCOPE(*a) == IN6_INTRA_COMM)

#define IN6_IS_ADDR_MC_GLOBAL(a) \
        (GET_IN6_MCASTSCOPE(*a) == IN6_GLOBAL)

/*
 * Additonal type information.
 */

#ifndef linux
/* For apps to port between 4.3-derived and 4.4-derived systems. */
#define SIN6_LEN

struct sockaddr_in6 {
  u_char sin6_len;
  u_char sin6_family;
  u_short sin6_port;
  uint32_t sin6_flowinfo;
  struct in6_addr sin6_addr;
};
#endif /* linux */

struct ipv6_mreq
{
  struct in6_addr ipv6mr_multiaddr;     /* Group addr. to join/leave. */
  unsigned int ipv6mr_interface;     /* Interface on which to do it. */
};

/* Cheesy hack for if net/route.h included... */
#ifdef RTM_VERSION
/*
 * sizeof(struct sockaddr_in6) > sizeof(struct sockaddr), therefore, I
 * need to define... 
 */ 
struct route6
{
  struct  rtentry *ro_rt;
  struct  sockaddr_in6 ro_dst;
};
#endif /* RTM_VERSION */

/*
 * IP options for IPv6.  Note I use the IPV6_* semantics for IPv6-
 * specific options.  Another reason for the inclusion of <netinet/in.h> is
 * for the options that are common between IPv6 and IPv4.
 */

#define	IPV6_OPTIONS		1    /* buf/ipv6_opts; set/get IP options */
#define	IPV6_HDRINCL		2    /* int; header is included with data */
#define	IPV6_TOS		3    /* int; IP type of service and preced. */
#define	IPV6_TTL		4    /* int; IP time to live */

#define IPV6_PKTINFO            5    /* struct in6_pktinfo: if. and addr */
#define IPV6_HOPLIMIT           6    /* uint8_t: IPv6 hop limit */
#define IPV6_CHECKSUM		7    /* int: checksum offset */
#define ICMPV6_FILTER		8    /* struct icmpv6_filter: type filter */

#define	IPV6_MULTICAST_IF	9    /* in6_addr; set/get IP multicast i/f  */
#define	IPV6_MULTICAST_HOPS	10   /* int; set/get IP multicast ttl */
#define	IPV6_MULTICAST_LOOP	11   /* int; set/get IP multicast loopback
				      */
#define	IPV6_ADD_MEMBERSHIP	12   /* ipv6_mreq; add an IP group membership
				      */
#define	IPV6_DROP_MEMBERSHIP	13   /* ipv6_mreq; drop an IP group membership
				      */

#define IPV6_ADDRFORM   0x16    /* int; get/set form of returned addrs. */
#define IPV6_RXHOPOPTS  0x19
#define IPV6_RXDSTOPTS  0x1a
#define IPV6_RXSRCRT    0x1b

#define IPV6_RXINFO     IPV6_PKTINFO
#define IPV6_TXINFO     IPV6_PKTINFO

#define IPV6_UNICAST_HOPS       IPV6_TTL

#define IN6_MAX_MEMBERSHIPS 20  /* Maximum number of multicast memberships. */
#define IPV6_DEFAULT_MCAST_HOPS 1
#define IPV6_DEFAULT_MCAST_LOOP 1

/*
 * Definitions for inet6 sysctl operations.
 *
 * Third level is protocol number.
 * Fourth level is desired variable within that protocol.
 */

#define	IPV6PROTO_MAXID	(2)	/* Only have 1 right now. */

#define	CTL_IPV6PROTO_NAMES { \
	{ "ipv6", CTLTYPE_NODE }, \
	{ 0, 0 }, \
}

/*
 * Names for IPv6 sysctl objects
 */

/* NOTE: IP forwarding taken care of in IP. */
/*#define	IPCTL_FORWARDING	1*/	/* act as router */

#define	IPV6CTL_SENDREDIRECTS	2	/* may send redirects when forwarding */
#define	IPV6CTL_DEFTTL		3	/* default TTL */
#ifdef notyet
#define	IPV6CTL_DEFMTU		4	/* default MTU */
#endif
#define	IPV6CTL_DEBUGLEVEL	5
#define	IPV6CTL_MAXID		6

#define	IPV6CTL_NAMES { \
	{ 0, 0 }, \
	{ 0, 0 }, \
	{ "redirect", CTLTYPE_INT }, \
	{ "ttl", CTLTYPE_INT }, \
	{ "mtu", CTLTYPE_INT }, \
	{ "debuglevel", CTLTYPE_INT }, \
	{ 0, 0 }, \
}

#ifdef KERNEL
/* Function prototypes go here. */
int in6_cksum __P((struct mbuf *,int, u_int, u_int));
#endif /* KERNEL */

struct in6_pktinfo {
  int ipi6_ifindex;
  struct in6_addr ipi6_addr;
};
#endif /* _NETINET6_IN6_H */
