/*
%%% copyright-cmetz-97
This software is Copyright 1997 by Craig Metz, All Rights Reserved.
The Inner Net License Version 2 applies to this software.
You should have received a copy of the license with this software. If
you didn't get a copy, you may request one from <license@inner.net>.

*/

#include <sys/types.h>
#include <stdio.h>
#include <sys/socket.h>
#include <netinet/in.h>
#if INET6
#include <netinet6/in6.h>
#endif /* INET6 */
#if LOCAL
#include <sys/un.h>
#endif /* LOCAL */
#include "support.h"
#if FASTCTO
#include <setjmp.h>
#endif /* FASTCTO */
#include <errno.h>
#include <netdb.h>

#if FASTCTO
static jmp_buf timeout_env;

static void timeout_handler(int i)
{
  longjmp(timeout_env, i);
};
#endif /* FASTCTO */

static int default_trying_callback(struct sockaddr *sa)
{
  char hbuf[NI_MAXHOST], sbuf[NI_MAXSERV];

  if (getnameinfo(sa, NRL_SA_LEN(sa), hbuf, sizeof(hbuf), sbuf, sizeof(sbuf), NI_NUMERICHOST | NI_NUMERICSERV)) {
    fprintf(stderr, "inner_getstream: getnameinfo failed\n");
    return -1;
  };

  fprintf(stderr, "Trying %s.%s...\n", hbuf, sbuf);
  return 0;
};

static int default_error_callback(char *myname, char *message)
{
  fprintf(stderr, "%s: %s\n", myname, message);
  return 0;
};

int inner_connect(struct addrinfo *ai, char *request, int requestlen, int (*trying_callback)(struct sockaddr *sa), int (*error_callback)(char *myname, char *message), char *myname, struct addrinfo **pai)
{
  int fd;
  char errorbuf[128];
#if FASTCTO
  void (*old_handler)(int);
#endif /* FASTCTO */

  if (!trying_callback)
    trying_callback = default_trying_callback;

  if (!error_callback)
    error_callback = default_error_callback;

#if FASTCTO
  old_handler = signal(SIGALRM, &timeout_handler);
#endif /* FASTCTO */

  for (; ai; ai = ai->ai_next) {
    if (trying_callback(ai->ai_addr))
      continue;

    if ((fd = socket(ai->ai_family, ai->ai_socktype, ai->ai_protocol)) < 0) {
      snprintf(errorbuf, sizeof(errorbuf), "socket: %s(%d)", strerror(errno), errno);
      error_callback(myname, errorbuf);
      continue;
    };

#if NETSEC
    if (setsockopt(fd, SOL_SOCKET, SO_SECURITY_REQUEST, request, requestlen) < 0) {
      snprintf(errorbuf, sizeof(errorbuf), "setsockopt(..., SO_SECURITY_REQUEST, ...): %s(%d)", strerror(errno), errno);
      error_callback(myname, errorbuf);
      close(fd);
      continue;
    };
#endif /* NETSEC */

#if FASTCTO
    if (setjmp(timeout_env)) {
      snprintf(errorbuf, sizeof(errorbuf), "Connection timed out");
      error_callback(myname, errorbuf);
      close(fd);
      continue;
    };

    alarm(FASTCTO);
#endif /* FASTCTO */

    if (connect(fd, ai->ai_addr, ai->ai_addrlen) < 0) {
#ifdef FASTCTO
      alarm(0);
#endif /* FASTCTO */
      snprintf(errorbuf, sizeof(errorbuf), "connect: %s(%d)", strerror(errno), errno);
      error_callback(myname, errorbuf);
      close(fd);
      continue;
    }

#ifdef FASTCTO
    alarm(0);
#endif /* FASTCTO */

    break;
  };

  if (ai) {
    if (pai)
      *pai = ai;
  } else {
    snprintf(errorbuf, sizeof(errorbuf), "no connections result");
    error_callback(myname, errorbuf);
    fd = -1;
  };

#if FASTCTO
  signal(SIGALRM, old_handler);
#endif /* FASTCTO */

  return fd;
};
