/*
 * Copyright (c) 1994, 1995, 1997
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
#ifndef lint
static char rcsid[] =
    "@(#) $Header: bootptab.c,v 1.7 97/01/28 18:17:59 leres Exp $ (LBL)";
#endif

#include <sys/types.h>
#include <sys/stat.h>

#include <netinet/in.h>

#include <arpa/inet.h>

#include <errno.h>
#include <memory.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include "gnuc.h"
#ifdef HAVE_OS_PROTO_H
#include "os-proto.h"
#endif

#include "bootptab.h"
#include "utils.h"

static	char *readbootptab __P((FILE *));
static	char *checkbootptab __P((void));

#define HASHSIZE 4096
#define HASH(a) (&addr_table[(a) & (HASHSIZE - 1)])

static u_int32_t addr_table[HASHSIZE];

static char *bootptab;

char *
setbootptab(file)
	register char *file;
{

	bootptab = file;
	return (checkbootptab());
}


/*
 * Determine if this is a "good client" or not.
 * Good clients have a bootptab entry.
 */
static char *
checkbootptab()
{
	register char *cp;
	register FILE *f;
	struct stat st;
	static char errmsg[128];
	static time_t mtime = 0, ctime = 0;

	if (bootptab == NULL)
		return ("no bootptab set");

	/* Stat the bootptab */
	if (stat(bootptab, &st) < 0) {
		(void)sprintf(errmsg, "stat on \"%s\": %s",
		    bootptab, strerror(errno));
		return (errmsg);
	}

	/* Nothing to do if nothing has changed */
	if (mtime == st.st_mtime && ctime == st.st_ctime)
		return (NULL);

	log(LOG_INFO, "%sreading \"%s\"", mtime != 0L ? "re-" : "", bootptab);

	/* Open bootptab file */
	if ((f = fopen(bootptab, "r")) == NULL) {
		(void)sprintf(errmsg, "error opening \"%s\": %s",
		    bootptab, strerror(errno));
		return (errmsg);
	}

	/*
	 * Record file modification time.
	 */
	if (fstat(fileno(f), &st) < 0) {
		(void)sprintf(errmsg, "fstat: %s", strerror(errno));
		fclose(f);
		return (errmsg);
	}
	mtime = st.st_mtime;
	ctime = st.st_ctime;
	cp = readbootptab(f);
	fclose(f);
	return (cp);
}

static char *
readbootptab(f)
	register FILE *f;
{
	register int i;
	register char *cp, *cp2;
	char line[BUFSIZ];
	register u_int32_t a, *p;

	memset(addr_table, 0, sizeof(addr_table));
	while (fgets(line, sizeof(line), f)) {
		cp = line;

		/* Kill trailing newline */
		cp2 = cp + strlen(cp) - 1;
		if (cp2 >= cp && *cp2 == '\n')
			*cp2++ = '\0';

		/* Scan for ip addresses */
		while (*cp != '\0') {
			if (*cp == '#')
				break;
			if (*cp != 'i') {
				++cp;
				continue;
			}
			if (*++cp != 'p' || *++cp != '=')
				continue;

			/* found one */
			++cp;
			cp2 = strchr(cp, ':');
			if (cp2 != NULL)
				*cp2 = '\0';
			a = inet_addr(cp);
			if ((int32_t)a == -1 || a == 0)
				break;

			p = HASH(a);
			i = HASHSIZE;
			while (*p != 0 && i-- > 0) {
				++p;
				if (p > &addr_table[HASHSIZE - 1])
					p = addr_table;
			}
/* XXX should check if the table is full? */
			*p = a;
			break;
		}
	}
	return (NULL);
}

/* Returns true if the specified address is a client in the bootptab file */
int
goodclient(a)
	register u_int32_t a;
{
	register char *cp;
	register int i;
	register u_int32_t *p;

	cp = checkbootptab();
	if (cp != NULL)
		log(LOG_ERR, "goodclient: checkbootptab: %s", cp);

	p = HASH(a);
	p = &addr_table[a & (HASHSIZE - 1)];
	i = HASHSIZE;
	while (*p != 0 && i-- > 0) {
		if (*p == a)
			return (1);
		++p;
		if (p > &addr_table[HASHSIZE - 1])
			p = addr_table;
	}
	return (0);
}
