/*
 *	(c) Copyright 1990, Kim Fabricius Storm.  All rights reserved.
 *
 *	Database access and update
 */
#include "config.h"
#include "db.h"

#include "hash.h"
#include "hdbm.h"
#include "newsoverview.h"

/* db.c */

static sort_gh __APROTO((group_header **g1, group_header **g2));
static void reloadactnewsrc __APROTO((void));
static void readactnewsrc __APROTO((void));

static void db_init_group __APROTO((group_header *gh, int num));
static void readtimfile __APROTO((void));
static void readactfile __APROTO((void));
static void db_fixup_cross_postings __APROTO((data_header *dhp,
					      data_dynamic_data *ddp,
					      struct novart *artp));

static struct novgroup *ngovp;
static struct novart *allarts;
static char **actlist, **grplist;
static HASHTABLE *timtbl;

extern long	atol();
extern void novclose();		/* trash last group's data */

#ifdef	CACHE_PURPOSE
#include <ctype.h>

struct purp_list {
    char *group_name;
    char *purpose;
} *purp_list;

static void cache_purpose __APROTO((void));
static int purp_cnt;
#endif	/* CACHE_PURPOSE */

/*
 * String pool allocation
 */
typedef struct stlisthdr {
    struct stlist *next;
} stlisthdr_t;

typedef struct stlist {
    stlisthdr_t n;
    char str[1];
} stlist_t;

static char *strkeep __APROTO((char *s, int hdr, int poolid));
static char *pmalloc __APROTO((int size, int poolid));
static void pfree __APROTO((int poolid));

#define	POOL_GRP 0	/* group names */
#define	POOL_PUR 1	/* cached purpose */
#define	POOL_ACT 2	/* active data */
#define	POOL_TIM 3	/* active.times data */
#define	POOL_MAX 3
#define	POOL_TMP 4	/* fake - this is not actually used */

import char
*master_directory, *db_directory, *db_data_directory, *news_directory,
  *news_lib_directory;
import int db_data_subdirs;

import int new_group_action;

import int nntp_progress;
import int newsrc_only;
import char *server_nntp;
import char newsrc_name[];


export master_header master;
#ifdef MALLOC_64K_LIMITATION
export group_header **active_groups = NULL;
#else
export group_header *active_groups = NULL;
#endif
export group_header **sorted_groups = NULL;

export int  reread_groups_file = 0;  

export int  check_group_access = 0;
export int32 db_read_counter = 0; 	/* articles read by db_read_art */

export data_header db_hdr;
export data_dynamic_data db_data;

/*
 * Init access to a group
 */

export group_header *current_group = NULL;

export char group_path_name[FILENAME];
export char *group_file_name = NULL;

static char *group_position = NULL;
static article_number current_digest_article = 0;

int
init_group(gh)
register group_header *gh;
{
    register char *p, *q;

    current_digest_article = 0;

    if (gh == NULL) 
      return 0;
    /*    if (gh->master_flag & M_IGNORE_GROUP) return 0; */	/* OBS */
    if (gh == current_group) 
      return 1;

    current_group = gh;

    if (gh->group_flag & G_FOLDER) {
	group_position = NULL;
	group_file_name = NULL;
	strcpy(group_path_name, gh->archive_file);
	return 1;
    }

#ifdef NNTP
    if (use_nntp && nntp_set_group(gh) < 0)
	return 0;
#endif /* NNTP */

    if (group_position == NULL)
	{
	    strcpy(group_path_name, news_directory);
	    group_position = group_path_name + strlen(group_path_name);
	    *group_position++ = '/';
	}

    for (p = group_position, q = gh->group_name; *q; q++)
	*p++ = (*q == '.') ? '/' : *q;

    /* client */
    if (gh->master_flag & M_NO_DIRECTORY) 
        return 0;

    if (check_group_access && !use_nntp) {
	*p = NUL;
	if (file_exist(group_path_name, "dxr") == 0) 
	    return 0;
    }

    *p++ = '/';
    *p = NUL;
    group_file_name = p;
    return 1;
}


/*
 *	Init the groups data from active file.
 */
void
open_master(mode)
int	mode;
{
    register group_header *gh;

    freeobj(sorted_groups);
    freeobj(active_groups);
    active_groups = NULL;
    sorted_groups = NULL;

    if (newsrc_only)
	readactnewsrc();
    else 
        readactfile();		/* Read in the active file - count groups */

    /* reading time file slow over serial link, don't if no need */
    if ((new_group_action >= 3) && !newsrc_only)
        readtimfile();		/* Read the newsgroup creation time file */

    db_expand_master();		/* uses count from readact() call! */
  
    Loop_Groups_Header(gh) {
        /* db_read_group opens a file per call; use db_init_group instead */
        db_init_group(gh, l_g_index);
    }

    /* Free actlist and timtbl space */
    pfree(POOL_ACT);
    free(actlist);

    pfree(POOL_TIM);
    free(timtbl);
    timtbl = NULL;

    sort_groups();
#ifdef	CACHE_PURPOSE
    cache_purpose();	/* cache sorted newsgroups and descriptions */
#endif
}

void
db_expand_master()
{
    master.free_groups = 20;

    active_groups = resizeobj(active_groups, group_header,
			      master.number_of_groups + master.free_groups);
    clearobj(active_groups + master.number_of_groups, group_header,
	     master.free_groups);
    sorted_groups = resizeobj(sorted_groups, group_header * , master.number_of_groups + master.free_groups);
}

void
close_master()
{
    if (ngovp) {
	novclose(ngovp);
	ngovp = NULL;
    }
}


static int
sort_gh(g1, g2)
group_header **g1, **g2;
{
    return strcmp((*g1)->group_name, (*g2)->group_name);
}


void
sort_groups()
{
    register group_header *gh;

    Loop_Groups_Header(gh)
	sorted_groups[l_g_index] = gh;

    quicksort(sorted_groups, master.number_of_groups, group_header *, sort_gh);

    s_g_first = 0;
    Loop_Groups_Sorted(gh)
	if (gh->group_name[0] != NUL) {
	    s_g_first = l_g_index;
	    break;
	}
}


group_header *lookup_no_alias(name)
char *name;
{
    register i, j, k, t;

    i = s_g_first;
    j = master.number_of_groups - 1;

    while (i <= j) {
	k = (i + j) / 2;

	if ( (t=strcmp(name, sorted_groups[k]->group_name)) > 0)
	    i = k+1;
	else
	if (t < 0)
	    j = k-1;
	else
	    return sorted_groups[k];
    }

    return NULL;
}

group_header *lookup(name)
char *name;
{
    register group_header *gh;
    group_header *gh_na;
    register int32 n, x;

    gh = lookup_no_alias(name);
    if (gh == NULL || (gh->master_flag & M_ALIASED) == 0) return gh;

    gh_na = gh;
    x = 16;
    do {
	if (--x == 0) {
	    log_entry('R', "Possible alias loop: %s", name);
	    return gh_na;
	}
	n = (int32)gh->data_write_offset;
	/* if alias info is unreliable, return original group
	   which will be ignored anyway */
	if (n < 0 || n >= master.number_of_groups) {
	    log_entry('R', "Bad aliasing of %s -> %d", gh->group_name, n);
	    return gh_na;
	}
	gh = ACTIVE_GROUP(n);
    } while (gh->master_flag & M_ALIASED);

    return gh;
}

int
art_collected(gh, art_num)
group_header *gh;
article_number art_num;
{
    return gh->first_db_article <= art_num && gh->last_db_article >= art_num;
}


static void
readactfile()
{
    char actline[512];
    int count = 0;
    int	i;
    FILE *actfp;
    stlist_t *sthead, *stp;

    if (actlist != NULL)
	return;

#ifdef NNTP 
    if (use_nntp) {
        tk_nntp_start();

        actfp = nntp_fopen_list("LIST");
    } else
#endif
	actfp = fopen(relative(news_lib_directory, "active"), "r");

#ifdef NNTP 
  if (actfp == NULL) {
    if (use_nntp)
      nn_exitmsg(1, "could not fetch active file from the NNTP server\n");
    else
      nn_exitmsg(1, "could not fetch active file: %s\n", 
					relative(news_lib_directory, "active"));
  }
#else
  if (actfp == NULL) {
    nn_exitmsg(1, "could not fetch active file: %s\n", 
					relative(news_lib_directory, "active"));
  }
#endif


    /*
     * Snarf all of active up in first pass.  This gives us a count
     * of the groups we can use for internal tables.
     */
    sthead = NULL;
    stp = NULL;
#ifdef NNTP
    while ( use_nntp  ?  nntp_fgets(actline, sizeof actline)
		      :  fgets(actline, sizeof actline, actfp) )
#else
    while ( fgets(actline, sizeof actline, actfp) )
#endif
    {
	stlist_t *stnew = (stlist_t *)strkeep(actline,sizeof(stlisthdr_t),POOL_ACT);
	if (stnew == NULL) {
	    nn_exitmsg(1, "out of mem for active file\n");
	}
	if (sthead != NULL) {
	    stp->n.next = stnew;
	    stp = stnew;
	}
	else {
	    sthead = stnew;
	    stp = sthead;
	}
	count++;
#ifdef NNTP
	if (nntp_progress && count % nntp_progress == 0) {
	  tk("nntp_lmsg {Reading active:%5d} 0",count);
	}
#endif /* NNTP */
    }
    stp->n.next = NULL;

    if (!use_nntp)
	(void) fclose(actfp);
    tk("nntp_kmsg");

    actlist = (char **)mem_obj(sizeof(char *), count + 1);
    if (actlist == NULL) {
	nn_exitmsg(1, "out of memory for active list\n");
    }
    grplist = (char **)mem_obj(sizeof(char *), count + 1);
    if (grplist == NULL) {
	nn_exitmsg(1, "out of memory for group list\n");
    }

    /*
     * Second pass (in core): Put active lines and group names into
     * string arrays.
     */
    for (i = 0, stp = sthead; stp && i < count; stp = stp->n.next, i++) {
	char *p = strchr(stp->str, ' ');

	if (p != NULL)
	    *p = NUL;
	grplist[i] = strkeep(stp->str, 0, POOL_GRP);
	actlist[i] = stp->str;
	if (p != NULL)
	    *p = ' ';
    }
    actlist[count] = NULL;
    grplist[count] = NULL;

    /* init the master struct */
    clearobj(&master, master, 1);
    master.number_of_groups = count;

}

static void
readtimfile()
{
    char timline[512];
    FILE *timfp;
    int hsize;
    int count = 0;

    if (timtbl != NULL)
	return;
    hsize = master.number_of_groups | 0x1ff;
    timtbl = hashcreate(hsize, (unsigned (*)())NULL);
    if (timtbl == NULL) {
	nn_exitmsg(1, "can't create time hash\n");
    }

#ifdef NNTP
    if (use_nntp) {
        tk_nntp_start();

	timfp = nntp_fopen_list("LIST active.times");
    } else
#endif
	timfp = fopen(relative(news_lib_directory, "active.times"), "r");

    if (timfp == NULL)
	return;		/* no great shakes if its missing */

#ifdef NNTP
    while ( use_nntp  ?  nntp_fgets(timline, sizeof timline)
		      :  fgets(timline, sizeof timline, timfp) )
#else
    while ( fgets(timline, sizeof timline, timfp) )
#endif
    {
	char *line = strkeep(timline, 0, POOL_TIM);
	char *p = strchr(line, ' ');

	if (p == NULL)
	    continue;
	*p++ = NUL;
	if (!hashstore(timtbl, line, p)) {
	    nn_exitmsg(1, "nn: time hashstore failed\n");
	}
	count++;
#ifdef NNTP
	if (nntp_progress && count % nntp_progress == 0) {
	  tk("nntp_lmsg {Reading active.times:%5d} 0",count);
	}
#endif /* NNTP */
    }

    if (!use_nntp)
	(void) fclose(timfp);
  tk("nntp_kmsg");

}

#ifdef	CACHE_PURPOSE
static int
purpcmp(p0, p1)
struct purp_list *p0, *p1;
{
    return strcmp(p0->group_name, p1->group_name);
}

/*
 * Open purpose file and cache sorted list of group purposes
 */
static void
cache_purpose()
{
    char buf[512];
    register char *p;
    FILE *fp;
    int i, ngrp;
    stlist_t *sthead, *stp, *stnew;
    struct purp_list *plp;

    if ((fp = open_purpose_file()) == NULL
	|| (ngrp = master.number_of_groups) == 0
	|| purp_list != NULL)
    {
	return;
    }

    ngrp = 0;
    sthead = NULL;
    stp = NULL;
    while (fgets(buf, sizeof(buf), fp) != NULL) {
	if ((p = strchr(buf, NL_)) != NULL) {
		*p = NUL;
	}
	stnew = (stlist_t *)strkeep(buf, sizeof(stlisthdr_t), POOL_PUR);
	if (stnew == NULL) {
	    /* tough cookies.  we'll just do without. */
	    pfree(POOL_PUR);
	    return;
	}
	if (sthead != NULL) {
	    stp->n.next = stnew;
	    stp = stnew;
	}
	else {
	    sthead = stnew;
	    stp = sthead;
	}
	stnew->n.next = NULL;
	ngrp++;
    }

    purp_list = (struct purp_list *)mem_obj(sizeof(struct purp_list), ngrp + 1);
    if (purp_list == NULL) {
	pfree(POOL_PUR);
	return;
    }

    for (i = 0, plp = purp_list, stp = sthead; stp; stp = stp->n.next) {
	p = stp->str;
	while (!isspace(*p)) {	/* skip newsgroup name */
	    if (*++p == NUL)
		goto next;
	}
	*p++ = NUL;

	while (isspace(*p)) {	/* skip to group description */
	    if (*++p == NUL)
		goto next;
	}
	plp->group_name = stp->str;
	plp->purpose = p;
	plp++;
	i++;
next:
	continue;
    }
    plp->group_name = NULL;
    plp->purpose = NULL;
    purp_cnt = i;

    quicksort(purp_list, purp_cnt, struct purp_list, purpcmp);

}

char *
purp_lookup(group)
char *group;
{
    register i, j, k, t;

    i = 0;
    j = purp_cnt - 1;

    while (i <= j) {
	k = (i + j) / 2;

	if ( (t=strcmp(group, purp_list[k].group_name)) > 0)
	    i = k+1;
	else
	if (t < 0)
	    j = k-1;
	else
	    return purp_list[k].purpose;
    }

    return "";
}
#endif	/* CACHE_PURPOSE */




/*
 * initialise *gh; this is much cheaper than calling db_read_group.
 */
static void
db_init_group(gh, num)
register group_header *gh;
int	num;
{
	register char	*line  = 0;

	/* tidy up the struct */
	clearobj(gh, struct group_header, 1);

	gh->group_num = num;
	if (gh->group_name == NULL) {
	    gh->group_name = grplist[num];
	    if (gh->group_name == NULL) {  
		nn_exitmsg(1, "can't map group %d to name\n", num);
	    }
	    gh->group_name_length = strlen(gh->group_name);
	}
	gh->master_flag = M_VALID;
	/* control.newgrp, etc are control groups */
	if (strncmp(gh->group_name, "control", 7) == 0)
		gh->master_flag |= M_CONTROL;
	/* these next two are subtle and we need to lie below */
	/* gh->first_db_article = 0;*/		/* lowest # in ov. data */
	/* gh->last_db_article = 0; */		/* highest # in ov. data */
	gh->first_a_article = 1; 		/* lowest # in active */
	/* gh->last_a_article = 0; */		/* highest number in active */
	/* gh->index_write_offset = 0; */	/* dunno */
	/* gh->data_write_offset = 0; */	/* dunno */

	/* set the creation time */
	gh->creation_time = 1;                  /* group creation date (~epoch) */
	if (timtbl)
		line = hashfetch(timtbl, gh->group_name);
	if (line != NULL) {
		gh->creation_time = atol(line);
	}

	line = actlist[num];
	if (line != NULL) {
		register char	*p = strchr(line, ' ');

		if (p == NULL)
			return;
		p++;
		gh->last_a_article = atol(p);
		p = strchr(p, ' ');
		if (p == NULL)
			return;
		p++;
		gh->first_a_article = atol(p);
	}
	gh->first_db_article = gh->first_a_article; /* lowest # in ov. data */
	gh->last_db_article = gh->last_a_article; /* highest # in ov. data */
	if (newsrc_only) {
	    gh->group_flag |= G_UNKNOWN;
	}
}

void
reloadactfile()
{
    char actline[512];
    int count = 0;
    int	tmp, alast, afirst;
    FILE *actfp;
    group_header *gh;

    if (newsrc_only) {
	reloadactnewsrc();
    } else {
#ifdef NNTP 
	if (use_nntp) {
	  tk_nntp_start();

	    actfp = nntp_fopen_list("LIST");
	} else
#endif
	    actfp = fopen(relative(news_lib_directory, "active"), "r");

	if (actfp == NULL) {
	    nn_exitmsg(1, "could not fetch active file\n");
	}

	/*
	 * Snarf all of active up in first pass.  This gives us a count
	 * of the groups we can use for internal tables.
	 */

#ifdef NNTP
	while ( use_nntp  ?  nntp_fgets(actline, sizeof actline)
		:  fgets(actline, sizeof actline, actfp) )
#else
	while ( fgets(actline, sizeof actline, actfp) )
#endif
	{
	    char *p = strchr(actline, ' ');
	    if (p == NULL)
		continue;
	    else {
		*p++ = NUL;
	    }
	    alast = atol(p++);
	    p = strchr(p, ' ');
	    if (p == NULL)
		continue;
	    afirst = atol(++p);
		    gh = lookup(actline);
	    if (gh) {
		tmp = gh->last_a_article;
		gh->last_db_article = gh->last_a_article = alast;
		gh->first_db_article = gh->first_a_article = afirst;
		gh->unread_count += (alast - tmp);
		gh->group_flag |= G_COUNTED;
		if (gh->newsrc_orig  && (gh->newsrc_orig != gh->newsrc_line)) {
		    free(gh->newsrc_orig);
		    gh->newsrc_orig =  copy_str(gh->newsrc_line);
		}
	    }
	    count++;
#ifdef NNTP
	    if (nntp_progress && count % nntp_progress == 0) {
		tk("nntp_lmsg {Reading active:%5d} 0",count);
	    }
#endif /* NNTP */
	}

	if (!use_nntp)
	    (void) fclose(actfp);

	tk("nntp_kmsg");
    }
}

static void
reloadactnewsrc()
{
  char	actline[512];
  FILE *actfp;
  char *newsrc_file;
  group_header *gh;
  
  newsrc_file = home_relative(newsrc_name);
  actfp = open_file(newsrc_file, OPEN_READ);

  if (actfp == NULL) {
    nn_exitmsg(1, "could not fetch newsrc file\n");
  }

  while ( fgets(actline, sizeof actline, actfp) )
  {
    char	*p;

    p = strchr(actline, ':');
    if (!p) {
	p = strchr(actline, '!');
    }
    if (!p)
	continue;

    *p = '\0';

    gh = lookup(actline);
    if (gh) {
      gh->group_flag |= G_UNKNOWN;
      gh->group_flag &= ~G_RDAHEAD;
      gh->master_flag |= M_BLOCKED;
      gh->first_db_article = gh->first_a_article = 1;
      gh->last_db_article = gh->last_a_article = ART_MAX_DUMMY;
      gh->unread_count = 1;
    }

  }

  fclose(actfp);

}

static void
readactnewsrc()
{
  register int	count = 0;
  char	actline[512];
  FILE *actfp;
  char *newsrc_file;
  int i;
  stlist_t *sthead, *stp;

  if (actlist != NULL)
    return;

  newsrc_file = home_relative(newsrc_name);
  actfp = open_file(newsrc_file, OPEN_READ);

  if (actfp == NULL) {
    nn_exitmsg(1, "could not fetch newsrc file\n");
  }

  /*
   * Snarf all of data up in first pass.  This gives us a count
   * of the groups we can use for internal tables.
   */
  sthead = NULL;
  stp = NULL;
  while ( fgets(actline, sizeof actline, actfp) )
  {
    char	line[100];
    char	*p;
    stlist_t 	*stnew;

    p = strchr(actline, ':');
    if (!p) {
	p = strchr(actline, '!');
    }
    if (!p)
	continue;

    *p = '\0';
    sprintf(line, "%s %d 000001 y", actline, ART_MAX_DUMMY);

    stnew = (stlist_t *)strkeep(line, sizeof(stlisthdr_t), POOL_ACT);
    if (stnew == NULL) {
      nn_exitmsg(1, "out of memory for newsrc file\n");
    }
    if (sthead != NULL) {
      stp->n.next = stnew;
      stp = stnew;
    }
    else {
      sthead = stnew;
      stp = sthead;
    }
    count++;
  }
  stp->n.next = NULL;

  fclose(actfp);

#ifdef NNTP 
  if (use_nntp) {
    actfp = nntp_fopen();
    if (actfp == NULL) {
	nn_exitmsg(1, "could not open nntp connection\n");
    }
  }
#endif


  actlist = (char **)mem_obj(sizeof(char *), count + 1);
  if (actlist == NULL) {
    nn_exitmsg(1, "out of memory for active list\n");
  }
  grplist = (char **)mem_obj(sizeof(char *), count + 1);
  if (grplist == NULL) {
    nn_exitmsg(1, "out of memory for group list\n");
  }

  /*
   * Second pass (in core): Put active lines and group names into
   * string arrays.
   */
  for (i = 0, stp = sthead; stp && i < count; stp = stp->n.next, i++) {
    char *p = strchr(stp->str, ' ');
    
    if (p != NULL)
      *p = NUL;
    grplist[i] = strkeep(stp->str, 0, POOL_GRP);
    actlist[i] = stp->str;
    if (p != NULL)
      *p = ' ';
  }
  actlist[count] = NULL;
  grplist[count] = NULL;

  /* init the master struct */
  clearobj(&master, master, 1);
  master.number_of_groups = count;

}

/*
 * slurp up the overview data for this group into *gh.
 * this costs a file open and so should not be done frivolously.
 */
void
db_read_group(gh, first, last)
register group_header *gh;
article_number first, last;
{
	register struct novart *artp, *lastartp = NULL;

	/* db_init_group(gh, group_num?? );  already done early at init time */

	if (ngovp != NULL)
	  	novclose(ngovp);		/* trash last group's data */

#ifdef NNTP
	if (use_nntp) {
	        tk_nntp_start();
		ngovp = nntp_get_overview(gh, first, last);
		if (newsrc_only) {
		  /* adjust first and last using info from GROUP
		     command, needed for -A mode                  */
		  if (gh->first_a_article > first)
		    first = gh->first_a_article;
		  if (gh->last_a_article < last)
		    last = gh->last_a_article;
		}
	} else
#endif
		ngovp = novopen(gh->group_name);

	if (ngovp == NULL) {
		printf("no overview data for group `%s'\n", gh->group_name);
		return;
	}
	allarts = novall(ngovp, first, last);
	if (allarts == NULL) {
		/*
		   printf("overview data inaccessible for group `%s'\n",
			   gh->group_name);
		 */
		return;
	}
	if (!use_nntp) {

	    if (first == gh->first_a_article)
		gh->first_db_article = atol(allarts->a_num); /* lowest # */

	    if (last == gh->last_a_article) {

		/* UGH! */
		for (artp = allarts; artp != NULL; artp = artp->a_nxtnum)
		  if (artp->a_num[0] != '0' || artp->a_num[1] != 0)
		    lastartp = artp;

		gh->last_db_article  = atol(lastartp->a_num); /* highest # */

	    }
	}

	novthread(ngovp);

      /* Handle Digest flag */
	if (gh->first_db_article < 0)
          gh->first_db_article = -gh->first_db_article;
	if (gh->last_db_article < 0)
          gh->last_db_article = -gh->last_db_article;
	if (newsrc_only) {
	    gh->first_a_article = gh->first_db_article;
	    gh->last_a_article = gh->last_db_article;
	}
}

/*
    Display the thread structure that a particular article
    belongs to
*/
thread_flag(a)
article_header *a;
{
  if (a->thread_art)
    tk("thread_nmark %d %d",a->thread_art->a_x,a->thread_art->a_y);	
}

thread_display_node(art)
struct novart *art;
{
  struct novart *artn;

  if (art->a_ptr) {
    tk_set_var("thread_text",art->a_ptr->sender);
  } else {
    if (art->a_num) 
      tk_set_var("thread_text"," - ");
    else	
      tk_set_var("thread_text","   ");
  }
  tk("thread_node %d %d %d %d",
     art->a_x, art->a_y,
     art->a_ptr ? (art->a_ptr->attr == A_SELECT || art->a_ptr->attr == A_AUTO_SELECT) : -1,
     art->a_ptr ? art->a_ptr->a_number: -1);
  if (art->a_child1) {
    artn = (struct novart *) hashfetch(ngovp->g_msgids,art->a_child1);
    if (artn)
      thread_display_node(artn);
    else
      printf("Null hash:%s\n",art->a_child1);
  }
  if (art->a_sibling) {
    artn = (struct novart *) hashfetch(ngovp->g_msgids,art->a_sibling);
    if (artn)
      thread_display_node(artn);
    else
      printf("Null hash:%s\n",art->a_sibling);
  }
}

thread_display(a)
article_header *a;
{
  struct novart *art, *nart;

  tk("thread_clear");
  if (!a->thread_head_ptr) {
    /*DEBUG*/
    printf("\n%s\n%s\n",a->sender,a->subject);
    return;
  }
  art = a->thread_head_ptr->thread_art;
  while (art->a_parent) {               /* find the root node */
    while (art->a_parent) {
      nart = (struct novart *) hashfetch(ngovp->g_msgids,art->a_parent);
      if (nart) {
	art = nart;
      } else {
	goto fin;
      }
    }
    while (art->a_sibling) {
      nart = (struct novart *) hashfetch(ngovp->g_msgids,art->a_sibling);
      if (nart) {
	art = nart;
      } else {
	goto fin;
      }
      if (art->a_parent)
	break;
    }
  }
fin:
  thread_display_node(art);
}

/*
     traverse thread trees, linking novart records to corresponding
     article header records and setting up info
*/
int cur_thread_cnt, cur_thread_line;
article_header *cur_head;

link_art(art,n)
struct novart *art; int n;
{
  art->a_x = n;
  art->a_y = cur_thread_line;
#ifdef THREAD_DEBUG
  printf("%*.*s%s:%d,%d %s: %s\n",n,n,"            ",art->a_num,art->a_x,art->a_y,art->a_msgid, art->a_subj);
  printf("%*.*s%s\n",n,n,"            ",art->a_refs);
  printf("%*.*s%s : %s : %s\n",n,n,"            ",art->a_parent,art->a_sibling,art->a_child1);
  printf("%*.*s%x : %x : %x\n",n,n,"            ",art->a_parent,art->a_sibling,art->a_child1);
  fflush(stdout);
#endif
  if (art->a_ptr) {
    cur_thread_cnt++;
    art->a_ptr->thread_cnt = cur_thread_cnt;
    art->a_ptr->thread_art = art;
    art->a_ptr->replies = n;
    if (!cur_head) {			/* the real head is a dummy */
      cur_head = art->a_ptr;
    }
    art->a_ptr->thread_head_ptr = cur_head;	
  } else {
#ifdef THREAD_DEBUG
    printf("No HEAD\n");
#endif
  }
#ifdef THREAD_DEBUG
  printf("%*.*s cnt=%d lev=%d head=%d\n\n",  n,n,"            ",cur_thread_cnt,n,cur_head);
#endif
}

other_thread(key,n)
int n; char *key;
{
  struct novart *art;
  art = (struct novart *) hashfetch(ngovp->g_msgids,key);
  if (!art) {
    printf ("THREAD: id not in hash: %s\n",key);
    return;
  }
  link_art(art,n);
  if (art->a_child1)
    other_thread(art->a_child1,n+1);
  if (art->a_sibling) {
    cur_thread_line++;
    other_thread(art->a_sibling,n);
  }
}

root_thread(key, data, hook)
char *key, *data, *hook;
{
  int n = 0;
  struct novart *art = (struct novart *)data;
  cur_thread_cnt = 0;
  cur_thread_line = 0;
  cur_head = art->a_ptr;

  link_art(art,n);
  if (art->a_child1) 
    other_thread(art->a_child1,1);
}

link_threads()
{
  if (ngovp && ngovp->g_roots) {
    hashwalk(ngovp->g_roots, root_thread, NULL);
  } else {
#ifdef THREAD_DEBUG
    printf("No ROOT\n");
#endif
  }
}

/*
 * fill in db_hdr and db_data from the overview data for the next
 * article in this group.  encodes header fields.
 */
off_t
db_read_art(artc)
struct novart **artc;
{
	register data_header *dhp = &db_hdr;
	register data_dynamic_data *ddp = &db_data;
	register struct novart *artp;
	int	recnt = 0;

	if (ngovp == NULL || ngovp->g_first == NULL)
		return 0;
	artp = novnext(ngovp);
	*artc = artp;

	if (artp == NULL)
		return 0;			/* group exhausted */
	if (artp->a_num == NULL)
	         return 0;                      /* end of real nodes */

	dhp->dh_number = atol(artp->a_num);
	/* printf("article #%ld\n", dhp->dh_number); */	/* DEBUG */
	dhp->dh_date = pack_date(artp->a_date);	/* "encoded Date: filed" */
	dhp->dh_hpos = 0;			/* 1st hdr byte */
	dhp->dh_lpos = 1L << 30;		/* last article byte */
	dhp->dh_fpos = 0;			/* 1st article text byte */
	dhp->dh_lines = -1;			/* -1 == "unknown" */
	if (isascii(artp->a_lines[0]) && isdigit(artp->a_lines[0]))
		dhp->dh_lines = atoi(artp->a_lines);
	dhp->dh_replies = 0;			/* # of References: */
	if (artp->a_refs != NULL) {
		register char	*p;

		for (p = artp->a_refs; *p != '\0'; p++)
			if (*p == '<')
				dhp->dh_replies++;
	}

	db_fixup_cross_postings(dhp, ddp, artp);

	if (dhp->dh_number < 0) {
		current_digest_article = dhp->dh_number = -dhp->dh_number;
		ddp->dh_type = DH_DIGEST_HEADER;
	} else if (dhp->dh_number == 0) {
#ifdef DO_NOV_DIGEST
		char *cp;
		if (artp->a_bytes && (cp = strchr(artp->a_bytes, ':'))) {
			dhp->dh_hpos = atol(++cp);
			if ((cp = strchr(cp, ':')) != NULL) {
				dhp->dh_fpos = atol(++cp);
				if ((cp = strchr(cp, ':')) != NULL) {
					dhp->dh_lpos = atol(++cp);
				}
			}
		}
#endif
		dhp->dh_number = current_digest_article;
		ddp->dh_type = DH_SUB_DIGEST;
	} else {
		current_digest_article = 0;
		ddp->dh_type = DH_NORMAL;
	}

	dhp->dh_sender_length =  pack_name(ddp->dh_sender, artp->a_from, NAME_LENGTH);
	dhp->dh_subject_length = pack_subject(ddp->dh_subject, artp->a_subj, &recnt, DBUF_SIZE);
	strncpy(ddp->dh_from, artp->a_from, DBUF_SIZE); 
	if (recnt)		
		dhp->dh_replies |= 0x80;

	db_read_counter++;
	return 1;
}

static void
db_fixup_cross_postings(dhp, ddp, artp)
data_header *dhp;
data_dynamic_data *ddp;
struct novart *artp;
{
    char *curg, *tmp;
    int numgrps = 0, numgrps_all = 0;

    dhp->dh_cross_postings = 0; /* assume none as default until we can show 
				   otherwise */

    /* If no "other" header lines are in NOV database, we're out of luck,
       can only assume no crosspostings, so return. */
    if ((artp->a_others) == NULL) return;

    /* Scan until we find a Xref: header line. */
    for (curg = artp->a_others; ; ++curg) {
	if (strncmp("Xref: ", curg, 6) == 0 ||
	    strncmp("xref: ", curg, 6) == 0)
	{
	    break;
	}
	curg = strchr(curg, '\t'); /* Not this header, skip to the next */
	if (curg == NULL) return;
    }

    curg += 6;			   /* Skip over "Xref: " */

    while (*curg == ' ') ++curg;   /* Skip to the hostname field after Xref: */

    /* Skip over the hostname to the space following hostname */
    if ( (curg = strchr(curg, ' ')) == NULL ) {
	return;			   /* header is malformed, punt. */
    }
    /*
     * Start reading the entries one at a time.  Each entry is of the
     * form "newsgroup:number", and entries are separated by spaces.
     * Algorithm loosely based on the orignal one in collect.c for
     * setting up the crosspost information.
     */
    while (*curg == ' ' && numgrps < DBUF_SIZE) {
	group_header *gh;

	while (*curg == ' ') ++curg;	/* Skip spaces to the next entry */

	/* Zap colon at end of current entry. */
	for (tmp = curg ; ; ++tmp) {
	    if (*tmp == ':' || *tmp == '\0' || *tmp == '\t')
		break;
	}
	if (*tmp != ':') break;		/* malformed entry, punt. */
	*tmp = '\0';

	/* Find gh struct for the group. */
	if ( (gh = lookup(curg)) != NULL) {
	    /* and add group number to the crosspost list. */
	    ddp->dh_cross[numgrps++] = gh->group_num;
	} else {
	    ddp->dh_cross[numgrps++] = -1;
	}	  
	curg = tmp + 1;
	while (isdigit(*curg)) ++curg; /* Skip over the article number */
    }
    if (numgrps > 1) {
	/* Note: if # of groups is only 1, we leave dh_cross_postings 
	   at its original value of zero. */
	dhp->dh_cross_postings = numgrps;

    }
    return;
}


/*
 * pmalloc()/pfree():
 *	A scheme to avoid malloc()/free() overhead; handles memory in
 *	STRCHUNK increments (deliberately same as STR_THUNK_SIZE).
 *	Unlike mark_str()/alloc_str()/release_str(), pfree()'d memory
 *	returns to the malloc() pool, which is arguably more social.
 *	More important, the alloc_str() family assumes only one active
 *	use at a time; interleaving uses or a misplaced release_str() has
 *	the potential to leak memory or corrupt the current_str_t pool.
 *
 *	Should probably be moved to global.h.
 */
#define	STRCHUNK	((1<<14) - 32)	/* leave room for malloc header */

typedef struct stpool {
    stlist_t *sthead;
    char *pool;
    int pfree;
    int pslop;
} stpool_t;

static stpool_t stpool[POOL_MAX+1];

static char *
pmalloc(size, poolid)
    int size, poolid;
{
    register stpool_t *pp;
    register stlist_t *stnew;
    register char *ret;
    register int alignsize;

    if (poolid < 0 || poolid > POOL_MAX)
	return NULL;
 
    /* this is required for worst case alignment */
    alignsize = sizeof(char *) > sizeof(long) ? sizeof(char *) : sizeof(long);
    size = (size + (alignsize - 1)) & ~(alignsize - 1);

    pp = &stpool[poolid];
    if (size <= pp->pfree) {
	/* Usually short; fits into current chunk */
	ret = pp->pool;
    }
    else if (size <= STRCHUNK) {
	/* Sometimes chunk is exhausted; chain new one to pool */
	stnew = (stlist_t *)malloc(sizeof(stlisthdr_t) + STRCHUNK);
	if (stnew == NULL)
	    return NULL;
	pp->pslop += pp->pfree;
	stnew->n.next = pp->sthead;
	pp->sthead = stnew;
	pp->pfree = STRCHUNK;
	ret = stnew->str;
    }
    else {
	/*
	 * Last resort: allocate oversize chunk and chain to pool
	 * behind current chunk.
	 */
	stnew = (stlist_t *)malloc(sizeof(stlisthdr_t) + size);
	if (stnew == NULL)
	    return NULL;
	if (pp->sthead != NULL) {
	    stnew->n.next = pp->sthead->n.next;
	    pp->sthead->n.next = stnew;
	}
	else {
	    stnew->n.next = NULL;
	    pp->sthead = stnew;
	}
	return stnew->str;
	/*NOTREACHED*/
    }

    pp->pool = ret + size;
    pp->pfree -= size;
    return ret;
}

static void
pfree(poolid)
    int poolid;
{
    register stpool_t *pp;
    register stlist_t *stp, *stnext;

    if (poolid < 0 || poolid > POOL_MAX)
	return;

    pp = &stpool[poolid];
    stp = pp->sthead;
    for (stp = pp->sthead; stp; stp = stnext) {
	stnext = stp->n.next;
	free(stp);
	stp = stnext;
    }
    pp->sthead = NULL;
    pp->pool = NULL;
    pp->pfree = 0;
    pp->pslop = 0;
}


/*
 * strkeep()
 *	Save a string, allowing space for a header.
 */
char *
strkeep(s, hdr, poolid)
char *s;
int hdr;
int poolid;
{
    register int size;
    register char *ret;

    size = hdr + strlen(s) + 1;
    if (poolid == POOL_TMP) {
	ret = alloc_str(size);
    }
    else {
	ret = pmalloc(size, poolid);
    }

    if (ret)
#ifdef NO_MEMCPY
	bcopy(s, ret+hdr, size-hdr);
#else
	memcpy(ret+hdr, s, size-hdr);
#endif /* NO_MEMCPY */

    return ret;
}
