/* $Id: display.c,v 1.6 2001/05/24 14:53:58 malekith Exp $ */

#include "serv.h"
#include <yw/display.h>
#include <yw/buffer.h>
#include <yw/map.h>
#include "win.h"

struct Display_struct {
	YwBuffer *buf;
	Conn *conn;
	
	int caps;
	int width;
	int height;

	int last_refresh;
};

static YwBufferCallback display_cb;

static int do_char_width(void *userdata, uint32_t ch)
{
	int r;
	
	(void)userdata;
	r = char_width(ch);
	if (r == -2)		/* XXX: hmm.. this is for ' ' */
		r = 1;
	return r;	
}

/* cache_widths() ommited, everything should be cached at this
 * point, otherwise we will get deadlock... */

static void remote_goto(void *userdata, int x, int y)
{
	Conn *c = userdata;

	printk(YL_DEBUG "goto(%d,%d)", x, y);
	yw_conn_queue_and_free(c->conn, yw_packet_make(yw_void_call_packet,
		"k", "goto",
		"i", x,
		"i", y,
		YW_END));
}

static void remote_puts(void *userdata, const YwString *str)
{
	Conn *c = userdata;

	printk(YL_DEBUG "puts(%d chars)", str->len);
	yw_conn_queue_and_free(c->conn, yw_packet_make(yw_void_call_packet,
		"k", "print",
		"s", str,
		YW_END));
}

static void remote_attr(void *userdata, YwBufAttr attr, int val)
{
	Conn *c = userdata;

	yw_conn_queue_and_free(c->conn, yw_packet_make(yw_void_call_packet,
		"k", "attr",
		"k", yw_bufattr_name(attr),
		"i", val,
		YW_END));
}

static void remote_flush(void *userdata)
{
	Conn *c = userdata;

	yw_conn_queue_and_free(c->conn, yw_packet_make(yw_void_call_packet,
		"k", "flush",
		YW_END));
	yw_conn_queue_flush(c->conn);
}

void display_present(int is_it);

static void set_buffer(Conn *c)
{
	Display *d;
	d = c->display;
	
	yw_buffer_free(d->buf);
	
	d->buf = yw_buffer_new(&display_cb, c, d->width, d->height, 1);
}

void init_display(Conn *c)
{
	if (c->display)
		return;		/* already done */
		
	lock();
	/* first kill the other display */
	yw_assert(conn_display != c);
	if (conn_display) {
		display_present(0);
		kill_connection(conn_display);
	}
	
	conn_display = c;

	c->display = YW_NEW_0(Display);
	
	c->display->width = 80;
	c->display->height = 24;
	c->display->conn = c;
	c->display->buf = NULL;
	
	c->type = display;

	set_buffer(c);

	display_present(1);
	refresh_display();

	unlock();
}

void free_display(Conn *c)
{
	if (c->display == NULL)
		return;
	
	yw_buffer_free(c->display->buf);
	yw_free(c->display);

	if (c == conn_display)
		display_present(0);
	conn_display = NULL;
}

static Display *get_display(CallbackData *cd)
{
	if (cd->conn->type != display) {
		cd->err = "non-display";
		return NULL;
	}
	
	return cd->conn->display;
}

static void kbd(CallbackData *cd)
{
	Display *d = get_display(cd);
	YwPacket *pkt;
	int wm_key = 0;
	const char *kw;
	Window *target;
	
	if (d == NULL)
		return;
		
	lock();
	for (target = wm_wins; target; target = target->next_global)
		if (target->layer >= focus_layer)
			break;
			
	if (conn_wm) {
		pkt = yw_packet_make(yw_call_packet,
				"i" "wm_kbd", target ? target->wm_win_id : -1,
				YW_END);
		yw_packet_append_words(pkt, yw_packet_word(cd->pkt, 1), -1);
		
		yw_conn_send(conn_wm->conn, pkt);
		pkt = yw_conn_recv_reply(conn_wm->conn, -1);

		if (pkt &&
		    yw_word_fetch_keyword(yw_packet_word(pkt, 0), &kw) == 0 &&
		    strcmp(kw, "wm_kbd_accepted") == 0)
			yw_word_fetch_int(yw_packet_word(pkt, 1), &wm_key);
		yw_packet_free(pkt);
	}
	
	for (target = wm_wins; target; target = target->next_global)
		if (target->layer >= focus_layer)
			break;

	if (target && !wm_key) {
		pkt = yw_packet_make(yw_void_call_packet,
				"i" "kbd", target->win_id,
				YW_END);
		yw_packet_append_words(pkt, yw_packet_word(cd->pkt, 1), -1);
		send_packet(target->parent, pkt, PRI_USER);
	}
	unlock();
}

static void have_cap(CallbackData *cd)
{
	Display *d = get_display(cd);
	YwWord *w;
	const char *c;
	int v;
	int state = -1;
	static YwMapCache *cache;
	enum {
		cap_color = YW_DISPLAY_CAP_COLOR,
		cap_boldface = YW_DISPLAY_CAP_BOLDFACE,
		cap_underline = YW_DISPLAY_CAP_UNDERLINE,
		cap_high_bg = YW_DISPLAY_CAP_HIGH_BG,
		cap_mouse = YW_DISPLAY_CAP_MOUSE,
		cap_width,
		cap_height,
		cap_end,
		cap_unknown
	};
	YwMapEntry e[] = {
		{ "color", cap_color },
		{ "boldface", cap_boldface },
		{ "underline", cap_underline },
		{ "high_bg", cap_high_bg },
		{ "mouse", cap_mouse },
		{ "width", cap_width },
		{ "height", cap_height },
		{ "end", cap_end },
		{ NULL, cap_unknown }
	};

	if (d == NULL)
		return;
	
	if (yw_word_fetch_mapped_keyword(yw_packet_word(cd->pkt, 1),
					 e, &cache, &v))
		return;
	
	w = yw_packet_word(cd->pkt, 2);
	
	if (yw_word_fetch_keyword(w, &c) == 0) {
		if (strcmp(c, "yes") == 0)
			state = 1;
		else if (strcmp(c, "no") == 0)
			state = 0;
	}
			
	
	switch (v) {
	case cap_color:
	case cap_boldface:
	case cap_underline:
	case cap_high_bg:
	case cap_mouse:
		switch (state) {
		case -1:
			cd->err = "bad bool option value";
			break;
		case 1:
			d->caps |= v;
			break;
		case 0:
			d->caps &= ~v;
			break;
		default:
			yw_halt();
		}
		break;
	case cap_height:
		if (yw_word_fetch_int(w, &d->height))
			cd->err = "height not followed by int";
		break;
	case cap_width:
		if (yw_word_fetch_int(w, &d->width))
			cd->err = "width not followed by int";
		break;
	case cap_end:
		printk(YL_DEBUG0 "display: caps=0x%04x %dx%d",
			d->caps, d->width, d->height);
		set_buffer(cd->conn);
		break;
	case cap_unknown:
		cd->err = "unknown capability";
		break;
	default:
		yw_halt();
	}
}

static void draw_win(Display *d, Window *w)
{
	yw_buffer_putbuffer(w->wm_buffer,
			    w->wm_off_x, w->wm_off_y,
			    w->width, w->height,
			    w->client_area_buffer,
			    w->off_x, w->off_y);

	yw_buffer_putbuffer(d->buf,
			    w->x, w->y,
			    w->wm_width, w->wm_height,
			    w->wm_buffer,
			    0, 0);
}

static void do_refresh_display(Conn *c, Event *e)
{
	Window *w, *p;
	Display *d;
	
	yw_assert(c->display);
	d = c->display;
	
	if (d->last_refresh >= e->frame)
		return;
		
	yw_buffer_clr(d->buf);

	lock();
	d->last_refresh = get_frame();
	if (wm_wins) {
		/* draw in reverse order... */
		for (w = wm_wins; w->next_global; w = w->next_global)
			/* nothing */ ;
		while (w != wm_wins) {
			draw_win(d, w);
			for (p = wm_wins;; p = p->next_global)
				if (p->next_global == w)
					break;
			w = p;
		}
		draw_win(d, w);
	}
	unlock();

	yw_buffer_sync(d->buf);
}

void refresh_display()
{
	Event *e;
	
	lock();
	if (conn_display) {
		e = new_event();
		e->pri = PRI_DRAW;
		e->execute = do_refresh_display;
		post_event(e, conn_display);
	}
	unlock();
}

void install_display_callbacks()
{
	display_cb.char_width = do_char_width;
	display_cb.cache_widths = NULL;
	display_cb.remote_goto = remote_goto;
	display_cb.remote_puts = remote_puts;
	display_cb.remote_attr = remote_attr;
	display_cb.remote_flush = remote_flush;
	
	install_callback("have_cap", ARG_TAGS, have_cap, NULL);
	install_callback("kbd", 0, kbd, NULL);
}
